/********************************************************************** 
 * Copyright (c) 2006, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractProjectSensitiveService.java,v 1.7 2009/11/23 20:40:57 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPlatformRunnable;

/**
 * Abstract service class to be extended by services that require association
 * with a project at run-time (based on input properties on their service for
 * example by the client). Project-related state is stored here as well as
 * methods that operate on this state.
 * 
 * @see org.eclipse.hyades.automation.server.AbstractWorkspaceSensitiveService
 * @see org.eclipse.hyades.automation.server.AbstractRelaunchableService
 * @see org.eclipse.hyades.automation.server.AbstractService
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractProjectSensitiveService extends AbstractWorkspaceSensitiveService {

	/**
	 * Project that is specified by the project specifier
	 */
	protected IProject project;

	/**
	 * The project selector, if specified, identifies one to potentially many
	 * projects used for this test execution, the concrete service determines
	 * how to use the selector specified.
	 */
	protected String projectSpecifier;

	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractProjectSensitiveService() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {

		// Execute common execute behavior before service specific
		Object object = super.execute();

		// If base class decides to abort this execution early
		//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
		if (object != IPlatformRunnable.EXIT_OK) {
			return object;
		}

		// Retrieve project specifier, might be null or empty string
		this.projectSpecifier = this.getProperty("project", "");//$NON-NLS-1$

		// Project qualifying the workspace
		if (this.projectSpecifier != null && this.projectSpecifier.trim().length() > 0) {
			this.project = this.getProject(this.projectSpecifier);
		}

		// Return OK back to the service consumer
		//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
		return IPlatformRunnable.EXIT_OK;

	}

	/**
	 * Return the project given the project specifier
	 * 
	 * @param projectSpecifier
	 *            identifies the project this service is associated with
	 * @return the project identified by the project specifier
	 */
	protected IProject getProject(String projectSpecifier) {
		return this.workspace.getRoot().getProject(projectSpecifier);
	}

	/**
	 * Synchronized project with file system contents
	 * 
	 * @param project
	 *            the project to synchronized, to all available depths
	 */
	protected void refreshProject() {
		try {
			this.project.refreshLocal(IProject.DEPTH_INFINITE, null);
		} catch (CoreException e) {
			//
		}
	}

	/**
	 * Synchronize the specified project with the file system.
	 * 
	 * @param theProject
	 * @provisional
	 */
	protected void refreshProject(IProject theProject) {
		try {
			theProject.refreshLocal(IProject.DEPTH_INFINITE, null);
		} catch (CoreException e) {
			//
		}
	}
	
}
