/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractRelaunchableService.java,v 1.10 2009/11/23 20:40:58 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IPlatformRunnable;

/**
 * Abstract class extended by services that need relaunch behavior -- for some
 * services it is desirable and even required to relaunch the Eclipe instance,
 * changing the parameters given at the command line for starting up Eclipse
 * (for example, workspace specification, VM arguments).
 * 
 * For services requiring workspace replacement at run-time, particular
 * implementations can extend from the workspace sensitive service referenced
 * below that in turn extends this class.
 * 
 * @see org.eclipse.hyades.automation.server.AbstractWorkspaceSensitiveService
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractRelaunchableService extends AbstractService {

	/**
	 * Used for command data
	 */
	private static final String COMMAND_DATA = "-data"; //$NON-NLS-1$

	/**
	 * New line constant
	 */
	private static final String NEW_LINE = "\n"; //$NON-NLS-1$

	/**
	 * Specify commands via properties
	 */
	private static final String PROPERTY_COMMANDS = "eclipse.commands"; //$NON-NLS-1$

	/**
	 * Indicates if the instance had already been relaunched by this class
	 */
	private static final String PROPERTY_RELAUNCH = "tptp.automation.relaunch"; //$NON-NLS-1$

	/**
	 * Specify VM value
	 */
	private static final String PROPERTY_VM = "eclipse.vm"; //$NON-NLS-1$

	/**
	 * Specify VM arguments
	 */
	private static final String PROPERTY_VM_ARGUMENTS = "eclipse.vmargs"; //$NON-NLS-1$

	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractRelaunchableService() {
		super();
	}

	/**
	 * Copy a file from source to destination optimized using java.nio
	 * 
	 * @param source
	 *            the source file of contents to copy
	 * @param destination
	 *            the destination file to create and copy contents into
	 * @throws IOException
	 */
	protected void copy(File source, File destination) throws IOException {

		// Establish channels to source and destination test suite files
		FileChannel sourceChannel = new FileInputStream(source).getChannel();
		FileChannel destinationChannel = new FileOutputStream(destination).getChannel();

		// Copy using channel transfer to (for copy optimization potential)
		sourceChannel.transferTo(0, sourceChannel.size(), destinationChannel);

		// Close channels and release any other resources
		sourceChannel.close();
		destinationChannel.close();

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {
		//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
		return IPlatformRunnable.EXIT_OK;
	}

	/**
	 * Indicates if this instance of eclipse has been relaunched from another
	 * automation framework request
	 * 
	 * @return false if the instance is not a relaunched instance (it is a first
	 *         time launch)
	 */
	boolean isRelaunched() {
		return System.getProperty(PROPERTY_RELAUNCH) != null;
	}

	/**
	 * Construct a new command line with the workspace substituted to the
	 * preferred value, reconstructed from the parts such as VM property, VM
	 * arguments property, commands property, command data and then followed by
	 * command VM arguments.
	 * 
	 * @param workspace
	 *            the workspace to switch over to
	 * @param additionalArguments
	 *            additional arguments for this relaunch of eclipse
	 * 
	 * @return the command line that has been reconstructed
	 */
	protected String reconstructCommandLine(String workspace, String additionalArguments) {

		// Retrieve VM property
		String property = System.getProperty(PROPERTY_VM);
		Assert.isNotNull(property);

		// Allocate a string buffer and start construction
		StringBuffer result = new StringBuffer();
		result.append(property);
		result.append(NEW_LINE);

		// Handle eclipse VM arguments
		String arguments = System.getProperty(PROPERTY_VM_ARGUMENTS);
		if (arguments != null) {
			StringTokenizer tokenizer = new StringTokenizer(additionalArguments);
			while(tokenizer.hasMoreTokens()) {
				String token = tokenizer.nextToken();
				result.append(token);
				result.append(NEW_LINE);
			}
			result.append("-D").append(PROPERTY_RELAUNCH);//$NON-NLS-1$
			result.append(NEW_LINE);
			result.append(arguments);
		}

		// Deal with commands
		property = System.getProperty(PROPERTY_COMMANDS);
		if (property == null) {
			result.append(COMMAND_DATA);
			result.append(NEW_LINE);
			result.append(workspace);
			result.append(NEW_LINE);
		} else {

			// Now deal with command data
			int index = property.indexOf(COMMAND_DATA);

			// Handle whether -data already appears or not
			if (index != -1) {
				index += COMMAND_DATA.length() + 1;
				result.append(property.substring(0, index));
				result.append(workspace);
				result.append(property.substring(property.indexOf('\n', index)));//$NON-NLS-1$
			} else {
				result.append(COMMAND_DATA);
				result.append(NEW_LINE);
				result.append(workspace);
				result.append(NEW_LINE);
				result.append(property);
			}
		}

		return result.toString();

	}

}