/********************************************************************** 
 * Copyright (c) 2006, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AbstractWorkspaceSensitiveService.java,v 1.15 2009/11/23 20:40:58 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.osgi.service.datalocation.Location;

/**
 * Abstract service class to be extended by services that require workspace
 * replacement at run-time (based on input properties on their service for
 * example by the client). Services that need other relaunchable use cases for
 * their services might consider extending directly from the abstract
 * relaunchable service.
 * 
 * @see org.eclipse.hyades.automation.server.AbstractRelaunchableService
 * @see org.eclipse.hyades.automation.server.AbstractService
 * 
 * @author Scott E. Schneider
 */
public abstract class AbstractWorkspaceSensitiveService extends
		AbstractRelaunchableService {

	/**
	 * Specify exit data via properties
	 */
	private static final String PROPERTY_EXIT_DATA = "eclipse.exitdata"; //$NON-NLS-1$

	/**
	 * The workspace object corresponding to the workspace specifier
	 */
	protected IWorkspace workspace;

	/**
	 * Specifies the workspace to use for this service execution
	 */
	protected String workspaceSpecifier;

	/**
	 * Default constructor invokes up hierarchy for any initialization required
	 */
	protected AbstractWorkspaceSensitiveService() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {

		// Execute common execute behavior before service specific
		Object object = super.execute();

		// If base class decides to abort this execution early
		//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
		if (object != IPlatformRunnable.EXIT_OK) {
			return object;
		}

		// Retrieve workspace preference against current workspace
		this.workspaceSpecifier = this.getProperty("workspace", "");//$NON-NLS-1$

		// Retrieve VM arguments specified
		String vmArguments = this.getProperty("vmargs", "");//$NON-NLS-1$
		

		// Retrieve the assigned workspace location
		Location location = Platform.getInstanceLocation();

		/*
		 * If assigned location is not preferred, switch workspace and if VM
		 * arguments are specified to the service, restart as well
		 */
		/* Bugzilla 179367, Modifications to OutOfProcessStrategy were made 
		 * long ago, so that all the proper arguments are used when executing
		 * eclipse, including vmargs and workspace, etc.  So, the conditional code
		 * that was used here to re-start eclipse is no longer needed.
		 *  
		 * It was actually a correction in this conditional code
		 * that caused the bugzilla regression when eclipse was set to re-start.
		 * 
		 * Further info from Scott Schneider and Joe Toomey indicated that re-start of 
		 * eclipse using the in-process strategy does not work either.  So, the re-start code
		 * is being removed.
		 * 
		 */

		// Retrieve actual objects from locators
		this.workspace = ResourcesPlugin.getWorkspace();

		// Return OK back to the service consumer
		//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
		return IPlatformRunnable.EXIT_OK;

	}
	
	
	/**
	 * This method is provided to allow graceful workbench shutdown
	 * 
	 * @provisional
	 * @throws CoreException
	 */
	public void saveWorkspace() throws CoreException {
		if (workspace != null) {
			workspace.save(true, new NullProgressMonitor());
		}
	}

}
