/********************************************************************** 
 * Copyright (c) 2005, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: AutomationServer.java,v 1.16 2009/11/23 20:40:58 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.IPlatformRunnable;
import org.eclipse.hyades.automation.core.Service;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask;

/**
 * <p>The automation server provides an internal front-end to automatable published
 * services but is primarily used by the supplied automation client to service
 * the requests of external automation clients. This public class is an internal
 * class that should not be used directly by external clients; external clients
 * should use the automation client adaptation layer to automate services.</p>
 * 
 * <p>Internal clients should go through the client layer as well for consistency
 * and maintainability purposes -- use the Java automation client adapters from
 * plug-ins.</p>
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Paul Slauenwhite
 * @version November 23, 2009
 * @since   March 30, 2005
 * @see     org.eclipse.hyades.automation.client.adapters.java.AutomationClientAdapter
 */
public class AutomationServer implements Service.Discoverable,
		Service.Executable,
		org.eclipse.hyades.automation.core.IAutomationServer, IPlatformRunnable {

	/**
	 * The command that identifies the automation value
	 */
	private static final String AUTOMATION_COMMAND_IDENTIFIER = "tptp.automation.command";//$NON-NLS-1$

	/**
	 * The command that identifies the automation data
	 */
	private static final String AUTOMATION_DATA_IDENTIFIER = "tptp.automation.data";//$NON-NLS-1$

	/**
	 * The command that identifies the automation value
	 */
	private static final String AUTOMATION_SYNCHRONICITY_DIRECTIVE = "tptp.automation.synchronicity";//$NON-NLS-1$

	/**
	 * Exposed for use by Eclipse extension mechanism
	 */
	public AutomationServer() {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service.Discoverable#discover()
	 */
	public List discover() {
		return new LinkedList();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service.Discoverable#discover(java.lang.String)
	 */
	public boolean discover(String identifier) {
		return true; // currently not implemented
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service)
	 */
	public Object execute(Service service) {
		return this.execute(service, ProgressiveTask.Synchronicity.SYNCHRONOUS);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service.Executable#execute(org.eclipse.hyades.automation.core.Service,
	 *      org.eclipse.hyades.automation.core.utils.ProgressiveTask.Synchronicity)
	 */
	public Object execute(Service service,
			ProgressiveTask.Synchronicity synchronicity) {
		return service.execute(synchronicity);
	}

	/**
	 * Returns the automation data file name, trims any surrounding spaces and
	 * removes all double quotes contained.
	 * 
	 * @return the fully-qualified automation data identifier
	 */
	private String getAutomationDataFileName() {

		// Retrieve the automation data identifier
		StringBuffer fileName = new StringBuffer(System.getProperty(
				AutomationServer.AUTOMATION_DATA_IDENTIFIER).trim());

		// Remove any spaces or quote around the name
		for (int i = fileName.length() - 1; i >= 0; i--) {
			if (fileName.charAt(i) == '"') {
				fileName.deleteCharAt(i);
			}
		}

		// Render string from buffer
		return fileName.toString();

	}

	/**
	 * Persists the memento back, can be re-loaded on the client side to find
	 * the updated properties (that serve as return values for services),
	 * basically the memento comes in, the service executes, potentially reading
	 * and writing to these properties and then they are written back out for
	 * retrieval by the interested callers.
	 * 
	 * @param candidate
	 *            the memento originally passed in at the start of the
	 *            automation server operation
	 * @param memento
	 *            the memento to persist back
	 */
	private void persistMemento(Object candidate, Service.Memento memento) {

		// If the candidate is not null and is a memento, not required
		if (candidate != null && candidate instanceof Service.Memento) {
			return;
		}

		// Opens the automation data file and writes persistable memento
		try {

			// Construct return file name
			String returnFileName = this.getAutomationDataFileName();
			int length = returnFileName.length();
			returnFileName = returnFileName.substring(0, length - 4).concat(
					"-out").concat(returnFileName.substring(length - 4));//$NON-NLS-1$

			FileOutputStream fileOutput = new FileOutputStream(returnFileName);
			ObjectOutputStream objectOutput = new ObjectOutputStream(fileOutput);
			objectOutput.writeObject(memento);
			objectOutput.close();
			fileOutput.close();
		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	/**
	 * Restores the service memento, either returns the input candidate if it
	 * meets the requirements or restores one from the disk, this is the
	 * captured state of the service from the client (such as service specific
	 * properties)
	 * 
	 * @param candidate
	 *            a potential memento that is acquired by other means via
	 *            strategy
	 * 
	 * @return the opaque memento
	 */
	private Service.Memento restoreMemento(Object candidate) {

		// If the candidate is not null and is a memento then use it
		if (candidate != null && candidate instanceof Service.Memento) {
			return (Service.Memento) candidate;
		}

		// If not, load the memento from the disk given the filename
		try {
			FileInputStream fileInput = new FileInputStream(this
					.getAutomationDataFileName());
			ObjectInputStream objectInput = new ObjectInputStream(fileInput);
			Service.Memento memento = (Service.Memento) objectInput
					.readObject();
			objectInput.close();
			fileInput.close();
			return memento;
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}

		// In exceptional cases
		return null;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.core.runtime.IPlatformRunnable#run(java.lang.Object)
	 */
	public Object run(Object object) {

		/*
		 * Retrieve the automation command specified (others might include
		 * discover, information, etc)
		 */
		String command = System
				.getProperty(AutomationServer.AUTOMATION_COMMAND_IDENTIFIER);

		/*
		 * Set synchronicity based on passed in directive (after this, it is set
		 * to a type-safe enumeration value, if null, default to synchronous,
		 * the execute call then blocks until service completion
		 */
		ProgressiveTask.Synchronicity synchronicity = ProgressiveTask.Synchronicity.ASYNCHRONOUS
				.toString()
				.equalsIgnoreCase(
						System
								.getProperty(AutomationServer.AUTOMATION_SYNCHRONICITY_DIRECTIVE)) ? ProgressiveTask.Synchronicity.ASYNCHRONOUS
				: ProgressiveTask.Synchronicity.SYNCHRONOUS;

		// Only execute is supported for this release
		if (command.equalsIgnoreCase("execute")) {//$NON-NLS-1$

			/*
			 * Restores the memento object per the memento design pattern, the
			 * memento encapsulates non-transient state that can be transferred
			 * from one service to another, reviving the object across process
			 * boundaries for example.
			 */
			Service.Memento memento = this.restoreMemento(object);

			/*
			 * Create a server-side service proxy that will hook up the concrete
			 * implementation based on extensions found with given extension --
			 * execute once instantiated and return result
			 */
			Service service = new ServiceProxy(memento);

			/*
			 * Execute service and keep return value
			 */
			Object returnValue = this.execute(service, synchronicity);

			/*
			 * Persist memento for properties return state if appropriate
			 */
			this.persistMemento(object, service.createMemento());

			// Return return value from service
			return returnValue;

		}

		// Unsupported commands return null (currently this return goes nowhere)
		//NOTE: When this is changed to IApplication.EXIT_OK, we will need to change it everywhere otherwise this condition may return false since the objects are not the same:
		return IPlatformRunnable.EXIT_OK;

	}

}