/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: KeepAliveService.java,v 1.5 2008/03/20 18:49:53 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import java.rmi.RMISecurityManager;
import java.rmi.RemoteException;
import java.rmi.registry.LocateRegistry;
import java.rmi.registry.Registry;
import java.rmi.server.UnicastRemoteObject;
import java.security.Permission;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.hyades.automation.core.IAutomationServer;
import org.eclipse.hyades.automation.server.internal.resources.AutomationServerResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * The keep-alive automation server is a service that serves as an RMI server
 * entry point into an automation server instance and provides access without
 * having to launch a new Eclipse instance each time a service is required. This
 * is basically a meta-service that helps maintain the automation server while
 * keep-alive is required.
 * 
 * This service starts up the RNI server appropriately and then keeps a
 * monitoring thread alive that shuts down the automation server after a given
 * timeout is exceeded. The timeout neglects the time taken to execute the
 * service.
 * 
 * The timeout should be compared against the time since the end of the service
 * invocation to the beginning of a new service request. This means the
 * background timer thread will be disabled as a request comes in to the
 * automation server and reset at the end of the invocation.
 * 
 * @author Scott E. Schneider
 */
public class KeepAliveService extends AbstractService {

	/**
	 * Remote object is the primary object exposed to the clients, it implements
	 * and is commanded through its automation server interface (that is a
	 * remote interface)
	 * 
	 * @author Scott E. Schneider
	 */
	public static class RemoteObject extends UnicastRemoteObject implements
			IAutomationServer {

		/**
		 * The stream unique identifier for this class, used for serialiation
		 * purposes
		 */
		private static final long serialVersionUID = -4260095335800707699L;

		/**
		 * Delegate automation server instance, handles the work
		 */
		private final AutomationServer delegate;

		/**
		 * Constructs the remotable object that implements the automation server
		 * remote interface
		 * 
		 * @param delegate
		 *            the delegate automation server to service the requests
		 * @param port
		 *            port number to listen for requests on
		 * 
		 * @throws RemoteException
		 *             any remote exceptions are thrown from this constructor
		 */
		public RemoteObject(AutomationServer delegate) throws RemoteException {

			// Invoke super constructor
			super();

			// Store automation server delegate object
			this.delegate = delegate;

		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.hyades.automation.server.IAutomationServer
		 *      #run(java.lang.Object)
		 */
		public Object run(Object args) throws RemoteException {
			return this.delegate.run(args);
		}

	}

	/**
	 * Specialized RMI security manager for use with keep alive automation
	 * server running code
	 * 
	 * @author Scott E. Schneider
	 */
	public static final class CustomSecurityManager extends RMISecurityManager {
		public void checkPermission(Permission permission) {
			System.out.println(permission);
		}
	}

	/**
	 * Default keep-alive timeout, the amount of time allowed to expire since
	 * the completion of one service invocation to the start of the next
	 * request.
	 */
	private static final int KEEP_ALIVE_TIMEOUT = 25000;

	/**
	 * Stream-Unique IDentifier (SUID) of this class
	 */
	private static final long serialVersionUID = 1037699386754320045L;

	/**
	 * Hook to the remote object that is the main entry point via RMI to the
	 * automation server
	 */
	private RemoteObject remoteObject;

	/**
	 * Timer that will schedule the timer task to monitor the timeout
	 */
	private Timer timer;

	/**
	 * The timer task that will shut down the automation server when appropriate
	 */
	private TimerTask timerTask;

	/**
	 * Default constructor supporting extension instantiation
	 */
	public KeepAliveService() {
		super();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {

		// Establish the remote object server
		try {

			// Retrieve port to use for RMI server
			String portString = this.getProperty("port");//$NON-NLS-1$
			int port = Integer.valueOf(portString).intValue();

			// Instantiate and keep track of remote object
			this.remoteObject = new KeepAliveService.RemoteObject(
					new AutomationServer());

			// Use RMI security manager
			if (System.getSecurityManager() == null) {
				System.setSecurityManager(new RMISecurityManager());
			}

			// Establish registry and bind this remote object
			try {
				Registry registry = LocateRegistry.createRegistry(port);
				registry.bind("server", this.remoteObject);//$NON-NLS-1$
			} catch (Throwable t) {
				t.printStackTrace();
			}

			// In case of any throwables
		} catch (Throwable e) {
			e.printStackTrace();
		}

		// Define keep-alive timer task (background task)
		this.timerTask = new TimerTask() {
			public void run() {
				System.out
				.println(NLS.bind(AutomationServerResourceBundle.KeepAliveService_TIME_OUT_, KeepAliveService.this.remoteObject));
			}
		};

		// Instantiate and schedule a new non-daemon timer
		this.timer = new Timer(false);
		this.timer.schedule(this.timerTask,
				KeepAliveService.KEEP_ALIVE_TIMEOUT,
				KeepAliveService.KEEP_ALIVE_TIMEOUT);

		// Return this object
		return this;

	}
}