/********************************************************************** 
 * Copyright (c) 2006, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ServiceProxy.java,v 1.19 2008/03/20 18:49:53 dmorris Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.automation.server;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.automation.core.AbstractService;
import org.eclipse.hyades.automation.core.Service;
import org.eclipse.hyades.automation.core.utils.MutableObject;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask;
import org.eclipse.hyades.automation.core.utils.ProgressiveTask.Synchronicity;
import org.eclipse.hyades.automation.server.internal.resources.AutomationServerResourceBundle;
import org.eclipse.osgi.util.NLS;

/**
 * The service proxy classes services as a proxy between the calling request and
 * the actual class that implements the extension point called out by the
 * service identifier (the service implementation is a java class that
 * implements the service interface and the appropriate extension point and id).
 * 
 * 
 * @author  Scott E. Schneider
 * @author  Paul E. Slauenwhite
 * @version February 26, 2008
 * @since   June 17, 2005
 */
class ServiceProxy extends AbstractService {

	/**
	 * Stream-Unique IDentifier (SUID) of this class
	 */
	private static final long serialVersionUID = -2274888399085487186L;

	/**
	 * The concrete service implementation given the requested service
	 * identifier
	 */
	private Service serviceImplementation;

	/**
	 * Construct a service proxy given the specified state, this serves as a
	 * proxy between the calling code and the concrete implementation that
	 * implements the appropriate extension point, this allows the memento state
	 * to be revived and the service identifier to be read without violating
	 * encapsulation.
	 * 
	 * @param state
	 *            the state of the service to revive
	 */
	ServiceProxy(Service.Memento state) {

		// Plug the memento state in hydrating service state
		super(state);

		// Find service implementation for this service identifier
		IExtension extension = Platform.getExtensionRegistry().getExtension(
				"org.eclipse.hyades.execution.service", this.getIdentifier());//$NON-NLS-1$

		if (extension != null) {

			// Retrieve the extension point implementors
			IConfigurationElement[] implementors = extension
					.getConfigurationElements();

			// Currently, we use the first returned service (future will have
			// execution strategies)
			try {
				Object executableExtension = null;
				try {
					executableExtension = implementors[0]
							.createExecutableExtension("class");//$NON-NLS-1$
				} catch (CoreException e) {
					executableExtension = implementors[0]
							.createExecutableExtension("implementation");//$NON-NLS-1$
				}
				if (executableExtension instanceof Service) {
					this.serviceImplementation = (Service) executableExtension;
					this.serviceImplementation.setMemento(state);

				}
			} catch (CoreException e) {
				e.printStackTrace();
			}

		}
		else {
			System.out.println(NLS.bind(AutomationServerResourceBundle.ServiceProxy_MISSING_SERVICE_ERROR_, this.getIdentifier()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {
		return this.execute(ProgressiveTask.Synchronicity.SYNCHRONOUS);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.hyades.automation.core.Service#execute(org.eclipse.hyades.automation.core.utils.ProgressiveTask.Synchronicity)
	 */
	public Object execute(final Synchronicity synchronicity) {

		// If service implementation and synchronicity valid
		if (this.serviceImplementation != null && synchronicity != null) {

			// A mutable object to hold the execute return
			final MutableObject executionResult = new MutableObject();

			// An object to hold potential thrown service throwable
			final MutableObject serviceThrowable = new MutableObject();

			/*
			 * Construct task that will execute the service
			 */
			ProgressiveTask executeService = new ProgressiveTask(
					"Executing service: " + this.getIdentifier(),
					new Runnable() {
						public void run() {
							try {
								executionResult
										.set(ServiceProxy.this.serviceImplementation
												.execute(synchronicity));
							} catch (Throwable t) {
								serviceThrowable.set(t);
							}
						}
					}, new NullProgressMonitor(), 3000);

			// Starting execution of service
			if (Boolean.getBoolean("SHOW_DEBUG")) {
				System.out.println("Executing service "
						+ (synchronicity == Synchronicity.SYNCHRONOUS ? ""
								: "a") + "synchronously:");
				System.out.println(this.serviceImplementation);
			}

			/*
			 * Execute progressive task synchronously, currently synchronicity
			 * is passed on to the automation server as a automation property
			 * but it is really acted on in the automation client layer,
			 * returning control immediately after initiating the execution
			 * launch strategy. In the future, there might be additional
			 * synchronicity actions when the Eclipse instance is kept-alive
			 * longing than one service execution.
			 */
			executeService.execute(Synchronicity.SYNCHRONOUS);
			
			if (serviceImplementation instanceof AbstractWorkspaceSensitiveService) {
				try {
					((AbstractWorkspaceSensitiveService)serviceImplementation).saveWorkspace();
				} catch (CoreException e) {
					// TODO Log and handle this exception properly
					e.printStackTrace();
				}
			}
			

			// Retrieve potential throwables thrown
			Throwable taskThrowable = null;

			// Throw uncaught throwables to caller
			if (!serviceThrowable.isNull()) {
				taskThrowable = (Throwable) serviceThrowable.getAndClear();
				throw new RuntimeException(taskThrowable);
			}

			// The execution result is unwrapped (null if canceled or not found)
			return executionResult.getAndClear();

		}

		// Null return indicates bad service implementation or synchronicity
		return null;

	}

}