/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestLogBugzillaProvider.java,v 1.9 2010/03/29 13:52:28 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.extensions.internal;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofileFactory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFRepositoryRecord;
import org.eclipse.hyades.test.ui.extensions.ExtensionsConstants;
import org.eclipse.hyades.test.ui.extensions.ExtensionsImages;
import org.eclipse.hyades.test.ui.extensions.ExtensionsPlugin;
import org.eclipse.hyades.test.ui.extensions.internal.resources.ExtensionsPluginResourceBundle;
import org.eclipse.hyades.test.ui.forms.editor.BaseFormEditor;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.IRecordRepositoryProvider;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.IRepositoryRecordListener;
import org.eclipse.hyades.test.ui.forms.util.FormsUtil;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.browser.IWebBrowser;
import org.eclipse.ui.browser.IWorkbenchBrowserSupport;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormPage;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.ImageHyperlink;
import org.eclipse.ui.forms.widgets.Section;
import org.eclipse.ui.forms.widgets.TableWrapData;
import org.eclipse.ui.forms.widgets.TableWrapLayout;

/**
 * <p>Test log Bugzilla provider.</p>
 * 
 * 
 * @author      Bianca Xue Jiang
 * @author      Paul E. Slauenwhite
 * @version     March 29, 2010
 * @since       May 14, 2006
 */
public class TestLogBugzillaProvider implements IRecordRepositoryProvider {
		
	private ListenerList recordListeners = null;
	private TPFExecutionEvent input = null;
	private IManagedForm mForm = null;
	private Composite parent = null;
	private Section defectSection = null;
	private Composite defectSectionClient = null;
	private List recordIDs = null;
	private Composite recordParent = null;
	private IWorkbenchBrowserSupport browserSupport = null;
	
	public TestLogBugzillaProvider() {
		recordListeners = new ListenerList();
	}

	public void addRecordListener(IRepositoryRecordListener listener) {
		recordListeners.add(listener);
	}

	public void removeRecordListener(IRepositoryRecordListener listener) {
		recordListeners.remove(listener);
	}

	public void createContent(final IManagedForm mForm, final Composite parent)
	{
		this.mForm = mForm;
		this.parent = parent;
		defectSection = FormsUtil.createSection(mForm, parent, ExtensionsPluginResourceBundle.TestLogViewer_Defects, "");  //$NON-NLS-1$
		defectSectionClient = (Composite)defectSection.getClient();
		
		Layout layout = new GridLayout(2, false);
		defectSectionClient.setLayout(layout);
		
		TableWrapLayout tLayout = new TableWrapLayout();
		tLayout.numColumns = 3;
		tLayout.makeColumnsEqualWidth = true;

		recordParent = mForm.getToolkit().createComposite(defectSectionClient);
		recordParent.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));		
		recordParent.setLayout(tLayout);

		recordIDs = new ArrayList();
		
		mForm.getToolkit().paintBordersFor(defectSectionClient);
		
		Composite actionParent = new Composite(defectSectionClient, SWT.NONE);
		actionParent.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END | GridData.VERTICAL_ALIGN_BEGINNING));
		actionParent.setLayout(new GridLayout());

		Button add = mForm.getToolkit().createButton(actionParent, ExtensionsPluginResourceBundle.WORD_ADD_DOT, SWT.PUSH); 
		add.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		add.addSelectionListener(new SelectionListener(){
			
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}
			
			public void widgetSelected(SelectionEvent e) {
				
				//Cancel this selection event if the editor is opened in read-only mode:
				if(((BaseFormEditor)(((FormPage)(mForm.getContainer())).getEditor())).isReadOnly()){
					e.doit = false;
				}
				else{

					InputDialog dialog = new InputDialog(Display.getCurrent().getActiveShell(), 
							ExtensionsPluginResourceBundle.TestLogViewer_addDefectTitle,  
							ExtensionsPluginResourceBundle.TestLogViewer_enterBugzillaNumber,  
							null, 
							new IInputValidator(){

						/* (non-Javadoc)
						 * @see org.eclipse.jface.dialogs.IInputValidator#isValid(java.lang.String)
						 */
						public String isValid(String newText) {

							//Only valid the new text if it contains one or more non-whitespace characters:
							if((newText != null) && (newText.trim().length() > 0)){

								//Do not trim the new text in the event of leading/trailing whitespace (e.g. invalid input):
								try {
									Integer.parseInt(newText);
								}
								catch (NumberFormatException n) {
									return (NLS.bind(ExtensionsPluginResourceBundle.TestLogViewer_invalidBugzillaNumber, newText));
								}

								//Must trim the new text since the record IDs are trimmed:
								if(recordIDs.contains(newText.trim())){
									return (NLS.bind(ExtensionsPluginResourceBundle.TestLogViewer_alreadyExists, newText)); 
								}
							}

							//Do not display an error message but disable the OK button if the user did not input one or more non-white space characters:
							else{
								return (""); //$NON-NLS-1$
							}

							return null;
						}
					});

					if(dialog.open() == InputDialog.OK){

						//Trim the record ID (required by the isValid() method for the IInputValidator associated with the InputDialog):
						String value = dialog.getValue().trim();

						TPFRepositoryRecord record = Common_TestprofileFactory.eINSTANCE.createTPFRepositoryRecord();
						record.setID(value);
						record.setURI(ExtensionsPlugin.getPreference(ExtensionsConstants.BUGZILLA_OPEN_URL) + value);
						record.setLabel(value);
						record.setType(ExtensionsConstants.BUGZILLA_PROVIDER_TYPE);

						fireRecordChanged(record, true);

						createRecordHyperLink(record);

						recordIDs.add(record.getID());

						recordParent.layout();
						defectSectionClient.layout();
						parent.layout();
					} 
				}
			}
		});

		Button submit = mForm.getToolkit().createButton(actionParent, ExtensionsPluginResourceBundle.TestLogViewer_Submit, SWT.PUSH); 
		submit.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		submit.addSelectionListener(new SelectionListener(){
			
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}
			
			public void widgetSelected(SelectionEvent e) {
				openURL(ExtensionsPlugin.getPreference(ExtensionsConstants.BUGZILLA_SUBMIT_URL), ExtensionsPlugin.PLUGIN_ID, ExtensionsPluginResourceBundle.TestLogViewer_NewBug, "");  //$NON-NLS-1$
			}
		});

		Button search = mForm.getToolkit().createButton(actionParent, ExtensionsPluginResourceBundle.TestLogViewer_Search, SWT.PUSH); 
		search.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		search.addSelectionListener(new SelectionListener(){
			
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}
			
			public void widgetSelected(SelectionEvent e) {
				openURL(ExtensionsPlugin.getPreference(ExtensionsConstants.BUGZILLA_SEARCH_URL), ExtensionsPlugin.PLUGIN_ID, ExtensionsPluginResourceBundle.TestLogViewer_SearchForBugs, "");  //$NON-NLS-1$
			}
		});

		mForm.getToolkit().paintBordersFor(defectSectionClient);
	}
	
	public void setInput(TPFExecutionEvent event){
	
		if(event == null || event.equals(input)){
			return;
		}
		
		input = event;
		
		if(recordParent != null && !recordParent.isDisposed())
		{
			Control[] children = recordParent.getChildren();
			if(children != null)
			{
				for(int i = 0; i < children.length; i++)
				{
					if(children[i] != null && !children[i].isDisposed())
					{
						children[i].dispose();
						children[i] = null;
					}
				}
			}	
			if(recordIDs == null){
				recordIDs = new ArrayList();
			}
			else{
				recordIDs.clear();
			}

			List records = event.getDefectRecords();
			if(records != null)
			{
				for(int i = 0; i < records.size(); i++){
					
					TPFRepositoryRecord record = (TPFRepositoryRecord)records.get(i);
					String type = record.getType();
					
					if((type != null) && (type.equals(ExtensionsConstants.BUGZILLA_PROVIDER_TYPE))){
						
						createRecordHyperLink(record);
						
						recordIDs.add(record.getID());
					}
				}
			}

			recordParent.layout();
			mForm.getToolkit().paintBordersFor(recordParent);
			defectSectionClient.layout();
			defectSection.redraw();
		}
	}
	
	protected void fireRecordChanged(TPFRepositoryRecord record, boolean isAdd)
	{
		Object[] listeners = recordListeners.getListeners();
		for(int i = 0; i < listeners.length; i++)
		{
			if(isAdd)
				((IRepositoryRecordListener)listeners[i]).recordAdded(record);
			else
				((IRepositoryRecordListener)listeners[i]).recordRemoved(record);
		}
	}
	
	public void selectionChanged(SelectionChangedEvent event) {
		if(event.getSelection() instanceof IStructuredSelection)
		{
			IStructuredSelection selection = (IStructuredSelection)event.getSelection();
			if(selection.getFirstElement() instanceof TPFExecutionEvent)
				setInput((TPFExecutionEvent)selection.getFirstElement());				
		}
	}
	
	private void openURL(String url, String browserId, String title, String tooltip)
	{
		IWebBrowser browser = null;
		int flags = 0;
		if(browserSupport == null)
			browserSupport = PlatformUI.getWorkbench().getBrowserSupport();
		if (browserSupport.isInternalWebBrowserAvailable()) {
			flags = IWorkbenchBrowserSupport.AS_EDITOR | IWorkbenchBrowserSupport.LOCATION_BAR
					| IWorkbenchBrowserSupport.NAVIGATION_BAR;

		} else {
			flags = IWorkbenchBrowserSupport.AS_EXTERNAL | IWorkbenchBrowserSupport.LOCATION_BAR
					| IWorkbenchBrowserSupport.NAVIGATION_BAR;
		}
		try {
			browser = browserSupport.createBrowser(flags, browserId, title, tooltip);
			browser.openURL(new URL(url));
		}
		catch (Exception e) {
			ExtensionsPlugin.logError(e);
		}
	}

	private void createRecordHyperLink(TPFRepositoryRecord record){
		
		TableWrapLayout tablewrapLayout = new TableWrapLayout();
		tablewrapLayout.numColumns = 2;
		tablewrapLayout.makeColumnsEqualWidth = false;
		tablewrapLayout.horizontalSpacing = 0;

		Composite recordLinkParent = mForm.getToolkit().createComposite(recordParent);
		recordLinkParent.setLayoutData(new TableWrapData(TableWrapData.FILL_GRAB, TableWrapData.MIDDLE));		
		recordLinkParent.setLayout(tablewrapLayout);
						
		ImageHyperlink recordLink = mForm.getToolkit().createImageHyperlink(recordLinkParent, SWT.NONE);
		recordLink.setLayoutData(new TableWrapData(TableWrapData.RIGHT, TableWrapData.MIDDLE));
		recordLink.setText(record.getLabel());
		recordLink.setImage(ExtensionsImages.INSTANCE.getImage(ExtensionsImages.IMG_BUG));
		recordLink.setData(record);
		recordLink.addHyperlinkListener(new HyperlinkAdapter(){
			
			public void linkActivated(HyperlinkEvent hyperlinkEvent) {
				
				TPFRepositoryRecord repositoryRecord = ((TPFRepositoryRecord)(hyperlinkEvent.widget.getData()));
				String url = (ExtensionsPlugin.getPreference(ExtensionsConstants.BUGZILLA_OPEN_URL) + (repositoryRecord.getLabel()));
						
				if((url != null) && (url.trim().length() > 0)){
					openURL(url, ExtensionsPlugin.PLUGIN_ID, repositoryRecord.getLabel(), ""); //$NON-NLS-1$
				}
			}			
		});

		ImageHyperlink removeRecordLink = mForm.getToolkit().createImageHyperlink(recordLinkParent, SWT.NONE);
		removeRecordLink.setLayoutData(new TableWrapData(TableWrapData.LEFT, TableWrapData.MIDDLE));
		removeRecordLink.setImage(ExtensionsImages.INSTANCE.getImage(ExtensionsImages.IMG_REMOVE_BUG));
		removeRecordLink.setData(record);
		removeRecordLink.addHyperlinkListener(new HyperlinkAdapter(){
			
			public void linkActivated(HyperlinkEvent hyperlinkEvent){
				
				//Only process this hyperlink event if the editor is not opened in read-only mode:
				if(!((BaseFormEditor)(((FormPage)(mForm.getContainer())).getEditor())).isReadOnly()){

					TPFRepositoryRecord repositoryRecord = ((TPFRepositoryRecord)(hyperlinkEvent.widget.getData()));

					if(MessageDialog.openConfirm(Display.getCurrent().getActiveShell(), ExtensionsPluginResourceBundle.TestLogViewer_removeDefectConfirmTitle, NLS.bind(ExtensionsPluginResourceBundle.TestLogViewer_removeDefectConfirmMessage, repositoryRecord.getLabel()))){

						//Remove the hyperlink:
						((ImageHyperlink)(hyperlinkEvent.getSource())).getParent().dispose();

						fireRecordChanged(repositoryRecord, false);
						recordIDs.remove(repositoryRecord.getID());

						recordParent.layout();
						parent.layout();
					}
				}
			}
		});
		
		recordLinkParent.layout();
	}
}
