/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PDContentProvider.java,v 1.8 2010/08/18 20:37:27 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.internal.ui;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.util.provisional.ITraceModelInteraction;
import org.eclipse.hyades.models.util.provisional.TraceModelInteraction;
import org.eclipse.hyades.trace.ui.internal.util.DeleteUtil;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.navigator.Navigator;
import org.eclipse.hyades.ui.provisional.extension.CustomContentProvider;
import org.eclipse.hyades.ui.provisional.extension.ICustomContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;

public class PDContentProvider implements ITreeContentProvider, IResourceChangeListener
{
	protected static Hashtable fMonitors = new Hashtable();

	protected Object[] objList;

	protected Navigator fViewer;
	
	/**
	 * A list of providers that implement the extension point pdCustomContentProvider in org.eclipse.tptp.platform.common.ui.
	 */
	protected ICustomContentProvider[] providers;

	class ResourceDeltaVisitor implements IResourceDeltaVisitor
	{
		public boolean visit(IResourceDelta delta) throws CoreException
		{
			IResource resource = delta.getResource();
			// skip workspace root
			if (resource.getType() == IResource.ROOT)
				return true;

			if (delta.getKind() == IResourceDelta.REMOVED)
			{
				if (resource instanceof IContainer)
					removeDoc((IContainer) resource);
				return true;
			} else if (delta.getKind() == IResourceDelta.ADDED)
			{
				return false;
			} else if (delta.getKind() == IResourceDelta.CHANGED)
			{
				return true;
			}

			return true;
		}
	}

	public PDContentProvider(Navigator viewer)
	{
		fViewer = viewer;
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		providers = CustomContentProvider.getContributors();
	}

	public static void addMonitor(IResource container, TRCMonitor monitor)
	{
		Object monitors = fMonitors.get(container);
		if (monitors == null)
		{
			monitors = new ArrayList();
			((ArrayList) monitors).add(monitor);

			fMonitors.put(container, monitors);
		} else
		{
			if (!((ArrayList) monitors).contains(monitor))
				((ArrayList) monitors).add(monitor);
		}

	}

	public void dispose()
	{
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);

	}

	public Object[] getChildren(Object elem)
	{
		final Object element = elem;
		// Reset the object list to an empty list to prevent it from checking the previous object list each time.
		objList = new Object[0];

		try
		{
			if (element instanceof IProject)
				objList = getContainerMonitors((IProject) element);
			else if (element instanceof IFolder)
				objList = getContainerMonitors((IFolder) element);
			else if (element instanceof TRCMonitor)
				objList = getMonitorChildren((TRCMonitor) element);
			else if (element instanceof TRCNode)
				objList = getNodeProcesses((TRCNode) element);
			else if (element instanceof TRCProcessProxy)
				objList = getProcessAgents((TRCProcessProxy) element);
			else if (element instanceof TRCAgentProxy)
				objList = null;
			else if (element instanceof INavigatorItem)
				objList = null;
		} catch (Exception e)
		{
			e.printStackTrace();
		}

		if (objList == null)
			objList = new Object[0];

		Object[] delegates = NavigatorExtensionUtil.getAllChildren(element, fViewer.getID()).toArray();
		if (delegates == null || delegates.length == 0) return objList;
		
		Object[] finalList = new Object[objList.length + delegates.length];
		System.arraycopy(objList, 0, finalList, 0, objList.length);
		System.arraycopy(delegates, 0, finalList, objList.length, delegates.length);
		
		return finalList;
	}

	public static TRCAgentProxy getAgent(String runtimeId) {
		if (runtimeId == null) return null;

		Iterator monitors = getMonitors().iterator();
		while (monitors.hasNext()) {
			TRCMonitor monitor = (TRCMonitor) monitors.next();
		
			Iterator nodes = monitor.getNodes().iterator();
			while (nodes.hasNext()) {
				TRCNode node = (TRCNode) nodes.next();
				if (node.eIsProxy()) continue;
			
				Iterator processes = node.getProcessProxies().iterator();
				while (processes.hasNext()) {
					TRCProcessProxy process = (TRCProcessProxy) processes.next();
					if (process.eIsProxy()) continue;
				
					Iterator agents = process.getAgentProxies().iterator();
					while (agents.hasNext()) {
						TRCAgentProxy agent = (TRCAgentProxy) agents.next();
						if (agent.eIsProxy()) continue;
					
						if (runtimeId.equals(agent.getRuntimeId())) return agent;
					}
				}
			}
		}
		
		return null;
	}
	
	protected Object[] getContainerMonitors(IContainer res)
	{
		return getContainerMonitorsArray(res).toArray();
	}

	protected ArrayList getContainerMonitorsArray(IContainer res)
	{
		ArrayList list = getContainerMonitors0(res);
		ArrayList result = new ArrayList();

		/**
		 * Process the items added to the list by getContainerMonitors0 and add them to the 
		 * result list to be displayed in the view.
		 */
		for (int idx = 0; idx < list.size(); idx++)
		{
			Object obj = list.get(idx);
			if (obj instanceof TRCMonitor)
			{
				TRCMonitor monitor = (TRCMonitor) obj;
				result.add(monitor);
			} else if (obj instanceof IContainer)
			{
				IContainer cont = (IContainer) obj;
				result.add(cont);
			} else if (obj instanceof INavigatorItem)
			{
				INavigatorItem navigatorItem = (INavigatorItem) obj;
				result.add(navigatorItem);
			}
		}

		return result;
	}

	protected ArrayList getContainerMonitors0(IContainer res)
	{
		ArrayList childList = new ArrayList();

		if (res == null || !res.exists())
			return childList;

		// add by James Zhou (zhouja@ca.ibm.com) for bug #117247 fix, Jan 16,
		// 2006
		if (res instanceof IProject && !((IProject) res).isOpen())
			return childList;
		// end of bug #117247 fix

		try
		{

			// get all folders
			IResource[] rmembers = res.members();
			for (int idx = 0; idx < rmembers.length; idx++)
			{
				IResource member = rmembers[idx];
				if (member == null || !member.exists())
					continue;

				if (member instanceof IFolder)
					childList.add(member);
			}
		} catch (Exception exc)
		{
			exc.printStackTrace();
		}

		// get mof objects
		Object monitors = fMonitors.get(res);

		if (monitors == null)
		{
			try
			{
				IResource[] members = res.members();
				for (int idx = 0; idx < members.length; idx++)
				{
					IResource member = members[idx];
					if (member == null || !member.exists())
						continue;

					boolean isMonitor = false;
					if (member instanceof IFile)
					{
						if (member.getFileExtension() != null && member.getFileExtension().equals(CommonUIConstants.MONITOR_EXT))
						{
							isMonitor = true;
							getMonitors(member, childList);
						}
					}
					
					/**
					 * If the resource is not a monitor, check if there is a custom provider using the pdCustomContentProvider extension
					 * point and add valid resources to the list of items to display in the navigator.
					 */
					if(!isMonitor) {
						for(int i = 0; i < providers.length; i++) {
							ICustomContentProvider provider = providers[i];
							if(provider != null) {
								if (provider.isValidResource(member)) {
									INavigatorItem navItem = provider.getContribution(member);
									if(navItem != null)
										childList.add(navItem);
								}
							}
						}
					}
				}
			} catch (CoreException exc)
			{

				CommonPlugin.logError(exc);
			}

			if (childList.size() > 0)
			{
				ArrayList v = new ArrayList();
				for (int i = 0; i < childList.size(); i++)
				{
					Object obj = childList.get(i);
					if (obj instanceof TRCMonitor)
						v.add(obj);
				}

				if (v.size() > 0)
					fMonitors.put(res, v);

			}

			return childList;
		}

		for (int idx = 0; idx < ((ArrayList) monitors).size(); idx++)
			childList.add(((ArrayList) monitors).get(idx));

		if (childList == null || childList.size() == 0)
			return new ArrayList();

		return childList;
	}

	public Object[] getElements(Object elem)
	{
		final Object element = elem;

		try
		{
			if (element instanceof IWorkspace)
				objList = ((IWorkspace) element).getRoot().getProjects();
			else if (element instanceof IWorkspaceRoot)
				objList = ((IWorkspaceRoot) element).getProjects();
		} catch (Exception e)
		{
			e.printStackTrace();
		}

		if (objList == null)
			objList = new Object[0];

		Object[] delegates = NavigatorExtensionUtil.getAllChildren(element, fViewer.getID()).toArray();
		Object[] finalList = new Object[objList.length + delegates.length];
		System.arraycopy(objList, 0, finalList, 0, objList.length);
		System.arraycopy(delegates, 0, finalList, objList.length, delegates.length);
		return finalList;
	}

	/**
	 * 
	 * @param monitor
	 * @return
	 */
	protected Object[] getMonitorChildren(TRCMonitor monitor)
	{
		return getMonitorChildren0(monitor).toArray();
	}

	/**
	 * 
	 * @param monitor
	 * @return
	 */
	protected ArrayList getMonitorChildren0(TRCMonitor monitor)
	{
		ArrayList list = new ArrayList();

		try
		{
			Iterator i = monitor.getNodes().iterator();
			while (i.hasNext())
			{
				TRCNode node = (TRCNode) i.next();
				if (node.eIsProxy())
					continue;
				// //*** adding support for multiple files
				// SaveUtil.addDocument(node.eResource());

				list.add(node);
			}
		} catch (Exception exc)
		{
			return list;
		}

		return list;
	}

	/**
	 * Insert the method's description here. Creation date: (10/06/2000 1:38:51
	 * PM)
	 * 
	 * @return java.util.Vector
	 */
	public static ArrayList getMonitors()
	{
		Enumeration list = fMonitors.elements();

		ArrayList v = new ArrayList();
		while (list.hasMoreElements())
		{
			ArrayList monitors = (ArrayList) list.nextElement();
			for (int idx = 0; idx < monitors.size(); idx++)
			{
				v.add(monitors.get(idx));
			}

		}

		return v;
	}

	/**
	 * Insert the method's description here. Creation date: (10/06/2000 1:38:51
	 * PM)
	 * 
	 * @return java.util.Vector
	 */
	public static ArrayList getMonitors(IContainer container)
	{
		Object monitors = fMonitors.get(container);
		if (monitors == null)
			return new ArrayList();

		return (ArrayList) monitors;
	}

	protected void getMonitors(IResource member, ArrayList childList)
	{
		ResourceSet resourceSet = TraceUIManager.getTraceUIManager().getResourceSet();
		Resource res = null;
		try
		{
			String monPath = member.getFullPath().toString();
			String name = monPath;

			res = resourceSet.getResource(SaveUtil.createURI("platform:/resource" + name + "#").trimFragment(), true);

			// Bug 322951
			ITraceModelInteraction[] contributors = TraceModelInteraction.getContributors();
			// System.out.println("trace file " + name);
			// populate monitors ...
			EList ext = res.getContents();
			Iterator i = ext.iterator();
			while (i.hasNext())
			{
				Object obj = i.next();
				if (obj instanceof TRCMonitor) {
					childList.add(obj);
					// Allow for implementers of the model interaction extension point to load additional data with the model load
					// such as on perspective start.
					if(contributors != null && contributors.length > 0) {
						for(int contributor = 0; contributor < contributors.length; contributor++) {
							contributors[contributor].traceModelLoad(((TRCMonitor)obj).eResource());
						}
					}
				}
			}

			// //*** adding support for multiple files
			// if(res != null)
			// SaveUtil.addDocument(res);

		} catch (Exception e)
		{

			CommonPlugin.logError(e);

			return;
		}
	}

	protected Object[] getNodeProcesses(TRCNode node)
	{
		return getNodeProcesses0(node).toArray();

	}

	protected ArrayList getNodeProcesses0(TRCNode node)
	{
		ArrayList list = new ArrayList();

		try
		{
			Iterator i = node.getProcessProxies().iterator();
			while (i.hasNext())
			{
				TRCProcessProxy process = (TRCProcessProxy) i.next();
				if (process.eIsProxy())
					continue;
				// //*** adding support for multiple files
				// SaveUtil.addDocument(process.eResource());

				list.add(process);
			}
		} catch (Exception exc)
		{
			return list;
		}

		return list;

	}

	public Object getParent(Object element)
	{

		return null;
	}

	protected Object[] getProcessAgents(TRCProcessProxy process)
	{
		return getProcessAgents0(process).toArray();
	}

	protected ArrayList getProcessAgents0(TRCProcessProxy process)
	{
		ArrayList list = new ArrayList();

		try
		{

			Object[] agents = process.getAgentProxies().toArray();
			for (int idx = 0; idx < agents.length; idx++)
			{
				TRCAgentProxy agent = (TRCAgentProxy) agents[idx];
				if (agent.eIsProxy())
					continue;

				// //*** adding support for multiple files
				// SaveUtil.addDocument(agent.getAgent().eResource());

				list.add(agent);
			}
		} catch (Exception exc)
		{
			return list;
		}

		return list;
	}

	public boolean hasChildren(Object element)
	{
		if (NavigatorExtensionUtil.hasChildren(element, fViewer.getID()))
		{
			return true;
		} else
		{
			return (getChildren(element).length > 0);
		}
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
	}

	public boolean isDeleted(Object element)
	{
		if (element instanceof IResource)
			return ((IResource) element).exists();
		return false;
	}

	public static void removeContainer(IContainer container)
	{
		Object list = fMonitors.get(container);
		if (list!=null && list instanceof ArrayList) {
			((ArrayList)list).clear();
		}
		fMonitors.remove(container);
	}

	protected void removeDoc(IContainer res)
	{
		removeContainer(res);

		try
		{
			DeleteUtil.deleteContainer(res, true, true, fViewer.getID());
		} catch (Exception e)
		{
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * @param container
	 * @param monitor
	 */
	public static void removeMonitor(IContainer container, TRCMonitor monitor)
	{
		if (container == null)
			return;

		removeMonitor(container.getFullPath().toOSString(), monitor);
	}

	public static void removeMonitor(String strPath, TRCMonitor monitor)
	{
		IPath path = new Path(strPath);

		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource res = root.findMember(path);
		if (res == null)
			return;

		Object monitors = fMonitors.get(res);
		if (monitors == null)
			return;

		((ArrayList) monitors).remove(monitor);
	}

	/**
	 * Insert the method's description here. Creation date: (10/06/2000 1:32:26
	 * PM)
	 */
	public static void resetMonitors()
	{
		Enumeration list = fMonitors.elements();
		while (list.hasMoreElements())
		{
			ArrayList monitors = (ArrayList) list.nextElement();
			monitors.clear();
		}
		
		fMonitors.clear();
	}

	/**
	 * Notifies this content provider that some resource changes have happened
	 * on the platform, and that the content provider should update
	 */
	public void resourceChanged(IResourceChangeEvent event)
	{

		if (event.getType() == IResourceChangeEvent.POST_CHANGE && event.getSource() instanceof IWorkspace)
		{

			IResourceDelta delta = event.getDelta();

			try
			{
				ResourceDeltaVisitor visitor = new ResourceDeltaVisitor();
				delta.accept(visitor);
			} catch (Exception exc)
			{
				exc.printStackTrace();
			}

			// Do a sync exec, not an async exec, since the resource delta
			// must be traversed in this method. It is destroyed
			// when this method returns.
			Display.getDefault().syncExec(new Runnable()
			{
				public void run()
				{
					if (fViewer != null && fViewer.getViewer() != null)
						fViewer.getViewer().refresh();
				}
			});
		}
	}

}
