/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PDLabelProvider.java,v 1.7 2008/01/24 02:29:56 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.internal.ui;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCEnvironmentVariable;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.tptp.platform.common.internal.util.ILabelProvider;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceImages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.util.CommonUITraceUtil;
import org.eclipse.ui.model.IWorkbenchAdapter;

public class PDLabelProvider extends LabelProvider
{

	protected Map imageTable;

	/**
	 * The images
	 */
	protected Image[] images;
	
	/**
	 * Stores images based on specific process types
	 * KEY = launch configuration id
	 * VALUE = image
	 */
	protected Map processTypeImages;
	
	protected ILabelProvider delegator;

	/**
	 * Default constructor
	 */
	public PDLabelProvider()
	{
		super();
		images = new Image[]
		{
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_MONITOR), 			// monitor
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_NODE), 			// node
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_PROCESS),			// process
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_MON_AGENT), 		// monitored agent
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_RUN_AGENT), 		// running agent
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_STOP_AGENT), 		// stopped agent
				CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_DETACHED_AGENT) 	// detached agent
		};
		delegator = new CommonLabelProviderDelegator(CommonUITracePlugin.getDefault().getPreferenceStore().getInt(CommonUITraceConstants.PROFILING_MONITOR_LAYOUT));
		processTypeImages = new Hashtable();
		populateProcessTypeImages();
	}

	protected void populateProcessTypeImages()
	{
		IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.debug.ui.launchConfigurationTypeImages");
		for (int i = 0; i < elements.length; i++)
		{
			String id = elements[i].getAttribute("configTypeID");
			ImageDescriptor imageDescriptor = CommonUITraceUtil.resolveImageDescriptor(elements[i], "icon");
			Image image = imageDescriptor == null ? null : imageDescriptor.createImage();
			
			if (id != null && image != null)
				processTypeImages.put(id, image);
		}
	}

	/**
	 * Disposes of all IWorkbenchAdapter based allocated images.
	 */
	public void dispose()
	{
		if (imageTable != null)
		{
			for (Iterator i = imageTable.values().iterator(); i.hasNext();)
			{
				((Image) i.next()).dispose();
			}
			imageTable = null;
		}
	}

	/**
	 * Returns the image to display for the passed resource element. NOTE: The
	 * resource label provider caches its images, so don't dispose them!
	 */
	public Image getImage(Object element)
	{
		if (element instanceof IAdaptable && ((IAdaptable) element).getAdapter(IWorkbenchAdapter.class) != null)
		{
			IWorkbenchAdapter adapter = (IWorkbenchAdapter) ((IAdaptable) element).getAdapter(IWorkbenchAdapter.class);

			// obtain the cached image corresponding to the descriptor
			if (imageTable == null)
			{
				imageTable = new Hashtable(40);
			}
			ImageDescriptor descriptor = adapter.getImageDescriptor(element);
			if (descriptor != null)
			{
				Image image = (Image) imageTable.get(descriptor);
				if (image == null)
				{
					image = descriptor.createImage();
					imageTable.put(descriptor, image);
				}
				return image;
			}
			return null;

		} 
		/* Monitor */
		else if (element instanceof TRCMonitor)
		{
			return images[0];
		}
		/* Node */
		else if (element instanceof TRCNode)
		{
			return images[1];
		}
		/* Process */
		else if (element instanceof TRCProcessProxy)
		{
			EList variables = ((TRCProcessProxy)element).getEnvironmentVariables();
			String launchConfigurationId = null;
			for (int i = 0, varCount = variables.size(); i < varCount; i++)
			{
				TRCEnvironmentVariable currentVariable = (TRCEnvironmentVariable)variables.get(i);
				if (CommonUITraceConstants.LAUNCH_CONFIGURATION_ID.equals(currentVariable.getName()))
					launchConfigurationId = currentVariable.getValue();
			}
			
			Image image = null;
			if (launchConfigurationId != null && (image = (Image)processTypeImages.get(launchConfigurationId)) != null)
				return image;
			
			return images[2];
		}
		/* Agent */
		else if (element instanceof TRCAgentProxy)
		{
			TRCAgentProxy agentProxy = (TRCAgentProxy)element;
			if (!agentProxy.isActive())
				return images[5]; // inactive
			else if (!agentProxy.isAttached())
				return images[6]; // detached
			else if (agentProxy.isMonitored())
				return images[3]; // monitored
			
			return images[4]; // running and attached

		} 
		/* Navigator item (aka analysis types) */
		else if (element instanceof INavigatorItem)
		{
			return ((INavigatorItem) element).getImage();
		}

		return null;
	}


	/*
	 * (non-Javadoc) Method declared on ILabelProvider.
	 */
	public String getText(Object element)
	{
		if (element instanceof IAdaptable && ((IAdaptable) element).getAdapter(IWorkbenchAdapter.class) != null)
		{
			IWorkbenchAdapter adapter = (IWorkbenchAdapter) ((IAdaptable) element).getAdapter(IWorkbenchAdapter.class);

			return adapter.getLabel(element);
		} else if (element instanceof INavigatorItem)
			return ((INavigatorItem) element).getText();

		return delegator.getText(element);
	}
}
