/********************************************************************** 
 * Copyright (c) 2005, 2008  IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: FilterTraceViewer.java,v 1.8 2008/04/12 18:09:58 apnan Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 * Eugene Chan, IBM
 * Valentina Popescu, IBM
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.ui.filters.IFilterQueryProvider;
import org.eclipse.hyades.ui.filters.IFilterViewer;
import org.eclipse.hyades.ui.filters.internal.actions.FiltersSelectionAction;
import org.eclipse.hyades.ui.filters.internal.util.FilterInformationManager;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;

/**
 * This class defines an abstract base class for viewers shown in the trace model that have
 * some filter type(s) associated with them.  This is for the profile monitor navigator,
 * log view naviator.  This class extends TraceViewer inheriting its properties,
 * and creates the filter menu actions for the implementors filter scope definition.
 */
public abstract class FilterTraceViewer extends TraceViewer implements IFilterViewer {

	protected FilterInformationManager _fi;
	protected IFilterQueryProvider _fqp;
	protected String _contentDescription = null;
	protected boolean _internalCall = false;
	protected Action _filtersAction;
	protected String defaultFilterName;

	/**
	 * @see TraceViewer#makeActions()
	 */
	public void makeActions()
	{
		if (isInitializedMenu())
			return; //create the toolbar only once

		initializedMenu(true);
		
		super.makeActions();
		
		IToolBarManager tbm = getViewSite().getActionBars().getToolBarManager();
		
		_fi = FilterInformationManager.instance(this);
		if (_fi.getCurrentFilter() == null) {
			_fi.selectedFilterName(defaultFilterName);
		}
		
		if (!tbm.isEmpty())
			tbm.add(new Separator());
		
		createFiltersAction();
		tbm.add(_filtersAction);
		
		// require update because toolbar control has been created by this point,
		// but manager does not update it automatically once it has been created
		//getViewSite().getActionBars().updateActionBars();		
	}
	
	public void createFiltersAction(){
		_filtersAction = new FiltersSelectionAction(_fi);
	}
	
	/**
	 * @see TraceViewer#dispose()
	 */
	public void dispose() {
		if (_fqp != null)
			_fqp.viewerDisposed();
		
		_fqp = null;
		
		super.dispose();
	}

	/**
	 * @see IFilterViewer#setFilterQueryProvider(IFilterQueryProvider)
	 */
	public void setFilterQueryProvider(IFilterQueryProvider filterQueryProvider)
	{
		_fqp = filterQueryProvider;
	}

	/**
	 * @see IFilterQueryProvider#getCurrentFilter()
	 */
	public SimpleSearchQuery getCurrentFilter()
	{
		return _fqp.getCurrentFilter();
	}

	/**
	 * @see IFilterQueryProvider#standardQuery()
	 */
	public SimpleSearchQuery standardQuery()
	{
		return _fqp.standardQuery();
	}	
	
	/**
	 * @see IFilterQueryProvider#advancedQuery()
	 */
	public SimpleSearchQuery advancedQuery()
	{
		return _fqp.advancedQuery();
	}
	
	/**
	 * @see IFilterQueryProvider#updateAdvancedQuery(SimpleSearchQuery, int)
	 */
	public void updateAdvancedQuery(SimpleSearchQuery query, int advancedTabSelectionIndex)
	{
		_fqp.updateAdvancedQuery(query, advancedTabSelectionIndex);	
	}

	/**
	 * Updates the description for this view for the filter applied to this view
	 *
	 */
	public void updateFilterAppliedDescription() {
		if (_contentDescription == null)
			_contentDescription = "";
		
		_internalCall = true;
		setContentDescription(_contentDescription);
		_internalCall = false;
	}

	/**
	 * @see org.eclipse.ui.part.WorkbenchPart#setContentDescription(java.lang.String)
	 */
	protected void setContentDescription(String description) {
		if (_internalCall)
			super.setContentDescription(description);
		else
		{
			_contentDescription = description;
			updateFilterAppliedDescription();
		}
	}

	public FilterInformationManager getFilterInformationManager() {
		return _fi;
	}
	
	public void setFilterActionEnabled(final boolean enabled) {
		if (_filtersAction == null) {
			return;
		}
		_filtersAction.setEnabled(enabled);
	}

	public boolean isFilterActionEnabled() {
		return (_filtersAction != null) ? _filtersAction.isEnabled() : false;
	}

	public void setDefaultFilterName(final String name) {
		defaultFilterName = name;
	}
	
	public Action getFiltersAction() {
		return _filtersAction;
	}

//	public void setFilterInformationManager(FilterInformationManager _fi) {
//		this._fi = _fi;
//	}
	
	//https://bugs.eclipse.org/bugs/show_bug.cgi?id=196332
	protected void disableToolBarActions() {
		if (_filtersAction!=null) {
			_filtersAction.setEnabled(false);
		}
		super.disableToolBarActions();
	}
	//https://bugs.eclipse.org/bugs/show_bug.cgi?id=196332
	protected void enableToolBarActions() {
		if (_filtersAction!=null) {
			_filtersAction.setEnabled(true);
		}
		super.enableToolBarActions();
	}
	
	public String getCurrentFilterName() {
		String filterAppliedName;
	
		if (_fqp != null && _fqp.getCurrentFilter() != null)
			filterAppliedName = _fqp.getCurrentFilter().getName();
		else
			filterAppliedName = CommonUITraceMessages.ST_FLTAPN;
		
		return filterAppliedName;
	}
}
