/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceViewer.java,v 1.15 2008/05/29 02:13:52 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.internal.actions.OpenTraceReportAction;
import org.eclipse.hyades.trace.ui.internal.navigator.SimpleLayout;
import org.eclipse.hyades.ui.extension.IExportViewer;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.part.IPage;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.part.ViewPart;

/**
 * This class defines an abstract base class for viewers shown in the trace model.
 * This is for the profile monitor navigator, log view naviator. 
 * 
 * Part of TraveViewer in trace.ui:  doesn't include:
 * menuAboutToShow, enableEditActions
 */
public abstract class TraceViewer extends ViewPart implements IMenuListener
           , ISelectionListener
           , IProfileEventListener
           , IDeleteListener
           , IExportViewer{

	public static final String VIEW_ID = "org.eclipse.hyades.trace.ui.TraceViewer";

	protected EObject _mofObject;
	protected boolean _initializedMenu = false;
	protected Hashtable _pages = new Hashtable();
	protected PageBook book;
	protected IPage currentPage; 
	protected IPage defaultPage;
	protected ILabelProvider _labelProvider = new SimpleLayout(null).getLabelProvider();

	protected Menu fContextMenu;

	public final String GROUP_ADD_VIEWS = "open.views";
	public final String GROUP_GENERIC = "generic.action";
	public final String GROUP_ADDITIONS = "additions";
	protected OpenTraceReportAction _printColumns;  

	/*
	 * Keep track of the refresh events ignored because the page view not visible
	 */
	protected boolean fRefreshView = false;

	/*
	 * Keep track of forced refreshes
	 */
	protected boolean fForceRefresh = false;	
	
	/*
	 * Keep track whether a context handler selection change event was sent 
	 */
	protected boolean contextHandlerSelectionChanged = false;

	/*
	 * Keep track of the selection changed event ignored because the page view not visible
	 */
	//	 protected boolean fSelectionChanged = false;

	/*
	 * Ensure refresh events being processed only if this part is visible.
	 */
	protected boolean fPartVisible = true;
	
	private EObject objToShow = null;
	private boolean initialized = false;
	private boolean loadingTrace = false;

	protected IPartListener2 fPartListener = new IPartListener2() {
		public void partActivated(IWorkbenchPartReference ref) {
		}
		public void partBroughtToTop(IWorkbenchPartReference ref) {
		}
		public void partInputChanged(IWorkbenchPartReference ref) {
		}
		public void partClosed(IWorkbenchPartReference ref) {
		}
		public void partDeactivated(IWorkbenchPartReference ref) {
		}
		public void partOpened(IWorkbenchPartReference ref) {
		}
		public void partVisible(IWorkbenchPartReference ref) {
			if (ref != null && ref.getPart(false) == TraceViewer.this) {
				fPartVisible = true;
				handlePartVisible();
			}
		}
		public void partHidden(IWorkbenchPartReference ref) {
			if (ref != null && ref.getPart(false) == TraceViewer.this) {
				fPartVisible = false;
				handlePartHidden();
			}
		}
	};
	protected int previousSelection=0;

	/**
	 * Constructor
	 */
	public TraceViewer() {
		super();

	}

	public synchronized void loadingTrace() {
		loadingTrace = true;
	}
	
	public synchronized void loadingTraceDone() {
		loadingTrace = false;
		showPage(defaultPage);
	}
	
	/**
	 * Add a page to the workbook. If the mofObject is null, show default empty page
	 * @param object an EObject, the EObject the page will be based on
	 * @param newPage a boolean, whether this is a new page to display or not
	 */
	
	public synchronized void addViewPage(EObject object, boolean newPage) {
		if (!initialized) {
			objToShow = object;
			return;
		}
			
		if (book == null) {
			//the widget was disposed...
			//remove the view from the pd listeners
			TraceUIManager.getTraceUIManager().removeSelectionListener(this);
			return;
		}

		_mofObject = getObjectToView(object);
		setViewTitle(_mofObject);

		if (_mofObject == null) {
			showPage(defaultPage);
			return;
		}

		TraceViewerPage page = getPage(_mofObject);
		if (defaultPage == page) {
			return;
		}

		if (page == null || page.isDisposed()) {
			//remove existing pages
			Enumeration pages = _pages.elements();
			while (pages.hasMoreElements()) {
				((IPage) pages.nextElement()).dispose();
			}
			_pages.clear();

			makeActions();						
			
			page = createPage(_mofObject);
			page.createControl(book);

			_pages.put(_mofObject, page);
						
			newPage = true;

		}
		//		page.refreshPage();
		page.update(newPage);

		if (page.isEmpty()) {
			showPage(defaultPage);
		}
		else {
			if (fPartVisible) handlePartVisible();
			showPage(page);
		}
	}
	
	/**
	 * Add a page to the workbook. If the mofObject is null, show default empty page
	 * @param object an EObject, the EObject the page will be based on
	 */
	public void addViewPage(EObject object) {
		addViewPage(object, false);
	}

	/**
	 * Returns the object that should be viewd if true if the selObject is selected in the monitors view For example, if the process node is selected, the view should display the profiling data if trace collects profiling data
	 * @param selObject an EObject
	 * @return an EObject, the object to view
	 */
	public EObject getObjectToView(EObject selObject) {
		return selObject;
	}

	/**
	 * Returns the default page.
	 * 
	 * @return an IPage, the default page.
	 */
	protected IPage createDefaultPage(PageBook book) {
		DefaultPage page = new DefaultPage();
		page.createControl(book);
		page.setMessage(getDefaultPageMessage());
		return page;
	}

	protected IPage createLoadingPage(PageBook book) {
		DefaultPage page = new DefaultPage();
		page.createControl(book);
		page.setMessage(CommonUITraceMessages.LOADING_DATA);
		return page;
	}
	
	/**
	 * Creates the TraceViewerPage instance associated with this trace viewer.
	 * @param mofObject an EObject, the object associate with this viewer and page
	 * @return a TraceViewerPage instance, the created page
	 */
	public abstract TraceViewerPage createPage(EObject mofObject);
	
	/**
	 * @see IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public synchronized void createPartControl(Composite parent) {
		// Create the page book.
		book = new PageBook(parent, SWT.NONE);

		// Create the default page.
		defaultPage = createDefaultPage(book);

		//ask the view to display the data for the curently selected object
		//INavigator navigator = TPTPUtil.getActiveNavigator(); 
		//if(navigator!=null && navigator.isLinkingEnabled())
		//else
		//	showPage(defaultPage);

		TraceUIManager.getTraceUIManager().addSelectionListener(this);
		TraceUIManager.getTraceUIManager().addProfileEventListener(this);
		TraceUIManager.getTraceUIManager().addDeleteListener(this);

		getViewSite().getPage().addPartListener(fPartListener);
		initialized = true;

		if (objToShow != null) {
			addViewPage(objToShow, true);
			objToShow = null;
		}
		else if (loadingTrace) {
			showPage(createLoadingPage(book));
		}
		else {
			addViewPage(HyadesUtil.getMofObject());
		}
	}
	/**
	 * Returns the current page in the view
	 * 
	 * @return org.eclipse.eclipse.ui.part.IPage
	 */
	public IPage getCurrentPage() {
		if (currentPage == null || !(currentPage instanceof TraceViewerPage))
			return null;

		return currentPage;
	}
	/**
	 * Returns the TraceViewerPage instance for the given EObject
	 * @param mofObject an EObject instance
	 * @return a TraceViewerPage instances
	 */
	public TraceViewerPage getPage(EObject mofObject) {

		if (mofObject != null) {
			Object page = _pages.get(mofObject);
			if (page != null)
				return (TraceViewerPage) page;
		}

		return null;
	}

	/**
	 * @see IWorkbenchPart#dispose()
	 */
	public void dispose() {
		TraceUIManager.getTraceUIManager().removeSelectionListener(this);
		TraceUIManager.getTraceUIManager().removeProfileEventListener(this);
		TraceUIManager.getTraceUIManager().removeDeleteListener(this);

		getSite().getPage().removePartListener(fPartListener);

		Enumeration pages = _pages.elements();
		while (pages.hasMoreElements()) {
			((IPage) pages.nextElement()).dispose();
		}

		_pages.clear();

		if (book != null) {
			book.dispose();
			book = null;
		}

		if (currentPage != null) {
			currentPage.dispose();
			currentPage = null;
		}

		if (defaultPage != null) {
			defaultPage.dispose();
			defaultPage = null;
		}

		fPartListener = null;

		super.dispose();
	}

	/**
	 * Sets where the menu has been initialized
	 * 
	 * @param init a boolean
	 */
	public void initializedMenu(boolean init) {
		_initializedMenu = init;
	}
	
	/**
 	 * Answer true if we want to track the given page.
	 * @param part an IWorkbenchPart instance
	 * @return a boolean
	 */
	protected boolean isImportant(IWorkbenchPart part) {
		//We only care about editors
		return true;
	}
	/**
	 * Returns whether the menu has been initialized
	 * 
	 * @return boolean
	 */
	public boolean isInitializedMenu() {
		return _initializedMenu;
	}
	/**
	 * Returns true if a save is needed for the view
	 * 
	 * @return a boolean
	 */
	public boolean isSaveNeeded() {
		return false;
	}
	
	/**
	 * This method is called to create the actions for the viewer 
	 */
	public void makeActions() {

		createReportAction();
		IToolBarManager tbm = getViewSite().getActionBars().getToolBarManager();
		tbm.removeAll();
		tbm.add(_printColumns);
		
	}
	
	public void createReportAction(){
		_printColumns = new OpenTraceReportAction(new StructuredSelection(this));		
	}
	
	/**
	 * Refreshes the page for the page associated with the given object
	 * 
	 * @param obj java.lang.Object
	 */
	protected void refreshPage(Object obj) {

		if (_pages == null)
			return;

		if (obj != null) {
			Object page = _pages.get(obj);
			if(page == null )
			{
				fRefreshView = true;
				return;
			}
			if (page instanceof TraceViewerPage) {
//				INavigator navigator = TPTPUtil.getActiveNavigator();
//				if (navigator != null && navigator.isLinkingEnabled())
				if(!((TraceViewerPage) page).isDisposed()){	 
					((TraceViewerPage) page).refreshPage();
				}

				if (((TraceViewerPage) page).isEmpty() || ((TraceViewerPage) page).isDisposed()) {
					showPage(defaultPage);
				} else {
					showPage((TraceViewerPage) page);
				}
			}
		}

	}

	/**
	 * Notify the viewer the selection has changed to the given object
	 * 
	 * @param obj java.lang.Object
	 */
	protected void selectionChanged(Object obj) {

		if (obj == null)
			return;

		Object page = _pages.get(obj);

		if (page != null && page instanceof TraceViewerPage) {
			((TraceViewerPage) page).selectionChanged();
		}

	}

	/**
	 * Removes the page associated with the given object
	 * 
	 * @param obj an Object
	 */
	public void removePage(Object obj) {

		if (obj == null || !(obj instanceof EObject))
			return;

		Object page = _pages.get(getObjectToView((EObject) obj));
		_pages.remove(obj);

		if (page != null) {
			if (page == currentPage) {
				showPage(defaultPage);
			}

			((IPage) page).dispose();
		}
	}
	
	/**
	 * Requests the part to take focus within the desktop.
	 */
	public void setFocus() {
	}
	
	/**
	 * Shows the given page
	 */
	public void showPage(IPage page) {
		book.showPage(page.getControl());
		currentPage = page;
		
		/* https://bugs.eclipse.org/bugs/show_bug.cgi?id=196332 */
		if (currentPage == defaultPage) {
			setViewTitle(null);
			disableToolBarActions();
		}
		else {
			enableToolBarActions();
		}

	}

	/**
	 * Selection has changed in the Monitors view (PDProjectExplorer)
	 */
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		fRefreshView = true;

		boolean openView = ((INavigator) part).isLinkingEnabled();
		if (!fPartVisible || part == null || part.getSite().getWorkbenchWindow() != getSite().getWorkbenchWindow()) {
			return; //get notifications only from the current workbench window
		}

		if (!openView) {
			//			fSelectionChanged = true;
			return;
		}
		
		if (selection == null || selection.isEmpty() || !(selection instanceof IStructuredSelection)) {
			return;
		}

		Object obj = ((IStructuredSelection) selection).getFirstElement();
		if (obj == null || !(obj instanceof EObject)) {
			return;
		}
		else {
			if(previousSelection != obj.hashCode())
			{
				fRefreshView = true;
				previousSelection = obj.hashCode();
			}
			else
				fRefreshView=false;
			
			addViewPage((EObject) obj);
		}
	}

	/**
	 * Returns the title of the view for this viewer 
	 * @return a String
	 */
	public abstract String getViewTitle();

	/**
	 * This method applies the appropriate function on this viewer for the given ProfileEvent
	 * 
	 * @param event a ProfileEvent instance
	 */
	public void handleProfileEvent(ProfileEvent event) {

		int type = event.getType();

		if (type == ProfileEvent.REFRESH_VIEWS)
		{
			if (!fPartVisible) {
				//ignore the events and keep track that the page need to be updated
				fRefreshView = true;
				return;
			} else {
				fRefreshView = false;
	
				if (event.getSource() instanceof EObject) {
					INavigator nav = HyadesUtil.getActiveNavigator();
					if (nav == null || nav.isLinkingEnabled() || _mofObject == null) {
						refreshPage(getObjectToView((EObject)event.getSource()));
					}
					else {
						
						if (_mofObject != event.getSource())
						{
							ProfileEvent newEvent = TraceUIManager.getTraceUIManager().getUpdateModelEvent(_mofObject);
							TraceUIManager.getTraceUIManager().notifyProfileEventListener(newEvent);
						}
						
						refreshPage(getObjectToView(_mofObject));
					}
				}
			}
		}
		else if (type == ProfileEvent.CONTEXT_HANDLER_SELECTION_CHANGED)
		{
			// we received a context handler selection changed event, in the case when
			//   the _mofObject of this view is non-null, we attempt a refresh event if it's
			//   visible, and if it's not visible simply set the contextHandlerSelectionChanged
			//   flag to true so that it can be updated when the page is loaded.  If the mofObject
			//   is null, we don't load the page since no data was loaded for it.
			if (_mofObject != null)
			{
				contextHandlerSelectionChanged = true;

				if (fPartVisible)
				{
					refreshPage(getObjectToView(_mofObject));
			    	contextHandlerSelectionChanged = false;
				}
			}
		}
	}

	/** 
	 * Sets the display view title for the given selection Object
	 * 
	 * @param selection an Object, the selection the view title is displayed for
	 */
	protected void setViewTitle(Object selection) {
		if (selection != null) {
			if (selection instanceof TRCAgentProxy) {
				TRCAgentProxy a = (TRCAgentProxy) selection;
				setContentDescription(getViewTitle() + " - " + _labelProvider.getText(a.getProcessProxy()));
				return;

			}
		}
		setContentDescription(getViewTitle());
	}

	/**
	 * Initializes the action bar for this viewer
	 */
	public void initializeActionBar() {
	}

	/**
	 * Sets the record selection for this viewer
	 * @param record an EObject, the record
	 * @param treeSel an EObject, the tree selection
	 */
	public void setRecordSelection(EObject record, EObject treeSel) {
	};

	/**
	 * Returns true of this viewer processes refresh events.
	 * 
	 * @return a boolean
	 */
	public boolean isProcessRefreshEvents() {
		return fPartVisible;
	}

	/**
	 * Refresh the visible page when the view has become visible
	 *  
	 */
	protected void refreshVisiblePage() {
		EObject obj = HyadesUtil.getMofObject();
		addViewPage(obj);
		EObject sel = getObjectToView(obj);
		refreshPage(sel);

		selectionChanged(sel);
	}


	/**
	 * Invoked when a top hierarchy object is deleted from the model.  Deregisters the given object.
	 */
	public void deregister(Object obj) {
		removePage(obj);
	}


	public boolean isFRefreshView() {
		return fRefreshView;
	}

	public void setFRefreshView(boolean refreshView) {
		fRefreshView = refreshView;
	}

	public boolean isFPartVisible() {
		return fPartVisible;
	}

	public void setFPartVisible(boolean partVisible) {
		fPartVisible = partVisible;
	}
	
	/**
	 * 
	 * @return control containing the data to be exported
	 * Subclasses should override this method and return the desired control to be exported
	 */
    public Object[] getViewerControls()
    {
    	return null;
    }
	
	/**
	 * 
	 * @return view title
	 */
    public String getViewerTitle()
    {
    	return getContentDescription();
    }
    
	/**
	 * 
	 * @return viewer associated with this view, if applicable
	 * can be a TreeViewer, TableTreeViewer or null if the view is not using a viewer
	 */
    public Viewer[] getExportViewer()
    {
    	return null;
    }
    
    public boolean contextHandlerSelectionChanged()
    {
    	return contextHandlerSelectionChanged;
    }
    
    /**
     * 
     * @return the model input for this view
     */
    public Object getModelObject()
    {
		if (currentPage == null || !(currentPage instanceof TraceViewerPage))
			return null;

		return ((TraceViewerPage)currentPage).getMOFObject();
    }
  
	/**
	 * Returns the default page message.
	 * 
	 */
	public String getDefaultPageMessage() 
	{		
		return CommonUITraceMessages.NODATA;
	}
    
	
	/**
	 * @see IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
	 */
	public void menuAboutToShow(IMenuManager menu) {

		if (getCurrentPage() != null && getCurrentPage() instanceof TraceViewerPage) {
			((TraceViewerPage) getCurrentPage()).fillContextMenu(menu);
		}

		menu.add(new Separator(GROUP_ADD_VIEWS));
		menu.add(new Separator(GROUP_GENERIC));

		menu.add(new Separator(GROUP_ADDITIONS));

	}
	
	public IPage getDefaultPage(){
		return defaultPage;
	}
	
	/*
	 * https://bugs.eclipse.org/bugs/show_bug.cgi?id=196332
	 * 
	 * This method is called when default page is shown
	 * Subclasses that contains additional toolbar actions
	 * may either override this method, or call the <code>super</code>
	 * implementation and add the logic for the additional actions.
	 */
	protected void disableToolBarActions() {
		if (_printColumns!=null) {
			_printColumns.setEnabled(false);
		}
	}

	/*
	 * https://bugs.eclipse.org/bugs/show_bug.cgi?id=196332
	 * 
	 * This method is called when default page is shown
	 * Subclasses that contains additional toolbar actions
	 * may either override this method, or call the <code>super</code>
	 * implementation and add the logic for the additional actions.
	 */	protected void enableToolBarActions() {
		if (_printColumns!=null) {
			_printColumns.setEnabled(true);
		}
	}

	public String getViewID() {
		return VIEW_ID;
	}

	public static EObject getObjectToView_(EObject selObject) {
		if (selObject == null)
			return selObject;
		if (selObject instanceof TRCProcessProxy) {
			int nbOfAgents = 0;
			TRCAgentProxy a = null;
			EList agents = ((TRCProcessProxy) selObject).getAgentProxies();
			for (int idx = 0; idx < agents.size(); idx++) {
				TRCAgentProxy agent = (TRCAgentProxy) agents.get(idx);
				if (agent == null || agent.eIsProxy())
					continue;
				if (agent.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE)) {
					nbOfAgents++;
					a = agent;
				}
			}
			if (nbOfAgents == 1) {
				return a;
			}
		}
		return selObject;
	}
	
	protected void handlePartVisible(){
		if (fRefreshView) {
			fRefreshView = false;
			INavigator navigator = HyadesUtil.getActiveNavigator();
			if (navigator != null && navigator.isLinkingEnabled())
			{
				fForceRefresh = false;
				refreshVisiblePage();
			}
		}
		
		if (fForceRefresh)
		{
			fForceRefresh = false;
			if (_mofObject != null)
				refreshPage(getObjectToView(_mofObject));
		}		
		
		// if the context selection has been changed we update the view
		//   if it has an mof object loaded, otherwise we don't force the view
		//   to load
		if (contextHandlerSelectionChanged)
		{
			if (_mofObject != null)
				refreshPage(getObjectToView(_mofObject));
			contextHandlerSelectionChanged = false;
		}
		
		propagateVisible();
	}

	protected void handlePartHidden(){
		propagateHidden();
	}
	
	protected void propagateVisible(){
		TraceViewerPage page = getPage(_mofObject);
		if(page != null){
			page.setVisible();
		}
	}

	protected void propagateHidden(){
		TraceViewerPage page = getPage(_mofObject);
		if(page != null){
			page.setHidden();
		}
	}
}
