/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceViewerPage.java,v 1.4 2008/04/17 07:49:38 aalexeev Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.part.Page;

/**
 * This class defines an abstract base class for Page instances shown in the trace model.
 * This is for the profile monitor navigator, log view naviator.
 */
public abstract class TraceViewerPage extends Page {
	protected TraceViewer _viewer;
	protected EObject _mofObject;
	protected Menu fContextMenu;
	protected boolean disposed = false;
	
	private boolean isVisible;

	/**
	 * The constructor
	 * 
	 * @param mofObject the EObject this page is associated with 
	 * @param viewer the TraceViewer this page is associated with
	 */
	public TraceViewerPage(EObject mofObject, TraceViewer viewer) {
		super();

		_viewer = viewer;
		_viewer.initializeActionBar();
		_mofObject = mofObject;
	}
	
	/**
	 * Returns the EObject instance this page is associated with.
	 * @return an EObject
	 */
	public EObject getMOFObject() {
		return _mofObject;
	}
	
	/**
	 * Returns the TraceViewer instance this page is associated with.
	 * @return a TraceViewer
	 */
	public TraceViewer getTraceViewer() {
		return _viewer;
	}
	
	/**
	 * Refreshes this page.
	 */
	public abstract void refreshPage();
	
	/**
	 * Notifies this page that the selection has changed.
	 */
	public abstract void selectionChanged();

	/**
	 * Sets the EObject instance associated with this page.
	 * 
	 * @param object an EObject
	 */
	public void setMofObject(EObject object) {
		_mofObject = object;
	}
	
	/**
	 * Updates this page appropriately, given whether it is a new page. 
	 *  
	 * @param newPage a boolean
	 */
	public abstract void update(boolean newPage);

	/**
	 * Contribute to the view popup menu
	 * @param menu an IMenuManager
	 */
	public void fillContextMenu(IMenuManager menu) {
	}
	
	/**
	 * Set the context menu for this view page
	 * @param parent the parent Control
	 * @param selProvider the ISelectionProvider for the context menu
	 */
	public void setContextMenu(Control parent, ISelectionProvider selProvider)
	{
		if(parent == null || parent.isDisposed())
		  return;
		  
		MenuManager menuMgr= new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(_viewer);
		fContextMenu= menuMgr.createContextMenu(parent);
		parent.setMenu(fContextMenu);
	
		// Register viewer with site. This must be done before making the actions.
		IWorkbenchPartSite site= _viewer.getSite();
		site.registerContextMenu(menuMgr, selProvider);
		site.setSelectionProvider(selProvider);
		  
	}
	
	/**
	 * @see Page#dispose()
	 */
	public void dispose()
	{
		super.dispose();
		
		_mofObject = null;  
		
		if(fContextMenu != null)
		   fContextMenu.dispose();
	   
		disposed = true;

	}
	
	/**
	 * Returns true if the page is empty and is to show no data.
	 * 
	 * @return a boolean
	 */
	public boolean isEmpty()
	{
		//assume page is not empty
		return false;
	}

	public boolean isDisposed() {
		return disposed;
	}
	
	final void setVisible(){
		if(isVisible)
			return;
		
		isVisible = true;
		handleVisible();
	}

	final void setHidden(){
		if(!isVisible)
			return;
		
		isVisible = false;
		handleHidden();
	}

	public final boolean isVisible(){
		return isVisible;
	}
	
	protected void handleVisible(){
	}

	protected void handleHidden(){
	}
}
