/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DeleteAction.java,v 1.14 2008/04/10 00:51:20 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.actions;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.internal.util.DeleteUtil;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.SelectionProviderAction;

/*
 * CONTEXT_ID delt0000 for delete popup action
 */

public class DeleteAction extends SelectionProviderAction implements ISelectionChangedListener
{
	protected INavigator fViewer;

	protected ArrayList deletedPaths = new ArrayList();

	public final static String SD_FILE_EXTENSION = "trcdbxmi";
	public final static String SYMPTOM_FILE_EXTENSION = "symptom";

	/**
	 * This dialog box asks the user whether they also want to delete the files saved on the 
	 * file system.  This dialog box is only presented when the user saves their data and tries
	 * to delete it.  If they haven't saved their data and try to delete their resources, they're
	 * presented by the dialog box in the ConfirmDeletionDialog class.
	 * 
	 */
	class DeleteResourceDialog extends MessageDialog
	{

		protected boolean deleteContent = false;

		protected Button radio1;

		protected Button radio2;

		DeleteResourceDialog(Shell parentShell)
		{
			super(parentShell, CommonUITraceMessages.DEL_TITLE, null, CommonUITraceMessages.CONFIRM_DELM, MessageDialog.QUESTION, new String[] { IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL }, 0);
			// yes is the default
		}

		protected Control createCustomArea(Composite parent)
		{
			Composite composite = new Composite(parent, SWT.NONE);
			composite.setLayout(new GridLayout());
			radio1 = new Button(composite, SWT.RADIO);
			radio1.addSelectionListener(selectionListener);
			String text1 = CommonUITraceMessages.DEL_RSP;
			radio1.setText(text1);

			radio2 = new Button(composite, SWT.RADIO);
			radio2.addSelectionListener(selectionListener);
			String text2 = CommonUITraceMessages.NDEL_RSP;
			radio2.setText(text2);

			// set initial state
			radio1.setSelection(deleteContent);
			radio2.setSelection(!deleteContent);

			return composite;
		}

		protected SelectionListener selectionListener = new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e)
			{
				Button button = (Button) e.widget;
				if (button.getSelection())
				{
					deleteContent = (button == radio1);
				}
			}
		};

		public boolean getDeleteContent()
		{
			return deleteContent;
		}
	}
	
	/**
	 * This dialog box is presented to the user when they try to delete the resources that 
	 * haven't been saved.  It's a simple dialog box confirming the deletion.
	 * This class was created as a result of bug 158058.
	 * 
	 * @author Navid Mehregani
	 *
	 */
	class ConfirmDeletionDialog extends MessageDialog
	{
		ConfirmDeletionDialog(Shell parentShell, String title, String message)
		{
			super(parentShell, title, null, message, MessageDialog.QUESTION, new String[] { IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL }, 0);			
		}

		protected Control createCustomArea(Composite parent)
		{
			Composite composite = new Composite(parent, SWT.NONE);
			composite.setLayout(new GridLayout());

			return composite;
		}
	}

	public DeleteAction(INavigator viewer)
	{
		super(viewer.getViewer(), CommonUITraceMessages.DEL);
		setDescription(CommonUITraceMessages.DEL_R);
		setToolTipText(CommonUITraceMessages.DEL_OTIP);

		ISharedImages images = CommonUITracePlugin.getDefault().getWorkbench().getSharedImages();
		setImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));

		PlatformUI.getWorkbench().getHelpSystem().setHelp(this, CommonUITracePlugin.PLUGIN_ID + ".delt0000");

		fViewer = viewer;
	}

	public void run()
	{
		final IStructuredSelection fSelections = (IStructuredSelection) fViewer.getViewer().getSelection();
		if (fSelections == null || fSelections.size() == 0)
			return;

		for (Iterator iterator = fSelections.iterator(); iterator.hasNext();) {
			Object sel  = iterator.next();
			if (!validForDelete(sel))
				return;			
		}
		
		
		/*
		 * Navid Mehregani
		 * Bugzilla 158058: First we need to determine, which dialog box we should present the user with.
		 * There are two dialog boxes. The first one is just a confirmation of the deletion; the second one
		 * asks the user whether they want to also delete the content from the file system. The second one
		 * should only be presented when the user tries to delete their *saved* resources.  The first one
		 * is presented when the user tries to delete *unsaved* resources.
		 */
		
		/* Determine the location of the corresponding file for the resource that's to be deleted */
		final String resource = ":/resource";
		String workspaceLocation = Platform.getLocation().toString();
		String selectedItem = null;
		String fullPath = null; /* Will store the absolute path to the corresponding file of the resource that's to be deleted */
		
		/* true if we're suppose to show the dialog box that asks whether the corresponding content on the
		 * file system should also be deleted.  false otherwise.*/
		boolean showDeleteFileContentDialog = true;
		
		/* true if only log files are being deleted, false otherwise */
		boolean LogDeleteDialog = true;
		
		boolean isSymptomCatalogResourceIncluded = false;
		boolean isOnlyDeletingSymptomCatalogResource = true;

		/* Navid Mehregani - bugzilla 158564: A different dialog box should be displayed if we're only deleting
		 * log files.  This was brought up as a usability issue in the RCP log analyzer. */
		for (Iterator iterator = fSelections.iterator(); iterator.hasNext();)
		{
			final Object sel = iterator.next();

			if ((sel != null) && (sel instanceof EObject))
			{
				isOnlyDeletingSymptomCatalogResource = false;

				Vector agentTypeList = new Vector();
				
				if (sel instanceof TRCMonitor)
					HyadesUtil.getAgentType((TRCMonitor)sel, agentTypeList);
				else if (sel instanceof TRCNode)
					HyadesUtil.getAgentType((TRCNode)sel, agentTypeList);
				else if (sel instanceof TRCProcessProxy)
					HyadesUtil.getAgentType((TRCProcessProxy)sel, agentTypeList);
				else if (sel instanceof TRCAgentProxy)
					agentTypeList.add(HyadesUtil.getAgentType((TRCAgentProxy)sel));
				
				if (agentTypeList.size() == 0)
					LogDeleteDialog = false;
				
				for (int agentTypeIndex=0; agentTypeIndex < agentTypeList.size(); agentTypeIndex++)
				{
					String agentType = (String)agentTypeList.get(agentTypeIndex);
					
					/* If the agent is not of type 'logging', there are other things being deleted */
					if ((agentType == null) || (!agentType.equalsIgnoreCase("Logging")))
					{
						LogDeleteDialog = false;
						break;  /* Break out of the first loop */
					}
				}
			} else if ((sel != null) && (sel instanceof IFile)) {
				IFile file = (IFile) sel;
				String ext = file.getFileExtension();
				if (ext != null) {
					if (ext.equals(SD_FILE_EXTENSION) || ext.equals(SYMPTOM_FILE_EXTENSION)) {
						isSymptomCatalogResourceIncluded = true;
					} else {
						isOnlyDeletingSymptomCatalogResource = false;
					}
				} else {
					isOnlyDeletingSymptomCatalogResource = false;
				}
			} else {
				isOnlyDeletingSymptomCatalogResource = false;
			}
		}
		/* End of bugzilla 158564 */
		
		if (!LogDeleteDialog)
		{
			for (Iterator iterator = fSelections.iterator(); iterator.hasNext();)
			{
				final Object sel = iterator.next();
	
				if (sel != null)
				{
					if (sel instanceof IResource)
					{
						fullPath = ((IResource) sel).getLocation().toString();
					} 
					else if (sel instanceof EObject)
					{
						selectedItem = ((EObject)sel).eResource().getURI().toString();
						fullPath = workspaceLocation + selectedItem.substring(selectedItem.indexOf(resource) + resource.length());
						
						/* Navid Mehregani - bugzilla 161453: In the large log support, a file is created regardless of whether the log file is saved or not */
						if (!((new File(fullPath)).exists()))
						{
							if ((sel instanceof TRCMonitor) && (areThereFilesOnDisk((TRCMonitor) sel)))
							{
								showDeleteFileContentDialog = true;
								break;
							}				
							else if ((sel instanceof TRCNode) && (areThereFilesOnDisk(((TRCNode) sel).getMonitor())))
							{
								showDeleteFileContentDialog = true;
								break;
							}				
							else if ((sel instanceof TRCProcessProxy) && (areThereFilesOnDisk(((TRCProcessProxy) sel).getNode().getMonitor())))
							{
								showDeleteFileContentDialog = true;
								break;
							}
							else if ((sel instanceof TRCAgentProxy) && (isAgentSavedOnDisk((TRCAgentProxy) sel)))
							{
								showDeleteFileContentDialog = true;
								break;
							}	
							else if ((sel instanceof CorrelationContainerProxy) && (areThereFilesOnDisk(((CorrelationContainerProxy) sel).getMonitor())))
							{
								showDeleteFileContentDialog = true;
								break;
							}									
						}
						/* End of bugzilla 161453 */
					}
						
				}
				
				if (fullPath != null) 
					showDeleteFileContentDialog = (new File(fullPath)).exists();
				
				if (showDeleteFileContentDialog)
					break; /* Break out of the for loop if we've found at least one associated file on disk */
			}
		}
		
		boolean deleteContents = false;
		
		if (LogDeleteDialog)
		{
			/* N.M - bugzilla 158564 */
			String message = null;
			if (isSymptomCatalogResourceIncluded) {
				if (isOnlyDeletingSymptomCatalogResource) {
					message = CommonUITraceMessages.CONFIRM_SYMPTOMCATALOG_DEL;
				} else {
					message = CommonUITraceMessages.CONFIRM_DELM;
				}
			} else {
				message = CommonUITraceMessages.CONFIRM_LOG_DEL;
			}
			ConfirmDeletionDialog confirmDeletionDialog = new ConfirmDeletionDialog(fViewer.getViewSite().getShell(), CommonUITraceMessages.DEL_TITLE, message);
			if (confirmDeletionDialog.open() != 0)
				return;
			
			deleteContents = true;
			/* End of bugzilla 158564 */
		}		
		/* Show the dialog box that asks whether the content on the file system should also be deleted */
		else if (showDeleteFileContentDialog)
		{
			DeleteResourceDialog dlg = new DeleteResourceDialog(fViewer.getViewSite().getShell());
			if (dlg.open() != 0)
				return;
			deleteContents = dlg.getDeleteContent();
		}
		/* Show the dialog box that simply confirms whether the user is sure they want to delete the selected resource(s)*/
		else
		{
			ConfirmDeletionDialog confirmDeletionDialog = new ConfirmDeletionDialog(fViewer.getViewSite().getShell(), CommonUITraceMessages.DEL_TITLE, CommonUITraceMessages.CONFIRM_DELM);
			if (confirmDeletionDialog.open() != 0)
				return;
		}
		
		/* End of bugzilla 158058 */
		final boolean fdeleteContents = deleteContents;
		Runnable refreshRunnable = new Runnable(){
			public void run() {
				BusyIndicator.showWhile(Display.getDefault(), new Runnable(){
					public void run(){ 
						runDeleteTask(fSelections, fdeleteContents);				
}
				});
			};
		};
		Display.getDefault().asyncExec(refreshRunnable);		

		

		ProfileEvent event = new ProfileEvent();
		event.setSource(null);
		event.setType(ProfileEvent.CLEAN_UP);
		TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		event = TraceUIManager.getTraceUIManager().getProfileEvent();
		if (event == null)
			event = new ProfileEvent();
		event.setSource(null);
		event.setType(ProfileEvent.UNSPECIFIED);
		TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
	}
	
	/**
	 * Convenient method to determine if there are any corresponding files on disk for the given monitor
	 * 
	 * @param monitor  The monitor to check
	 * @return  true if there are any files on disk for the corresponding monitor; false otherwise.
	 */
	protected boolean areThereFilesOnDisk(TRCMonitor monitor)
	{
		if (monitor == null)
			return false;
		
		/* Step through all the agents of a monitor and return true as soon as a corresponding file for 
		 * one of its agents is found on disk */
		Iterator i = monitor.getNodes().iterator();
		while (i.hasNext()) {
			TRCNode node = (TRCNode)i.next();
			Iterator j = node.getProcessProxies().iterator();
			while (j.hasNext()) {
				TRCProcessProxy process = (TRCProcessProxy)j.next();
				Iterator k = process.getAgentProxies().iterator();
				while (k.hasNext()) {
					TRCAgentProxy agent = (TRCAgentProxy)k.next();
					
					if (isAgentSavedOnDisk(agent))
						return true;
				}				
			}
		}
		
		return false;
	}
	
	/**
	 * Helper method to determine whether a corresponding file for the given agent exists on disk
	 * 
	 * @param agentProxy  The agent to check for
	 * @return  True if a corresponding file exists for the given agent; false otherwise.
	 */
	protected boolean isAgentSavedOnDisk(TRCAgentProxy agentProxy)
	{
		try {
			final String RESOURCE = ":/resource";
			String workspaceLocation = Platform.getLocation().toString();		
			String agentPath = agentProxy.getAgent().eResource().getURI().toString();
			String fullPath = workspaceLocation + agentPath.substring(agentPath.indexOf(RESOURCE) + RESOURCE.length());
			
			if ((new File(fullPath)).exists()) 
				return true;
		}
		catch (Exception e) {
			if(ModelDebugger.INSTANCE.debug)
			{
			  ModelDebugger.log(e);
			}
			return false;
		}
		return false;
	}

	/**
	 * Method containsPath.
	 * 
	 * @param innerPath
	 * @return boolean
	 */
	protected boolean containsPath(IPath innerPath)
	{
		for (Iterator iter = deletedPaths.iterator(); iter.hasNext();)
		{
			IPath path = (IPath) iter.next();

			if (path.isPrefixOf(innerPath))
				return true;
		}

		return false;
	}

	protected boolean checkContainer(IContainer container, int deep)
	{
		if (deep != 0)
		{
			if (deep != -1)
				--deep;
			try
			{
				IResource[] resources = container.members();
				for (int j = 0; j < resources.length; j++)
				{
					if (resources[j] instanceof IContainer)
					{
						if (!checkContainer((IContainer) resources[j], deep))
							return false;
					}
				}
			} catch (CoreException e)
			{
				e.printStackTrace();
				return false;
			}
		}
		return true;
	}

	protected boolean validForDelete(Object sel)
	{
		if (sel != null)
		{
			if (sel instanceof IContainer)
			{
				return checkContainer((IContainer) sel, -1);
			} else if (sel instanceof TRCAgentProxy)
			{
				if (!checkAgentState((TRCAgentProxy) sel))
					return false;
			} else if (sel instanceof TRCProcessProxy)
			{
				if (!checkAgentsState(((TRCProcessProxy) sel).getAgentProxies()))
					return false;
			} else if (sel instanceof TRCNode)
			{
				if (!checkAgentsState(((TRCNode) sel).getProcessProxies()))
					return false;
			} else if (sel instanceof TRCMonitor)
			{
				if (!checkAgentsState(((TRCMonitor) sel).getNodes()))
					return false;
			}
			return true;
		} else
			return true;
	}

	/**
	 * Method checkAgentState.
	 * 
	 * @param eList
	 */
	protected boolean checkAgentsState(EList eList)
	{
		for (Iterator iter = eList.iterator(); iter.hasNext();)
		{
			Object temp = iter.next();
			if (temp instanceof TRCAgentProxy)
			{
				TRCAgentProxy agent = (TRCAgentProxy) temp;
				if (agent.eIsProxy())
					continue;
				if (!checkAgentState(agent))
					return false;
			} else if (temp instanceof TRCProcessProxy)
			{
				if (!validForDelete(temp))
					return false;
			} else if (temp instanceof TRCNode)
			{
				if (!validForDelete(temp))
					return false;
			}
		}
		return true;
	}

	protected boolean checkAgentState(TRCAgentProxy a)
	{
		if ((a.isActive() && (a.isAttached() || a.isMonitored())) || a.isCollectionData())
		{
			treatActiveAgentWhenDelete(a);
			return false;
		}
		return true;
	}

	protected void treatActiveAgentWhenDelete(TRCAgentProxy a)
	{

		final Shell shell = Display.getDefault().getActiveShell();
		if (shell != null)
		{
			final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, NLS.bind(CommonUITraceMessages.DELAG_ERR, a.getName()), null);
			shell.getDisplay().asyncExec(new Runnable()
			{
				public void run()
				{
					ErrorDialog.openError(shell, CommonUITraceMessages.TRC_MSGT, CommonUITraceMessages.DELF_ERRT, err);
				}
			});
		}
	}

	/**
	 * Let target decide.
	 */
	public void selectionChanged(IStructuredSelection selection)
	{
	}

	/**
	 * 
	 */
	public void dispose()
	{

		super.dispose();

		fViewer = null;
		if (deletedPaths != null)
			deletedPaths.clear();
		deletedPaths = null;
	}
	
	protected void runDeleteTask(IStructuredSelection fSelections, boolean deleteContents){		
		deletedPaths.clear();
		DeleteUtil.setDeleteInProgress(true);
		try
		{
			for (Iterator iterator = fSelections.iterator(); iterator.hasNext();)
			{
				final Object sel = iterator.next();

				if (sel != null)
				{					
					if (sel instanceof IResource)
					{
						if (!containsPath(((IResource) sel).getFullPath()))
						{
							DeleteUtil.deleteResource((IResource) sel, deleteContents, deleteContents, fViewer.getID());
							deletedPaths.add(((IResource) sel).getFullPath());
						}
					} 
					else if (sel instanceof TRCMonitor){
						DeleteUtil.deleteMonitor((TRCMonitor) sel, true, deleteContents, deleteContents, fViewer.getID());
						TraceUIManager.getTraceUIManager().removeSelectionModel((TRCMonitor) sel);
					}
					else if (sel instanceof TRCNode){
						DeleteUtil.deleteNode((TRCNode) sel, true, deleteContents, deleteContents, fViewer.getID());
						TraceUIManager.getTraceUIManager().removeSelectionModel((TRCNode) sel);
					}
					else if (sel instanceof TRCAgentProxy){						
						DeleteUtil.deleteAgent((TRCAgentProxy) sel, true, deleteContents, deleteContents, fViewer.getID());
						TraceUIManager.getTraceUIManager().removeSelectionModel((TRCAgentProxy) sel);
					}
					else if (sel instanceof TRCProcessProxy){
						DeleteUtil.deleteProcess((TRCProcessProxy) sel, true, deleteContents, deleteContents, fViewer.getID());
						TraceUIManager.getTraceUIManager().removeSelectionModel((TRCProcessProxy) sel);
					}
					else if (sel instanceof CorrelationContainerProxy){
						DeleteUtil.deleteCorrelationContainer((CorrelationContainerProxy) sel, true, deleteContents, deleteContents, fViewer.getID());
						TraceUIManager.getTraceUIManager().removeSelectionModel((CorrelationContainerProxy) sel);
					}
					else if (sel instanceof INavigatorItem)
						((INavigatorItem) sel).delete(true, deleteContents);

				}
			}
		} catch (Exception e)
		{
			e.printStackTrace();
		}
		DeleteUtil.setDeleteInProgress(false);
		fSelections = null;
		deletedPaths.clear();
	}


}
