/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.navigator;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.trace.internal.ui.PDLabelProvider;
import org.eclipse.hyades.ui.internal.extension.INavigatorLayout;
import org.eclipse.hyades.ui.internal.navigator.Navigator;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;

/**
 * Presents the contents of the Profiling Monitor view in a simple, or
 * application-oriented layout. The projects, monitors and nodes are hidden,
 * and the process labels are more descriptive (they indicate which host on
 * which the process is running).
 * 
 * @author Curtis d'Entremont
 * @since 3.3
 */
public class SimpleLayout implements INavigatorLayout {
	
	protected IContentProvider _contentProvider;
	protected ILabelProvider _labelProvider;
	protected Navigator _viewer;
	
	/**
	 * Constructs a new FlatLayout for the given Profiling Monitor view
	 * instance.
	 * 
	 * @param viewer the Profiling Monitor view.
	 */
	public SimpleLayout(Navigator viewer) {
		_viewer = viewer;
	}
	
	/**
	 * Returns this layout's content provider, which does not supply
	 * projects, monitors, or nodes. The content provider is lazily
	 * loaded and cached.
	 * 
	 * @return this layout's content provider.
	 * @see org.eclipse.hyades.ui.internal.extension.INavigatorLayout#getContentProvider()
	 */
	public IContentProvider getContentProvider() 
	{
		if (_contentProvider == null) 
		{		
			_contentProvider = new SimpleContentProvider(_viewer.getViewContentProvider());
		}
		return _contentProvider;
	}
	
	/**
	 * Returns this layout's label provider, which appends an extra bit
	 * of information to the process labels which indicates on which host the
	 * process is running. The label provider is lazily loaded and cached.
	 * 
	 * @return this layout's label provider.
	 * @see org.eclipse.hyades.ui.internal.extension.INavigatorLayout#getLabelProvider()
	 */
	public ILabelProvider getLabelProvider() {
		if (_labelProvider == null) {
			
			if(_viewer == null)
				_labelProvider = new PDLabelProvider();
			else
				_labelProvider = _viewer.getViewLabelProvider();
		}
		return _labelProvider;
	}
}

/**
 * Hides monitors, nodes, and possibly projects, depending on the user preference.
 * This provides an application-oriented view, which is useful when all the processes
 * are running on the same machine.
 */
class SimpleContentProvider extends AbstractFilteredContentProvider {

	protected IPreferenceStore _store = CommonUITracePlugin.getDefault().getPreferenceStore();
	
	/**
	 * Constructs a new simple view of the profiling elements in the profiling
	 * monitor view.
	 * 
	 * @param provider The underlying content provider that this provider filters.
	 */
	public SimpleContentProvider(ITreeContentProvider provider) {
		super(provider);
	}
	
	/**
	 * Returns true if the given element and all its children should be filtered
	 * out entirely.
	 * 
	 * @param element The element to check.
	 * @return Whether or not to filter out the element and all its children.
	 * @see org.eclipse.hyades.trace.ui.internal.navigator.AbstractFilteredContentProvider#isFiltered(java.lang.Object)
	 */
	public boolean isFiltered(Object element) {
		return false;
	}
	
	/**
	 * Returns true if the given element (but not its children) should be filtered
	 * out, or "skipped". In this case, monitors and nodes are always skipped, and
	 * projects are skipped when the user preference for this is selected.
	 * 
	 * @param element The element to check.
	 * @return Whether or not to skip the given element.
	 * @see org.eclipse.hyades.trace.ui.internal.navigator.AbstractFilteredContentProvider#isSkipped(java.lang.Object)
	 */
	public boolean isSkipped(Object element) {
		if (element instanceof IProject) {
			return !_store.getBoolean(CommonUITraceConstants.DISPLAY_PROJECTS);
		}
		return (element instanceof TRCMonitor) || (element instanceof TRCNode) || (element instanceof IFolder);
	}
}
