/**********************************************************************
 * Copyright (c) 2005, 2011 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLTraceDataProcessor.java,v 1.12 2011/01/17 21:47:49 mreid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.piclient;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.net.InetAddress;
import java.util.Date;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.internal.execution.local.common.ExtendedDataServerListener;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.loaders.util.InvalidXMLException;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.BinaryLoader;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.util.internal.InvalidEventException;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.internal.wizard.TraceWizardMessages;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;
import org.eclipse.tptp.platform.common.ui.trace.internal.util.PDCoreUtil;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.text.MessageFormat;

public class XMLTraceDataProcessor extends BinaryLoader implements ExtendedDataServerListener {
	//~ Instance fields
	// ----------------------------------------------------------------------------

	protected boolean newFile;

	protected static final String NEW_LINE = "\n";

	protected static final int BUFFER_SIZE = 64 * 1024;

	protected boolean firstEvent = true;

	//	private File profilingFile = null;
	protected OutputStream fileWriter = null;

	protected String profileFileName = null;

	protected final String START_TAG = "<TRACE>";

	protected final String END_TAG = "</TRACE>";

	protected final String XML_VERSION_TAG = "<?xml version=\"1.0\"?>";

	protected final String END_TAG_REPLACE = "\n<!-- new trace starts-->\n";

	protected final String START_TRACE = START_TAG + NEW_LINE;

	protected final String END_TRACE = END_TAG + NEW_LINE;

	protected final String XML_VERSION = XML_VERSION_TAG + NEW_LINE;

	protected static final char[] ID1_IN = new char[] { 'i', 'D', '1', '>' };

	protected static final char[] ID1_OUT = new char[] { 'i', 'D', '1', '<' };

	protected static final char[] ID2_IN = new char[] { 'i', 'D', '2', '>' };

	protected static final char[] ID2_OUT = new char[] { 'i', 'D', '2', '<' };

	protected static final String INCOMMING_DATA = "incommingData";

	protected boolean specialProcessing = true;

	protected long totalBytesRead = 0;

	protected long prevTotalBytesRead = 0;

	protected long startTime;

	protected boolean validTag;

	protected int tempRemaining;
	//~ Constructors
	// -------------------------------------------------------------------------------

	public XMLTraceDataProcessor(TRCAgent agent) {
		super(agent);
		setCollectionMode(agent.getCollectionMode().getValue());

		(new RefreshUI()).start();
	}

	/**
	 * XMLDataProcessor constructor comment.
	 */
	public XMLTraceDataProcessor(TRCAgentProxy agentProxy) {
		super(agentProxy);

		setCollectionMode(agentProxy.getCollectionMode().getValue());

		(new RefreshUI()).start();
	}

	/**
	 * XMLDataProcessor constructor comment.
	 */
	public XMLTraceDataProcessor(TRCMonitor monitor) {
		super(monitor);
		(new RefreshUI()).start();
	}

	//~ Methods
	// ------------------------------------------------------------------------------------

	/**
	 * Sets the fw.
	 * 
	 * @param fw
	 *            The fw to set
	 */
	public void setWriter(OutputStream fw) {
		this.fileWriter = fw;
	}

	/**
	 * Returns the fw.
	 * 
	 * @return Writer
	 */
	public OutputStream getWriter() {
		return fileWriter;
	}

	/**
	 * Sets the profileFileName.
	 * 
	 * @param profileFileName
	 *            The profileFileName to set
	 */
	public void setProfileFileName(String profileFileName) {
		this.profileFileName = profileFileName;
	}

	/**
	 * Returns the profileFileName.
	 * 
	 * @return String
	 */
	public String getProfileFileName() {
		return profileFileName;
	}

	//    /**
	//     * Sets the pFile.
	//     * @param pFile The pFile to set
	//     */
	//    public void setProfilingFile(File pFile) {
	//        this.profilingFile = pFile;
	//    }
	//
	//    /**
	//     * Returns the pFile.
	//     * @return File
	//     */
	//    public File getProfilingFile() {
	//        return profilingFile;
	//    }

	public void createWriter() {

		try {
			if (profileFileName != null) {
				fileWriter = makeWriter();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	protected String getDumpFileName(){
		return "TraceContents-" + getTimeStamp() + ".xml";
	}
	
	protected OutputStream makeWriter() throws IOException {
		File f = new File(profileFileName);
		if (f.exists()) {
			if (!PDCoreUtil.isZipFile(profileFileName)) {
				checkAndRemoveEndTag(profileFileName);
				newFile = false;
				return new BufferedOutputStream(new FileOutputStream(profileFileName, true), BUFFER_SIZE);
			} else {

				File newFileName = new File(new StringBuffer(f.getParentFile().getAbsolutePath()).append(System.getProperty("file.separator")).append("Temp-").append((new Date()).getTime()).toString());
				renameFile(f, newFileName);

				ZipOutputStream zipOutputStream = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(f), BUFFER_SIZE));
				ZipInputStream zipInputStream = new ZipInputStream(new BufferedInputStream(new FileInputStream(newFileName), BUFFER_SIZE));

				try
				{
					copyZip(zipInputStream, zipOutputStream);
				}catch (Exception e) {
					zipOutputStream.close();
					zipOutputStream = new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(f), BUFFER_SIZE));
				}

				zipInputStream.close();

				newFileName.delete();

				ZipEntry traceEntry = new ZipEntry(getDumpFileName());
				zipOutputStream.putNextEntry(traceEntry);

				newFile = true;

				return zipOutputStream;
			}
		} else {
			OutputStream fos = new BufferedOutputStream(new FileOutputStream(profileFileName));
			ZipOutputStream zipStream = new ZipOutputStream(fos);
			ZipEntry traceEntry = new ZipEntry(getDumpFileName());
			zipStream.putNextEntry(traceEntry);
			newFile = true;

			return zipStream;
		}
	}

	protected boolean renameFile(File f, File newFileName) {
		boolean ret = f.renameTo(newFileName);
		if (ret == false) {
			ret = copyFile(f, newFileName);
			f.delete();

		}
		return ret;
	}

	/**
	 * @param f
	 * @param newFileName
	 */
	protected boolean copyFile(File f, File newFileName) {
		try {
			BufferedInputStream bufferedInputStream = new BufferedInputStream(new FileInputStream(f), BUFFER_SIZE);
			BufferedOutputStream bufferedOutputStream = new BufferedOutputStream(new FileOutputStream(newFileName), BUFFER_SIZE);
			byte[] buffer = new byte[BUFFER_SIZE];
			while (bufferedInputStream.available() > 0) {
				bufferedInputStream.read(buffer);
				bufferedOutputStream.write(buffer);
			}
			bufferedInputStream.close();
			bufferedOutputStream.close();
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}

	/**
	 * @param zipInputStream
	 * @param zipStream
	 */
	protected void copyZip(ZipInputStream zipInputStream, ZipOutputStream zipOutputStream) throws IOException {
		ZipEntry zipEntry;
		int byteCount;
		byte[] data = new byte[BUFFER_SIZE];
		while ((zipEntry = zipInputStream.getNextEntry()) != null) {
			zipOutputStream.putNextEntry(zipEntry);
			while ((byteCount = zipInputStream.read(data, 0, BUFFER_SIZE)) != -1) {
				zipOutputStream.write(data, 0, byteCount);
			}
			zipOutputStream.closeEntry();
		}
	}

	/**
	 * @return
	 */
	protected String getTimeStamp() {
		return DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date()).replace(' ', '_').replace(':', '_').replace(',', '_');
	}

	public void dataServerExited() {
		try {
			if (getAgentProxy() != null) {
				prevTotalBytesRead = totalBytesRead;
				getAgentProxy().setCollectionData(false);
				//set the current time if not already set
				if (!getAgentProxy().eIsSet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_StopTime())) {
					LoadersUtils.setSessionStopTime(System.currentTimeMillis(), getAgentProxy());
				}

				endTrace(getWriter());
				setProfileFileName(null);
				//                setProfilingFile(null);
				setWriter(null);

				cleanUp();

				Display d = Display.getDefault();

				d.asyncExec(new Runnable() {
					public void run() {
						ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();

						event.setSource(getAgentProxy());
						event.setType(ProfileEvent.STOP_COLLECTING);
						TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
					}
				});
			}
		} catch (Exception exc) {
			//do nothing
			ModelDebugger.log(exc);
		}
	}

	public void dispose() {
		super.cleanUp();

		fileWriter = null;
	}

	public void endTrace(OutputStream randomFile) {
		try {
			if (randomFile != null) {
				randomFile.write(END_TRACE.getBytes());
				randomFile.flush();
				randomFile.close();
				randomFile = null;
			}
		} catch (IOException e) {
		}
	}

	protected boolean isValidTag(byte[] buf, int offset, int length) {
		if(validTag)
			return true;
		if (startsWith(buf, offset, length, START_TAG.getBytes()) || startsWith(buf, offset, length, XML_VERSION_TAG.getBytes()))
			return false;
		else
		{
			validTag=true;
			return true;
		}

	}

	/**
	 * @param container
	 * @param bytes
	 * @return
	 */
	protected boolean startsWith(byte[] container, int offset, int length, byte[] bytes) {
		if (length < bytes.length)
			return false;
		for (int i = 0; i < bytes.length; i++) {
			if (container[offset + i] != bytes[i]) {
				return false;
			}
		}
		return true;
	}

	protected void loadEventHook(byte[] buffer, int offset, int length) {
		if (firstEvent) {
			if (length > 1 && !(buffer[offset] == '<' && buffer[offset + 1] == '?')) {
				LoadersUtils.loadRootEvent(this);
			}
			firstEvent = false;
		}
		super.loadEvent(buffer, offset, length);
	}
	
	protected void loadEventHook(byte[] buffer, int length) {
		loadEventHook(buffer, 0, length);
	}
	
	/**
	 * Insert the method's description here. Creation date: (3/1/01 12:06:07 PM)
	 * 
	 * @param buffer
	 *            byte[]
	 * @param length
	 *            int
	 * @param peer
	 *            java.net.InetAddress
	 */
	public void incommingData(byte[] buffer, int offset, int length, java.net.InetAddress peer) {
		if (ModelDebugger.INSTANCE.debugEventValue) {
			ModelDebugger.INSTANCE.writeBinaryLog(INCOMMING_DATA, buffer, offset, length);
		}
		if (ModelDebugger.INSTANCE.debugEventFlow) {
			System.out.println(ID1_IN);
		}

		try {
			if ((getAgentProxy() != null) && getAgentProxy().isToProfileFile()) {
				totalBytesRead += length;
				//            	bufStr=new String(buffer, 0, length, "UTF8");
				if (isValidTag(buffer, offset, length)) {
					writeByte(buffer, offset, length);
				}
			} else {
				loadEventHook(buffer, offset, length);
			}
		} catch (InvalidXMLException e) {
		} catch (java.lang.OutOfMemoryError e) {
			handleOutOfMemoryError();
			stopAgent();
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(ID1_OUT);
				System.out.println(" = " + e.getLocalizedMessage());
			}

			ModelDebugger.log(e);
			throw e;
		}
		if (ModelDebugger.INSTANCE.debugEventFlow) {
			System.out.println(ID1_OUT);
		}
	}

	/**
	 * Insert the method's description here. Creation date: (3/1/01 12:06:07 PM)
	 * 
	 * @param buffer
	 *            char[]
	 * @param length
	 *            int
	 * @param peer
	 *            java.net.InetAddress
	 */
	public void incommingData(char[] buffer, int offset, int length, java.net.InetAddress peer) {
		if (ModelDebugger.INSTANCE.debugEventFlow) {
			System.out.println(ID2_IN);
		}

		byte[] newBuffer = new byte[length];

		for (int i = 0; i < length; i++) {
			newBuffer[i] = (byte) buffer[offset + i];
		}
		if (ModelDebugger.INSTANCE.debugEventValue) {
			ModelDebugger.INSTANCE.writeBinaryLog(INCOMMING_DATA, newBuffer, 0, length);
		}

		try {
			if ((getAgentProxy() != null) && getAgentProxy().isToProfileFile()) {
				totalBytesRead += length;
				//				bufStr=new String(newBuffer, "UTF8");
				if (isValidTag(newBuffer, 0, length)) {
					writeByte(newBuffer, 0, length);
				}

				//new emf
				//                super.loadEvent(newBuffer, length, false, true);
			} else {
				loadEventHook(newBuffer, length);
			}
		} catch (InvalidXMLException e) {
		} catch (java.lang.OutOfMemoryError e) {
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(ID2_OUT);
				System.out.println(" = " + e.getLocalizedMessage());
			}
			handleOutOfMemoryError();
			stopAgent();
			ModelDebugger.log(e);
			throw e;
		}
		System.out.println(ID2_OUT);

	}

	/**
	 * Insert the method's description here. Creation date: (3/1/01 12:06:07 PM)
	 * 
	 * @param data
	 *            byte[]
	 * @param length
	 *            int
	 * @param peer
	 *            java.net.InetAddress
	 */
	public void invalidDataType(byte[] data, int offset, int length, java.net.InetAddress peer) {
	}

	/**
	 * Insert the method's description here. Creation date: (3/1/01 12:06:07 PM)
	 */
	public void waitingForData() {
	}

	public void startTrace(OutputStream randomFile) {
		try {
			if (getAgentProxy() != null) {
				randomFile.write(START_TRACE.getBytes());
				randomFile.flush();
			}
		} catch (IOException e) {
		}
	}

	public void writeHeader(Writer randomFile) {
		// // try {
		//            if (getAgentProxy() != null) {
		//                startTrace(randomFile);
		//// writeNode(randomFile);
		//// writeProcessCreate(randomFile);
		//// writeAgentCreate(randomFile);
		//// writeTraceStart(randomFile);
		//// writeFilter(randomFile);
		//// writeOption(randomFile);
		//            }
		//// } catch (IOException e) {
		//// }
	}

	/*
	 * Write the necessary trace event to the head to the head of the file
	 */
	public void writeXMLVersion(OutputStream randomFile) {
		try {
			randomFile.write(XML_VERSION.getBytes());
			randomFile.flush();
		} catch (IOException e) {
		}
	}

	/**
	 *  
	 */
	protected TRCAgentProxy getAgentProxy() {
		return getContext().getAgentProxy();
	}

	protected void handleOutOfMemoryError() {
		/* Inform the user we are running out of memory */
		final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, CommonUITraceMessages.POUTMEM_ERRT, null);

		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				ErrorDialog.openError(new Shell(), CommonUITraceMessages.TRC_MSGT, CommonUITraceMessages.OUTMEM_ERRT, err);
			}
		});
	}

	protected synchronized void notifyListener(Agent a, TRCAgentProxy agent) {
		final TRCAgentProxy ag = agent;

		Display d = Display.getDefault();

		d.asyncExec(new Runnable() {
			public void run() {
				//update ui
				ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();

				event.setSource(ag);
				event.setType(ProfileEvent.STOP_MONITOR);
				TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
			}
		});
	}

	protected void stopAgent() {
		if (getAgentProxy() != null) {

			try {
				Agent ai = (Agent) LoadersUtils.locateAgentInstance(getAgentProxy());
				if (ai != null)
					ai.stopMonitoring();
			} catch (InactiveAgentException e) {
				e.printStackTrace();
			}
			getAgentProxy().setMonitored(false);
			getAgentProxy().setAttached(false);
			getAgentProxy().setActive(false);

			LoadersUtils.deregisterDataProcessor(getAgentProxy());

			if (fileWriter != null) {
				endTrace(fileWriter);
			}

			notifyListener((Agent) LoadersUtils.locateAgentInstance(getAgentProxy()), getAgentProxy());
		}
	}

	//    private void writeAgentCreate(Writer randomFile) throws IOException {
	//        if (getAgentProxy() != null) {
	//            randomFile.write(new StringBuffer("<agentCreate
	// agentId=\"").append(getAgentProxy().getRuntimeId()).append("\"
	// version=\"1.000").append("\"
	// processIdRef=\"").append(getAgentProxy().getProcessProxy().getRuntimeId()).append("\"
	// agentName=\"").append(getAgentProxy().getName()).append("\"
	// agentType=\"").append(getAgentProxy().getType()).append("\"
	// agentParameters=\"").append((getAgentProxy().getProcessProxy().getLaunchMode()
	// == 0) ? "server=controlled" : "").append("\" time=\"").append(new
	// Date().getTime() / 1000).append("\"/>\n").toString());
	//            randomFile.flush();
	//        }
	//    }

	protected void writeByte(byte[] data, int offset, int length) {

		try {
			if (fileWriter != null) {
				fileWriter.write(data, offset, length);
				fileWriter.write(NEW_LINE.getBytes());
				fileWriter.flush();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	//    private void writeConfigFilter(EList filters, Writer randomFile) throws
	// IOException {
	//        if (filters != null) {
	//            Iterator i = filters.iterator();
	//
	//            while (i.hasNext()) {
	//                TRCFilter filter = (TRCFilter) i.next();
	//                String method = filter.getOperation();
	//
	//                randomFile.write(new StringBuffer("<filter
	// pattern=\"").append(filter.getPattern()).append("\"
	// mode=\"").append(filter.getMode()).append("\"
	// genericPattern=\"SUFFIX\"/>\n").toString());
	//                randomFile.flush();
	//            }
	//        }
	//    }
	//
	//    private void writeConfigOption(EList options, Writer randomFile) throws
	// IOException {
	//        if (options != null) {
	//            Iterator it = options.iterator();
	//
	//            while (it.hasNext()) {
	//                TRCOption option = (TRCOption) it.next();
	//
	//                randomFile.write(new StringBuffer("<option
	// key=\"").append(option.getKey()).append("\"
	// value=\"").append(option.getValue()).append("\"/>\n").toString());
	//                randomFile.flush();
	//            }
	//        }
	//    }

	//    private void writeFilter(Writer randomFile) throws IOException {
	//        if (getAgentProxy() != null) {
	//            EList filters = null;
	//
	//            EList configs = getAgentProxy().getConfigurations();
	//            Iterator i = configs.iterator();
	//
	//            if (i.hasNext()) {
	//                TRCConfiguration config = (TRCConfiguration) i.next();
	//
	//                filters = config.getFilters();
	//
	//                writeConfigFilter(filters, randomFile);
	//
	//                return;
	//            }
	//
	//            writeConfigFilter(filters, randomFile);
	//        }
	//    }

	//    private void writeNode(Writer randomFile) throws IOException {
	//        if (getAgentProxy() != null) {
	//            TRCNode node = getAgentProxy().getProcessProxy().getNode();
	//
	//            if (node == null) {
	//                return;
	//            }
	//
	//            randomFile.write(new StringBuffer("<node
	// nodeId=\"").append(getAgentProxy().getProcessProxy().getNode().getRuntimeId()).append("\"
	// hostname=\"").append(node.getName()).append("\"
	// ipaddress=\"").append((node.getIpAddress() != null) ? node.getIpAddress()
	// : "-Unavailable-").append("\"
	// timezone=\"").append(node.getTimezone()).append("\" time=\"").append(new
	// Date().getTime() / 1000).append("\"/>\n").toString());
	//            randomFile.flush();
	//        }
	//    }

	//    private void writeOption(Writer randomFile) throws IOException {
	//        EList options = null;
	//
	//        EList configs = getAgentProxy().getConfigurations();
	//        Iterator i = configs.iterator();
	//
	//        if (i.hasNext()) {
	//            TRCConfiguration config = (TRCConfiguration) i.next();
	//
	//            options = config.getOptions();
	//
	//            writeConfigOption(options, randomFile);
	//        }
	//    }

	//    private void writeProcessCreate(Writer randomFile) throws IOException {
	//        if (getAgentProxy() != null) {
	//            TRCProcessProxy processCreate = getAgentProxy().getProcessProxy();
	//
	//            if (processCreate == null) {
	//                return;
	//            }
	//
	//            randomFile.write(new StringBuffer("<processCreate
	// processId=\"").append(processCreate.getRuntimeId()).append("\" pid=\"" +
	// processCreate.getPid()).append("\"
	// nodeIdRef=\"").append(processCreate.getNode().getRuntimeId()).append("\"
	// time=\"").append(new Date().getTime() / 1000).append("\"").append("
	// application_executable=\"").append(getApplicationExecutableString(processCreate)).append("\"/>\n").toString());
	//            randomFile.flush();
	//        }
	//    }

	//    private void writeTraceStart(Writer randomFile) throws IOException {
	//        if (getAgentProxy() != null) {
	//            randomFile.write(new StringBuffer("<traceStart traceId=\"").append("\"
	// agentIdRef=\"").append(getAgentProxy().getRuntimeId()).append("\"
	// time=\"\"/>\n").toString());
	//            randomFile.flush();
	//        }
	//    }

	//~ Inner Classes
	// ------------------------------------------------------------------------------

	class RefreshUI extends Thread {
		int collectingState;
		int prevProcessedFragments=0;
		long prevTime;
		
		// These three variables are used to calculate and report the bytes per second value
		
		/** The time that the bytes per second value was last calculated and updated */
		long bpsLastReportedTime = 0;
		
		/** The value of the total number of bytes variable when bytes/second was last calculated and updated*/
		long bpsLastReportedTotalBytes = 0;
		
		/** The last bytes per second time that was reported and calculated (this cached value is used instead, if 
		 * not enough time [< 1000 msec] has elapsed to make a proper calculation)*/
		int bpsLastBytesPerSecondTime = 0;
		
		public RefreshUI() {
			super("Profile UI");
		}

		public void updateStatusBar() {
			Display d = Display.getDefault();
			if(d==null || d.isDisposed())
				return;
			d.asyncExec(new Runnable() {
				protected long t;
				protected long t1;

				public void run() {
					INavigator navigator = HyadesUtil.getActiveNavigator();
					if (navigator == null)
						return;
					if (HyadesUtil.getMofObject() != getAgentProxy())
						return;

					t = System.currentTimeMillis();
					t1=(t-prevTime)/1000;
					prevTime=t;
					t = (t - startTime) / 1000;
					if (t == 0)
						t = 1;
					if (t1 == 0)
						t1 = 1;
					
					int deltaEvents = getProcessedFragments()-prevProcessedFragments;
					prevProcessedFragments=getProcessedFragments();

					// Bytes per second calculation code, start ------
					if(bpsLastReportedTime == 0) {
						bpsLastReportedTime = System.currentTimeMillis();
					}

					long reportedBytesDelta = totalBytesRead - bpsLastReportedTotalBytes;
					long reportedTimeDelta = System.currentTimeMillis() - bpsLastReportedTime;
					
					int bytesPerSecondResult = 0;
					
					// If less than 1 second has passed since we last did the calculation, then wait
					// until at least 1 second has passed (and use the previously reported value) 
					if(reportedTimeDelta < 1000) {
						// Use the last value, because a full second has not yet passed
						bytesPerSecondResult = bpsLastBytesPerSecondTime;

					} else {
						// Enough time has passed, so do the calculation and save the results
						bpsLastReportedTotalBytes = totalBytesRead;
						bpsLastReportedTime = System.currentTimeMillis();
						bytesPerSecondResult = (int)(1000.0*((float)reportedBytesDelta/(float)reportedTimeDelta));
						bpsLastBytesPerSecondTime = bytesPerSecondResult;
					}
					// Byte per second calculation code , end -----

					
					String statusBarMessage = MessageFormat.format(TraceWizardMessages.LIVE_PROGRESS, new String[] {"" + prevProcessedFragments, "" + (deltaEvents / t1), "" + t ,""+totalBytesRead,""+bytesPerSecondResult});
										

					IStatusLineManager statusLineManager = navigator.getViewSite().getActionBars().getStatusLineManager();
					statusLineManager.setMessage(statusBarMessage);
					if(ModelDebugger.INSTANCE.debug)
						ModelDebugger.log("Live connection refresh thread - update status bar");

				}
			});
				
		}

		public void run() {
			startTime = System.currentTimeMillis();
			prevTime=startTime;
			while (scanner!=null) {
				try {
					updateStatusBar();
					if(getAgentProxy() != null)
					{
						if (prevTotalBytesRead == totalBytesRead) {
							if (collectingState != 1) {
								if (getAgentProxy().isCollectionData()) {
									getAgentProxy().setCollectionData(false);
								}
		
								Display d = Display.getDefault();
								if(d==null || d.isDisposed())
									return;
								d.asyncExec(new Runnable() {
									public void run() {
										ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
		
										event.setSource(getAgentProxy());
										event.setType(ProfileEvent.STOP_COLLECTING);
										TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
										if(ModelDebugger.INSTANCE.debug)
											ModelDebugger.log("Live connection refresh thread - set STOP_COLLECTING ");
									}
								});
								collectingState = 1;
							}
						} else if (totalBytesRead > prevTotalBytesRead) {
							if (collectingState != 2) {
								if (!getAgentProxy().isCollectionData()) {
									getAgentProxy().setCollectionData(true);
								}
		
								Display d = Display.getDefault();
								if(d==null || d.isDisposed())
									return;
								d.asyncExec(new Runnable() {
									public void run() {
										ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
		
										event.setSource(getAgentProxy());
										event.setType(ProfileEvent.COLLECTING);
										TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
										if(ModelDebugger.INSTANCE.debug)
											ModelDebugger.log("Live connection refresh thread - set COLLECTING ");
									}
								});
								collectingState = 2;
							}
						}
					}
					prevTotalBytesRead = totalBytesRead;

					sleep(1000);
				} catch (InterruptedException exc) {
				}

			}
			if(ModelDebugger.INSTANCE.debug)
			{
				ModelDebugger.log("RefreshUI stopped");
			}
			Display d = Display.getDefault();
			if (d != null && !d.isDisposed()) {
				d.syncExec(new Runnable() {
					public void run() {
						try {
							INavigator navigator = HyadesUtil.getActiveNavigator();
							if (navigator == null)
								return;

							String statusBarMessage = "";
							IStatusLineManager statusLineManager = navigator.getViewSite().getActionBars().getStatusLineManager();
							statusLineManager.setMessage(statusBarMessage);
							
						} catch (Exception e) {
						}
						
					}
				});
			}

		}
	}

	/**
	 * @return
	 */
	public boolean isNewProfileFile() {
		return newFile;
	}

	/** 
	 * Check the non-zip format profiling file for an trace end tag and remove it
	 * if there exist one. 
	 * @param filename  
	 * 		file to be checked
	 * @return true if end tag exists and removed, false if nothing change.
	 */
	public boolean checkAndRemoveEndTag(String filename) {
		File checkFile = new File(filename);
		if (checkFile.exists() && checkFile.isFile()) {
			try {
				RandomAccessFile raf = new RandomAccessFile(checkFile, "rw");
				long checkFileSize = raf.length();
				StringBuffer b = new StringBuffer(END_TAG);
				//Assuming profile file is not modified manually and end tag is 
				// always the last entry of the file. 
				long seekLocation = checkFileSize - b.length() - 2;
				if (seekLocation < 0)
					seekLocation = 0;
				raf.seek(seekLocation);
				while (seekLocation <= checkFileSize) {
					String readLine = raf.readLine();
					StringBuffer t = new StringBuffer(readLine);
					if (readLine.indexOf(END_TAG) >= 0) {
						raf.seek(seekLocation);
						//mark the new trace
						raf.writeBytes(END_TAG_REPLACE);
						raf.close();
						return true;
					}
					seekLocation = seekLocation + t.length();
				}
				raf.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
		return false;
	}

	public void incommingData(byte[] data, int length, InetAddress peer) {
		incommingData(data, 0, length, peer);

	}

	public void incommingData(char[] data, int length, InetAddress peer) {
		incommingData(data, 0, length, peer);
	}

	public void invalidDataType(byte[] data, int length, InetAddress peer) {
		invalidDataType(data, 0, length, peer);
	}

	public void incommingStream(InputStream inputStream, InetAddress peer) {
		if ((getAgentProxy() != null) && getAgentProxy().isToProfileFile()) {
			try {
				int readBytes = 0;
				totalBytesRead = 0;
				prevTotalBytesRead = 0;
				byte[] buffer = new byte[BUFFER_SIZE];
				while (true) {
					readBytes = inputStream.read(buffer);
					if(readBytes==-1)
						break;
					totalBytesRead += readBytes;
					fileWriter.write(buffer, 0, readBytes);
				}
				fileWriter.flush();
				fileWriter.close();
			} catch (IOException e) {
				Status status = new Status(IStatus.ERROR, CommonUITracePlugin.PLUGIN_ID, IStatus.ERROR, "", e);
				CommonPlugin.logError(status);
			}

		} else
			loadEvents(inputStream, 0, -1);
		cleanUp();
	}

	public void loadEvents(InputStream inputStream, int offset, int length) throws InvalidEventException {
		if(scanner==null)
			return;
		final InputStream delegatedInputStream = inputStream;
		InputStream monitoredInputStream = new InputStream() {
			byte[] temp;

			byte[] oneByte = new byte[1];

			public synchronized int read(byte[] b, int off, int len) throws IOException {

				int readBytes = 0;
				if (specialProcessing) {
					if (firstEvent) {
						readBytes = delegatedInputStream.read(b, off, 2);
						if ((readBytes > 1 && b[off] == '<' && b[off + 1] != '?') || (readBytes > 0 && b[off] != '<')) {
							readBytes = readBytes+ "<TRACE>".getBytes().length;
							temp = new byte[readBytes];
							System.arraycopy("<TRACE>".getBytes(), 0, temp, 0, "<TRACE>".getBytes().length);
							temp[temp.length - 2] = b[off];
							temp[temp.length - 1] = b[off + 1];

							readBytes = Math.min(readBytes, len);
							System.arraycopy(temp, 0, b, off, readBytes);
							len -= readBytes;
							if (len > 0) {
								readBytes += delegatedInputStream.read(b, off + readBytes, len);
								specialProcessing = false;
								temp = null;
							} else if (len < 0)
								tempRemaining = temp.length - readBytes;
							else {
								specialProcessing = false;
								temp = null;
							}
						} else {
							readBytes = delegatedInputStream.read(b, off, len);
							specialProcessing = false;
						}
						firstEvent = false;
					} else {
						readBytes = Math.min(tempRemaining, len);
						System.arraycopy(temp, 0, b, off, readBytes);
						off += readBytes;
						len -= readBytes;
						if (len > 0)
							readBytes += delegatedInputStream.read(b, off, len);
						tempRemaining -= readBytes;
						if (tempRemaining == 0) {
							specialProcessing = false;
							temp = null;
						}
					}
				} else
					readBytes = delegatedInputStream.read(b, off, len);

				totalBytesRead += readBytes;

				if (ModelDebugger.INSTANCE.debugEventValue) {
					ModelDebugger.INSTANCE.writeBinaryLog(INCOMMING_DATA, b, off, readBytes);
				}
				return readBytes;
			}

			public int available() throws IOException {
				return delegatedInputStream.available();
			}

			public void close() throws IOException {
				delegatedInputStream.close();
				tempRemaining=-1;
			}

			public synchronized void mark(int readlimit) {
				delegatedInputStream.mark(readlimit);
			}

			public boolean markSupported() {
				return delegatedInputStream.markSupported();
			}

			public int read() throws IOException {
				if (read(oneByte, 0, 1) == -1)
					return -1;
				else
					return oneByte[0];
			}

			public int read(byte[] b) throws IOException {
				return read(b, 0, b.length);
			}

			public synchronized void reset() throws IOException {
				delegatedInputStream.reset();
			}

			public long skip(long n) throws IOException {
				return delegatedInputStream.skip(n);
			}
		};

		try {
			scanner.scanContent(new BufferedInputStream(monitoredInputStream,16*1024), offset, length);
		} catch (Throwable e) {
			if(tempRemaining!=-1)
			{
				logLiveProgress( getProcessedFragments(), startTime, totalBytesRead );
				
			}
			return;
		}
		
		logLiveProgress( getProcessedFragments(), startTime, totalBytesRead );

	}
	
	public synchronized void cleanUp() {

		logLiveProgress( getProcessedFragments(), startTime, totalBytesRead );
		
		Display d = Display.getDefault();
		if(d==null || d.isDisposed())
			return;
		if (getAgentProxy()!=null && getAgentProxy().isCollectionData()) {
			getAgentProxy().setCollectionData(false);
		}
		d.asyncExec(new Runnable() {
			public void run() {
				ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();

				event.setSource(getAgentProxy());
				event.setType(ProfileEvent.STOP_COLLECTING);
				TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
				if(ModelDebugger.INSTANCE.debug)
					ModelDebugger.log("Live connection refresh thread - set STOP_COLLECTING ");
			}
		});
		super.cleanUp();
	}
	
	private void logLiveProgress( int fragmentsProcessed, long startTime, long totalBytes ) {
		
		if( ModelDebugger.INSTANCE.debug ) {
			
			long deltaTime = (System.currentTimeMillis() - startTime) / 1000;
			if (deltaTime == 0)
				deltaTime = 1;
			String status =  MessageFormat.format(TraceWizardMessages.LIVE_PROGRESS, new String[] { "" + getProcessedFragments(), "" + (getProcessedFragments() / deltaTime), "" + deltaTime, "" + totalBytesRead, "" + (totalBytesRead / deltaTime) });

			ModelDebugger.log(status);
			
		}
		
	}
	
	public void loadEvent(byte[] buffer, int offset, int length, boolean loadToModel, boolean toProfileFile) throws InvalidXMLException {
		if(toProfileFile)
			totalBytesRead+=length;
		super.loadEvent(buffer, offset, length, loadToModel, toProfileFile);
	}
	
	public void loadEvent(byte[] buffer, int offset, int length, boolean loadToModel) throws InvalidXMLException {
		totalBytesRead+=length;
		super.loadEvent(buffer, offset, length, loadToModel);
	}
	
}
