/************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HTMLTraceStatReportWizard.java,v 1.4 2010/05/07 18:59:50 jwest Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/
package org.eclipse.hyades.trace.ui.internal.reports;


import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.extension.IExportViewer;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;


/**
 *
 */
public class HTMLTraceStatReportWizard extends ReportGeneratorWizard {

	protected final static String newLine = System.getProperties().getProperty("line.separator");
	
    protected String FILE_EXTENSION = "html"; //$NON-NLS-1$
	
    protected LocationPage locationPage;
	
	public HTMLTraceStatReportWizard() {
		super();
        setWindowTitle(CommonUITraceMessages._16); //$NON-NLS-1$
	}

	public boolean isAvailable(ISelection selection) {
		
		if(selection == null || selection.isEmpty())
			return false;
		
		Object sel = ((StructuredSelection)selection).getFirstElement();
		if(!(sel instanceof IExportViewer))
			return false;
		
		Object[] controls = ((IExportViewer)sel).getViewerControls();
		if(controls == null)
			return false;
		
		for(int idx=0; idx<controls.length; idx++)
		{
			if(controls[idx] instanceof Tree)
				return true;
		}
				
		return false;
	}
	
    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
        locationPage = new LocationPage("location", adjustLocation()); //$NON-NLS-1$
        locationPage.setTitle(CommonUITraceMessages._14); //$NON-NLS-1$
        locationPage.setDescription(CommonUITraceMessages._15); //$NON-NLS-1$
        locationPage.setFileExtension(FILE_EXTENSION);
    }

    protected IStructuredSelection adjustLocation() {
        IStructuredSelection structuredSelection = getSelection();
        
        Object selection = structuredSelection.getFirstElement();
        if (selection instanceof IResource) {
            IResource res = (IResource) selection;
            IContainer parent = res.getParent();
            if ((parent != null) && parent.exists()) {
                structuredSelection = new StructuredSelection(parent);
            }
        }
        return structuredSelection;
    }

	
    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#generate(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IFile generate(IProgressMonitor monitor) throws Exception {
    	
    	Object sel = getSelection().getFirstElement();
    	if(sel == null || !(sel instanceof IExportViewer))
    		return null;
      
        monitor.beginTask("", 4); //$NON-NLS-1$             
    	
        IFile reportFile = getReportFile();
        monitor.worked(1);
        export((IExportViewer)sel, reportFile.getLocation().toOSString());
        monitor.worked(1);

        reportFile.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        monitor.done();
           
        return reportFile;
    }
	
	protected IFile getReportFile() {
        IFile file = null;
        IPath path = locationPage.getContainerFullPath();
        IResource res = createContainer(path);
        path = new Path(locationPage.getFileName()); //$NON-NLS-1$
        file = ((IContainer) res).getFile(path);
        locationPage.allowOverwrite();
        return file;
    }
    
    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#addReportPages()
     */
    protected void addReportPages() throws Exception {
        addPage(locationPage);
    }
  
    /**
     * @see org.eclipse.jface.wizard.IWizard#canFinish()
     */
    public boolean canFinish() {
    	
        return locationPage.getItemName().length() > 0 && locationPage.getContainerFullPath() != null;
    }
    
    protected IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = ResourcesPlugin.getWorkspace();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {
            if (idx == 0) {
                //project
                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
                if (project == null || !project.exists()) {
                    //create the project
                    try {
                        project.create(null);
                    } catch (Exception e) {
                    	CommonPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                try {
                    project.open(null);
                } catch (Exception e) {
                }
                container = project;
            } else // (idx > 1)
            { //folder
                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
                if (folder == null || !folder.exists()) {
                    //create the folder
                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
                    	CommonPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                container = folder;
            }
        }
        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
        	CommonPlugin.logError(exc);
            exc.printStackTrace();
        }
        return container;
    }
    
    /**
     * 
     * @param viewer
     * @return
     */
	protected boolean export(IExportViewer viewer, String fileName) {
		
		if(fileName == null)
		{
			MessageDialog.openError(getShell(), CommonUITraceMessages.TRC_MSGT, CommonUITraceMessages._6);
		    return false;
		}
		
		Writer writer = null;
		try {
			File file = new File(fileName);
			if(file.isDirectory())
			{
				MessageDialog.openError(getShell(), CommonUITraceMessages.TRC_MSGT, CommonUITraceMessages._7);
				return false;				
			}
			OutputStream outputStream = new FileOutputStream(file.toString(), false);
			writer = new OutputStreamWriter(outputStream, "UTF8");
		} catch (IOException ioException) {
			ioException.printStackTrace();			
			return false;
		}
		
		StringBuffer buffer = new StringBuffer();        
		buffer.append(printHeader());
		buffer.append(printTitle(viewer.getViewerTitle()));
		
		for(int idx=0; idx<viewer.getViewerControls().length; idx++)
		{
			if(viewer.getViewerControls()[idx] instanceof Tree)
				buffer.append(printColumns((Tree)viewer.getViewerControls()[idx]));
		}
		
		buffer.append(printFooter());
		
		try {
			writer.write(buffer.toString());
			writer.flush();
			writer.close();
		} catch (Exception exception) {
			exception.printStackTrace();
		}
		
		return true;
	}
    
	protected String printHeader()
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		StringBuffer buffer = new StringBuffer();
		buffer.append("<html").append(getDirection()).append(getLang()).append(">").append(newLine);
		buffer.append("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">").append(newLine)
		              .append("<head>").append(newLine)
		              .append("<STYLE TYPE=\"text/css\">").append(newLine)
		              .append("p, table, td, th {  font-family: arial, helvetica, geneva; font-size: 10pt}").append(newLine)
		              .append("body {  font-family: arial, helvetica, geneva; font-size: 10pt; clip:   rect(   ); margin-top: 5mm; margin-left: 3mm}").append(newLine)
		              .append("</STYLE>").append(newLine)
		              .append("</head>").append(newLine);
		buffer.append("<body text=\"#000000\" bgcolor=\"#ffffff\">").append(newLine);
		
		return buffer.toString();
	}
	
	protected String printTitle(String title)
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		StringBuffer buffer = new StringBuffer();
		
		buffer.append("<caption align=top><B>").append(newLine);
		buffer.append(replaceEntities(title)).append(newLine);
		buffer.append("</B></caption>").append(newLine);
        return buffer.toString();		
	}
	
	protected String printFooter()
	{
		final String newLine = System.getProperties().getProperty("line.separator");
		
		StringBuffer buffer = new StringBuffer();		
		buffer.append("</body>").append(newLine);
		buffer.append("</html>").append(newLine);
		return buffer.toString();
	}
	
	protected String printColumns(Tree tree)
	{		
		StringBuffer buffer = new StringBuffer();
		buffer.append("<table border=0 cellspacing=2 cellpadding=2 WIDTH=\"100%\">").append(newLine);

		buffer.append("<tr>").append(newLine);
		for (int idx = 0; idx < tree.getColumnCount(); idx++) {
			String columnText = tree.getColumn(idx).getText();

			if (columnText.indexOf("<") == 0)
				columnText = "&lt;" + columnText.substring(1);
			else if (columnText.indexOf(">") == 0)
				columnText = "&gt;" + columnText.substring(1);

			buffer.append("<th align=").append(getAlign()).append(">")
			.append(columnText).append("</th>").append(newLine);
		}
		buffer.append("</tr>").append(newLine);

		printTreeItems(tree.getItems(), 0, true, buffer, tree.getColumnCount());

		buffer.append("</table>").append(newLine);
		
		return buffer.toString();
	}
	
	protected boolean printTreeItems(TreeItem[] treeItems, int level, boolean color, StringBuffer buffer, int columnCount)
	{
		for (int kdx = 0; kdx < treeItems.length; kdx++)
		{
			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">").append(newLine);
			else   
				buffer.append("<tr>").append(newLine);
	
			for (int ndx = 0; ndx < columnCount; ndx++)
				buffer.append("<td align=").append(getAlign()).append(">")
						.append(ndx==0?getFirstColumnIndentString(treeItems[kdx], level):"")
						.append(treeItems[kdx].getText(ndx))
						.append("</td>").append(newLine);
			
			buffer.append("</tr>").append(newLine);

			color = !color;
			if (treeItems[kdx].getExpanded())
				color = printTreeItems(treeItems[kdx].getItems(), level+1, color, buffer, columnCount);
		}
		
		return color;
	}
	
	protected String getFirstColumnIndentString(TreeItem item, int level)
	{
		StringBuffer buffer = new StringBuffer();
		
		for (int i = 0; i < level; i ++)
			buffer.append("&nbsp;&nbsp;&nbsp;&nbsp;");
			
		if (item.getItemCount() > 0)
		{
			if (item.getExpanded())
				buffer.append("[-]&nbsp;");
			else
				buffer.append("[+]&nbsp;");				
		}
		else
			buffer.append("&nbsp;&nbsp;&nbsp;&nbsp;");
		
		return buffer.toString();
	}
	
	protected String getDirection()
    {
    	if (Window.getDefaultOrientation() == SWT.RIGHT_TO_LEFT)
    		return " dir=\"rtl\"";
    	else
    		return "";
    }
    
	// Bug 308112: Provide lang attribute on HTML element for assistive technologies
	protected String getLang()
	{
		return " lang=\"" + Platform.getNL().replace('_', '-') + "\"";
	}
	
    protected String getAlign()
    {
    	if (Window.getDefaultOrientation() == SWT.RIGHT_TO_LEFT)
    		return "right";
    	else
    		return "left";
    }    

	protected String getCSVFormat(String text)
	{
		if(text == null)
			return text;
		
		if(text.indexOf(",") == -1 && text.indexOf("\"") == -1)
			return text;
		
		text = text.replaceAll("\"", "\"\"");
		
		return (new StringBuffer("\"")).append(text).append("\"").toString();			
	}

   public boolean performFinish() {
	   
       IFile reportFile = getReportFile();
       if (reportFile.exists()) {
           if (!MessageDialog.openQuestion(getShell(), CommonUIMessages._158, //$NON-NLS-1$
        		   CommonUIMessages._159)) //$NON-NLS-1$
            	
               return false;
       }
	   
	   return super.performFinish();
   
   }

   //replace XML entities in a string 
   public String replaceEntities(String original) {
	   String result=original;
	   result = result.replaceAll("\\&","&amp;");
	   result = result.replaceAll("\\<","&lt;");
	   result = result.replaceAll("\\>","&gt;");
	   result = result.replaceAll("\\'","&apos;");
	   result = result.replaceAll("\\\"","&guot;");	   
	   return result;
   }

}
