/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DeleteUtil.java,v 1.11 2010/09/02 17:07:38 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.net.UnknownHostException;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.CorrelationContainer;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.EMFUtil;
import org.eclipse.hyades.models.hierarchy.util.IHyadesExtendedResource;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.util.provisional.ITraceModelInteraction;
import org.eclipse.hyades.models.util.provisional.TraceModelInteraction;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.ui.IDeleteListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;


public class DeleteUtil 
{
	
	protected static boolean isDeleteInProgress = false;
	public static void deleteAgent(TRCAgentProxy agent, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (agent == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		TRCProcessProxy process = agent.getProcessProxy();

		if (process == null)
			return;
		
		deleteExternalItems(agent, refresh, deleteContents, saveRequired, navigatorID);

		//remove documents
		removeDoc(agent, deleteContents, saveRequired, navigatorID);
		agent.setProcessProxy(null);

		if (process.getAgentProxies().size() == 0) {
			deleteProcess(process, refresh, deleteContents, saveRequired, navigatorID);
		}
		
		
		if (refresh) {
			ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.REFRESH_VIEWS);
			TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		}
	}
	
	public static void deleteEObject(EObject eObject, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID){
		if(eObject instanceof TRCAgent){
			deleteAgent(((TRCAgent)eObject).getAgentProxy(),refresh, deleteContents, saveRequired, navigatorID);
		}else if(eObject instanceof TRCProcessProxy){
			deleteProcess((TRCProcessProxy)eObject, refresh, deleteContents, saveRequired, navigatorID);
		}else if(eObject instanceof TRCNode){
			deleteNode((TRCNode)eObject, refresh, deleteContents, saveRequired, navigatorID);
		}else if(eObject instanceof TRCMonitor){
			deleteMonitor((TRCMonitor)eObject, refresh, deleteContents, saveRequired, navigatorID);
		}else if(eObject instanceof CorrelationContainer){
			deleteCorrelationContainer(((CorrelationContainer)eObject).getCorrelationContainerProxy(), refresh, deleteContents, saveRequired, navigatorID);
		}
	}
	
	
	public static void deleteProcess(TRCProcessProxy process, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID) {

		if (process == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		TRCNode node = process.getNode();

		if (node == null)
			return;

		if (node.getProcessProxies().size() == 1) {
			deleteNode(node, refresh, deleteContents, saveRequired, navigatorID);
		} else {

			//remove documents
			removeDoc(process, deleteContents, saveRequired, navigatorID);
			deleteExternalItems(process, refresh, deleteContents, saveRequired, navigatorID);
			process.setNode(null);
		}

		if (refresh) {
			ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);
		}
	}
	public static void deleteMonitor(TRCMonitor monitor, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (monitor == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		// traverse and prune all external elements first
		Iterator i = monitor.getNodes().iterator();
		while (i.hasNext()) {
			TRCNode node = (TRCNode)i.next();
			Iterator j = node.getProcessProxies().iterator();
			while (j.hasNext()) {
				TRCProcessProxy process = (TRCProcessProxy)j.next();
				Iterator k = process.getAgentProxies().iterator();
				while (k.hasNext()) {
					TRCAgentProxy agent = (TRCAgentProxy)k.next();
					deleteExternalItems(agent, refresh, deleteContents, saveRequired, navigatorID);
				}
				deleteExternalItems(process, refresh, deleteContents, saveRequired, navigatorID);
			}
			deleteExternalItems(node, refresh, deleteContents, saveRequired, navigatorID);
		}
		deleteExternalItems(monitor, refresh, deleteContents, saveRequired, navigatorID);

		IPath path = new Path(TString.resourcePath(monitor.eResource().getURI()));
		if (path.segmentCount() > 1)
			path = path.removeLastSegments(1);

		PDContentProvider.removeMonitor(path.toOSString(), monitor);

		//remove documents
		removeDoc(monitor, deleteContents, saveRequired, navigatorID);

		if (refresh) {

			ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		}

	}
	
	
	public static void deleteNode(TRCNode node, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (node == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		TRCMonitor monitor = node.getMonitor();

		if (monitor == null)
			return;

		if (monitor.getNodes().size() == 1) {
			deleteMonitor(monitor, refresh, deleteContents, saveRequired, navigatorID);
		} else {
			//remove documents
			removeDoc(node, deleteContents, saveRequired, navigatorID);
			deleteExternalItems(node, refresh, deleteContents, saveRequired, navigatorID);
			node.setMonitor(null);
		}

		if (refresh) {
			ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		}
	}


	public static void deleteContainer(IContainer container, boolean deleteContents, boolean saveRequired, String navigatorID) throws CoreException {

		if (container == null || !container.exists()
			   || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		IResource[] members = container.members();
		if (members.length > 0) {
			for (int i = 0; i < members.length; i++) {
				if (members[i] instanceof IFolder) {
					IFolder member = (IFolder) members[i];
					deleteContainer(member, deleteContents, saveRequired, navigatorID);
				}
			}
		}

		List monitors = PDContentProvider.getMonitors(container);
		for (int idx = 0; idx < monitors.size(); idx++) {
			if (!(monitors.get(idx) instanceof TRCMonitor))
				continue;

			TRCMonitor mon = (TRCMonitor) monitors.get(idx);
			if (mon != null)
				deleteMonitor(mon, true, deleteContents, saveRequired, navigatorID);
		}

		deleteExternalItems(container, true, deleteContents, saveRequired, navigatorID);
		PDContentProvider.removeContainer(container);
		container.delete(true, null);

	}

	public static void deleteResource(IResource resource, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (resource == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		try {

			if (resource == null || !resource.exists()) {
				return;
			}
			//remove documents
			else if (resource instanceof IProject || resource instanceof IFolder)
				deleteContainer((IContainer) resource, deleteContents, saveRequired, navigatorID);
			else
				resource.delete(true, null);

			ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		} catch (CoreException exc) {
			exc.printStackTrace();
		}

	}

	public static void deleteExternalItems(Object parent, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID) {
		INavigatorContribution[] delegates = NavigatorExtensionUtil.getContributions(navigatorID);
		for (int i=0;i<delegates.length;++i) {
			List children = delegates[i].getChildren(parent);
			Iterator iter = children.iterator();
			while (iter.hasNext()) {
				INavigatorItem item = (INavigatorItem)iter.next();
				if (item.isDeleteEnabled())
					item.delete(refresh, deleteContents);
			}
		}
	}
	
	public static void removeDoc(TRCAgentProxy agent, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (agent == null || agent.getAgent() == null || agent.getAgent().eResource() == null
				|| navigatorID == null || navigatorID.trim().length() == 0)
			return;

		TraceUIManager.getTraceUIManager().removeSelectionModel((TRCAgentProxy) agent);//bugzilla 2006665
		LoadersUtils.deregisterDataProcessor(agent);
		LoadersUtils.deregisterAgentListener(agent);

		LoadersUtils.deregisterAgentInstance(agent);
		LoadersUtils.deregisterAgentProxyContext(agent);

		if (deleteContents) {
			IPath path = new Path(TString.resourcePath(agent.getAgent().eResource().getURI()));
			IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
			deleteResource(res, true, saveRequired, navigatorID);
		}
		// Bug 322951 and 324325
		ITraceModelInteraction[] contributors = TraceModelInteraction.getContributors();
		if(contributors != null && contributors.length > 0) {
			// Allow for implementers of the Trace model extension point to contribute to the delete action
			// such as on a context menu delete.
			for(int contributor = 0; contributor < contributors.length; contributor++) {
				contributors[contributor].traceModelDelete(agent.eResource(), deleteContents);
			}
		}

		List list = TraceUIManager.getTraceUIManager().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(agent);
		}

		TRCProcessProxy process = agent.getProcessProxy();
		TRCNode trcnode = process.getNode();

		if (process == null || trcnode == null)
			return;

		//process.getAgentProxies().remove(agent);

		try {
			String nodeName = trcnode.getName();
			Node node = NodeFactory.createNode(nodeName);
			if (node.isConnected()) {

				Object instance = LoadersUtils.locateAgentInstance(agent);
				if (instance != null && instance instanceof Agent) {
					Agent a = (Agent) instance;
					if (a.isAttached()) {
						a.detach();
					}
				}
			}

		} catch (UnknownHostException uhe) {
			//do nothing
		} catch (Exception exc) {
			exc.printStackTrace();
		}

		//unload resource
		unloadDoc(agent.getAgent());
		for (Iterator iter = EMFUtil.delete(agent, agent.eResource().getResourceSet()).iterator();
		iter.hasNext();
		) {
			Resource res = (Resource) iter.next();

			if (res != null) {
				res.setModified(true);
				if(res.getURI().toString().endsWith("trcmxmi"))
				{
					try {
						cleanupEmptyCorrelationContainersProxies((TRCMonitor)res.getContents().get(0),deleteContents,saveRequired,navigatorID);
					} catch (Exception e) {
						// ignore exceptions
					}
				}
			}
		}
	}
	protected static void cleanupEmptyCorrelationContainersProxies(TRCMonitor monitor,boolean deleteContents, boolean saveRequired, String navigatorID) {
		CorrelationContainerProxy[] containerProxies = (CorrelationContainerProxy[])monitor.getCorrelationContainerProxies().toArray();
		for (int i = 0; i < containerProxies.length; i++) {
			if(containerProxies[i].getCorrelatedAgents().isEmpty())
			{
				deleteCorrelationContainer(containerProxies[i],true,deleteContents,saveRequired,navigatorID);
			}
		}
	}

	public static void removeDoc(TRCMonitor mon, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (mon == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		List list = TraceUIManager.getTraceUIManager().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(mon);
		}

		if (mon.eResource() != null) {
			//			SaveUtil.removeDocument(mon.eResource());

			if (deleteContents) {
				IPath path = new Path(TString.resourcePath(mon.eResource().getURI()));
				IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
				deleteResource(res, true, saveRequired, navigatorID);
			}

		}

		Object[] nodes = mon.getNodes().toArray();
		for (int idx = 0; idx < nodes.length; idx++) {
			TRCNode node = (TRCNode) nodes[idx];

			if (node.eIsProxy())
				continue;

			removeDoc(node, deleteContents, saveRequired, navigatorID);
		}

		//unload resource
		unloadDoc(mon);

	}

	public static void removeDoc(TRCNode node, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (node == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		List list = TraceUIManager.getTraceUIManager().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(node);
		}

		if (node.eResource() != null) {
			//			SaveUtil.removeDocument(node.eResource());
			if (deleteContents) {
				IPath path = new Path(TString.resourcePath(node.eResource().getURI()));
				IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
				deleteResource(res, true, saveRequired, navigatorID);
			}

		}

		Object[] processes = node.getProcessProxies().toArray();
		for (int idx = 0; idx < processes.length; idx++) {

			TRCProcessProxy process = (TRCProcessProxy) processes[idx];

			if (process.eIsProxy())
				continue;

			removeDoc(process, deleteContents, saveRequired, navigatorID);
		}

		TRCMonitor monitor = node.getMonitor();
		if (monitor == null)
			return;

		monitor.getNodes().remove(node);

		if (saveRequired) { //update the monitor xmi file
			IPath path = new Path(TString.resourcePath(monitor.eResource().getURI()));
			IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);

			if (res != null) {
				try {
					SaveUtil.saveResource(null, monitor.eResource());
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		}

		//unload resource
		unloadDoc(node);

	}

	public static void removeDoc(TRCProcessProxy process, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (process == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;

		if (deleteContents) {
			IPath path = new Path(TString.resourcePath(process.eResource().getURI()));
			IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
			deleteResource(res, true, saveRequired, navigatorID);
		}

		Object[] agents = process.getAgentProxies().toArray();
		for (int idx = 0; idx < agents.length; idx++) {
			TRCAgentProxy agent = (TRCAgentProxy) agents[idx];

			if (agent.eIsProxy())
				continue;

			removeDoc(agent, deleteContents, saveRequired, navigatorID);
		}

		TRCNode node = process.getNode();
		if (node != null) {
			node.getProcessProxies().remove(process);

			if (saveRequired) { //update the node xmi file
				IPath path = new Path(TString.resourcePath(node.eResource().getURI()));
				IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);

				if (res != null) {
					try {
						SaveUtil.saveResource(null, node.eResource());
					} catch (Exception exc) {
						exc.printStackTrace();
					}
				}
			}
		}
  
		List list = TraceUIManager.getTraceUIManager().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(process);
		}

		//unload resource
		unloadDoc(process);
	}

	public static void unloadDoc(EObject object) {

		if (object == null || object.eResource() == null)
			return;

		TraceUIManager.getTraceUIManager().removeSelectionModel(object);
		((IHyadesExtendedResource) object.eResource()).delete();

	}

	/**
	 * @param proxy
	 * @param b
	 * @param deleteContents
	 */
	public static void deleteCorrelationContainer(CorrelationContainerProxy correlationContainerProxy, boolean refresh, boolean deleteContents, boolean saveRequired, String navigatorID) {
		if (correlationContainerProxy == null || navigatorID == null || navigatorID.trim().length() == 0)
			return;


		List list = TraceUIManager.getTraceUIManager().getDeleteListeners();
		for (int idx = 0; idx < list.size(); idx++) {
			((IDeleteListener) list.get(idx)).deregister(correlationContainerProxy);
		}
		
		//remove documents
		CorrelationContainer correlationContainer = correlationContainerProxy.getCorrelationContainer();
		if (correlationContainer != null && !correlationContainer.eIsProxy()) {
			if (deleteContents) { //update the node xmi file
				IPath path = new Path(TString.resourcePath(correlationContainer.eResource().getURI()));
				IResource res = ResourcesPlugin.getWorkspace().getRoot().findMember(path);
				deleteResource(res, true, saveRequired, navigatorID);

			}
			IHyadesExtendedResource corrRes = (IHyadesExtendedResource)correlationContainer.eResource();
			if(corrRes != null){
				corrRes.delete();
			}
			correlationContainer.setCorrelationContainerProxy(null);
		}

		if(correlationContainerProxy.getMonitor()!=null)
		{
		Resource res = correlationContainerProxy.getMonitor().eResource();
		
		correlationContainerProxy.setMonitor(null);

		if (res != null && saveRequired) {
			try {
				SaveUtil.saveResource(null, res);
			} catch (Exception exc) {
				exc.printStackTrace();
				}
			}
		}
		
		for (Iterator iter = correlationContainerProxy.getCorrelatedAgents().iterator(); iter.hasNext();) {
			TRCAgentProxy element = (TRCAgentProxy) iter.next();
			if (element.getInternalCorrelations().contains(correlationContainerProxy))
				element.getInternalCorrelations().remove(correlationContainerProxy);
		}
		correlationContainerProxy.getCorrelatedAgents().clear();

		if (refresh) {
			ProfileEvent event = TraceUIManager.getTraceUIManager().getProfileEvent();
			event.setSource(null);
			event.setType(ProfileEvent.UNSPECIFIED);
			TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		}

	}
	
	public static void setDeleteInProgress(boolean newValue){
		isDeleteInProgress = newValue;
	}
	public static boolean isDeleteInProgress(){
		return isDeleteInProgress;
	}

}
