/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SaveUtil.java,v 1.7 2009/09/10 21:17:47 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;

/**
 * Utility methods for saving navigator items.
 */
public class SaveUtil {
	/**
	 * Save all the given resources. Will also recursively
	 * save the parents and children.
	 */
	public static void save(Collection c, IProgressMonitor monitor, String navigatorID) {
		//Bugzilla 111536 
		//Need to handle the special case when a correlation container proxy is saved then also it's associated agent proxies and parents (process proxy, node, monitor) must be saved.
		//If we don't include the agent proxies then the findItems method will traverse and save only the correlation container proxy, it's parent - the monitor, and the agent proxies. 
		// The parents of the agent proxies will be missed and therefore the correlation data will be lost when the perspectve is closed and the agent proxies are not saved explicitly. 
		Collection sel = new UniqueEList();
		for (Iterator iter = c.iterator(); iter.hasNext();) {
			Object element = iter.next();			
			if(element instanceof CorrelationContainerProxy){
				//if a correlation container is found add first the agent proxies to the list
				//so that they are processed first by findItems(i.next(), c, true, true, navigatorID);
				// and as a consequence their parents will also be saved
				sel.addAll(((CorrelationContainerProxy) element).getCorrelatedAgents());
			}			
		}
		sel.addAll(c);
		Collection items = findItems(sel, navigatorID);
		
		if (monitor != null)
			monitor.beginTask(Action.removeMnemonics(CommonUITraceMessages.SAVE), items.size());

		if (isValid(items)) {
			Iterator i = items.iterator();
			while (i.hasNext()) {
				saveItem(i.next(), monitor);
				if (monitor != null && monitor.isCanceled())
					break;
			}
		}
		
		if (monitor != null)
			monitor.done();
	}

	/**
	 * Save a single item (non-recursive).
	 */
	public static void saveItem(Object obj, IProgressMonitor monitor) {
		if (obj instanceof EObject) {
			EObject eobj = (EObject)obj;
			
			if (obj instanceof TRCMonitor) {
				try {
					org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, eobj.eResource());
				}
				catch (Exception e) {
					handleSaveException(CommonUITraceMessages.SMON_TXT, ((TRCMonitor)obj).getName(), eobj);
				}
			}
			else if (obj instanceof TRCNode) {
				try {
					org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, eobj.eResource());
				}
				catch (Exception e) {
					handleSaveException(CommonUITraceMessages.SNOD_TXT, ((TRCNode)obj).getName(), eobj);
				}
			}
			else if (obj instanceof TRCProcessProxy) {
				try {
					org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, eobj.eResource());
				}
				catch (Exception e) {
					handleSaveException(CommonUITraceMessages.SPRC_TXT, ((TRCProcessProxy)obj).getName(), eobj);
				}
			}
			else if (obj instanceof TRCAgentProxy) {
				TRCAgentProxy agent = (TRCAgentProxy)obj;
				if (agent.getAgent() != null && agent.getAgent().eResource() != null) {
					try {
						org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, agent.getAgent().eResource());
					}
					catch (Exception e) {
						handleSaveException(CommonUITraceMessages.SAG_TXT, agent.getName(), agent);
					}
				}
			}
			else if (obj instanceof CorrelationContainerProxy) {
				CorrelationContainerProxy proxy = (CorrelationContainerProxy)obj;
				if (proxy.getCorrelationContainer() != null) {
					try {
						if(!(proxy.getCorrelationContainer().eResource()!=null && proxy.getCorrelationContainer().eResource().getURI().toString().endsWith(".corrdb")))
						{
							org.eclipse.hyades.models.hierarchy.util.SaveUtil.saveResource(monitor, proxy.getCorrelationContainer().eResource());
						}
					}
					catch (Exception e) {
						handleSaveException(CommonUITraceMessages.SCOR_TXT, proxy.getName(), eobj);
					}
				}
			}
		}
		else if (obj instanceof INavigatorItem) {
			if (monitor != null) {
				monitor.setTaskName(((INavigatorItem)obj).getText());
				monitor.worked(1);
			}
			INavigatorItem item = (INavigatorItem)obj;
			if (item.isSaveEnabled())
				item.save(true);
		}
	}

	/*
	 * Collect all items related to the given selection. This
	 * collects parents (ancestors) and children (descendants).
	 */
	protected static Collection findItems(Collection selection, String navigatorID) {
		Collection c = new HashSet();
		Iterator i = selection.iterator();
		while (i.hasNext())
			findItems(i.next(), c, true, true, navigatorID);
		return c;
	}
	
	protected static void findItems(Object obj, Collection c, boolean findAncestors, boolean findDescendants, String navigatorID) {
		if (obj != null && !c.contains(obj)) {
			if (!(obj instanceof IContainer))
				c.add(obj);
			
			if (findAncestors) {
				findItems(getParent(obj), c, true, false, navigatorID);
			}
			if (findDescendants) {
				Iterator i = getChildren(obj, navigatorID).iterator();
				while (i.hasNext())
					findItems(i.next(), c, false, true, navigatorID);
			}
		}
	}

	protected static Object getParent(Object obj) {
		if (obj instanceof TRCNode)
			return ((TRCNode)obj).getMonitor();
		else if (obj instanceof TRCProcessProxy)
			return ((TRCProcessProxy)obj).getNode();
		else if (obj instanceof TRCAgentProxy)
			return ((TRCAgentProxy)obj).getProcessProxy();
		else if (obj instanceof CorrelationContainerProxy)
			return ((CorrelationContainerProxy)obj).getMonitor();
		else if (obj instanceof INavigatorItem)
			return ((INavigatorItem)obj).getParent();
		else
			return null;
	}
	
	protected static Collection getChildren(Object obj, String navigatorID) {
		Collection c = new HashSet();

		if (obj instanceof IContainer) {
			IContainer container = (IContainer)obj;
			try {
				Object[] members = container.members();
				for (int i=0;i<members.length;++i)
					if (members[i] instanceof IContainer)
						c.add(members[i]);
			}
			catch (CoreException e) {
				e.printStackTrace();
			}

			c.addAll(PDContentProvider.getMonitors(container));
		}
		else if (obj instanceof TRCMonitor)
			c.addAll(((TRCMonitor)obj).getNodes());
		else if (obj instanceof TRCNode)
			c.addAll(((TRCNode)obj).getProcessProxies());
		else if (obj instanceof TRCProcessProxy)
			c.addAll(((TRCProcessProxy)obj).getAgentProxies());
		else if (obj instanceof CorrelationContainerProxy)
			c.addAll(((CorrelationContainerProxy)obj).getCorrelatedAgents());
		
		c.addAll(NavigatorExtensionUtil.getAllChildren(obj, navigatorID));
		return c;
	}

	/*
	 * Is this a valid collection to save?
	 */
	protected static boolean isValid(Collection c) {
		Iterator i = c.iterator();
		while (i.hasNext())
			if (!isValid(i.next()))
				return false;
		return true;
	}
	
	/*
	 * Is this a valid item to save?
	 */
	protected static boolean isValid(Object obj) {
		if (obj instanceof EObject) {
			if (obj instanceof TRCMonitor) {
				TRCMonitor mon = (TRCMonitor)obj;
				if (isReadOnly(mon.eResource())) {
					handleSaveReadOnlyException(mon.getName(), mon);
					return false;
				}
			}
			else if (obj instanceof TRCNode) {
				TRCNode node = (TRCNode)obj;
				if (!node.eIsProxy()) {
					if (isReadOnly(node.eResource())) {
						handleSaveReadOnlyException(node.getName(), node);
						return false;
					}
				}
			}
			else if (obj instanceof TRCProcessProxy) {
				TRCProcessProxy p = (TRCProcessProxy)obj;
				if (!p.eIsProxy()) {
					if (isReadOnly(p.eResource())) {
						handleSaveReadOnlyException(p.getName(), p);
						return false;
					}
				}
			}
			else if (obj instanceof TRCAgentProxy) {
				TRCAgentProxy a = (TRCAgentProxy)obj;
				if (!a.eIsProxy() && a.getAgent() != null && a.getAgent().eResource() != null) {
					if (isReadOnly(a.getAgent().eResource())) {
						handleSaveReadOnlyException(a.getName(), a);
						return false;
					}
					if (a.isCollectionData()) {
						handleAgentCollectingWhenSave(a);
						return false;
					}
				}
			}
			else if (obj instanceof CorrelationContainerProxy) {
				CorrelationContainerProxy proxy = (CorrelationContainerProxy)obj;
				if (isReadOnly(proxy.getCorrelationContainer().eResource())) {
					handleSaveReadOnlyException(proxy.getName(), proxy);
					return false;
				}
			}
		}
		return true;
	}
	
	/*
	 * Is this resource read-only?
	 */
	protected static boolean isReadOnly(Resource rsc) {
		IPath path = new org.eclipse.core.runtime.Path(rsc.getURI().toString());
		IResource foundrsc =
			ResourcesPlugin.getWorkspace().getRoot().findMember(path);
		if ((foundrsc != null) && (foundrsc.isAccessible()))
			return foundrsc.getResourceAttributes().isReadOnly();

		return false;
	}

	protected static void handleAgentCollectingWhenSave(TRCAgentProxy a) {
		if (a.getAgent() == null)
			return;
		final String text = NLS.bind(CommonUITraceMessages.SAGC_ERROR_, a.getName());

		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				final Shell shell = CommonUITracePlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
				if (shell != null) {
					final Status err =
						new Status(
							Status.WARNING,
							ResourcesPlugin.PI_RESOURCES,
							IResourceStatus.INTERNAL_ERROR,
							text,
							null);
					ErrorDialog.openError(
						shell,
						CommonUITraceMessages.TRC_MSGT,
						CommonUITraceMessages.SFLT_ERROR_,
						err);
				}
			}
		});
	}

	protected static void handleSaveException(String objectType, String objectName, EObject obj) {
		Resource res = obj.eResource();

		String text;
		if (res != null)
			text = NLS.bind(CommonUITraceMessages.SFL_ERROR_, new Object[] {objectType, res.getURI().toString()});
		else
			text = NLS.bind(CommonUITraceMessages.SFL_ERROR_, new Object[] {objectType, objectName});

		final String finalText = text;
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				Shell shell = CommonUITracePlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
				if (shell != null) {
					Status err =
						new Status(
							Status.WARNING,
							ResourcesPlugin.PI_RESOURCES,
							IResourceStatus.INTERNAL_ERROR,
							finalText,
							null);
					ErrorDialog.openError(
						shell,
						CommonUITraceMessages.TRC_MSGT,
						CommonUITraceMessages.SFLT_ERROR_,
						err);
				}
			}
		});
	}

	protected static void handleSaveReadOnlyException(String objectName, EObject obj) {
		Resource res = obj.eResource();
		String text="";
		if (res != null)
			text = NLS.bind(CommonUITraceMessages.SROF_ERROR_, new Object[] {objectName, res.getURI().toString()});
		if (objectName != null)
			text = NLS.bind(CommonUITraceMessages.SROF_ERROR_, new Object[] {objectName, objectName});

		final String finalText = text;
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				Shell shell = CommonUITracePlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
				if (shell != null) {
					final Status err =
						new Status(
							Status.WARNING,
							ResourcesPlugin.PI_RESOURCES,
							IResourceStatus.INTERNAL_ERROR,
							finalText,
							null);
					ErrorDialog.openError(
					    shell,
						CommonUITraceMessages.TRC_MSGT,
						CommonUITraceMessages.SFLT_ERROR_,
						err);
				}
			}
		});
	}
}
