/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceLogActionGroup.java,v 1.6 2008/01/24 02:30:00 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.common.ui.trace.internal.actions;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.internal.actions.DeleteAction;
import org.eclipse.hyades.trace.ui.internal.actions.ImportActionGroup;
import org.eclipse.hyades.trace.ui.internal.actions.OpenTraceReportAction;
import org.eclipse.hyades.trace.ui.internal.actions.OpenWizardAction;
import org.eclipse.hyades.trace.ui.internal.actions.SaveAction;
import org.eclipse.hyades.trace.ui.internal.actions.ToggleLinkingAction;
import org.eclipse.hyades.trace.ui.internal.core.TraceAssociationManager;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenNodeWizard;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.action.CollapseAllAction;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIImages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.actions.ActionGroup;


public class TraceLogActionGroup extends NavigatorActionGroup 
{ 
	protected DeleteAction fDeleteAction;
	protected SaveAction fSaveToFile;
	protected CollapseAllAction collapseAllAction;
	protected ToggleLinkingAction toggleLinkingAction;
	protected RefreshTreeAction fRefreshTree;
	protected RefreshFromLocalAction fRefreshLocal;	
	protected Separator fRefreshGrp;
	protected ActionGroup fImportActionGroup;
	protected OpenTraceReportAction openReportAction;
	protected OpenWizardAction fOpenNodeAction;		
	protected Separator fAttachGrp;
	protected Separator fMonitorGrp;
	protected Separator fTerminateGrp;
	protected Separator fNewSnapshotGrp;
	protected Separator fGCGroup;
	
	protected final String GROUP_ADD_VIEWS = "group.add.views";
	
	protected Map traceViewsMap;
	
	/*
	 * Constructor for TraceLogActionGroup
	 * @param logNavigator
	 */
	public TraceLogActionGroup(INavigator viewer)
	{
		super(viewer);
		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.navigator.action.NavigatorActionGroup#createActions()
	 */
	protected void createActions() {
		super.createActions();
		
		traceViewsMap = new HashMap();
		toggleLinkingAction = new ToggleLinkingAction(getNavigator(), CommonUITraceMessages.LNKV); 
		toggleLinkingAction.setToolTipText(CommonUITraceMessages.LNKVT); 
		CommonUIImages.setActionImage(CommonUIImages.INSTANCE, toggleLinkingAction, CommonUIImages.IMG_LINK_WITH_VIEWER);

		collapseAllAction = new CollapseAllAction((TreeViewer)getNavigator().getViewer());
		collapseAllAction.useDefaultDecorators();

		fImportActionGroup = new ImportActionGroup(getNavigator());
		fNewSnapshotGrp = new Separator(CommonUITraceConstants.PROFILE_NEW_SNAPSHOT_GROUP);
		fGCGroup = new Separator(CommonUITraceConstants.PROFILE_GC_GROUP);		
		fRefreshGrp = new Separator(CommonUITraceConstants.PROFILE_REFRESH_GROUP);
		fAttachGrp = new Separator(CommonUITraceConstants.PROFILE_ATTACH_GROUP);
		fTerminateGrp = new Separator(CommonUITraceConstants.PROFILE_TERMINATE_GROUP);
		fRefreshTree = new RefreshTreeAction(getNavigator());		
		fMonitorGrp = new Separator(CommonUITraceConstants.PROFILE_MONITOR_GROUP);
		fSaveToFile = new SaveAction(getNavigator());
		fRefreshLocal = new RefreshFromLocalAction(getNavigator());	
		fDeleteAction = new DeleteAction(getNavigator());		
		openReportAction = new OpenTraceReportAction();	
		fOpenNodeAction = new OpenWizardAction(getNavigator(), OpenNodeWizard.class, CommonUITraceMessages.PROP);
	}

	/**
	 * @see org.eclipse.ui.actions.ActionGroup#dispose()
	 */
	
	
	public void dispose()
	{		
		if(traceViewsMap!=null)
		{
			traceViewsMap.clear();
			traceViewsMap = null;
		}
			
		
		if (fImportActionGroup != null)
		{
			fImportActionGroup.dispose();
			fImportActionGroup = null;
		}
			
		deleteSeparators();	
		deleteActions();	
		super.dispose();
	}
		
	protected void deleteSeparators()
	{
	   
		if(fAttachGrp != null)
		{
			fAttachGrp.dispose();
			fAttachGrp = null;
		}   
		if(fRefreshGrp != null)
		{
			fRefreshGrp.dispose();
			fRefreshGrp = null;
		}   
		if(fTerminateGrp != null)
		{
			fTerminateGrp.dispose();
			fTerminateGrp = null;
		}  
		if(fMonitorGrp != null)
		{
			fMonitorGrp.dispose();
			fMonitorGrp = null;
		}   
	
		if(fNewSnapshotGrp != null)
		{
			fNewSnapshotGrp.dispose();
			fNewSnapshotGrp = null;
		}   
		if(fGCGroup != null)
		{
			fGCGroup.dispose();
			fGCGroup = null;
		}   
	
	}

	protected void deleteActions()
	{
	
		if (fSaveToFile != null)
		{
			fSaveToFile.dispose();
			fSaveToFile = null;
		}
		if (fDeleteAction != null)
		{
				fDeleteAction.dispose();
				fDeleteAction = null;
		}
		if (fRefreshLocal != null)
		{
			fRefreshLocal.dispose();
			fRefreshLocal = null;
		}
		if (fRefreshTree != null)
		{
			fRefreshTree.dispose();
			fRefreshTree = null;
		}
		if (toggleLinkingAction != null)
		{
			toggleLinkingAction.dispose();
			toggleLinkingAction= null;
		}
	}


	
	public void fillActionBars(IActionBars actionBars)
	{
		super.fillActionBars(actionBars);
		
		IMenuManager menu = actionBars.getMenuManager();
		menu.add(new Separator());
		menu.add(toggleLinkingAction);
		
		IToolBarManager toolbar = actionBars.getToolBarManager();		
		toolbar.add(new Separator("monitoringGroup"));
		toolbar.add(new Separator("gcGroup"));
		toolbar.add(new Separator("refreshGroup"));
		toolbar.add(new Separator("navigatorGroup"));
		toolbar.add(collapseAllAction);
		toolbar.add(toggleLinkingAction);
	}
	
	/**
	 * @see org.eclipse.ui.actions.ActionGroup#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	public void fillContextMenu(IMenuManager menu)
	{		
		super.fillContextMenu(menu);			
	}
	
	
	public void fillContextMenuWithGroups(IMenuManager menu)
	{
		super.fillContextMenu(menu);
	}
	
	/**
	 * Returns true if the selections have the same type
	 * @param selection
	 * @return
	 */
	protected boolean areSameClass(IStructuredSelection selection) 
	{	
		if(selection == null || selection.isEmpty())
		  return true;
		  
		Class objClass = selection.getFirstElement().getClass();  
		Iterator iterator = selection.iterator();  
		while(iterator.hasNext())
		{
			if (!(objClass.equals(iterator.next().getClass()))) {
				return false;
			}
		}
		
		return true;
	}



	protected void updateActions(IStructuredSelection selection) {
		boolean saveEnabled = true;
		boolean deleteEnabled = true;
		
		if (selection.size() == 0)
			saveEnabled = deleteEnabled = false;
		
		Iterator i = selection.iterator();
		while (i.hasNext()) {
			Object next = i.next();
			if (next instanceof INavigatorItem) {
				INavigatorItem item = (INavigatorItem)next;
				if (!item.isSaveEnabled())
					saveEnabled = false;
				if (!item.isDeleteEnabled())
					deleteEnabled = false;
			}
		}
		
		fSaveToFile.setEnabled(saveEnabled);
		fDeleteAction.setEnabled(deleteEnabled);
		fRefreshLocal.setEnabled(fRefreshLocal.areLoadedResources());
	}

	/**
	 * Returns the list of actions available for this trace type
	 * @param type
	 */
	protected IAction[] getViewMappingForType(String type) 
	{
		// sort the items alphabetically
		IAction[] actions = (IAction[])getViewMappingForTypeUnsorted(type).clone();
		Arrays.sort(actions, new Comparator() {
			public int compare(Object o1, Object o2) {
				IAction a = (IAction)o1;
				IAction b = (IAction)o2;
				if (a != null && b != null)
					return a.getText().compareTo(b.getText());
				return 0;
			}
		});
		return actions;
	}
	
	protected IAction[] getViewMappingForTypeUnsorted(String type) 
	{
		Object mapp = traceViewsMap.get(type);
    			
		if(mapp != null)
			return ((IAction[])mapp);
    	    
		IAssociationMappingRegistry registry = TraceAssociationManager.getTraceViewMappingRegistry();
		IAssociationMapping mapping = registry.getAssociationMapping(IAssociationConstants.EP_ANALYZER_EXTENSIONS);
		
		if(mapping == null)
		{
			IAction[] actions = new IAction[0];
			traceViewsMap.put(type, actions);
			
			return actions;
		}
    		    		  
		IAssociationDescriptor[] array = mapping.getAssociationDescriptors(type);
		IAction[] actions = new IAction[array.length];
		
		for(int idx=0; idx<array.length; idx++)
		{
			IAssociationDescriptor desc = array[idx];
				
			try {
					
				IAction action = (IAction)desc.createImplementationClassInstance();
				
				if(action != null)
				{
					action.setText(desc.getName());
					action.setImageDescriptor(desc.getImageDescriptor());
				
					actions[idx] = action;										
				}								
			} catch (Exception exc) {
				exc.printStackTrace();
			}
								
		}
		
		traceViewsMap.put(type, actions);
		return actions;
    	        	  
	}
	
	
	/**
	 * Indicates that a property has changed.
	 * @see org.eclipse.ui.IPropertyListener
	 */
	public void propertyChanged(Object source, int propId)
	{
		//reset trace view actions
		if(source instanceof IAssociationMappingRegistry)
			traceViewsMap.clear();
	}
	
	public void handleDoubleClick(DoubleClickEvent event) {
		
		IStructuredSelection s = (IStructuredSelection) event.getSelection();
		Object obj = s.getFirstElement();

		if(obj == null || !((obj instanceof EObject) || (obj instanceof INavigatorItem)))
		  return;

		IAction action = getDefaultAction(obj);
		if (action != null) {
			action.run();
		}
	}
	
	/**
	 * Returns the default action for the specified object
	 * (for when the user double clicks on it).
	 */
	protected IAction getDefaultAction(Object obj) {
		// this set stays sorted
		Set children = new TreeSet(new Comparator() {
			public int compare(Object o1, Object o2) {
				if (o1 instanceof INavigatorItem && o2 instanceof INavigatorItem) {
					return ((INavigatorItem)o1).getText().compareTo(((INavigatorItem)o2).getText());
				}
				return o1.hashCode() - o2.hashCode();
			}
		});

		if(obj instanceof TRCMonitor) {
			IAction action = getDefaultViewMappingForType(CommonUITraceConstants.MONITOR_TYPE);
			if (action != null) {
				return action;
			}
			children.addAll(((TRCMonitor)obj).getNodes());
		}
		else if(obj instanceof TRCNode) {
			IAction action = getDefaultViewMappingForType(CommonUITraceConstants.HOST_TYPE);
			if (action != null) {
				return action;
			}
			children.addAll(((TRCNode)obj).getProcessProxies());
		}
		else if(obj instanceof TRCProcessProxy)	{
			children.addAll(((TRCProcessProxy)obj).getAgentProxies());
		}
		else if(obj instanceof CorrelationContainerProxy)	{
			children.addAll(((CorrelationContainerProxy)obj).getCorrelatedAgents());
		}
		else if(obj instanceof TRCAgentProxy) 
		{
			/*
			 * Find out whether there exists a viewer for any
			 * of the profiling types.
			 */
			boolean hasViewer = false;
			Iterator iter = children.iterator();
			while (iter.hasNext()) {
				if (getDefaultAction(iter.next()) != null) {
					hasViewer = true;
					break;
				}
			}
			
			/*
			 * If not, fall back to the agent's default viewer.
			 */
			if (!hasViewer) {
				IAction action = getDefaultViewMappingForType(((TRCAgentProxy)obj).getType());
				if (action != null) {
					return action;
				}
			}
		}
		else if(obj instanceof INavigatorItem) {
			IAction action = getDefaultViewMappingForType(((INavigatorItem)obj).getType());
			if (action != null) {
				return action;
			}
		}
	
		return null;
	}
	
	
	/**
	 * Returns the default action for this type; when the user
	 * double-clicks.
	 */
	protected IAction getDefaultViewMappingForType(String type) 
	{
		IAction[] actions = getViewMappingForTypeUnsorted(type);
		if (actions.length >= 1) {
			return actions[0];
		}
		return null;
	}
}
