/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PDCoreUtil.java,v 1.8 2009/12/14 22:20:34 ewchan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.common.ui.trace.internal.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfiguration;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessImpl;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.security.internal.util.ConnectUtil;
import org.eclipse.hyades.trace.internal.ui.PDContentProvider;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITracePlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.ui.PlatformUI;

import com.ibm.icu.text.DateFormat;


public class PDCoreUtil {
	
	/* The modes that the PI agent can run in */
    public static final byte PI_MODE_CONTROLLED = 0x01;
    public static final byte PI_MODE_APPLICATION = 0x02;
    public static final byte PI_MODE_STANDALONE = 0x03;
    public static final byte PI_MODE_ENABLED = 0x04;
	 

	public static void activate(TRCAgentProxy agentProxy) {
		//deactivate previous agents
		for (Iterator iter = agentProxy.getProcessProxy().getAgentProxies().iterator(); iter.hasNext();) {
			TRCAgentProxy element = (TRCAgentProxy) iter.next();
			try {
				if(agentProxy.getName().equals(element.getName()) && agentProxy.getType().equals(element.getType()))
				{
					element.setActive(false);
					element.setAttached(false);
				}
			} catch (NullPointerException e) {
				// ignore null pointer exceptions
			}
		}
		agentProxy.setActive(true);
		agentProxy.setAttached(true);
		
	}

    public static TRCAgentProxy createAgent(TRCProcessProxy process, Agent a) {
        TRCAgentProxy agentProxy = createAgent(process, a, "");

        //	setSessionStartTime(a.get, agentProxy);
        return agentProxy;
    }

	public static TRCAgentProxy getCorrespondingAgent(TRCProcessProxy processProxy,Agent ag,boolean activeProxy) {
		Iterator i = processProxy.getAgentProxies().iterator();

		while (i.hasNext()) {
			TRCAgentProxy agentProxy = (TRCAgentProxy) i.next();

			if (agentProxy.eIsProxy()) {
				continue;
			}

			if ((agentProxy != null) && agentProxy.getName().equals(ag.getName()) && (agentProxy.getRuntimeId() == null || agentProxy.getRuntimeId().equals(ag.getUUID())) && agentProxy.isActive()==activeProxy) {
				return agentProxy;
			}
		}

		return null;
	}

    public static TRCAgentProxy createAgent(TRCProcessProxy processProxy, Agent agent, String sessionId) 
    {
        TRCAgentProxy agentProxy = getCorrespondingAgent(processProxy,agent,false);
        if(agentProxy!=null)
        	return agentProxy;

        return createAgentNoCheck(processProxy, agent, sessionId);
    }
    


    /**
     * Insert the method's description here.
     * Creation date: (10/05/2000 10:03:14 AM)
     * @param path org.eclipse.core.resources.IPath
     */
    public static final IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = ResourcesPlugin.getWorkspace();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {

            if (idx == 0) { //project

                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());

                if ((project == null) || !project.exists()) { //create the project

                    try {
                        project.create(null);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                try {
                    project.open(null);
                } catch (Exception e) {
                }

                container = project;
            } else // (idx > 1)
             { //folder

                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));

                if ((folder == null) || !folder.exists()) { //create the folder

                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
//                        e.printStackTrace();
                    }
                }

                container = folder;
            }
        }

        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
//            exc.printStackTrace();
        }

        return container;
    }

    public static final TRCMonitor createMonitor(IContainer res, String name) {

        TRCMonitor monitor = null;
        ArrayList monitors = PDContentProvider.getMonitors(res);

        for (int idx = 0; idx < monitors.size(); idx++) {
            Object mon = monitors.get(idx);

            if (mon instanceof TRCMonitor && ((TRCMonitor) mon).getName().equals(name)) {
                return (TRCMonitor) mon;
            }
        }

        String monitorName = name;//TString.change(name, " ", "");
        String monitorFileName = monitorName + "." + CommonUIConstants.MONITOR_EXT;

        IPath filePath = res.getFullPath().append(monitorFileName);

        URI uri = URI.createPlatformResourceURI(filePath.toString(),true);
        Resource monDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        monDoc.setModified(true);
        EList monExt = monDoc.getContents();

        TraceUIManager.getTraceUIManager().getResourceSet().getResources().add(monDoc); // prevents reloading later

        HierarchyFactory factory = TraceUIManager.getTraceUIManager().getPerftraceFactory();

        monitor = factory.createTRCMonitor();
        monitor.setName(name);

        monExt.add(monitor);

        PDContentProvider.addMonitor(res, monitor);

        return monitor;
    }

    /**  
     * This is a helper class that search for any existing node under a monitor for a given node name
     * node name given is compared with the list of nodes under the monitor,ignoring case considerations.
     * 
     * @param monitor monitor to be searched under
     * @param name hostname
     * @return null if node does not exist, otherwise node with matching name.
     */
    public static TRCNode findNode(TRCMonitor monitor, String name) {    	
    	Object[] nodes = monitor.getNodes().toArray();

        for (int idx = 0; idx < nodes.length; idx++) {
            TRCNode node = (TRCNode) nodes[idx];

            if (node.eIsProxy()) {
                continue;
            }

            if (node.getName().equalsIgnoreCase(name)) {
                return (TRCNode) node;
            }
        }
    	
    	return null;
    }
    
    public static final TRCNode createNode(TRCMonitor monitor, String name, String portNumber) {
        String ipAddress = "";

        try {
            Node fnode = NodeFactory.createNode(name);
            name = fnode.getInetAddress().getHostName();
            ipAddress = fnode.getInetAddress().getHostAddress();
        } catch (UnknownHostException exc) {
        }

        TRCNode node = findNode(monitor, name);
        
        if (node!=null) {
        	node.setPort(Integer.parseInt(portNumber));
        	return node;
        }
        
        String nodeName = monitor.getName()+ "_" + TString.change(name, " ", "");//TString.change(monitor.getName(), " ", "") + "_" + TString.change(name, " ", "");
        String nodeFileName = nodeName + "." + CommonUIConstants.NODE_EXT;

        String monPath = monitor.eResource().getURI().toString();
        IPath path = new Path(monPath);

        if (path.segmentCount() > 1) {
            monPath = path.removeLastSegments(1).toString();
        }

        IPath filePath = new Path(monPath).append(nodeFileName);

        URI uri = SaveUtil.createURI(filePath.toString()+"#").trimFragment();
        Resource nodeDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        nodeDoc.setModified(true);
        EList nodeExt = nodeDoc.getContents();

        TraceUIManager.getTraceUIManager().getResourceSet().getResources().add(nodeDoc); // prevents reloading later

        HierarchyFactory factory = TraceUIManager.getTraceUIManager().getPerftraceFactory();

        node = factory.createTRCNode();
        node.setName(name);
        node.setIpAddress(ipAddress);
        node.setMonitor(monitor);
        node.setPort(Integer.parseInt(portNumber));
        nodeExt.add(node);

        EList mon_nodes = monitor.getNodes();

        try {
            mon_nodes.add(node);
        } catch (Exception e) {
            e.printStackTrace();
        }

        return node;
    }

    public static TRCProcessProxy createProcess(TRCNode node, Object process, boolean isProcessLaunched) 
    {
    	Process oldProcess = null;
    	IProcess newProcess = null;
    	
    	if (process instanceof Process)
    		oldProcess = (Process)process;
    	else if (process instanceof IProcess)
    		newProcess = (IProcess)process;
    	else
    		return null;
    	
    	
    	long processId;
    	String processUUID, processName, processExecutable;
    	
    	try
    	{
	    	if (oldProcess == null)
	    	{
	    		processId = newProcess.getProcessId();	
	    		processUUID = isProcessLaunched ? newProcess.getUUID() : null;
	    		processName = newProcess.getName();
	    		processExecutable = newProcess.getExecutable();
	    	}
	    	else
	    	{
	    		processId = Long.parseLong(oldProcess.getProcessId());
	    		processUUID = isProcessLaunched ? oldProcess.getUUID() : null;
	    		processName = oldProcess.getName();
	    		processExecutable = oldProcess.getExecutable();
	    	}
    	}
    	catch (Exception e)
    	{
    		CommonUITracePlugin.getDefault().log(e);
    		return null;
    	}
    	
    	
        TRCProcessProxy trcProcessProxy = null;
        EList processes = node.getProcessProxies();
        Iterator nodeProcesses = processes.iterator();        
        
        /* For every process under the node check to see if the process already exists */
        while (nodeProcesses.hasNext()) 
        {
            trcProcessProxy = (TRCProcessProxy) nodeProcesses.next();
            if (trcProcessProxy.eIsProxy() || trcProcessProxy.getPid() != processId) 
            {
                continue;
            }

            
            /* Process already exists - return it */
            if (isProcessLaunched && (trcProcessProxy.getRuntimeId() != null) && trcProcessProxy.getRuntimeId().equals(processUUID)) 
            {
                return trcProcessProxy;
            }

            /* Check if the process is active; if not, this is a different process whith same id */
            Iterator agents = trcProcessProxy.getAgentProxies().iterator();
            while (agents.hasNext()) 
            {
                TRCAgentProxy agent = (TRCAgentProxy) agents.next();

                if (agent.eIsProxy()) 
                {
                    continue;
                }

                if (agent.isActive()) 
                {
                    return trcProcessProxy;
                }
            }
        }

        TRCMonitor monitor = node.getMonitor();

        String rID = (new Date()).toString();
        rID = TString.change(rID, " ", "");
        rID = TString.change(rID, ":", "");

        String processQualifiedName = monitor.getName() + "_" + node.getName() + "_" + processId + "_" + rID;
        String fileName = processQualifiedName + "." + CommonUIConstants.PROCESS_EXT;

        String monPath = node.eResource().getURI().toString();
        IPath path = new Path(monPath);

        if (path.segmentCount() > 1) {
            monPath = path.removeLastSegments(1).toString();
        }

        IPath filePath = new Path(monPath).append(fileName);

        URI uri = SaveUtil.createURI(filePath.toString()+"#").trimFragment();

        Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        pDoc.setModified(true);
        EList pExt = pDoc.getContents();

        TraceUIManager.getTraceUIManager().getResourceSet().getResources().add(pDoc); // prevents reloading later
        HierarchyFactory factory = TraceUIManager.getTraceUIManager().getPerftraceFactory();

        //create process element
        trcProcessProxy = factory.createTRCProcessProxy();
       
        /* Ensure the runtime UUID is set on the process object */
        if (isProcessLaunched)
        {
        	trcProcessProxy.setPid((int)processId);
        	trcProcessProxy.setRuntimeId(processUUID);
        }
        	            
        if (processName != null) 
        {
            trcProcessProxy.setName(processName);
        } 
        else 
        {
            trcProcessProxy.setName(processExecutable);
        }
                    

        trcProcessProxy.setLocation(process instanceof ProcessImpl ? ((ProcessImpl)process).getlocation() : null);
        trcProcessProxy.setActive(true);
        trcProcessProxy.setLaunchMode(0);                     

        trcProcessProxy.setNode(node);
        pExt.add(trcProcessProxy);

        return trcProcessProxy;
    }
    
    public static TRCProcessProxy createProcess(TRCNode node, Object process) 
    {
    	return createProcess(node, process, true);
    }
    
    
    public static void detachAgents() {
		
        for (Iterator e = HierarchyResourceSetImpl.getInstance().getResources().iterator(); e.hasNext();) {
            Resource res = ((Resource) e.next());
            EList ext = res.getContents();
            Iterator i = ext.iterator();

            while (i.hasNext()) {
                Object obj = i.next();

				TRCAgentProxy agent = null;
				if(obj instanceof TRCAgent)
					agent = ((TRCAgent)obj).getAgentProxy();
				else if(obj instanceof TRCAgentProxy)
                   agent = (TRCAgentProxy) obj;					
					
                if (agent != null) {
                    TRCProcessProxy process = agent.getProcessProxy();

                    if ((process == null) || !process.isActive() || !agent.isActive()) {
                        continue;
                    }
					
                    try {
                        Object instance = LoadersUtils.locateAgentInstance(agent);

                        if ((instance != null) && instance instanceof Agent) {
                            Agent a = (Agent) instance;

							if(a != null)
							{
								
								if(process.getLaunchMode() != 0)//launch
								{
									a.getProcess().kill(0);									
								}
								else
								{//attached
		                            if (a.isAttached()) {
		                                a.detach();
		                            }
									
								}
							}
							
                        }
                    } catch (Exception exc) {
                        exc.printStackTrace();
                    }
                }
            }
        }
    }

	
	/**
	 * Returns the appropriate VM argument that will cause the application
	 * to run with the PI agent in the mode that is specified.
	 * 
	 * @param mode The mode that the PI agent should run in.  See PI_MODE_* constants.
	 */
	public static String getPIVMArgument (byte mode)
	{
		String param = "";
		
		String piFlag = "-XrunpiAgent:server=";
		switch (mode)
		{
			case PI_MODE_CONTROLLED:
				piFlag += "controlled";
				break;
			case PI_MODE_APPLICATION:
				piFlag += "application";
				break;
			case PI_MODE_STANDALONE:
				piFlag += "standalone";
				break;
			case PI_MODE_ENABLED:
				piFlag += "enabled";
				break;
		}
		
		param = piFlag + param;
		return param;
	}
	
	   
    public static void setAgentConfiguration(TRCAgentProxy agent, Agent a) {
        AgentConfiguration aconfig = a.getConfiguration();

        aconfig.clear();

        //set profiling options
        if (agent.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE)) {
            Object[] configs = agent.getConfigurations().toArray();

            if (configs.length > 0) {
                TRCConfiguration config = (TRCConfiguration) configs[0];

                Object[] filters = config.getFilters().toArray();

                for (int idx = 0; idx < filters.length; idx++) {
                    TRCFilter filter = (TRCFilter) filters[idx];

                    if (filter.getActive().booleanValue()) {
                        AgentConfigurationEntry entry = new AgentConfigurationEntry();

                        /* RKD:  If the method portion of the filter is an empty String then this is a class filter */
                        if (filter.getOperation().equals("") || filter.getOperation().equals("\"\"")) {
                            entry.setName(filter.getPattern());
                            entry.setType("SETFILTER");
                            entry.setValue(filter.getMode());
                            entry.setEnabled(filter.getActive().booleanValue());
                        } else {
                            /* Method Filter */
                            entry.setName(filter.getPattern() + ' ' + filter.getOperation());
                            entry.setType("SETMETHODFILTER");
                            entry.setValue(filter.getMode());
                            entry.setEnabled(filter.getActive().booleanValue());
                        }

                        aconfig.addEntry(entry);
                    }
                }

                Object[] options = config.getOptions().toArray();

                for (int idx = 0; idx < options.length; idx++) {
                    TRCOption option = (TRCOption) options[idx];

                    AgentConfigurationEntry entry = new AgentConfigurationEntry();

                    entry.setType("SETOPTION");
                    entry.setName(option.getKey());
                    entry.setValue(option.getValue());

                    aconfig.addEntry(entry);
                }
            }
        }
    }

    /*
     * get the folder structure of a fully qualify file name
     */
    public static String getDirectoryName(String fName) {
        int i = fName.lastIndexOf(File.separatorChar);

        if (i >= 0) {
            fName = fName.substring(0, i);
        }

        return fName;
    }

    
     /*
     * check the folders existence of a fully qualify file path
     */
    public static boolean isPathExists(String fdir) {
        File pf = new File(getDirectoryName(fdir));

        return pf.exists();
    }

    public static boolean isZipFile(InputStream readStream) throws IOException {
        byte[] magic = new byte[2];

        if (readStream.read(magic) > 1) {
            if ((magic[0] == 'P') && (magic[1] == 'K')) {
                return true;
            }
        }

        return false;
    }

    public static boolean isZipFile(String fileName) throws IOException {
        FileInputStream readStream = new FileInputStream(fileName);
        boolean ret = isZipFile(readStream);

        readStream.close();
        readStream = null;

        return ret;
    }

    public static boolean createDirectoryMessage(String fDir, Shell shell) {
        File directory = new File(PDCoreUtil.getDirectoryName(fDir));
        String msg = NLS.bind(CommonUITraceMessages.FCT_M, directory.getPath());

        if (!MessageDialog.openQuestion(shell, CommonUITraceMessages.TRC_MSGT, msg)) {
            return false;
        }

        if (!directory.mkdirs()) {
            String errMsg = CommonUITraceMessages.ERR_FCT;

            MessageDialog.openError(shell, CommonUITraceMessages.TRC_MSGT, errMsg);

            return false;
        }

        return true;
    }

    public static TRCAgentProxy createUniqueAgent(TRCProcessProxy processProxy, Agent agent) {
        long timeStamp = (new Date()).getTime();
        double time = ((double) timeStamp) / 1000;
        TRCAgentProxy agentProxy = createAgentNoCheck(processProxy, agent, timeStamp + "");
		LoadersUtils.registerAgentInstance(agentProxy,agent);
        LoadersUtils.setSessionStartTime(time, agentProxy);

        return agentProxy;
    }
  

    protected static TRCAgentProxy createAgentNoCheck(TRCProcessProxy process, Agent agent, String sessionId) {
        TRCAgentProxy agentProxy;
        String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date());
        String rID = timestamp;

        rID = TString.change(rID, " ", "");
        rID = TString.change(rID, ":", "");

        String agentName = new StringBuffer(process.getNode().getMonitor().getName()).append("_").append(process.getNode().getName()).append("_").append(process.getPid()).append("_").append(rID).append("_").append(sessionId).append("_").append(agent.getName()).toString();


        String pPath = process.eResource().getURI().toString();
        IPath path = new Path(pPath);

        if (path.segmentCount() > 1) {
            pPath = path.removeLastSegments(1).toString();
        }

        String fileName = agentName + "." + CommonUIConstants.AGENT_EXT;
        IPath filePath = new Path(pPath).append(fileName);

        URI uri = SaveUtil.createURI(filePath.toString()+"#").trimFragment();

        Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList agExt = agDoc.getContents();

        TraceUIManager.getTraceUIManager().getResourceSet().getResources().add(agDoc); // prevents reloading later

        HierarchyFactory factory =  TraceUIManager.getTraceUIManager().getPerftraceFactory();

        agentProxy = factory.createTRCAgentProxy();
        agentProxy.setName(agent.getName());

        if (agent.getType() != null) {
            agentProxy.setType(agent.getType());
        }

        agentProxy.setProcessProxy(process);

        TRCAgent iAgent = factory.createTRCAgent();

        iAgent.setAgentProxy(agentProxy);
        agExt.add(iAgent);

		markModifiedResources(iAgent,true);		

        if (agentProxy.getAgent() != null) {
            agentProxy.getAgent().setCollectionMode(agentProxy.getCollectionMode());
        }

        iAgent.setName(agentProxy.getName());
        iAgent.setRuntimeId(agentProxy.getRuntimeId());
        iAgent.setType(agentProxy.getType());
        iAgent.setCollectionMode(agentProxy.getCollectionMode());

        return agentProxy;
    }

    /**
     * Sets the modified flag on the Hierarchy resources up to the monitor.
     * 
	 */
	public static void markModifiedResources(EObject eObject, boolean modifiedState) {
		while(true)
		{
			if(eObject==null)
				break;
			if(eObject instanceof TRCAgent)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCAgent)eObject).getAgentProxy().getProcessProxy();
				continue;
			}
			if(eObject instanceof TRCAgentProxy)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCAgentProxy)eObject).getProcessProxy().getNode();
				continue;
			}
			if(eObject instanceof TRCProcessProxy)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCProcessProxy)eObject).getNode();
				continue;
			}
			if(eObject instanceof TRCNode)
			{
				eObject.eResource().setModified(modifiedState);
				eObject = ((TRCNode)eObject).getMonitor();
				continue;
			}
			if(eObject instanceof TRCMonitor)
			{
				eObject.eResource().setModified(modifiedState);
			}
			break;
		}
	}

	
	/**
	 * Returns true if the currently chosed perspective is the profiling & logging
	 * perspective. 
	 */
	public static boolean isProfilingPerspective()
	{
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getPerspective().getId().equals(CommonUITraceConstants.ID_TRACE_PERSPECTIVE);
	}
	
	
	public static boolean isLocalHost(String hostname)
	{
		String name = CommonUITraceConstants.LOCAL_HOST;
		if (name.equals(hostname))
			return true;
		
		try {
			  Node fnode = NodeFactory.createNode(name);
			  name = fnode.getInetAddress().getHostName();
		}
		catch(UnknownHostException exc)
		{
			return false;
		}

	    if(hostname.equals(name))
	      return true;
		
		return false;
	}
	
    public static Node profileConnect(String host, String port, boolean showErrorMsgs) {
        String text = "";

        if (((host == null) || (host.trim() == "") || (port == null) || (port.trim() == "")) && showErrorMsgs) {
            text = CommonUITraceMessages.INVALID_HOST;

            final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, text, null);

            Display.getDefault().syncExec(new Runnable() {
            	public void run() {
            		ErrorDialog.openError(CommonUITracePlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), CommonUITraceMessages.TRC_MSGT
            				, CommonUITraceMessages.CON_FL, err);
            	}
            });

            return null;
        }

        IPreferenceStore store = CommonUITracePlugin.getDefault().getPreferenceStore();
        ConnectUtil util = new ConnectUtil(host, port, store.getString(CommonUIConstants.SECURE_USERID), CommonUITracePlugin.getDefault());
        int result = util.connect(showErrorMsgs);

        if (result == ConnectUtil.CONNECTION_SUCCESS) { //store the user id
            store.setValue(CommonUIConstants.SECURE_USERID, util.getUserId());

            return util.getNode();
        }

        return null;
    }
    
    public static Node profileConnect(String host, String port) {
    	return profileConnect(host, port, true);
    } 
	
}
