/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DXYSurface.java,v 1.4 2008/05/23 14:11:50 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.StringTokenizer;

import org.eclipse.tptp.platform.report.core.internal.DChildrenController;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DItem;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlError;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlReader;
import org.eclipse.tptp.platform.report.drivers.xml.internal.DXmlWriter;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.extension.internal.DExtensionRegistry;
import org.eclipse.tptp.platform.report.extension.internal.IDExtension;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;


/**
 * Implementation of IXYSurface that can be inserted as child of DGraphic object.
 * All Z values are stored inside one dimensionnal double array.
 * 
 * @see IXYSurface, DGraphic
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DXYSurface extends DItem implements IXYSurface
{  
  
  protected int sizex_,sizey_; // number of points along axis X and Y 
  protected double z_[], zmin_,zmax_; //Z values and Z min, Z max
  
  /** 
   * Create an empty surface, use setZ() to set surface data.
   */
  public DXYSurface()
  {
    z_ = new double[1];
  }
  
  /**
   * Create a surface, all z values are 0
   */
  public DXYSurface( int sizex, int sizey )
  {
    sizex_=sizex; sizey_=sizey;
    z_ = new double[ sizex_*sizey_ ];
  }
  
  /**
   * Create a surface using external parameters, array is not copied.
   */
  public DXYSurface( int sizex, int sizey, double []z, double zmin, double zmax )
  {
    sizex_=sizex; sizey_=sizey;
    z_=z;
    zmin_=zmin; zmax_=zmax;
  }
  
  /**
   * Change surface parameters, array is taken "as is", must have sizex*sizey size
   * and is not copyied.
   * This method allows to set all Z value in a array before create surface.
   */
  public void setZ( int sizex, int sizey, double []z, double zmin, double zmax )
  {
    sizex_=sizex; sizey_=sizey;
    z_=z;
    zmin_=zmin; zmax_=zmax;
  }
  
  /** 
   * Change the value of one Z at given place
   * zmiin and  zmax aren't updated.
   */
  public void setZ( int x, int y, double z )
  {
    z_[y*sizex_+x] = z;
  }
  
  public int getSizeX() { return sizex_; }
  public int getSizeY() { return sizey_; }
  public double getZMin() { return zmin_; }
  public double getZMax() { return zmax_; }
  public double getZ( int x, int y ) { return z_[(int)(y*sizex_+x)]; }
  
  /**
   * Parse all Z array to setup zmin and zmax.
   */
  public void updateZMinMax()
  {
    if( z_==null || z_.length==0 ) return ;
    zmin_ = zmax_ = z_[0];
    for( int i=0; i<z_.length; ++i )
    {
      double z=z_[i];
      if( z<zmin_ ) zmin_=z; else if ( z>zmax_) zmax_=z;
    }
  }
  
  
  public static class Extension implements IDExtension, DXmlReader.IDItemFactory
  {
    public void updateExtensible(DExtensible ext)
    {
          if (ext instanceof DXmlWriter)
               ext.installDoMethods( "writeXml", this);
          else 
          if (ext instanceof DXmlReader) {
             ext.installDoMethods( "readXml", this);
             ((DXmlReader)ext).addIDItemFactory( this );
          }
    }
    /** write JSML */
    public void writeXml( DXYSurface surf, DExtensible p, Object arg )
    {
      DXmlWriter writer = (DXmlWriter)p;
      
      writer.writeExtension( surf );
      
      writer.startElement("XYSURFACE",surf);
	  writer.outAttribute( "SIZEX", surf.sizex_ );
	  writer.outAttribute( "SIZEY", surf.sizey_ );
	  writer.outAttribute( "ZMIN", surf.zmin_ );
	  writer.outAttribute( "ZMAX", surf.zmax_ );	  

	  //save data as hexadecimal in CDATA
	  if( surf.z_!=null )
	  {
	    int len = surf.z_.length;
	    ByteArrayOutputStream bos = new ByteArrayOutputStream(len*9);
	    try{
	      int i=0;
	      for( int j=0; j<surf.sizey_; j++ )
	      {
	        for( int k=0; k<surf.sizex_; ++k,i++ )
	        {
	          bos.write( Double.toString(surf.z_[i]).getBytes("UTF-8") );
	          bos.write(' ');
	        }
	        bos.write('\n');
	      }
	      writer.outCData( bos.toString("UTF-8") );
	    } catch( IOException ex ) {
	      ex.printStackTrace();
	      throw new DXmlError( ex.getMessage() );
	    }
	  }
	  
	  writer.doChildrenItem(surf,p,arg);
		
	  writer.endElement();
    }
    
    private int getInteger( String name, NamedNodeMap attr )
    {
      Node n = attr.getNamedItem( name );
	  if (n == null) 
		throw new DXmlError(name +" attribute not defined while parsing XYSURFACE element");
	  String val = n.getNodeValue();
	  try{
	    int v = Integer.parseInt( val );
	    return v;
	  } catch( NumberFormatException e ) {
	    throw new DXmlError(name+" must be an integer value");
	  }
    }
    private double getDouble( String name, NamedNodeMap attr )
    {
      Node n = attr.getNamedItem( name );
	  if (n == null) 
		throw new DXmlError(name +" attribute not defined while parsing XYSURFACE element");
	  String val = n.getNodeValue();
	  try{
	    double v = Double.parseDouble( val );
	    return v;
	  } catch( NumberFormatException e ) {
	    throw new DXmlError(name+" must be a double value");
	  }
    }
    
    /** read JSML */
    public void readXml( DXYSurface surf, DExtensible p, Object p_arg )
    {
      DXmlReader reader = (DXmlReader)p ;
      DXmlReader.Arg arg = (DXmlReader.Arg)p_arg;

      NamedNodeMap attr = arg.node.getAttributes();
      reader.setNodeStyle(attr,surf);
      
      int sizex = getInteger( "SIZEX", attr );
      int sizey = getInteger( "SIZEY", attr );
      double zmin = getDouble( "ZMIN", attr );
      double zmax = getDouble( "ZMAX", attr );
      
      
      //as some parser "cut" the text data into several node, sometimes they cut the text
      //inside a number, meaning we got a NumberFormatException, or bad number parsing
      //meaning we must to have in memory the whole text....
      String text="";
      nodes:
      for( Node cdata=arg.node.getFirstChild(); (
           cdata!=null); 
           cdata=cdata.getNextSibling()) 
      {
        if( cdata.getNodeType()==Node.TEXT_NODE )
        {
          text += cdata.getNodeValue();
        }
      }
      int index=0;
      int len = sizex*sizey;
      double values[] = new double[len];
      StringTokenizer tok = new StringTokenizer( text, " \n" );
      //read only len doubles as text can contains other stuff 
      //I hope "other stuff" isn't inserted before the number I expect...
      while( tok.hasMoreTokens() )
      {
        String stok = tok.nextToken();
        try{             
          values[index++] = Double.parseDouble( stok );
          if( index>=len ) break;
        } catch( NumberFormatException ex ) {          
          throw new DXmlError( ex.getMessage() );
        }
      }
      surf.setZ( sizex, sizey, values, zmin, zmax );
      if( index!=len ) surf.updateZMinMax();

      reader.doChildrenItem( surf, p, arg );
    }
    
    public IDItem createIDItem(Node node) 
    {
      if ( node != null && node.getNodeType()==Node.ELEMENT_NODE 
        && ( "XYSURFACE".equals( node.getNodeName())
             || "xysurface".equals( node.getNodeName())))
      {
        return new DXYSurface();
      }
      return null;
    } 
  }
  
  // install extension...
  static private Extension extension ;
  static
  {    
    extension = new Extension();
    //allow DXYSurface to be a child of DGraphic 
    DChildrenController.installChildRule( DGraphic.class, DXYSurface.class );
    DExtensionRegistry.addExtension( extension, DXmlWriter.class );
    DExtensionRegistry.addExtension( extension, DXmlReader.class );
  }
  
  
}