/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultChartRender.java,v 1.10 2008/12/13 06:38:29 ewchan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;

import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DI18N;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DPropertyStore;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.DWallpaper;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.core.internal.IDCoord;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDRenderable;
import org.eclipse.tptp.platform.report.core.internal.IDWallpaper;
import org.eclipse.tptp.platform.report.drawutil.internal.DrawUtilIGC;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.igc.alg.internal.LineAlg;
import org.eclipse.tptp.platform.report.igc.internal.IBrush;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IPen;
import org.eclipse.tptp.platform.report.igc.internal.IRect;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Point;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.render.internal.IRender;
import org.eclipse.tptp.platform.report.render.internal.IRenderLocation;
import org.eclipse.tptp.platform.report.render.internal.IRenderMonitor;
import org.eclipse.tptp.platform.report.tools.internal.DAlignment;
import org.eclipse.tptp.platform.report.tools.internal.DAlignmentPair;
import org.eclipse.tptp.platform.report.tools.internal.IDIImageProvider;
import org.eclipse.tptp.platform.report.tools.internal.IVObject;
import org.eclipse.tptp.platform.report.tools.internal.VDouble;

import com.ibm.icu.util.ULocale;


/**
 * ChartRender is made to 
 * - draw DGraphic item in an area
 * - locate a point on a DGraphic area.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DefaultChartRender implements IRender
{
  /** temporary data during draw/locate */
  protected DefaultChartRenderData d_ ; 

  /** Create a chart render */
  public DefaultChartRender()
  {
    d_=null;
  }
  
  /** @return true since default render can draw anything...
   * even if don't know how to do...
   */
  public boolean canRender( IDRenderable _r )
  {
    return (_r instanceof DGraphic);
  }
  
  /** Draw graphic _g on rectangle _R using GC _gc. */
  public void draw( IGC _gc, IRect _r, IDRenderable _g, float _scale, IDIImageProvider _ip, IRenderMonitor _m, IVObject _data )
  {
    if( _g==null ) return ;
    if (!( _g instanceof DGraphic)) return;
    
    d_ = new DefaultChartRenderData( true, _gc, _scale );
    try {
      render( (DGraphic)_g, _r, _scale, _ip,  _data );
    }
    catch( DefaultRenderChartLocation _l) {}//musn't appears
    d_ = null;
  }
  
  /** Return a location for the given point, GC must be provided to compute text extends.
   * can return null if no location is found.
   */
  public IRenderLocation locate( int _x, int _y, IGC _gc, IRect _r, IDRenderable _g, float _scale, IDIImageProvider _ip, IVObject _data )
  {
    if(  _g==null) return null;
    if (!( _g instanceof DGraphic)) return null;
    if(!_r.contains(_x,_y) ) return null;

    d_ = new DefaultChartRenderData( false, _gc, _scale );
    d_.lx_ = _x;
    d_.ly_ = _y;
    
    try {
      render( (DGraphic)_g, _r, _scale, _ip, _data );
    }
    catch( DefaultRenderChartLocation _l )
    {
      return _l;
    }
    finally
    {
      d_=null;
    }
    return null;
  }

  private static DefaultChartRenderHBars hbars_render;
  
  /**
   * draw or locate point _G graphic in rectangle _R
   * data if not null will be used to store data from one rendering to another.
   * since those data can contains system resource such ad Font,Color ...
   * This is of respnsability of the render() caller who set data!=null to check if
   * store object is an IDisposable implementation, and if it is to call dispose...
   * @see IVObject IDisposable
   */
  protected void render( DGraphic _G, IRect _R, float scale_, IDIImageProvider _ip, IVObject data ) throws DefaultRenderChartLocation
  {
    RenderPersistData rpd = null;
    boolean dispose_rpd = false;
    //setup margin because it depends on graphic property and device
    d_.setupMargin( _G );
    try
    {
      {
        Object d = (data ==null ? null : data.getObject());
        if( d instanceof RenderPersistData )
        {
          rpd = (RenderPersistData)d;
          //check if the same graphic ... and same rectangle area...
          if( rpd.graphic != _G 
           || rpd.graphic_type != _G.getRenderableId()
           || !rpd.graphic_rect.equals( _R ) )
          {
            //do a full invalidation
            rpd = null;
          }
        }
      }
      if( rpd==null )
      {
        rpd = new RenderPersistData( _G );
        rpd.graphic_rect = new Rect( _R );//copy rect to prevent value changes
        rpd.graphic_type = _G.getRenderableId();
        rpd.ag_rect = new Rect(_R); //this "final inner drawing area" will be modified by title/legend/...
        rpd.ag_rect.shrink( d_.margin_ );

        rpd.g_style = new IGCDStyle( _G, d_.scale_ );

        //wallpaper ?
        if( rpd.graphic.getStyle()!=null )
        {
          IDWallpaper wp = rpd.graphic.getStyle().getWallpaper();
          if( wp instanceof DWallpaper )
          {
            if( rpd.g_wallpaper==null && _ip!=null  )
            {
              rpd.g_wallpaper = _ip.getImage( wp.getName() );
              rpd.e_wallpaper = wp.getEffect();
            }
          }
        }

        if( data!=null ) 
        {
          data.setObject( rpd );
        } else {
          dispose_rpd = true; 
        }
      }
      
      //graphics background
      if( d_.drawing() )
      {
        d_.gc_.setBrush( new SolidBrush( rpd.g_style.getBack()) );
        d_.gc_.fillRect( rpd.graphic_rect );
        //wallpaper:
        if( rpd.g_wallpaper!=null )
        {
          DrawUtilIGC.drawImage( d_.gc_, rpd.g_wallpaper, rpd.graphic_rect, scale_, rpd.e_wallpaper );
        }
      } 
      

      //if number of curve changed .. rebuild curves array and invalidate legend
      int n_curves=0;
      for( IDItem i=rpd.graphic.getFirstChild(); i!=null; i=i.getNext())
      {
        if( i instanceof DCurve ) n_curves++;
      }
      //changed ?
      if( rpd.curve_styles==null || n_curves !=rpd.curve_styles.length )
      {
        rpd.rebuildCurveStyles( n_curves, d_.scale_ );
        rpd.invalidate( RenderPersistData.V_LEGEND );
      }
      
      //graphic title, update rg bounds (if rpd invalid)
      renderGraphicTitle( rpd );

      //legend, update rg bounds (if rpd invalid )
      renderLegend( rpd );
      
      //graphic is rendered depending on it type:
           if ( _G.isRenderableId( DGraphic.T_SECTORS        )) renderSector( rpd, false );
      else if ( _G.isRenderableId( DGraphic.T_SECTORS3D      )) renderSector( rpd, true  );
      else if ( _G.isRenderableId( DGraphic.T_HISTOGRAM      )) renderHistogram( rpd, false );
      else if ( _G.isRenderableId( DGraphic.T_XY             )) DefaultChartRenderXY.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_XYZ            )) DefaultChartRenderXYZ.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_WIREFRAME      )) DefaultChartRenderXYZ.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_HISTOGRAM_IM   )) renderHistogram( rpd, true);
      else if ( _G.isRenderableId( DGraphic.T_H_HISTOGRAM_IM )) renderHHistogram( rpd, true);
      else if ( _G.isRenderableId( DGraphic.T_HISTORS        )) renderHistors( rpd );
      else if ( _G.isRenderableId( DGraphic.T_HBARS          )) DefaultChartRenderHBars.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_STACKBARS      )) DefaultChartRenderStackBars.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_H_STACKBARS    )) DefaultChartRenderHStackBars.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_METER          )) DefaultChartRenderMeter.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_SERIES3D       )) DefaultChartRenderSeries3D.render( d_, rpd );
      else if ( _G.isRenderableId( DGraphic.T_PIE3D          )) DefaultChartRenderPie3D.render( d_, rpd, true );
      else if ( _G.isRenderableId( DGraphic.T_TORUS3D        )) DefaultChartRenderPie3D.render( d_, rpd, false );
      else if ( _G.isRenderableId( DGraphic.T_XY_SURFACE     )) DefaultChartRenderXYSurface.render( d_, rpd );
      else
      {
        d_.error( "Unknown Graphic type :\n'"+_G.getRenderableId()+"'", rpd );
        return ;
      }
    
      //point to locate in graphic rectangle, might interest render's caller.
      if( d_.locating() && rpd.ag_rect.contains( d_.lx_, d_.ly_ ) )
      {
        throw new DefaultRenderChartLocation( DLocated.InGraphicArea, _G, rpd.ag_rect );
      }
    }
    finally
    {
      //dispose rpd since IVObject can't store it ?
      if( dispose_rpd && rpd!=null )
      {
        rpd.dispose();
        rpd=null;
      }      
    }
  }
  
  
  /** 
   * render graphic title on the top of graphic's rectangle
   * update _rg bounds only if persistent data is recomputed here
   */
  protected void renderGraphicTitle(  RenderPersistData rpd )
     throws DefaultRenderChartLocation
  {
    //validity of title is based on :
    // - property P_SHOW_TITLE
    // - property P_TITLE_LAYOUT
    // - title text.
    
    boolean display_title = rpd.graphic.getProperties().get(DGraphic.P_SHOW_TITLE, true);
    
    String title = DefaultChartRenderData.getResourceString(rpd.graphic.getTitle(), 
            (DI18N)rpd.graphic.getChildOfClass(DI18N.class));

    display_title &= (title!=null) && (!"".equals(title));
    
    //alignment and style for graphic title. 
    DAlignmentPair aln=null;
    try{
      aln=(DAlignmentPair)rpd.graphic.getProperties().get( DGraphic.P_TITLE_LAYOUT );
    }
    finally{
      //default title style:
      if(aln==null) aln = new DAlignmentPair( IDAlignment.TOP, IDAlignment.HCENTER);
    }
    
    boolean valid = false ;
    
    //check if validity is always good
    if( rpd.isValid( RenderPersistData.V_TITLE ) )
    {
      valid = rpd.title_displayed == display_title 
            && aln.equals( rpd.title_aln ) 
            && title == rpd.title_text ;
    }

    //need to update rpd title's data 
    if( !valid )
    {
      rpd.invalidate( RenderPersistData.V_TITLE );
      //as title is invalid, legend is invalid too
      rpd.invalidate( RenderPersistData.V_LEGEND );
    
      //remember this for next validation time.    
      rpd.title_displayed = display_title ;
      rpd.title_aln = aln;
      rpd.title_text = title;
    
      //clear title data:
      rpd.title_displayed_text = null;
      rpd.title_rect = null;
      rpd.title_render_aln = 0;
    
      //no title ?
      if( !rpd.title_displayed ) {
        rpd.validate( RenderPersistData.V_TITLE );
        return ; //ag_rect is good
      }

      //compute title rectangle.
      d_.gc_.setFont( rpd.g_style.getFont() );
      ISize size = DrawUtilIGC.textExtent( d_.gc_, title, aln.getAlignment() );      

      Rect r_ttl = new Rect( rpd.graphic_rect );
      r_ttl.shrink( d_.margin_ );
      
      if( aln.haveAlignment( IDAlignment.BOTTOM ))
      {
        r_ttl.setTop( r_ttl.bottom()-size.getH() );
        rpd.ag_rect.setBottom( rpd.ag_rect.bottom()-r_ttl.h()-d_.margin_ );
      }
      else if( aln.haveAlignment( IDAlignment.LEFT ))
      {
        r_ttl.setWidth( size.getW() );
        rpd.ag_rect.setLeft(  rpd.ag_rect.left()+r_ttl.w()+d_.margin_ );
      }
      else if( aln.haveAlignment( IDAlignment.RIGHT ))
      {
        r_ttl.setLeft( r_ttl.right()-size.getW() );
        rpd.ag_rect.setRight( rpd.ag_rect.right()-r_ttl.w()-d_.margin_ );
      }
      else //TOP
      {
        r_ttl.setHeight( size.getH() );
        rpd.ag_rect.setTop( rpd.ag_rect.top()+r_ttl.h()+d_.margin_ );
      }
    
      rpd.title_displayed_text = DrawUtilIGC.truncateText( d_.gc_, title, r_ttl.w(),r_ttl.h(), aln.getAlignment() );
      
      //combine VERTICAL/.../ with text alignment
      rpd.title_render_aln = aln.getAlignment(DrawUtilIGC.MASK) | aln.getSecondaryAlignment();
      rpd.title_rect = r_ttl;
      
      rpd.validate( RenderPersistData.V_TITLE );
    }

    if( !rpd.title_displayed ) return ;

    //now time to render/locate
    d_.renderText( rpd.title_displayed_text, rpd.title_rect, rpd.title_render_aln, rpd.g_style, DLocated.InGraphicTitleText, rpd.graphic );
    if( d_.locating() )
    {
      if( rpd.title_rect.contains( d_.lx_, d_.ly_ ) )
      {
        throw new DefaultRenderChartLocation( DLocated.InGraphicTitleArea, rpd.graphic, rpd.title_rect );
      }
    }
  }
  
  /** render legend and update rg bounds accordingly alignment */
  protected void renderLegend( RenderPersistData rpd )  throws DefaultRenderChartLocation 
  {
    //Legend validity is based on
    // - current V_LEGEND value
    // - P_SHOW_LEGEND property
    // - P_LEGEND_LAYOUT property
    //missing font changement will produce visual changement if legend is displayed
    
    //display legend ?
    boolean display_legend = rpd.graphic.getProperties().get(DGraphic.P_SHOW_LEGEND, true);
    //alignment and style for graphic legend. 
    DAlignmentPair aln=null;
    try{
       aln=(DAlignmentPair)rpd.graphic.getProperties().get( DGraphic.P_LEGEND_LAYOUT );
    }
    finally{
      //default legend style:
      if(aln==null) aln = new DAlignmentPair( IDAlignment.RIGHT, IDAlignment.VCENTER);
    }
    
    
    boolean valid = false;
    if( rpd.isValid( RenderPersistData.V_LEGEND ))
    {
      valid = true;
      if( display_legend != rpd.legend_displayed )
      {
        valid=false;
      }
      else if ( !aln.equals( rpd.legend_aln ) )
      {
        valid=false;
      }
    }
    
    //space between symbol and text
    int lgd_sym_spacing= d_.dpiX(10);    
    //space between two legend entry layouted vertically
    int lgd_vert_spacing = d_.dpiY( 4 ); 

    //need to recompute ?
    if( valid )
    {
      if( !display_legend ) return ;
    }
    else 
    {
      //TBD: when legend will be invalidate but title still valid,
      //we lost ag_rect that renderLegend require to compute its layout
      //-> need to memorize it.
      rpd.legend_displayed = display_legend;
      rpd.legend_aln = new DAlignmentPair(aln);

      rpd.legend_rect = null;      
    
      rpd.validate( RenderPersistData.V_LEGEND );
      
      if( !display_legend ) 
      {
        return ;
      }
      
      //final graphic area will be reduce
      Rect rg = rpd.ag_rect;
      
      //vertical layout if primary alignment is left or right
      //otherwise, layout is horizontal.
      boolean vert_layout = aln.haveAlignment( IDAlignment.LEFT )
                          | aln.haveAlignment( IDAlignment.RIGHT) 
                          | aln.haveAlignment( IDAlignment.HCENTER) ;      

      int n_legend = rpd.curve_styles.length;
    
      int entry_max_height = 0;
      int entry_max_width  = 0;
    
      int entries_width  = 0; //vert: entry_max_width, horz: cumul width of entries (no spacing)
      int entries_height = 0; //vert: cumul height of entries(no spacing), horz: entry_max_height
      //used to set same entries symbol size.
      int entry_min_height =0;
      boolean init=false;    
      for( int i=0; i<n_legend; ++i )
      {
        CurveStyle ccurve = (CurveStyle)rpd.curve_styles[i];
        d_.gc_.setFont( ccurve.style_.getFont() );
        String name = DefaultChartRenderData.getResourceString(ccurve.curve_.getName(), 
                (DI18N)rpd.graphic.getChildOfClass(DI18N.class));
        
        if( !DrawUtilIGC.isEmpty( name ) )
        {
          ISize sz = d_.gc_.textExtent( name );
          int sz_w = sz.getW(), sz_h = sz.getH();
          entry_max_width  = Math.max( entry_max_width,  sz_w );
          entry_max_height = Math.max( entry_max_height, sz_h );
          if( vert_layout)
          {
            entries_height += sz_h;
          } else {
            entries_width  += sz_w;
          }          
          if ( !init || entry_min_height > sz_h ) entry_min_height=sz_h;
          init=true;
        }
      }
      if( !init ) //all names are empty.
      {
        //no entry with names: do not display legend.
        rpd.legend_displayed = false;
        return ;
      }
      if( vert_layout )
      {
        entries_width = entry_max_width;
      } else {
        entries_height= entry_max_height;
      }

      //size (wxh) for symbol in legend.
      int lgd_sym_size = (entry_min_height/2);
      if( lgd_sym_size < d_.dpiX(7) ) lgd_sym_size = (entry_min_height-2);
    
      int legend_x=0;
      int legend_y=0;
      int legend_H=0; //global height
      int legend_W=0; //global width
      int n_lgd_by_line;
      int n_lines ;
      
      int p_legend_limit = vert_layout ? 4 : 8; //default
      
      try{
        int pll = rpd.graphic.getProperties().get( DGraphic.P_LEGEND_LIMIT, p_legend_limit );
        if( pll > 0 ) p_legend_limit = pll;
      }
      catch( Exception e ) {} //use default legend limit
    
      if( vert_layout )
      {
        //try one column
        n_lgd_by_line = 1;
        n_lines = n_legend;// /n_lgd_by_line ;
        //let space for symbol and margin...
        legend_W = n_lgd_by_line*(lgd_sym_size+lgd_sym_spacing+entry_max_width+d_.margin_)-d_.margin_;
        legend_H = n_lines*(entry_max_height + lgd_vert_spacing) - lgd_vert_spacing ;
        //need to use  multi row ?
        if( legend_H > rg.h())
        {
          n_lines = (rg.h()+lgd_vert_spacing) / (entry_max_height+lgd_vert_spacing) ;
          //not enough place for one line ?
          if( n_lines == 0 )
          {
            //try to reduce entry_max_width
//TODO...
            rpd.legend_displayed = false;
            return ;          
          }
          n_lgd_by_line = n_legend / n_lines + (n_legend%n_lines==0 ? 0:1);
          //take care of rest ...
          n_lines = n_legend / n_lgd_by_line + (n_legend % n_lgd_by_line ==0 ? 0 : 1);
          legend_W = n_lgd_by_line*(lgd_sym_size+lgd_sym_spacing+entry_max_width+d_.margin_)-d_.margin_;
          legend_H = n_lines*(entry_max_height + lgd_vert_spacing) - lgd_vert_spacing ;
        }
      
        //limit legend width to 1/th of graphic area
        if ( legend_W+d_.margin_ > (rg.w()/p_legend_limit) )
        {
          entry_max_width =  ((rg.w()/p_legend_limit-d_.margin_)/n_lgd_by_line) - lgd_sym_size -lgd_sym_spacing;
          legend_W = n_lgd_by_line*(lgd_sym_size+lgd_sym_spacing+entry_max_width+d_.margin_)-d_.margin_;
          //limit for entry_max_width
          if( entry_max_width < 15 )
          {
          	rpd.legend_displayed=false;
          	return;
          } 
        }
        //extrem limit
        if( legend_W < d_.dpiX(25) ) 
        {
        	rpd.legend_displayed=false;
        	return ;
        } 
         //now place legends location, from alignement.
        if( aln.haveAlignment( IDAlignment.LEFT ) )
        {
          legend_x = rg.x();
          rg.setLeft( rg.left()+legend_W+d_.margin_ );
        }
        else //by default: if ( _aln.haveAlignment( IDAlignment.RIGHT))
        {
          legend_x = rg.right()-legend_W;
          rg.setRight( legend_x-d_.margin_);
        }
        //use secondary alignment for vertical placing
        if( aln.haveSecondaryAlignment( IDAlignment.TOP ))
        {
          legend_y = rg.y();        
        }
        else if( aln.haveSecondaryAlignment( IDAlignment.BOTTOM ))
        {
          legend_y = rg.bottom()-legend_H;
        }
        else //VCENTER
        {
          legend_y = rg.centerY()-legend_H/2;
        }
      }
      else // horizontal layout
      {
        //try one row 
        n_lgd_by_line = n_legend;
        n_lines = 1; //is: n_legend/n_lgd_by_line
        //let space for symbol and margin...
        legend_W = n_lgd_by_line*(lgd_sym_size+lgd_sym_spacing+entry_max_width+d_.margin_)-d_.margin_;
        legend_H = n_lines*(entry_max_height + lgd_vert_spacing) - lgd_vert_spacing ;
        //not enough space to display one row ?
        //try multi-rows
        if( legend_W+d_.margin_ > rg.w())
        {
          n_lgd_by_line = (rg.w()/*+d_.margin_-d_.margin_*/)/(lgd_sym_size+lgd_sym_spacing+entry_max_width+d_.margin_);
          //can't set only one using default width, let one entry by line and try to reduce entry width
          if( n_lgd_by_line == 0 )
          {
            n_lgd_by_line = 1;
            n_lines = n_legend ;
            entry_max_width =  (rg.w()/n_lgd_by_line) - lgd_sym_size -lgd_sym_spacing;
            legend_W = n_lgd_by_line*(lgd_sym_size+lgd_sym_spacing+entry_max_width);
            //extrem limit
            if( legend_W < 30 ) 
            {
            	rpd.legend_displayed = false;
            	return ;
            } 
            legend_H = n_lines*(entry_max_height + lgd_vert_spacing) - lgd_vert_spacing ;
          }
          else 
          {
            n_lines = n_legend / n_lgd_by_line + (n_legend % n_lgd_by_line ==0 ? 0 : 1);
            n_lgd_by_line = n_legend / n_lines + (n_legend % n_lines==0 ? 0 : 1);
            legend_W = n_lgd_by_line*(lgd_sym_size+lgd_sym_spacing+entry_max_width+d_.margin_)-d_.margin_;
            legend_H = n_lines*(entry_max_height + lgd_vert_spacing) - lgd_vert_spacing ;
          }
        }
        //limit height of legend...     
        if ( legend_H > (rg.h()/p_legend_limit) )
        {
          //as less height, this is not the entry_max_height will take less place (font height is constant)
          //but try to limit entry_max_width to have another n_lines, and finally take less height...
          n_lines = (rg.h()/p_legend_limit + lgd_vert_spacing)/(entry_max_height+lgd_vert_spacing);
          if( n_lines == 0 )
          {
            rpd.legend_displayed = false;
            return ;
          }
          n_lgd_by_line = n_legend / n_lines;
          n_lines = n_legend / n_lgd_by_line + (n_legend % n_lgd_by_line ==0 ? 0 : 1);
          entry_max_width =  ((legend_W+d_.margin_)/n_lgd_by_line) - lgd_sym_size -lgd_sym_spacing -d_.margin_;
          legend_H = n_lines*(entry_max_height + lgd_vert_spacing) - lgd_vert_spacing ;
          //extrem limit for new entry width:
          if( entry_max_width < d_.dpiX(30) ) 
          {
          	rpd.legend_displayed = false;
          	return ;
          } 
        }      
        //place legend ...     
        if( aln.haveAlignment( IDAlignment.TOP ) )
        {
          legend_y = rg.top();
          rg.setTop( rg.top()+legend_H+d_.margin_ );       
        }
        else //by default: if ( _aln.haveAlignment( IDAlignment.BOTTOM))
        {
          legend_y = rg.bottom()-legend_H;
          rg.setBottom( legend_y-d_.margin_);
        }
        //use secondary alignment for left/right/hcenter
        if( aln.haveSecondaryAlignment( IDAlignment.LEFT ))
        {
          legend_x = rg.x();        
        }
        else if( aln.haveSecondaryAlignment( IDAlignment.RIGHT ))
        {
          legend_x = rg.right()-legend_W;
        }
        else //VCENTER
        {
          legend_x = rg.centerX()-legend_W/2;
        }      
      }
    
      Rect lgd_area= new Rect( legend_x, legend_y, legend_W, legend_H );
      
      rpd.legend_rect = lgd_area;
      rpd.entry_max_width = entry_max_width;
      rpd.entry_max_height= entry_max_height;
      rpd.legend_symbol_size = lgd_sym_size ;
      rpd.legend_vertical_layout = vert_layout;
      rpd.legend_n_lines = n_lines;
      rpd.legend_n_entry_by_line = n_lgd_by_line;
      
      rpd.ag_rect = rg;
    }
    

    //shortcut: on on y est pas, on passe la main ...
    //shortcut for locating: outside ?
    if( d_.locating() && !rpd.legend_rect.contains( d_.lx_, d_.ly_ ) )
    {
      return  ;
    }
    
    Rect r_sym=new Rect();
    Rect r_txt_area=new Rect();

    int lgx = rpd.legend_rect.x();
    int lgy = rpd.legend_rect.y();
    for( int i=0; i<rpd.curve_styles.length; ++i )
    {
      CurveStyle ccurve = rpd.curve_styles[i];
      
      String name = DefaultChartRenderData.getResourceString(ccurve.curve_.getName(), 
              (DI18N)rpd.graphic.getChildOfClass(DI18N.class));
      
//? may have symbol only ?      
      if( DrawUtilIGC.isEmpty( name )) continue;
      //text is choosen black, because fore color is curve color
      d_.gc_.setFont( ccurve.style_.getFont() );
  //    Point legend_ext = d_.gc_.textExtent( name );
      r_sym.setRect( lgx,lgy+(rpd.entry_max_height-rpd.legend_symbol_size)/2, rpd.legend_symbol_size,rpd.legend_symbol_size );

      String txt = DrawUtilIGC.truncateTextH( d_.gc_, name, rpd.entry_max_width );

      if( d_.drawing() )
      {
        d_.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
        d_.gc_.setBrush( new SolidBrush( ccurve.style_.getBack()) );
        ccurve.symbol_.draw( d_.gc_, r_sym );
      }
      else //locating
      {
        //inside color rectangle
        if( ccurve.symbol_.contains( d_.gc_, r_sym, d_.lx_, d_.ly_ ) )
        {
          throw new DefaultRenderChartLocation( DLocated.LegendColor, ccurve.curve_, r_sym );
        }
      }
    
      r_txt_area.setRect( lgx+rpd.legend_symbol_size+lgd_sym_spacing, lgy, rpd.entry_max_width, rpd.entry_max_height );
      d_.renderText( txt, r_txt_area, IDAlignment.LEFT|IDAlignment.VCENTER,
                     ccurve.style_, DLocated.LegendText, ccurve.curve_ );

      if( rpd.legend_vertical_layout )
      {
        lgy += rpd.entry_max_height+lgd_vert_spacing;
        if( (i)%rpd.legend_n_lines == rpd.legend_n_lines-1 ) //lgy > legend_H ) //multi column
        {
          lgx += rpd.entry_max_width + rpd.legend_symbol_size+lgd_sym_spacing+d_.margin_;
          lgy  = rpd.legend_rect.y();
        }
      }
      else //horizontal layout
      {
        lgx += rpd.legend_symbol_size+lgd_sym_spacing+rpd.entry_max_width +d_.margin_;
        if( (i)%rpd.legend_n_entry_by_line == rpd.legend_n_entry_by_line-1 )//gx > legend_W ) //multi row
        {  
          lgy += rpd.entry_max_height +lgd_vert_spacing;
          lgx  = rpd.legend_rect.x();
        }
      }
    }//for

    //dans la zone de la legende ??
    if( d_.locating() && rpd.legend_rect.contains( d_.lx_, d_.ly_ ) )
    {
      throw new DefaultRenderChartLocation( DLocated.LegendArea, null, rpd.legend_rect );
    }
  }    
  
  /** Render Histogram graphic family */
  protected void renderHistogram( RenderPersistData rpd, /*boolean _3d, boolean _bs, */boolean _im )
    throws DefaultRenderChartLocation
  {
    //need rebuild ?
    if( !(rpd.g instanceof RenderPersistData.Histogram ))
    {
      if( rpd.g != null )
      {
        rpd.g.dispose();
      }
      //histogram must have (only) one axis.
      DAxis axis=null;
      for( IDItem i=rpd.graphic.getFirstChild(); i!=null; i=i.getNext() )
      {
        if( i instanceof DAxis && axis==null )
        {
          axis=(DAxis)i;
          break;
        }
      }

      if ( axis==null )
      {
        if(d_.drawing()) d_.error( "Error: Histogram without axis", rpd);
        return ;
      }

      RenderPersistData.Histogram rpdh = new RenderPersistData.Histogram();
      rpdh.axis_style = new IGCDStyle( axis, d_.scale_ );
    
      //min/max values for axis, P_MIN/P_MAX win the game, otherwise must compute...
      Object ay_min=axis.getProperties().get( DAxis.P_MIN ),
             ay_max=axis.getProperties().get( DAxis.P_MAX );

      //compute min/max values..
      if( ay_max==null || ay_min==null )
      {
        MinMax mm = new MinMax( axis );
        for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
        {
          CurveStyle ccurve = rpd.curve_styles[icrv];    
          for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
          {
            if( !(item instanceof DPoint) ) continue;
            for( IDItem ic=item.getFirstChild(); ic!=null; ic=ic.getNext() )
            {
              if( !(ic instanceof IDCoord) ) continue;
              mm.update( (IDCoord)ic );
            }
          }
        }
        if( ay_min ==null ) ay_min = mm.getVMin();
        if( ay_max ==null ) ay_max = mm.getVMax();
      }
      
      rpdh.y_scale = DefaultChartRenderData.getScale( axis, 0,0, ay_min,ay_max);
      rpdh.axis = axis ;

      if( rpdh.y_scale==null || !rpdh.y_scale.isValid() )
      {
        //smell like update document problem (MIN/MAX=0.0f), or invalid P_DATA_CLASS in DAxis.
        if( d_.drawing() ) d_.error("Error: invalid axis", rpd);
        return ;
      }
      d_.gc_.setFont( rpdh.axis_style.getFont() );
      rpdh.yam= d_.computeYAxisAmounts( true, true, axis, rpdh.y_scale, rpd.ag_rect, IDAlignment.ROTCCW90, null);
 
      int rg_x = rpd.ag_rect.x();
      int rg_y = rpd.ag_rect.y();
      int rg_w = rpd.ag_rect.w();
      int rg_h = rpd.ag_rect.h();

      IDItem item=null;

      rpdh.n_bar=0;
      rpdh.n_max_points_by_curve = 0;
      rpdh.h_label_height =0;

      boolean is_rot90 = false;
      {
        Object prop=rpdh.axis.getProperties().get(DAxis.P_LABEL_ALIGNMENT);
        if( prop instanceof IDAlignment )
        {
          rpdh.h_axis_alignment = (IDAlignment)prop;
          is_rot90 = rpdh.h_axis_alignment.getAlignment( IDAlignment.ROTCCW90|IDAlignment.ROTCW90 )!=0; 
        }
        else
        {
          //create default alignment for axis
          rpdh.h_axis_alignment = new DAlignment( DAlignment.HCENTER );
        }
      }
      //count bars, and get max font height
      IGCDStyle sty_point = new IGCDStyle( d_.scale_ );
      Object value =null;
      for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
      {
        CurveStyle ccurve = rpd.curve_styles[icrv];    
        int npoint=0;
        for( item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
        {
          if( !(item instanceof DPoint) ) continue;
         
          npoint++;
          rpdh.n_bar++;

          //point can have a label:
          DText label=null;
          for( IDItem it=item.getFirstChild(); (it!=null); it=it.getNext())
          {
            if( it instanceof DText ) { label=(DText)it; break; }
          }
          String full_text=null;
          if( label!=null )
          {
            sty_point.styleOf( label, d_.scale_ );
            full_text = label.getText();
          } else {
            sty_point.styleOf( item, d_.scale_ );
            IDCoord coord = ((DPoint)item).getCoord( rpdh.axis );
            value = coord.getValue( value );
            full_text = rpdh.y_scale.valueText( value );
          }
          d_.gc_.setFont( sty_point.getFont() );
          int lh = DrawUtilIGC.textExtent( d_.gc_, full_text, rpdh.h_axis_alignment.getAlignment() ).getH();
          if( lh > rpdh.h_label_height ) rpdh.h_label_height = lh;          
        } 
        if( npoint > rpdh.n_max_points_by_curve ) rpdh.n_max_points_by_curve=npoint;        
      }
      if( _im ) rpdh.n_bar+=rpdh.n_max_points_by_curve-1; //space between blocks

      //limit text heigh to 50% of graphic rectangle
      if( rpdh.h_label_height > (int)(0.5f*rg_h) )
      {
        rpdh.h_label_height = (int)(0.5f*rg_h);
      }
      rg_x += rpdh.yam.getL();  rg_w -= rpdh.yam.getL();
      rg_y += rpdh.yam.getT();  rg_h -= rpdh.yam.getT();
      //value of bar below them.
      rg_h -= rpdh.h_label_height+d_.axis_unit_to_dot_spacing ;

      if( rpdh.n_bar > 0 )
      {
        rpdh.thin_bar  = ( rpd.graphic.getProperties().get( DGraphic.P_THIN_BAR, false ) );
        boolean _3d = ( rpd.graphic.getProperties().get( DGraphic.P_3D, false ) );
        float kthinbar = rpdh.thin_bar ? 0.7f : 1.0f;
        float k3d = 0.5f;
        
        if( _3d )
        {
          rpdh.wbartobar = rg_w/(float)(rpdh.n_bar+k3d*kthinbar);
        } else {
          rpdh.wbartobar = rg_w/(float)(rpdh.n_bar);
        }
    
        rpdh.wbar = kthinbar*rpdh.wbartobar;
        rpdh.xdecbar= rpdh.thin_bar ? (rpdh.wbartobar-rpdh.wbar)/2 : 0.0f;

        rpdh._3d_z = 0 ;
        if ( _3d )
        {
          rpdh._3d_z = (int)(k3d*rpdh.wbar);
          //3d means, reduce chart area...
          rg_w -= rpdh._3d_z ;
          rg_y += rpdh._3d_z ;
          rg_h -= rpdh._3d_z ;
        }  
      }
      //now can set scale pixel bounds
      rpdh.y_scale.setScaleRange( rg_y+rg_h, rg_y );
    
      rpd.ag_rect.setRect( rg_x, rg_y, rg_w, rg_h );
      rpd.g = rpdh;
      
    } //end-of-rebuild
    
    if( rpd.ag_rect.w()<=0 || rpd.ag_rect.h()<=0 ) return ;
         
    //now time to render ...
    RenderPersistData.Histogram h = (RenderPersistData.Histogram)rpd.g;

    //draw Y axis
    d_.renderYAxis( true, true, true, h.axis,h.y_scale,rpd.ag_rect,h.yam,h.axis_style,IDAlignment.ROTCCW90, h._3d_z,h._3d_z);

    // marker line detection
    d_.renderMarkerLine( true, false, h.axis, rpd.ag_rect, h.y_scale); 
    
    int rg_left = rpd.ag_rect.left();
    int rg_right= rpd.ag_rect.right();
    
    //base lines
    if(d_.drawing())
    {
      d_.gc_.setPen( new LineStylePen( h.axis_style.getFore() ) );
      int btom = rpd.ag_rect.bottom();
      d_.gc_.drawLine( rg_left, rpd.ag_rect.top(),        rg_left, btom ); // Y base line
      d_.gc_.drawLine( rg_left, btom, rg_right, btom ); // X base line
    }
    
    Object vf = DefaultChartRenderData.getNewInstanceValue( h.axis );
    float xbar=rg_left;
    int   y_bar_zero = (int)h.y_scale.getScaleForBar();
    if( d_.drawing())
    {
      d_.gc_.drawLine( rg_left, y_bar_zero, rg_right, y_bar_zero );
    }
    

    DefaultRenderChartLocation loc=null;
    int n_curves = rpd.curve_styles.length;
    
    IShape save_clip = null;    
    if( d_.drawing() )
    {
      save_clip = d_.gc_.getClipping();
      d_.gc_.setClipping( new Rect( rpd.ag_rect.x(), rpd.ag_rect.y()-h._3d_z, rpd.ag_rect.w()+h._3d_z+1, rpd.ag_rect.h()+h._3d_z ) );
    }
    
    if( _im )
    {
      //histogram drawing  mixed curves
      //(ie point #0 for all curves, then point #i...)     
      for( int ipt=0; ipt<h.n_max_points_by_curve; ++ipt )
      {
        for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
        {
          CurveStyle ccurve = rpd.curve_styles[icrv];    
          //get point at ipt in curve:
          DPoint point=null;
          int i=0;
          for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
          {
            if( !( item instanceof DPoint) ) continue;
            if( i==ipt)
            {
              point = (DPoint)item;
              break;
            }
            ++i;
          } 
          if( point ==null ) continue;
          
          try{                     
            if( renderHistogramBar(true,  rpd.ag_rect, rpd.locale, h,n_curves, vf,point,icrv,ipt,xbar,y_bar_zero,_im))
            {
              xbar+=h.wbartobar;
            }
          }
          catch( DefaultRenderChartLocation _loc )
          {
            //trap bar location and keep only last, due to
            //drawing left to right and 3D mode (bar overlap)
            loc=_loc;
          }
        }//for 
        
        //separate islands
        xbar += h.wbartobar; 
      }//for
    }
    else
    {
      //histogram drawing curves prior points
      for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
      {
        CurveStyle ccurve = rpd.curve_styles[icrv];    
        int ipt=0;
        for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
        {
          if( !(item instanceof DPoint) ) continue;
          
          DPoint point = (DPoint)item;
          try{
            if( renderHistogramBar(true, rpd.ag_rect,rpd.locale, h,n_curves,vf,point,icrv,ipt,xbar,y_bar_zero,_im))
            {
              ipt++;
              xbar+=h.wbartobar;
            }  
          }
          catch( DefaultRenderChartLocation _l )
          { 
            //check all bar (to handle correctly 3d drawing)         
            loc=_l; 
            ipt++;
            xbar+=h.wbartobar;
          }
        }//for 
      }//for
    }
    
    if( save_clip!=null ) d_.gc_.setClipping( save_clip );
    
    if( d_.locating() && loc!=null )
    {
      throw loc;
    }
    
    //time to locate on Y axis:
    d_.renderYAxis( true, false, true, h.axis,h.y_scale,rpd.ag_rect,h.yam,h.axis_style,IDAlignment.ROTCCW90, h._3d_z,h._3d_z);
    
    // time to marker line
    d_.renderMarkerLine( true, true, h.axis, rpd.ag_rect, h.y_scale); 
  }
  
  /** Render Histogram graphic family */
  protected void renderHHistogram( RenderPersistData rpd, boolean _im )
    throws DefaultRenderChartLocation
  {
      //need rebuild ?
      if( !(rpd.g instanceof RenderPersistData.Histogram ))
      {
        if( rpd.g != null )
        {
          rpd.g.dispose();
        }
        //histogram must have (only) one axis.
        DAxis axis=null;
        for( IDItem i=rpd.graphic.getFirstChild(); i!=null; i=i.getNext() )
        {
          if( i instanceof DAxis && axis==null )
          {
            axis=(DAxis)i;
            break;
          }
        }

        if ( axis==null )
        {
          if(d_.drawing()) d_.error( "Error: Histogram without axis", rpd);
          return ;
        }

        RenderPersistData.Histogram rpdh = new RenderPersistData.Histogram();
        rpdh.axis_style = new IGCDStyle( axis, d_.scale_ );
      
        //min/max values for axis, P_MIN/P_MAX win the game, otherwise must compute...
        Object ay_min=axis.getProperties().get( DAxis.P_MIN ),
               ay_max=axis.getProperties().get( DAxis.P_MAX );

        //compute min/max values..
        if( ay_max==null || ay_min==null )
        {
          MinMax mm = new MinMax( axis );
          for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
          {
            CurveStyle ccurve = rpd.curve_styles[icrv];    
            for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
            {
              if( !(item instanceof DPoint) ) continue;
              for( IDItem ic=item.getFirstChild(); ic!=null; ic=ic.getNext() )
              {
                if( !(ic instanceof IDCoord) ) continue;
                mm.update( (IDCoord)ic );
              }
            }
          }
          if( ay_min ==null ) ay_min = mm.getVMin();
          if( ay_max ==null ) ay_max = mm.getVMax();
        }
        
        rpdh.y_scale = DefaultChartRenderData.getScale( axis, 0,0, ay_min,ay_max);
        rpdh.axis = axis ;

        if( rpdh.y_scale==null || !rpdh.y_scale.isValid() )
        {
          //smell like update document problem (MIN/MAX=0.0f), or invalid P_DATA_CLASS in DAxis.
          if( d_.drawing() ) d_.error("Error: invalid axis", rpd);
          return ;
        }
        d_.gc_.setFont( rpdh.axis_style.getFont() );
        rpdh.yam= d_.computeYAxisAmounts( false, true, axis, rpdh.y_scale, rpd.ag_rect, IDAlignment.ROTCCW90, null);
   
        int rg_x = rpd.ag_rect.x();
        int rg_y = rpd.ag_rect.y();
        int rg_w = rpd.ag_rect.w();
        int rg_h = rpd.ag_rect.h();

        IDItem item=null;

        rpdh.n_bar=0;
        rpdh.n_max_points_by_curve = 0;
        rpdh.h_label_height =0;

        //boolean is_rot90 = false;
        {
          Object prop=rpdh.axis.getProperties().get(DAxis.P_LABEL_ALIGNMENT);
          if( prop instanceof IDAlignment )
          {
            rpdh.h_axis_alignment = (IDAlignment)prop;
            //is_rot90 = rpdh.h_axis_alignment.getAlignment( IDAlignment.ROTCCW90|IDAlignment.ROTCW90 )!=0; 
          }
          else
          {
            //create default alignment for axis
            rpdh.h_axis_alignment = new DAlignment( DAlignment.HCENTER );
          }
        }
        //count bars, and get max font height
        IGCDStyle sty_point = new IGCDStyle( d_.scale_ );
        Object value =null;
        for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
        {
          CurveStyle ccurve = rpd.curve_styles[icrv];    
          int npoint=0;
          for( item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
          {
            if( !(item instanceof DPoint) ) continue;
           
            npoint++;
            rpdh.n_bar++;

            //point can have a label:
            DText label=null;
            for( IDItem it=item.getFirstChild(); (it!=null); it=it.getNext())
            {
              if( it instanceof DText ) { label=(DText)it; break; }
            }
            String full_text=null;
            if( label!=null )
            {
              sty_point.styleOf( label, d_.scale_ );
              full_text = label.getText();
            } else {
              sty_point.styleOf( item, d_.scale_ );
              IDCoord coord = ((DPoint)item).getCoord( rpdh.axis );
              value = coord.getValue( value );
              full_text = rpdh.y_scale.valueText( value );
            }
            d_.gc_.setFont( sty_point.getFont() );
            int lh = DrawUtilIGC.textExtent( d_.gc_, full_text, rpdh.h_axis_alignment.getAlignment() ).getW();
            if( lh > rpdh.h_label_height ) rpdh.h_label_height = lh;          
          } 
          if( npoint > rpdh.n_max_points_by_curve ) rpdh.n_max_points_by_curve=npoint;        
        }
        if( _im ) rpdh.n_bar+=rpdh.n_max_points_by_curve-1; //space between blocks

        //limit text heigh to 50% of graphic rectangle
        if( rpdh.h_label_height > (int)(0.5f*rg_h) )
        {
          rpdh.h_label_height = (int)(0.5f*rg_h);
        }
        rg_x += rpdh.yam.getL() + rpdh.h_label_height+d_.axis_unit_to_dot_spacing;  rg_w -= rpdh.yam.getL();
        rg_y += rpdh.yam.getT();  rg_h -= rpdh.yam.getT();
        //value of bar below them.
        rg_w -= rpdh.h_label_height+d_.axis_unit_to_dot_spacing ;

        if( rpdh.n_bar > 0 )
        {
          rpdh.thin_bar  = ( rpd.graphic.getProperties().get( DGraphic.P_THIN_BAR, false ) );
          boolean _3d = ( rpd.graphic.getProperties().get( DGraphic.P_3D, false ) );
          float kthinbar = rpdh.thin_bar ? 0.7f : 1.0f;
          float k3d = 0.5f;
          
          if( _3d )
          {
            rpdh.wbartobar = rg_h/(float)(rpdh.n_bar+k3d*kthinbar);
          } else {
            rpdh.wbartobar = rg_h/(float)(rpdh.n_bar);
          }
      
          rpdh.wbar = kthinbar*rpdh.wbartobar;
          rpdh.xdecbar= rpdh.thin_bar ? (rpdh.wbartobar-rpdh.wbar)/2 : 0.0f;

          rpdh._3d_z = 0 ;
          if ( _3d )
          {
            rpdh._3d_z = (int)(k3d*rpdh.wbar);
            //3d means, reduce chart area...
            rg_w -= rpdh._3d_z ;
            rg_x += rpdh._3d_z ;
            rg_h -= rpdh._3d_z ;
          }  
        }
        //now can set scale pixel bounds
        rpdh.y_scale.setScaleRange(  rg_x, rg_x+rg_w);
      
        rpd.ag_rect.setRect( rg_x, rg_y, rg_w, rg_h );
        rpd.g = rpdh;
        
      } //end-of-rebuild
      
      if( rpd.ag_rect.w()<=0 || rpd.ag_rect.h()<=0 ) return ;
           
      //now time to render ...
      RenderPersistData.Histogram h = (RenderPersistData.Histogram)rpd.g;

      //draw Y axis
      d_.renderYAxis( false, true, true, h.axis,h.y_scale,rpd.ag_rect,h.yam,h.axis_style,IDAlignment.HCENTER, h._3d_z,h._3d_z);

      // marker line detection
      d_.renderMarkerLine( false, false, h.axis, rpd.ag_rect, h.y_scale); 
      
      int rg_left = rpd.ag_rect.left();
      int rg_right= rpd.ag_rect.right();
      int rg_top = rpd.ag_rect.top();
      int rg_bottom= rpd.ag_rect.bottom();
      
      //base lines
      if(d_.drawing())
      {
        d_.gc_.setPen( new LineStylePen( h.axis_style.getFore() ) );
        int btop = rpd.ag_rect.top();
        d_.gc_.drawLine( rg_left, rpd.ag_rect.bottom(), rg_left, btop ); // Y base line
        d_.gc_.drawLine( rg_left, btop, rg_right, btop ); // X base line
      }
      
      Object vf = DefaultChartRenderData.getNewInstanceValue( h.axis );
      float xbar=rg_left;
      int   y_bar_zero = (int)h.y_scale.getScaleForBar();
      if( d_.drawing())
      {
        d_.gc_.drawLine(y_bar_zero, rg_top, y_bar_zero, rg_bottom );
      }
      

      DefaultRenderChartLocation loc=null;
      int n_curves = rpd.curve_styles.length;
      
      IShape save_clip = null;    
      if( d_.drawing() )
      {
        save_clip = d_.gc_.getClipping();
        d_.gc_.setClipping( new Rect( rpd.ag_rect.x()-h._3d_z, rpd.ag_rect.y(), rpd.ag_rect.w()+h._3d_z+1, rpd.ag_rect.h()+h._3d_z ) );
      }
      
      if( _im )
      {
        //histogram drawing  mixed curves
        //(ie point #0 for all curves, then point #i...)     
        for( int ipt=0; ipt<h.n_max_points_by_curve; ++ipt )
        {
          for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
          {
            CurveStyle ccurve = rpd.curve_styles[icrv];    
            //get point at ipt in curve:
            DPoint point=null;
            int i=0;
            for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
            {
              if( !( item instanceof DPoint) ) continue;
              if( i==ipt)
              {
                point = (DPoint)item;
                break;
              }
              ++i;
            } 
            if( point ==null ) continue;
            
            try{                     
              if( renderHistogramBar( false, rpd.ag_rect,rpd.locale,h,n_curves, vf,point,icrv,ipt,xbar,y_bar_zero,_im))
              {
                xbar+=h.wbartobar;
              }
            }
            catch( DefaultRenderChartLocation _loc )
            {
              //trap bar location and keep only last, due to
              //drawing left to right and 3D mode (bar overlap)
              loc=_loc;
            }
          }//for 
          
          //separate islands
          xbar += h.wbartobar; 
        }//for
      }
      else
      {
        //histogram drawing curves prior points
        for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
        {
          CurveStyle ccurve = rpd.curve_styles[icrv];    
          int ipt=0;
          for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
          {
            if( !(item instanceof DPoint) ) continue;
            
            DPoint point = (DPoint)item;
            try{
              if( renderHistogramBar( false,  rpd.ag_rect, rpd.locale, h,n_curves,vf,point,icrv,ipt,xbar,y_bar_zero,_im))
              {
                ipt++;
                xbar+=h.wbartobar;
              }  
            }
            catch( DefaultRenderChartLocation _l )
            { 
              //check all bar (to handle correctly 3d drawing)         
              loc=_l; 
              ipt++;
              xbar+=h.wbartobar;
            }
          }//for 
        }//for
      }
      
      if( save_clip!=null ) d_.gc_.setClipping( save_clip );
      
      if( d_.locating() && loc!=null )
      {
        throw loc;
      }
      
      //time to locate on Y axis:
      d_.renderYAxis( false, false, true, h.axis,h.y_scale,rpd.ag_rect,h.yam,h.axis_style,IDAlignment.ROTCCW90, h._3d_z,h._3d_z);
      
      // time to marker line
      d_.renderMarkerLine( false, true, h.axis, rpd.ag_rect, h.y_scale); 
  }
  
  /**
   *  Render on histogram bar ... internal use only.
   * @return true if the a bar have been rendered, false otherwise
   */
  private boolean renderHistogramBar( boolean vertical, Rect _rg, ULocale locale, RenderPersistData.Histogram h, int n_curves, Object _value, DPoint _point, int _icrv, int _ipt, float _xbar, int _y_bar_zero, boolean _im )
    throws DefaultRenderChartLocation
  {    
    IDCoord coord = _point.getCoord( h.axis );
    if ( coord ==null ) return false;
    
    IGCDStyle sty_point = null;
    IGCDStyle sty_label = null; //style for labeled bar (on DText)
    try
    {
      _value = coord.getValue( _value );
      int ibh = (int)h.y_scale.toScale(_value)-_y_bar_zero;    
      Rect rbar;
      
      if ( _im ) // island mixed
      {
          if (vertical)
            _xbar = _rg.x() + (_ipt*(n_curves+1)+_icrv)*h.wbartobar;
          else
            _xbar = _rg.y() + (_ipt*(n_curves+1)+_icrv)*h.wbartobar;
      }
      if( h.thin_bar )  // tick bar
      {
        if (vertical)
            rbar = new Rect( (int)(_xbar+h.xdecbar), _y_bar_zero, (int)h.wbar, ibh );
        else
            rbar = new Rect( _y_bar_zero, (int)(_xbar+h.xdecbar), ibh, (int)h.wbar );
      }
      else
      {
        //why not (int)_wbar ... to correct the (int) round values... indeed.
        //using this bar are connected without 'hole'
        if (vertical)
           rbar = new Rect( (int)_xbar, _y_bar_zero, (int)( _xbar+h.wbartobar-(int)_xbar), ibh );
        else
           rbar = new Rect( _y_bar_zero, (int)_xbar, ibh, (int)( _xbar+h.wbartobar-(int)_xbar) );
      }
      rbar.normalize();

      sty_point = new IGCDStyle( _point, d_.scale_ );
      
      d_.renderBar( vertical, rbar, sty_point.getBack(), _point, h._3d_z );

      //point can have a label as child:
      DText label=null;
      for( IDItem it=_point.getFirstChild(); it!=null; it=it.getNext() )
      {
        if(it instanceof DText)
        {
          label=(DText)it;
          break;
        }
      }
      
      //label or value ?
      String full_text=null;
      String loc_id  =DLocated.CurvePointText;
      IDItem loc_item=_point;
      if( label!=null )
      {
        full_text = label.getText();
        loc_id   = DLocated.LabelText;
        loc_item = label;
        if( !DrawUtilIGC.isEmpty( full_text ) )
        {
          sty_label = new IGCDStyle( label, d_.scale_ );         
        }
      }
      else //value is displayed below the bar.
      {        
        full_text = h.y_scale.valueText( _value);
      }
      if( !DrawUtilIGC.isEmpty(full_text) )
      {
        IGCDStyle sty = sty_label !=null ? sty_label : sty_point ;
        d_.gc_.setFont( sty.getFont() );
        int aln = h.h_axis_alignment.getAlignment();
     
        String txt; 
        if (vertical)
            txt = DrawUtilIGC.truncateText( d_.gc_, full_text, (int)h.wbartobar, h.h_label_height, aln );
        else
            txt = DrawUtilIGC.truncateText( d_.gc_, full_text, _rg.left() , h.h_label_height, aln);
        if( !DrawUtilIGC.isEmpty( txt ) )
        {
          IShape rc = d_.gc_.getClipping(); //disable clipping of drawing area.
          d_.gc_.setClipping( (IShape)null );
          Rect rtxt;
          if (vertical)
              rtxt = new Rect( (int)_xbar, _rg.bottom()+ d_.axis_unit_to_dot_spacing,
                                          (int)h.wbartobar-d_.dpiX(2), h.h_label_height );
          else
              rtxt = new Rect( _rg.left() - d_.axis_unit_to_dot_spacing - h.h_label_height, 
                                   (int)_xbar, h.h_label_height, (int)h.wbartobar-d_.dpiX(2));//h.yam.getL());
          
          d_.renderText( txt, rtxt, aln, sty, loc_id, loc_item );
          d_.gc_.setClipping( rc ); //reset clipping.
        }
      }      
    }
    finally
    {
//TODO: remove try/finally as it's empty (was sty_xx .dispose())      
    }
    return true;
  }
  

  /** Render sector-ish graphic */
  protected   void renderSector( RenderPersistData rpd, boolean _3d ) throws DefaultRenderChartLocation
  {
    //true to display value of sector
    boolean display_values = rpd.graphic.getProperties().get(DGraphic.P_SHOW_VALUES, true )
                              || rpd.graphic.getProperties().get(DGraphic.P_SHOW_PERCENTAGE, false);
    
    int rg_w = rpd.ag_rect.w();
    int rg_h = rpd.ag_rect.h();
    int d = rg_w;
    int h_for_text = display_values ? d_.dpiX(40) : 0;
    if ( d > rg_h-h_for_text ) d=rg_h-h_for_text;

    //si on affiche les texte:
    if ( display_values )
    {
      //an other example of choose-by-nose ... ok it's done for text display...
      d = 3*d/4;
    }

    double v_sum = 0.0;

    CurveStyle ccurve=null;
    IDItem item=null;
    
    //get an axis..
    DAxis axis=null;
    for( item=rpd.graphic.getFirstChild(); item!=null; item=item.getNext() )
    {
      if( item instanceof DAxis ) { axis=(DAxis)item; break; }
    }
    if( axis==null )
    {
      d_.error("Error: missing Axis in graphic", rpd);
      return;
    }
    if( !axis.useNumbers() )
    {
      d_.error("Error: Axis in Pie Chart must use numbers", rpd);
      return;
    }
    String axis_unit = axis.getUnit();
    if( axis_unit==null ) axis_unit="";
    
    //check for 3D property for sectors.
    if( !_3d )
    {
      _3d = rpd.graphic.getProperties().get( DGraphic.P_3D, false );
    }

    //sum all value (correspond to 100% of circle)    
    int n_points=0;
    Object val = null;
    for( int i=0; i<rpd.curve_styles.length; ++i )
    {
      ccurve = rpd.curve_styles[i];
      for( item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext())
      {
        if ( !(item instanceof DPoint) ) continue;
        
        IDCoord coord= ((DPoint)item).getCoord( axis );
        if ( coord==null ) continue;
        
        val = coord.getValue(val); 
        if( !(val instanceof Number) ) continue; //remember axis must use numbers...
        v_sum += Math.abs( ((Number)val).doubleValue() );
        n_points++;
      }
    }

    //no value to display ?
    if ( v_sum == 0.0 )
    {
      //d_.error( "Error: can't print sectors (sum:0)", rpd );
      return ;
    }

    int x3d = 0, y3d = 0, r3d=0, a3d=0;

    if ( _3d )
    {
      //choose an "3d" shadow depth    
      if( d >= d_.dpiX(48) ) //limit shadow depth
      {
        x3d = -d_.dpiX( 8);
        y3d = +d_.dpiX(12);
      } else {
        y3d = d / 4;
        x3d = -(int)(2.0f*y3d/3.0f);
        // if depth is too small, don't draw 3d shadow...
        if( Math.abs(x3d) < d_.dpiX(2) )
        {
          _3d = false;
          x3d = y3d = 0;
        }
      }
      if( _3d )
      {
        r3d = (int)Math.sqrt( (double)(x3d*x3d + y3d*y3d) );
        float a = (float)Math.atan2( (float)y3d, (float)x3d ); //-PI..PI
        if( a < 0.0f ) a += 2*Math.PI ; 
        a3d = (int)(180*a/(Math.PI));
      }
    }

    LineAlg l3d = new LineAlg( 0, 0, x3d, y3d );

    //locating: sectors are checked in drawing order, all are checked, last win.
    DefaultRenderChartLocation loc_sector=null;

    //because line separating two sector must be drawn after sector
    //otherwise fill next sector will overwrite line.
    //this array store x,y coordinates of point to draw joined to center or pie.
    int between_sectors[] = new int[n_points*2];
    Rect r = new Rect();
    int rg_x = rpd.ag_rect.x();
    int rg_y = rpd.ag_rect.y();
    
    //just for text formating (..for the moment)
    IScale scale = DefaultChartRenderData.getScale( axis, 0,0,null,null);
    
    while( l3d.nextPoint(null) )
    {
      int dx3d = l3d.getX();
      int dy3d = l3d.getY();
      boolean top=(dx3d==x3d)&&(dy3d==y3d);

      int ray = d >> 1; 

      r.setRect( rg_x+(rg_w -d)/2 +dx3d,
                 rg_y+(rg_h -d)/2 -dy3d, d,d ); 
      if( r.w()==0 || r.h() == 0 ) continue;

      //locating: (here for optimisation)
      double loc_dist=-1.0;
      double loc_angle=-1; 
      if(d_.locating())
      {
        Point rc=r.center();
        int dx=d_.lx_-rc.getX(), dy=d_.ly_-rc.getY();
        loc_dist=Math.sqrt( dx*dx + dy*dy );
        if(loc_dist<=ray)
        {
          double a = Math.atan2( -dy, dx ); //-PI..PI
          if( a < 0.0f ) a += 2*Math.PI ; 
          loc_angle = a;
        }
      }
      
      int wbs=0;      
      int ipt=0;
      double angle=0;
      for( int i=0; i<rpd.curve_styles.length; ++i )
      {
        ccurve = rpd.curve_styles[i];
        for( item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
        {
          if ( !(item instanceof DPoint) ) continue;
          
          DPoint point = (DPoint)item;
          IDCoord coord = point.getCoord( axis );
          if ( coord ==null ) continue;
          
          val = coord.getValue( val );
          if( !(val instanceof Number )) continue; //remember axis must use numbers..
          
          ipt++;
          double rv = ((Number)val).doubleValue();
          double v = Math.abs( rv )/v_sum; //% of disk
          double end_angle = angle+(Radian._2PI*v); //radian;
          if( ipt >= n_points ) //last sector ?
          {
            end_angle=Radian._2PI;
          }
          
          //false: double delta_angle = end_angle-angle;
          //must "integrate" in arc length the round effect due to internal use of degree 
          //(at least in SWTGC), but as others IGC can have problem...
          double delta_angle = Radian.D2R( Radian.iR2D(end_angle) -Radian.iR2D(angle) );

          //SWT doesn't support delta_angle==0 ... pfff!!
          if( d_.drawing() )
          {
             int cx = r.centerX(), cy=r.centerY();
             int rx = r.getW()/2,  ry=r.getH()/2;
             //use color from DPoint or DCurve or ...
             int pc = IGCDStyle.GetBackColor(  point );
             if(top)
             {
               d_.gc_.setPen( new LineStylePen( RGBA.BLACK) );
               d_.gc_.setBrush( new SolidBrush( pc ) );
               if( delta_angle != 0 )
               {
                 d_.gc_.fillArc( cx,cy, 0.0, rx,ry, angle, delta_angle );
                 d_.gc_.drawArc( cx,cy, 0.0, rx,ry, angle, delta_angle );
               }
             } else {
               if( delta_angle != 0 )
               {
                 int dark = RGBA.Darker(  pc, 0.35f );
                 //d_.gc_.setForeground( dark );
                 d_.gc_.setBrush( new SolidBrush(dark) );
                 d_.gc_.fillArc( cx,cy, 0.0, rx,ry, angle, delta_angle );
               }
             }
             between_sectors[wbs++]= (int)( cx+r.w()*0.5*Math.cos( angle ) );
             between_sectors[wbs++]= (int)( cy-r.h()*0.5*Math.sin( angle ) );              
          }
          else //locating
          {
            if( (loc_dist <= ray)
              &&(loc_angle>=angle)&&(loc_angle<=end_angle) )
            {
              //memorise: last is the winner !
              loc_sector = new DefaultRenderChartLocation( DLocated.CurveSector, point, r );
              loc_sector.setAngles( Radian.iR2D(angle), Radian.iR2D(end_angle) );
            }
          }              
          //render value + unit (from axis)
          if( top && display_values )
          {
            double atxt_rad = angle+(end_angle-angle)/2;
            int atxt = Radian.iR2D( atxt_rad );
            //compensation on ray for 3d mode in order text doesn't overlap shadow.
            int r3=0;
            if ( _3d )
            {
              /* Ok, few further word :
               * for 3d we add a ray else text overlap shadow ...
               * the best is this ray depend on text angle AND
               * angle choosen for 3d shadow (amazing no?).
               */
              int da = (int)Math.abs( atxt-a3d-180 );
              //compensation only in shadow area:        
              if( da < 90 )
              {
                //this is first order approximation:
                float k = 1.0f - da/90.f;
                r3 = (int)(k * r3d);
              }
            }
 
            //point might have a label
            String txt=null;
            DText label=null;
            for( IDItem it=point.getFirstChild(); it!=null; it=it.getNext() )
            {
              if( it instanceof DText ) { label=(DText)it; break; }
            }
            IGCDStyle sty_txt = ccurve.style_;
            String loc_id=DLocated.CurvePointText;
            IDItem loc_item=point;
            if( label!=null )
            {
              txt = label.getText();
              loc_id=DLocated.LabelText;
              loc_item=label;
              if( !DrawUtilIGC.isEmpty( txt ) )
              {
                sty_txt = new IGCDStyle( label, d_.scale_ );
              }
            } else {
              //label is value plus axis unit.
              txt = scale.valueText( val ) + axis_unit;
              sty_txt = new IGCDStyle( point, d_.scale_ );
            }
            d_.renderSectorText( atxt_rad, ray+r3+d_.margin_, r, txt, loc_item, sty_txt, loc_id);
         }

         angle = end_angle;
        }//item
      }//curve
      
      if( d_.drawing() )
      {
        //separation lines for sectors, doing at each 3D step let us able
        //to draw "vertical" separation of sectors...
        d_.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
        int cx = r.centerX(), cy=r.centerY();      
        for(int i=0; i<between_sectors.length;)
        {
          int x = between_sectors[i++];
          int y = between_sectors[i++];  
          if( top )
            d_.gc_.drawLine( cx,cy, x,y );
          else
            d_.gc_.drawPoint( x, y );
        }
      }
    }//3d
    
    //found a sector under point to locate ?
    if( loc_sector!=null ) throw loc_sector;
  }

  private void renderHistorsStep( DAxis _axis, IScale _scale, String _step, int _line_style, int _cx,int _cy, String loc_id )
    throws DefaultRenderChartLocation
  {
    Object step = _axis.getProperties().get( _step, null );
    if( step==null ) return ; //nothing to do

    IPen save_pen = null;
    if( d_.drawing() )
    {
      save_pen = d_.gc_.getPen();
      d_.gc_.setPen( new LineStylePen( d_.getLineColor( _axis), _line_style ));
    }
    
    //circles using dots ...
    for( Object vs = _scale.stepFirst(step); vs!=null;  vs=_scale.stepNext(step,vs) )
    {
      int radius=(int)_scale.toScale( vs );      
      int diameter = radius*2;
      if( d_.drawing() )
      {
        d_.gc_.drawOval( _cx-radius, _cy-radius, diameter, diameter );
      } else { //locating:
        if( DrawUtilIGC.ovalUnder( _cx-radius, _cy-radius, diameter, diameter, d_.lx_,d_.ly_ ) )
        {
          throw new DefaultRenderChartLocation( loc_id, _axis, new Rect(_cx-radius,_cy-radius,diameter,diameter) );
        }
      }
    }      
    if( save_pen!=null )
    {
      d_.gc_.setPen( save_pen );
    }
  }
  
  private void renderHistorsStepUnit( DAxis _axis, ULocale locale, IScale _scale, int _center_x, int _center_y, String _axis_unit )
    throws DefaultRenderChartLocation
  {
    Object step_unit = _axis.getProperties().get( DAxis.P_STEP_UNIT );
    //because axis is under sectors, and must be located after them 
    if (  (step_unit!=null) )
    {
      IPen   save_pen   = d_.gc_.getPen();
      IBrush save_brush = d_.gc_.getBrush();
      IFont  save_font  = d_.gc_.getFont();
      IGCDStyle style = new IGCDStyle(  _axis, d_.scale_ );
      try{
        IBrush btext = new SolidBrush( style.getFore() );
        IBrush bfill = new SolidBrush( style.getBack() );        
        d_.gc_.setPen( new LineStylePen( style.getFore() ) );
        d_.gc_.setFont( style.getFont() );
      
        //circles
        Rect rtxt = new Rect();
        //int dpix4 = d_.dpiX(4);
        int aln = IDAlignment.DEFAULT_ALIGN;
        String loc_id = DLocated.InAxisYStepUnitText;
        boolean first=true;
        boolean opaque=_axis.getProperties().get( DAxis.P_HISTORS_UNIT_OPAQUE, false );
        boolean draw_top = true;
        boolean draw_left= true;
        boolean draw_right=true;
        boolean draw_bottom=true;
        Object phu= _axis.getProperties().get( DAxis.P_HISTORS_UNIT, null );
        if( phu instanceof IDAlignment )
        {
          IDAlignment a = (IDAlignment)phu;
          draw_top    = a.haveAlignment( IDAlignment.TOP    );
          draw_left   = a.haveAlignment( IDAlignment.LEFT   );
          draw_right  = a.haveAlignment( IDAlignment.RIGHT  );
          draw_bottom = a.haveAlignment( IDAlignment.BOTTOM );
        }
        DefaultRenderChartLocation loc_last_unit_circle_win = null;
        for( Object vs = _scale.stepFirst(step_unit); vs!=null;  vs=_scale.stepNext(step_unit,vs) )
        {
          int xray=(int)_scale.toScale( vs );      
          int xray2 = xray*2;
          if( d_.drawing() ) {
            d_.gc_.setBrush( bfill );
            d_.gc_.drawOval( _center_x-xray, _center_y-xray, xray2, xray2 );
          } else {
            if( DrawUtilIGC.ovalUnder(_center_x-xray, _center_y-xray, xray2, xray2, d_.lx_,d_.ly_))
            {
              //ok: here 'dot' in "InAxisYStepUnitTextDot" means a circle ...
              loc_last_unit_circle_win = new DefaultRenderChartLocation(DLocated.InAxisYStepUnitTextDot, _axis, new Rect(_center_x-xray, _center_y-xray, xray2, xray2) );
            }
          }
          String unit = _scale.valueText(vs) + _axis_unit;
          ISize size =DrawUtilIGC.textExtent( d_.gc_, unit, aln );
          int sw = size.getW(), sh = size.getH();
          boolean first_drawn = false;
          if( draw_top )
          {
            rtxt.setRect( _center_x-sw/2, _center_y-xray-sh/2, sw, sh );
            if( opaque && d_.drawing() ) {
              d_.gc_.setBrush( bfill );
              d_.gc_.fillRect( rtxt );
            }
            d_.gc_.setBrush( btext );
            d_.renderText( unit, rtxt, aln, style, loc_id, _axis );
            first_drawn=first;
          }
          if( draw_bottom && (!first || first&&!first_drawn) )
          {
            rtxt.setRect( _center_x-sw/2, _center_y+xray-sh/2, sw, sh );
            if( opaque && d_.drawing() ) {
              d_.gc_.setBrush( bfill );
              d_.gc_.fillRect( rtxt );
            }
            d_.gc_.setBrush( btext );
            d_.renderText( unit, rtxt, aln, style, loc_id, _axis );
            first_drawn = first;
          }
          if( draw_left && (!first || first&&!first_drawn) )
          {
            rtxt.setRect( _center_x-xray-sh/2, _center_y-sw/2, sh, sw );
            if( opaque && d_.drawing() ) {
              d_.gc_.setBrush( bfill );
              d_.gc_.fillRect( rtxt );
            }
            d_.gc_.setBrush( btext );
            d_.renderText( unit, rtxt, IDAlignment.ROTCCW90, style, loc_id, _axis );
            first_drawn=first;
          }
          if( draw_right && (!first || first&&!first_drawn))
          {
            rtxt.setRect( _center_x+xray-sh/2, _center_y-sw/2, sh, sw );
            if( opaque && d_.drawing() ) {
              d_.gc_.setBrush( bfill );
              d_.gc_.fillRect( rtxt );
            }
            d_.gc_.setBrush( btext );
            d_.renderText( unit, rtxt, IDAlignment.ROTCW90, style, loc_id, _axis );
          }
          first=false;
        }
        d_.gc_.setPen( save_pen );
        d_.gc_.setBrush( save_brush );
        d_.gc_.setFont( save_font );
        //prefer to locate a TextUnit rather than a ...Dot but don't forget it if here it is!
        if( loc_last_unit_circle_win!=null ) throw loc_last_unit_circle_win;
      } finally {
//TODO:remove finally because t's empty now !        
      }
    }
  }
  
  /**
   * Render HISTOR graph (each values have fixed angle sector but ray differ)
   */
  private void renderHistors( RenderPersistData rpd ) throws DefaultRenderChartLocation
  {
    boolean display_decorations = true;

    int  rg_w = rpd.ag_rect.w();
    int  rg_h = rpd.ag_rect.h();
    
    int d = rg_w;
    if ( d > rg_h ) d=rg_h;

    // let space to render values as text outside histor
    if ( display_decorations ) d = 3*d/4; 

    int ray=d>>1; //rayon corresp. a la valeur maxi.

    DAxis  axis   = null;
    IDItem item ;
    int    nsector=0;   
    for( item=rpd.graphic.getFirstChild(); item!=null; item=item.getNext() )
    {
      if( item instanceof DAxis ) { axis = (DAxis)item; break; }
    }
    if( axis==null )
    {
      if( d_.drawing() ) d_.error("Missing axis", rpd);        
      return ;
    }

    DPropertyStore props = axis.getProperties();
    Object vmax = props.get( DAxis.P_MAX );
    Object vmin = props.get( DAxis.P_MIN );
    boolean vmin_tryied_to_be_zero=false;
    if( vmin==null && vmax!=null && axis.useNumbers() )
    {
      vmin = new VDouble(0.0);
      vmin_tryied_to_be_zero=false;
    }
    
    //compute min/max and get number of sectors...
    {
      boolean do_mm = (vmax==null)||(vmin==null);
      MinMax mm = do_mm ? new MinMax( axis ) :null;
      for( item=rpd.graphic.getFirstChild(); item!=null; item=item.getNext() )
      {
        if ( !(item instanceof DCurve) ) continue;
        for( IDItem it=item.getFirstChild(); it!=null; it=it.getNext())
        {
          if( !(it instanceof DPoint) ) continue;
          IDCoord coord = ((DPoint)it).getCoord( axis );
          if( coord!=null )
          {
            //mmhh.. need to have non negatives values (if values are numbers ...)
            if(do_mm) mm.update( coord );
            nsector++;
          }
        }
      }
      if( do_mm )
      {
        if( vmin==null )
        {
          vmin = mm.getVMin();
          Object v = props.get( DAxis.P_MIN_HINT );
          //take min( mm.getVMin(), P_MIN_HINT )
          if( v!=null )
          {
            try {
              if( vmin==null )
              {
                vmin = v;
              } else if ( vmin instanceof Comparable ) {
                if( ((Comparable)vmin).compareTo( v ) > 0 ) vmin=v;
              } else if ( v instanceof Comparable ) {
                if( ((Comparable)v).compareTo( vmin ) < 0 ) vmin=v;
              }
            } catch( Exception e ) {};
          }
        }
        else if ( vmin_tryied_to_be_zero )
        {
          //for historical reason, only P_MIN_HINT apply here
          vmin = props.get( DAxis.P_MIN_HINT, null );
        }
        if( vmax==null )
        {
          vmax= mm.getVMax();
          Object v = props.get( DAxis.P_MAX_HINT );
          //take max( mm.getVmax(), P_MAX_HINT )
          if( v!=null )
          {
            try {
              if( vmax==null )
              {
                vmax = v;
              } else if ( vmax instanceof Comparable ) {
                if( ((Comparable)vmax).compareTo( v ) < 0 ) vmax=v;
              } else if ( v instanceof Comparable ) {
                if( ((Comparable)v).compareTo( vmax ) > 0 ) vmax=v;
              }
            } catch( Exception e ) {};
          }
        }
      }
    }


    double s_angle = Radian._2PI/(float)nsector; //same angle for all values...

    int center_x = rpd.ag_rect.centerX();
    int center_y = rpd.ag_rect.centerY();

    Rect r = new Rect();

    //pixel scale means along a ray..
    IScale scale = DefaultChartRenderData.getScale( axis, 0,d/2, vmin,vmax );
    if( !scale.isValid() )
    {
      if(d_.drawing()) d_.error("Error: invalid scale",rpd);
      return ;
    }
 
    String axis_unit=axis.getUnit();
    if( axis_unit==null) axis_unit="";

    //Note: for locating mode on "axis", the located found might be reseted by one of
    //the three layer of axis (properties).
    //But the locating found is only thrown if no sector are above it.
    DefaultRenderChartLocation axis_located = null;
    
    int x3d = 0, y3d = 0, r3d=0, a3d=0;
    boolean _3d = rpd.graphic.getProperties().get( DGraphic.P_3D, false );
    if ( _3d )
    {
      //choose an "3d" shadow depth    
      if( d >= d_.dpiX(48) ) //limit shadow depth
      {
        x3d = -d_.dpiX( 8);
        y3d = +d_.dpiX(12);
      } else {
        y3d = d / 4;
        x3d = -(int)(2.0f*y3d/3.0f);
        // if depth is too small, don't draw 3d shadow...
        if( Math.abs(x3d) < d_.dpiX(2) )
        {
          _3d = false;
          x3d = y3d = 0;
        }
      }
      if( _3d )
      {
        r3d = (int)Math.sqrt( (double)(x3d*x3d + y3d*y3d) );
        float a = (float)Math.atan2( (float)y3d, (float)x3d ); //-PI..PI
        if( a < 0.0f ) a += 2*Math.PI ; 
        a3d = (int)(180*a/(Math.PI));
      }
    }
    
    LineAlg l3d = new LineAlg( 0, 0, x3d, y3d );
    int scx=center_x, scy=center_y;
    
    while( l3d.nextPoint(null) )
    {
      int dx3d = l3d.getX();
      int dy3d = l3d.getY();
      boolean bottom = (dx3d==0)&&(dy3d==0);
      boolean top=(dx3d==x3d)&&(dy3d==y3d);
      center_x = scx+dx3d;
      center_y = scy+dx3d;

      double angle=0.0;    
      Object val=null;
      
      //render axis, this is done at "top" level to not disturb user in case of 3D property.
      if( top )
      {
        try {
          renderHistorsStep( axis, scale, DAxis.P_STEP_DOT, LineStylePen.DOT, center_x, center_y, DLocated.InAxisYStepDot );
        } catch( DefaultRenderChartLocation loc ) { axis_located=loc; }
        //render  "STEP_LINE" ... here must be named circle...
        try{
          renderHistorsStep( axis, scale, DAxis.P_STEP_LINE, LineStylePen.SOLID, center_x, center_y, DLocated.InAxisYStepLine );
        } catch( DefaultRenderChartLocation loc ) { axis_located=loc; }
        //draw "STEP_UNIT" ... here must be named circle...
        try {
          renderHistorsStepUnit( axis, rpd.locale, scale, center_x, center_y, axis_unit );
        } catch( DefaultRenderChartLocation loc ) { axis_located=loc; }      
      }
      
      int max_radius = (int)scale.toScale( vmax );
      for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
      {
        CurveStyle ccurve = rpd.curve_styles[icrv];    
        
        //curves may contains several sectors
        for( item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
        {
          if( !(item instanceof DPoint ) ) continue;
          
          DPoint point = (DPoint)item;
          IDCoord coord = point.getCoord( axis );
          if( coord == null ) continue;
          
          int xray = (int)scale.toScale( val=coord.getValue(val) );
          boolean over_radius = xray > max_radius;
          if( over_radius ) xray = max_radius;        
          int xray2 = 2*xray;
          
          if( xray>0 )
          {
            r.setRect( 0,0,xray2,xray2);
            r.moveCenter( center_x, center_y );
            if(d_.drawing())
            {
              //take care of float -> int truncation
//TODO:cf renderSector:             int start_angle = (int)(angle+s_angle-(int)s_angle);
              double start_angle = angle;
              //use style of points
              IGCDStyle style = new IGCDStyle( point, d_.scale_ );
              int dark=0;           
              if( !top ) {
                dark = RGBA.Darker( style.getBack(), 0.35f );
                d_.gc_.setBrush( new SolidBrush( dark ) );
              } else {
                d_.gc_.setBrush( new SolidBrush( style.getBack() ) );
              }
              if( top||bottom ) {
                d_.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
              } else { 
                d_.gc_.setPen( new LineStylePen( dark ) );              
              }
              //another time without this, SWT bark, but if this cond failed... 
              //we will se nothing !
              if( s_angle != 0 )        
              {  
                d_.gc_.fillArc( center_x, center_y, 0.0, xray, xray, start_angle, s_angle );
                if( !over_radius && (top || bottom) )
                  d_.gc_.drawArc( center_x, center_y, 0.0, xray, xray, start_angle, s_angle );
              }
              
              int px = center_x+(int)(xray*Math.cos( angle ));
              int py = center_y-(int)(xray*Math.sin( angle ));
              if( top || bottom )  
                d_.gc_.drawLine( center_x, center_y, px, py );
              else
                d_.gc_.drawPoint( px, py );
              double r_angle = angle + s_angle ;
              px = center_x+(int)(xray*Math.cos( r_angle ));
              py = center_y-(int)(xray*Math.sin( r_angle ));
              if(top||bottom)
                d_.gc_.drawLine( center_x, center_y, px, py );
              else
                d_.gc_.drawPoint( px, py );
              
            }
            else //locating
            {
              if( r.contains( d_.lx_, d_.ly_ ) ) //check first if point is in sector rectangle
              {
                //from renderSector:
                float loc_dist=-1.0f;
                double loc_angle=-1;
                int dx=d_.lx_-center_x, dy=d_.ly_-center_y;
                loc_dist=(float)Math.sqrt( (double)(dx*dx + dy*dy) );
                if(loc_dist<=xray)
                {
                  float a = (float)Math.atan2( (float)-dy, (float)dx ); //-PI..PI
                  if( a < 0.0f ) a += 2*Math.PI ; 
                  //a est dans 0..2PI 
                 // loc_angle = (float)(360.0f*a/(2*Math.PI));
                  loc_angle = a;
                  
                  if( (loc_angle>=angle)&&(loc_angle<=(angle+s_angle)) )
                  {
                    DefaultRenderChartLocation loc=new DefaultRenderChartLocation( DLocated.CurveSector, point, r);
                    loc.setAngles( Radian.iR2D(angle), Radian.iR2D(angle+s_angle) );
                    if( !top ) {
                      //must wait to check above render...
                      //this is mandatory only if axis is rendered at top level.
                      axis_located = loc;
                    } else {
                      throw loc;
                    }
                  }
                }
              }
            }
          }
          
          if ( top && display_decorations )
          {
            //histors might have labels
            DText label=null;
            for( IDItem it=point.getFirstChild(); it!=null; it=it.getNext())
            {
              if(it instanceof DText) { label=(DText)it; break; }
            }
            String txt=null;
            IGCDStyle style=null;
            String loc_id=DLocated.CurvePointText;
            IDItem loc_item=point;
            if( label!=null )
            {
              txt = label.getText();
              loc_id=DLocated.LabelText;
              loc_item=label;
              if( !DrawUtilIGC.isEmpty(txt) )
              {
                style = new IGCDStyle( label, d_.scale_ );
              }
            } else {
              txt = scale.valueText( val ) + axis_unit;
              style = new IGCDStyle( point, d_.scale_ );
            }
            double atxt = angle+s_angle/2;
            //use this if label must appear on "bottom": r.moveCenter(scx,scy);
            //in case of 3D, text might be really close to sector, if it's a problem,
            //refer to renderSector() to known how to (try to) correct this.
            d_.renderSectorText( atxt, ray+d_.margin_, r, txt, loc_item, style, loc_id );
          }
          
          angle+=s_angle ;
      }
    }//for
    }//while  
    
    //time to remember that we might have located somthing in axis and no sector overlap it...
    if( d_.locating() && axis_located !=null) 
    {
      throw axis_located;
    }    
  }
  
  

}
