/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultChartRender3D.java,v 1.2 2008/05/23 14:11:50 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;

import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.drawutil.internal.DrawUtilIGC;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.Point2D;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;


/**
 * This class contains common method for 3D rendering
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
class DefaultChartRender3D
{
  /** @return the name of axis for a XYZ rendering (merge title and unit) */
  public static String axis3DName( DAxis _axis )
  {
    String title = _axis.getTitle();

    String unit = _axis.getUnit();
    if( !DrawUtilIGC.isEmpty( unit ) )
    {
      if( title != null )
      {
        title += " (" + unit +")";
      } else {
        title = "("+unit+")";
      }
    }
  
    return title ;
  }

  /** 
   * draw the 3D box using view, this method must be call twice one for "back" box lines and
   * the other one for "front" box lines
   */
  public static void drawBox3D( DefaultChartRenderData _d, View3D _view, Box3D _bb, boolean x_incr, boolean y_incr, boolean z_incr, boolean for_front )
  {
    Point2D p[] = new Point2D[8];
    double xn = _bb.getXMin(), xx = _bb.getXMax();
    double yn = _bb.getYMin(), yx = _bb.getYMax();
    double zn = _bb.getZMin(), zx = _bb.getZMax();
    p[0] = _view.projection( xn,yn,zn, new Point2D() );
    p[1] = _view.projection( xx,yn,zn, new Point2D() );
    p[2] = _view.projection( xx,yx,zn, new Point2D() );
    p[3] = _view.projection( xn,yx,zn, new Point2D() );
    p[4] = _view.projection( xn,yn,zx, new Point2D() );
    p[5] = _view.projection( xx,yn,zx, new Point2D() );
    p[6] = _view.projection( xx,yx,zx, new Point2D() );
    p[7] = _view.projection( xn,yx,zx, new Point2D() );
    
    LineStylePen pen = new LineStylePen( RGBA.LIGHT_GRAY );
    _d.gc_.setPen( pen );

    final double xzplan; //y (min or max) where z axis is rendered, depends on point of view
    final double yzplan; //x (min or max) where z axis is rendered, depends on point of view
    
    if( for_front )
    {
      if( y_incr ) xzplan = _bb.getYMax(); else xzplan = _bb.getYMin();    
      if( x_incr ) yzplan = _bb.getXMax(); else yzplan = _bb.getXMin();
    } else {
      if( y_incr ) xzplan = _bb.getYMin(); else xzplan = _bb.getYMax();    
      if( x_incr ) yzplan = _bb.getXMin(); else yzplan = _bb.getXMax();
    }
    
    p[0] = _view.projection( xn, xzplan, zn, new Point2D() );
    p[1] = _view.projection( xx, xzplan, zn, new Point2D() );
    p[2] = _view.projection( yzplan, yn, zn, new Point2D() );
    p[3] = _view.projection( yzplan, yx, zn, new Point2D() );
    p[4] = _view.projection( xn, xzplan, zx, new Point2D() );
    p[5] = _view.projection( xx, xzplan, zx, new Point2D() );
    p[6] = _view.projection( yzplan, yn, zx, new Point2D() );
    p[7] = _view.projection( yzplan, yx, zx, new Point2D() );
    
    _d.gc_.drawLine( (int)p[0].getX(),(int)p[0].getY(), (int)p[1].getX(),(int)p[1].getY() );
    _d.gc_.drawLine( (int)p[2].getX(),(int)p[2].getY(), (int)p[3].getX(),(int)p[3].getY() );
    _d.gc_.drawLine( (int)p[4].getX(),(int)p[4].getY(), (int)p[5].getX(),(int)p[5].getY() );
    _d.gc_.drawLine( (int)p[6].getX(),(int)p[6].getY(), (int)p[7].getX(),(int)p[7].getY() );

    _d.gc_.drawLine( (int)p[0].getX(),(int)p[0].getY(), (int)p[4].getX(),(int)p[4].getY() );
    _d.gc_.drawLine( (int)p[1].getX(),(int)p[1].getY(), (int)p[5].getX(),(int)p[5].getY() );
    _d.gc_.drawLine( (int)p[2].getX(),(int)p[2].getY(), (int)p[6].getX(),(int)p[6].getY() );
    _d.gc_.drawLine( (int)p[3].getX(),(int)p[3].getY(), (int)p[7].getX(),(int)p[7].getY() );
  }
  
  /**
   * Render Z axis 3D choosing side of box where axis isn't hidden by drawing.
   * Again this method must be called twice:
   * - "back" part of Z axis render axis lines
   * - "front" part of Z axis render texts and dots.
   * A third call can be done to get axis insets: call with insets !=null.
   */
  public static void renderZAxis3D( 
      DefaultChartRenderData _d, View3D _view, Box3D _bb,
      DAxis _axis, IScale _scale, IGCDStyle _axis_style,
      boolean x_incr, boolean y_incr, boolean z_incr, boolean for_front,
      Insets insets
    )
    throws DefaultRenderChartLocation
  {
    boolean get_insets=insets!=null;
    double xzplan; //y (min or max) where z axis is rendered, depends on point of view
    double yzplan; //x (min or max) where z axis is rendered, depends on point of view
    
    if( y_incr ) xzplan = _bb.getYMin(); else xzplan = _bb.getYMax();    
    if( x_incr ) yzplan = _bb.getXMin(); else yzplan = _bb.getXMax();
    
    final double xmin = _bb.getXMin(), xmax = _bb.getXMax(),
                 ymin = _bb.getYMin(), ymax = _bb.getYMax(),
                 zmin = _bb.getZMin(), zmax = _bb.getZMax();

    Point2D p0 = new Point2D(), 
            p1 = new Point2D();

    /* 
     * LINES
     */
    if( !for_front && !get_insets )
    {
      Object step = _axis.getProperties().get( DAxis.P_STEP_LINE );
      if ( _d.drawing() && step!=null )
      {
        _d.gc_.setPen( new LineStylePen( _d.getLineColor( _axis ) ) );
        
        //more code but switch is done once..        
        for( Object vs=_scale.stepFirst(step); vs!=null; vs=_scale.stepNext(step,vs) )
        {
          double v = _scale.toScale( vs );
          _view.projection( xmax, xzplan,  v, p0 );
          _view.projection( xmin, xzplan,  v, p1 );
          _d.gc_.drawLine( (int)p0.getX(), (int)p0.getY(), (int)p1.getX(), (int)p1.getY() );
          
          _view.projection( yzplan, ymin, v, p0 );
          _view.projection( yzplan, ymax, v, p1 );
          _d.gc_.drawLine( (int)p0.getX(), (int)p0.getY(), (int)p1.getX(), (int)p1.getY() );
        }
      }
    }
    
    //step unit and axis title are rendered at for_front level..
    if( !for_front && !get_insets ) return ;
    
    //location for Z axis units and title:
    double x_axis, y_axis;
    if( x_incr ) {
      if( y_incr ) {
        x_axis = xmin;
        y_axis = ymax;
      } else {
        x_axis = xmax;
        y_axis = ymax;
      }
    } else {
      if( y_incr ) {
        x_axis = xmin;
        y_axis = ymin;
      } else {
        x_axis = xmax;
        y_axis = ymin;
      }
    }
    
    double nphi = Radian.normalize( _view.getPhi() );
    boolean text_on_the_right = nphi<=Radian._PI2 || nphi >= Radian._3PI2;

    /*
     * Step Dot.
     */
    Object step = _axis.getProperties().get( DAxis.P_STEP_DOT );
    if( step!=null )
    {
      int dot_length = text_on_the_right ? _d.axis_dot_size : -_d.axis_dot_size;
      if( get_insets )
      {
        if( text_on_the_right ) insets.addR( dot_length );
        else insets.addL( -dot_length );
      }
      else if(  _d.drawing() )
      {
        //use line color (if any) to draw dot
        _d.gc_.setPen( new LineStylePen( _d.getLineColor(_axis) ));
        
        for( Object vs = _scale.stepFirst(step); vs!=null; vs=_scale.stepNext(step,vs) )
        {
          double v = _scale.toScale( vs );
          _view.projection(  x_axis, y_axis, v,  p0 ); 
          _d.gc_.drawLine( (int)p0.getX(),(int)p0.getY(), (int)p0.getX()+dot_length, (int)p0.getY());
        }
      }
    }
    
    /*
     * Step Unit.
     */
    boolean have_step_unit = false;    
    step = _axis.getProperties().get( DAxis.P_STEP_UNIT );
    int wtx=0;
    if ( step!=null )
    {
      have_step_unit = true;
      
      if( _d.drawing() )
      {
        _d.gc_.setBrush( new SolidBrush( _axis_style.getFore() ) );
        _d.gc_.setFont( _axis_style.getFont() );
        //use axis foreground color for unit text's dot.
        _d.gc_.setPen( new LineStylePen( _axis_style.getFore() ) );
      }
      
      Rect rtxt = new Rect();
      int fh = _d.gc_.getFontMetrics().getHeight();
      
      for( Object vs = _scale.stepFirst(step); vs!=null; vs=_scale.stepNext(step,vs) )
      {
        double v = _scale.toScale( vs );
        _view.projection(  x_axis, y_axis, v,  p0 ); 
            
        String txt = _scale.valueText( vs );
        int wt = _d.gc_.textExtent( txt ).getW();
        if( wt > wtx ) wtx=wt;
        if( text_on_the_right ) {
          rtxt.setRect( _d.axis_unit_dot_size+_d.axis_unit_to_dot_spacing, -fh/2, wt, fh );
        } else {
          rtxt.setRect( -wt-_d.axis_unit_dot_size-_d.axis_unit_to_dot_spacing, -fh/2, wt, fh );
        }
        rtxt.moveBy( (int)p0.getX(), (int)p0.getY() );            

        if( get_insets )
        {
        } 
        else if( _d.drawing() )
        {
          //dot for this unit text
          if( text_on_the_right ) {
            _d.gc_.drawLine( (int)p0.getX(),(int)p0.getY(), (int)p0.getX()+_d.axis_unit_dot_size, (int)p0.getY());
          } else {
            _d.gc_.drawLine( (int)p0.getX(),(int)p0.getY(), (int)p0.getX()-_d.axis_unit_dot_size, (int)p0.getY());
          }
          DrawUtilIGC.drawText( _d.gc_, rtxt, IDAlignment.CENTER, txt );
        }
        else //locating
        {
          if( rtxt.contains( _d.lx_, _d.ly_ ) )
          {
            throw new DefaultRenderChartLocation( DLocated.InAxisZStepUnitText, _axis, rtxt );
          }
        }
      }
      wtx += _d.axis_unit_dot_size+_d.axis_unit_to_dot_spacing;
      if( get_insets )
      {
        if( text_on_the_right ) {
          insets.maxR( wtx );
        } else {
          insets.maxL( wtx );
        }
      }
    }

    /*
     * Axis title.
     */
    String axis_title = DefaultChartRender3D.axis3DName( _axis );
    if ( !DrawUtilIGC.isEmpty( axis_title ) )
    {
      Rect rtxt = new Rect();
      int fh = _d.gc_.getFontMetrics().getHeight();       
      int wt = _d.gc_.textExtent( axis_title ).getW();
      
      if( get_insets )
      {
        if( text_on_the_right )
          insets.addR( _d.axis_title_to_unit_spacing+fh );
        else
          insets.addL( _d.axis_title_to_unit_spacing+fh );
      }
      else
      {
        rtxt.setRect( 0, 0, fh, wt );
      
        _view.projection( x_axis, y_axis, zmax, p0 );
        _view.projection( x_axis, y_axis, zmin, p1 );
        int x = ((int)p0.getX()+(int)p1.getX())/2;
        int y = ((int)p0.getY()+(int)p1.getY())/2;
        if( text_on_the_right )
        {
          rtxt.moveCenter( x+wtx+_d.axis_title_to_unit_spacing+fh/2, y );
        } else {
          rtxt.moveCenter( x-wtx-_d.axis_title_to_unit_spacing-fh/2, y );
        }
        
        if( _d.drawing() )
        {
          _d.gc_.setBrush( new SolidBrush( _axis_style.getFore() ) );
          _d.gc_.setFont( _axis_style.getFont() );
          int aln = text_on_the_right ? IDAlignment.CENTER | IDAlignment.ROTCW90
              : IDAlignment.CENTER | IDAlignment.ROTCCW90;
          DrawUtilIGC.drawText( _d.gc_, rtxt, aln, axis_title );
        }
        else //locating
        {
          if( rtxt.contains( _d.lx_, _d.ly_ ))
          {
            throw new DefaultRenderChartLocation( DLocated.InAxisZTitle, _axis, rtxt ); 
          }
        }
      }
    }
  }
  
  /**
   * Render X or Y axis 3D choosing side of box where axis isn't hidden by drawing.
   * Again this method must be called twice:
   * - "back" part of axis render axis lines
   * - "front" part of axis render texts and dots.
   * A third call can be done to get axis insets: call with insets !=null.
   */
  public static void renderXYAxis3D(
      boolean for_x_axis,
      DefaultChartRenderData _d, View3D _view, Box3D _bb,
      DAxis _axis, IScale _scale, IGCDStyle _axis_style,
      boolean x_incr, boolean y_incr, boolean z_incr, boolean for_front,
      Insets insets
    )
    throws DefaultRenderChartLocation
  {
    boolean get_insets=insets!=null;
    double xplan; //x (min or max) where axis is rendered, depends on point of view
    double yplan; //y (min or max) where axis is rendered, depends on point of view
    double zplan; //z (min or max) where axis is rendered, depends on point of view
   
    if( x_incr ) xplan = _bb.getXMin(); else xplan = _bb.getXMax();    
    if( y_incr ) yplan = _bb.getYMin(); else yplan = _bb.getYMax();    
    if( z_incr ) zplan = _bb.getZMin(); else zplan = _bb.getZMax();
    
    final double xmin = _bb.getXMin(), xmax = _bb.getXMax(),
                 ymin = _bb.getYMin(), ymax = _bb.getYMax(),
                 zmin = _bb.getZMin(), zmax = _bb.getZMax();

    Point2D p0 = new Point2D(), p1 = new Point2D();

    /* 
     * LINES
     */
    if( !for_front && !get_insets )
    {
      Object step = _axis.getProperties().get( DAxis.P_STEP_LINE );
      if ( _d.drawing() && step!=null )
      {
        _d.gc_.setPen( new LineStylePen( _d.getLineColor( _axis ) ) );
        
        //more code but switch is done once..        
        for( Object vs=_scale.stepFirst(step); vs!=null; vs=_scale.stepNext(step,vs) )
        {
          double v = _scale.toScale( vs );
          if( for_x_axis ) {
            _view.projection( v, yplan, zmin, p0 );
            _view.projection( v, yplan, zmax, p1 );
          } else {
            _view.projection( xplan, v, zmin, p0 );
            _view.projection( xplan, v, zmax, p1 );
          }
          _d.gc_.drawLine( (int)p0.getX(), (int)p0.getY(), (int)p1.getX(), (int)p1.getY() );
          
          if( for_x_axis ) {
            _view.projection( v, ymin, zplan, p0 );
            _view.projection( v, ymax, zplan, p1 );
          } else {
            _view.projection( xmin, v, zplan, p0 );
            _view.projection( xmax, v, zplan, p1 );
          }
          _d.gc_.drawLine( (int)p0.getX(), (int)p0.getY(), (int)p1.getX(), (int)p1.getY() );
        }
      }
    }
    
    //step unit and axis title are rendered at for_front level..
    if( !for_front && !get_insets ) return ;
    
    boolean text_on_top, text_on_right ;
    //location for axis units and title:
    double z_axis=0, x_axis=0, y_axis=0, dangle=0;
   int code = (x_incr ?100:0) | (y_incr?10:0) | (z_incr?1:0);
   if( for_x_axis )
   {
     z_axis = ( code==10 || code==11 || code ==110 || code==111 ) ? zmin : zmax;
     y_axis = ( code==0 || code==10 || code==100||code==110 ) ? ymin : ymax;
     text_on_top = (code==0 || code==11 || code==100 || code==111 ) ? false: true;
     text_on_right= (code>=100) ? true : false; //1xx
     dangle = Radian._PI2;
   } else {
     if( code==0 || code==101 || code==110 ) z_axis=zmin; else z_axis=zmax;
     if( code==10 || code==110 || code==111 )  x_axis=xmin; else x_axis=xmax;
     if( code==10 || code==100 || code==101 ) text_on_top=false; else text_on_top=true;
     if( code==1 || code==111 || code==101 ) text_on_right=true; else text_on_right=false;
     if( code==1 || code==10 || code==11 || code==101 || code==110 ) dangle=Radian._3PI2; else dangle=Radian._PI2;
   }   
   //inverse top and right value due to bottom-head view:
   int p = Radian.iR2D( Radian.normalize(_view.getPhi()));
   if( p > 90 && p<180 || p >270 ) {
     text_on_top = !text_on_top;
     text_on_right = !text_on_right;
   }
    
   if( for_x_axis ) {
     _view.projection( xmin, y_axis, z_axis, p0 );
     _view.projection( xmax, y_axis, z_axis, p1 );
   } else {
     _view.projection( x_axis, ymin, z_axis, p0 );
     _view.projection( x_axis, ymax, z_axis, p1 );
   }
    double angle = Radian.normalize( Math.atan2(-(int)(p1.getY()-p0.getY()), (int)(p1.getX()-p0.getX())));
    angle += dangle;
        
    Rect r_view=null;
    if( get_insets )
    {
      r_view = _view.getScaledTo();
    }
    
    /*
     * Step Dot.
     */
    Object step = _axis.getProperties().get( DAxis.P_STEP_DOT );
    boolean have_step_dot = step!=null ;
    if( step!=null )
    {
      int dot_length = _d.axis_dot_size;
      int dx = (int)Math.round( dot_length*Math.cos( angle ));
      int dy = (int)Math.round( dot_length*Math.sin( angle ));
      if( get_insets )
      {
        //as 'dot' is so tiny, I compute insets as they are outside rectangle 
        //(simplest case to do)
        if( dx > 0 ) insets.maxR( dx ); else insets.maxL( -dx );
        if( dy > 0 ) insets.maxT( dy ); else insets.maxB( -dy );
      }
      else if(  _d.drawing() )
      {
        //use line color (if any) to draw dot
        _d.gc_.setPen( new LineStylePen( _d.getLineColor(_axis) ));

        for( Object vs = _scale.stepFirst(step); vs!=null; vs=_scale.stepNext(step,vs) )
        {
          double v = _scale.toScale( vs );
          if( for_x_axis ) {
            _view.projection(  v, y_axis, z_axis, p0 );
          } else {
            _view.projection(  x_axis, v, z_axis, p0 );
          }
          _d.gc_.drawLine( (int)p0.getX(),(int)p0.getY(), (int)p0.getX()+dx, (int)p0.getY()-dy );
        }
      }
    }
    
    /*
     * Step Unit.
     */
    boolean have_step_unit = false;    
    step = _axis.getProperties().get( DAxis.P_STEP_UNIT );
    int wtx=0;
    if ( step!=null )
    {
      have_step_unit = true;
      
      if( _d.drawing() )
      {
        _d.gc_.setBrush( new SolidBrush( _axis_style.getFore() ) );
        _d.gc_.setFont( _axis_style.getFont() );
        //use axis foreground color for unit text's dot.
        _d.gc_.setPen( new LineStylePen( _axis_style.getFore() ) );
      }
      
      Rect rtxt = new Rect();
      int fh = _d.gc_.getFontMetrics().getHeight();
      
      int tl = _d.axis_unit_dot_size+_d.axis_unit_to_dot_spacing; 
      int dxd = (int)Math.round( _d.axis_unit_dot_size*Math.cos( angle ));
      int dyd = (int)Math.round( _d.axis_unit_dot_size*Math.sin( angle ));
      int dxt = (int)Math.round( tl*Math.cos( angle ));
      int dyt = (int)Math.round( tl*Math.sin( angle ));
      for( Object vs = _scale.stepFirst(step); vs!=null; vs=_scale.stepNext(step,vs) )
      {
        double v = _scale.toScale( vs );
        if( for_x_axis ) {
          _view.projection( v, y_axis, z_axis,  p0 );
        } else {
          _view.projection( x_axis, v, z_axis,  p0 );
        }
            
        String txt = _scale.valueText( vs );
        int wt = _d.gc_.textExtent( txt ).getW();
        if( wt > wtx ) wtx=wt;
        if( text_on_top ) {
          rtxt.setRect( (int)p0.getX()+dxt, (int)p0.getY()-dyt-fh, wt, fh  );
        } else {
          rtxt.setRect( (int)p0.getX()+dxt, (int)p0.getY()-dyt, wt, fh  );
        }
        if( !text_on_right )  rtxt.moveBy( -wt, 0 );

        if( get_insets )
        {
          //note: unit_step_dot and spacing is included in rtxt bounds checking below:
          insets.max( r_view, rtxt );
        } 
        else if( _d.drawing() )
        {
          //dot for this unit text
          _d.gc_.drawLine( (int)p0.getX(),(int)p0.getY(), (int)p0.getX()+dxd, (int)p0.getY()-dyd);
          DrawUtilIGC.drawText( _d.gc_, rtxt, IDAlignment.CENTER, txt );
        }
        else //locating
        {
          if( rtxt.contains( _d.lx_, _d.ly_ ) )
          {
            throw new DefaultRenderChartLocation( 
                for_x_axis ? DLocated.InAxisXStepUnitText : DLocated.InAxisYStepUnitText, _axis, rtxt );
          }
        }
      }
      wtx += _d.axis_unit_dot_size+_d.axis_unit_to_dot_spacing;
    }

    /*
     * Axis title.
     */
    String axis_title = DefaultChartRender3D.axis3DName( _axis );
    if ( !DrawUtilIGC.isEmpty( axis_title ) )
    {
      Rect rtxt = new Rect();
      int fh = _d.gc_.getFontMetrics().getHeight();       
      int wt = _d.gc_.textExtent( axis_title ).getW();
      
      if( for_x_axis )  {
        _view.projection( xmin, y_axis, z_axis, p0 );
        _view.projection( xmax, y_axis, z_axis, p1 ); 
      } else {
        _view.projection( x_axis, ymin, z_axis, p0 );
        _view.projection( x_axis, ymax, z_axis, p1 );
      }
      int x = ((int)p0.getX()+(int)p1.getX())/2;
      int y = ((int)p0.getY()+(int)p1.getY())/2;
      int d;
      //impact on (wtx,fh) (unit texts) rectangle (units) on d
      if(have_step_unit )
      {
        int ia = Radian.iR2D( Radian.normalize(angle) );
        int wtx_fh_proj;
        if( ia == 0 || ia==180 ) {
          wtx_fh_proj = wtx;
        } else if ( ia==90 || ia==270 ) {
          wtx_fh_proj = fh;
        } else {
          //(0,0),(0,fh),(wtx,fh),(wtx,0) the for corner of text rectangle:
          //y = tan(angle)*x the line all corner must be projected on
          //take max/min projections (distance from 0), them you got the "projection" of
          //rectangle on the line.
          //or faster: take projection of Y size and X size on line which we know angle
          double d1 = Math.abs( fh*Math.cos(Radian._PI2-angle));
          double d2 = Math.abs( wtx*Math.cos( angle ));
          wtx_fh_proj = (int)( d1+d2 );
        }
        d = _d.axis_unit_dot_size+_d.axis_unit_to_dot_spacing + wtx_fh_proj  + _d.axis_title_to_unit_spacing;
      } else if ( have_step_dot ) {
        //no step unit
        d = _d.axis_dot_size + _d.axis_title_to_unit_spacing;
      } else {
        d = _d.axis_title_to_unit_spacing;
      }  
      int dxt = (int)Math.round( d*Math.cos( angle ));
      int dyt = (int)Math.round( d*Math.sin( angle ));
      x += dxt;
      y -= dyt;
      if( text_on_top ) y -= fh;
      if(!text_on_right) x -= wt; 
      rtxt.setRect( x, y, wt, fh );

      if( get_insets )
      {
        insets.max( r_view, rtxt );
      }
      else if( _d.drawing() )
      {
        _d.gc_.setBrush( new SolidBrush( _axis_style.getFore() ) );
        _d.gc_.setFont( _axis_style.getFont() );
        int aln =  IDAlignment.CENTER ;
        DrawUtilIGC.drawText( _d.gc_, rtxt, aln, axis_title );
      }
      else //locating
      {
        if( rtxt.contains( _d.lx_, _d.ly_ ))
        {    
          throw new DefaultRenderChartLocation( for_x_axis ? DLocated.InAxisXTitle : DLocated.InAxisYTitle, _axis, rtxt ); 
        }
      }      
    }
  }
}
