/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultChartRenderData.java,v 1.10 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;

import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;
import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DCategory;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DI18N;
import org.eclipse.tptp.platform.report.core.internal.DMarkerLine;
import org.eclipse.tptp.platform.report.core.internal.DPropertyStore;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.core.internal.IDColor;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.drawutil.internal.DrawUtilIGC;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.igc.internal.IFont;
import org.eclipse.tptp.platform.report.igc.internal.IGC;
import org.eclipse.tptp.platform.report.igc.internal.IGCDirect;
import org.eclipse.tptp.platform.report.igc.internal.IPen;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.util.internal.Font;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.tools.internal.DAlignment;
import org.eclipse.tptp.platform.report.tools.internal.VDouble;

import com.ibm.icu.text.UFormat;
import com.ibm.icu.util.ULocale;
/**
 * Contains metrics and (dynamic) data to render the chart part of a DGraphic.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DefaultChartRenderData
{
    public final int axis_title_to_unit_spacing;
    public final int axis_unit_to_dot_spacing ;
    public final int axis_dot_size;
    public final int axis_unit_dot_size;
    public final int max_axis_dot_size ;

    /** @return true if render is in drawing mode*/ 
    public boolean drawing() { return draw_; }
    /** @return true if render is in locating mode */
    public boolean locating() { return !draw_; }

    /** margin for render */
    public int margin_; 
    
    public boolean draw_;
    public IGC gc_; //mode drawing
    public IGCDirect gd_; //stricly gc_.getIGCDirect();
    public float   scale_;
    public int lx_, ly_; //mode locating
    
    public DefaultChartRenderData( boolean _draw,  IGC _gc, float _scale )
    {
      gc_=_gc;
      gd_=_gc.getIGCDirect();
      scale_=_scale;
      draw_=_draw;
      lx_=ly_=-1;
      margin_ = Math.min( dpiX(10), dpiY(10) ); //default margin
      
      axis_title_to_unit_spacing = dpiX(5);
      axis_unit_to_dot_spacing   = dpiX(2);
      axis_dot_size              = dpiX(2);
      axis_unit_dot_size         = dpiX(4);
      max_axis_dot_size = Math.max( axis_dot_size, axis_unit_dot_size );
    }        
    
    public void setupMargin( DGraphic _graphic )
    {
      margin_ = _graphic.getProperties().get( DGraphic.P_MARGIN, 10 ); //default margin
      margin_ = Math.min( dpiX(margin_), dpiY(margin_) );
    }
   
    /** print error occured while rendering chart. */
    public void error( String _message, RenderPersistData rpd )
    {
      if( draw_ )
      {
        IFont font = gc_.getFont();
        Font nf = new Font( font );
        nf.setFontStyle( font.getFontStyle() | IFont.BOLD );
      
        gc_.setFont( nf );
        gc_.setBrush( new SolidBrush( RGBA.RED) ); 
        gc_.drawText( _message, rpd.graphic_rect.getX()+5, rpd.graphic_rect.getY()+5+12 );
        gc_.setFont( font );
      }
    }
    
    public int dpiX( int x )
    {
      return x;
      //NO!since IGC works only with pixel: return gd_.devX( x );
      //TODO/remove:return (int)( scale_*DrawUtil.convertDPIX( x, display_, device_ ));
    }
    public int dpiY( int y )
    {
      return y;
      //NO!since IGC works only with pixel:return gd_.devY( y );
      //TODO/remove:return (int)( scale_*DrawUtil.convertDPIY( y, display_, device_ ));
    }
 
    /** 
     * @return line RGBA color from P_LINE_COLOR property or lightGray by default.
     */
    protected int getLineColor( DAxis axis )
    {
      IDColor c = (IDColor)axis.getProperties().get( DAxis.P_LINE_COLOR );
      if( c!=null )
      {
        return IGCDStyle.GetRGBA( c );
      }
      return RGBA.LIGHT_GRAY;
    }
  
    /** 
     * Draw or locate for an horizontal/vertical text.
     * @param _text text to render.
     * @param _rect rectangle to draw text
     * @param _align LEFT/RIGHT/HCENTER/... flag
     * @param _style style to paint
     * @param _locid identifier for location
     * @param _item item for location
     */
    public void renderText( String _text, Rect _rect, int _align, IGCDStyle _style,
                            String _locid, IDItem _item )
      throws DefaultRenderChartLocation                           
    {
      if( draw_ )
      {
        //value is drawn using curve style
        gc_.setBrush( new SolidBrush(_style.getFore()) );
        gc_.setFont( _style.getFont() );
        DrawUtilIGC.drawText( gc_, _rect, _align, _text );
      }
      else //locating
      {
        Rect nrtxt = DrawUtilIGC.boundingRect( gc_, _rect, _align, _text );
        //nrtxt might be bigger than _rect, clip it
        nrtxt.intersection( _rect );

        if( (nrtxt.w()>0) && nrtxt.contains( lx_, ly_ ) )
        {
          throw new DefaultRenderChartLocation( _locid, _item, nrtxt, _text );
        }
      }
    }
    
    private static DI18N _i18n;
    private static ULocale _locale;
    private static NLString _nls;
    
    /**
     * @return a text contained in a resourcebundle file
     */
    public static String getResourceString( String tag, DI18N i18n) {
        if (i18n!=null) {
          if (i18n != _i18n) {
              _locale = new ULocale(i18n.getLanguage(), i18n.getCountry());
              if (i18n.getResourceBundle() != null) {
                  _nls = new NLString(i18n.getResourceBundle(), _locale);
              } else {
                  _nls = new NLString();
              }
          }
    
          return _nls.getString(tag);
        }
        else {
            if (_nls==null)
               _nls = new NLString();
            return _nls.getString(tag);
        }
    }
    
    /**
     * @return the title to display for axis "title (unit)" or "(unit)" or null. 
     */
    public static String getAxisTitle( DAxis _axis )
    {
      if( _axis==null ) return null;
      
      String text = _axis.getTitle();
      String unit = _axis.getUnit();
      if (_axis.getParent() instanceof DGraphic) {
          DGraphic graph = (DGraphic)_axis.getParent();
          DI18N child = (DI18N)graph.getChildOfClass(DI18N.class);
          text = getResourceString(_axis.getTitle(), child);
          unit = getResourceString(_axis.getUnit(), child);
      }
      
      if( !DrawUtilIGC.isEmpty(unit))
      {
        if( text==null) text = "("+unit+")"; else text += " ("+unit+")";
      }
      return text;
    }
    
    /**
     * Compute and return amounts (border) required to render an <b>horinzontal</b> axis at the
     * bottom of rectangle _R. Axis font must be set in current gc_.
     * A configured pixel scale is requiered to get min/max values and units text locations.
     * Amounts get bounds from untruncated texts.
     */
    public Insets computeXAxisAmounts( DAxis _axis, IScale _scale, Rect _R, int _title_alignment, Insets _insets )
    {
      if( _insets==null ) _insets = new Insets();
      //top: no amount
      //bottom: the height of axis title, axis units, axis dots and spacings...
      String title = getAxisTitle( _axis );
      boolean have_title = !DrawUtilIGC.isEmpty(title);
      Object step_unit  = _axis.getProperties().get( DAxis.P_STEP_UNIT );
      
      if( have_title && _axis.getProperties().get( DAxis.P_SHOW_TITLE, have_title ) )
      {
        _insets.addB( DrawUtilIGC.textExtent( gc_, title, _title_alignment ).getH()
                      +  axis_title_to_unit_spacing );
      }
      
      IDAlignment align;
      if (_axis.getProperties().hasKey(DAxis.P_LABEL_ALIGNMENT))
        align = (IDAlignment)_axis.getProperties().get(DAxis.P_LABEL_ALIGNMENT);
      else
        align = new DAlignment(IDAlignment.CENTER);
      boolean rotate90 = align.haveOneOfAlignment(IDAlignment.ROTCW90|IDAlignment.ROTCCW90);
      
      int vmin_text_width=0;
      int value_text_max_width=0;
      if( step_unit != null )
      {
        //as we don't know which text the scale want to draw, iterate on all text, to get
        //maxmimum width of text.
        int max_text_h = 0;
            boolean first=true;
            for( Object vs=_scale.stepFirst(step_unit); vs!=null; vs=_scale.stepNext(step_unit,vs) )
            {
              String txt = _scale.valueText( vs );
              if( txt==null ) continue;
              ISize size = gc_.textExtent( txt ); 
              int txt_width = rotate90 ? size.getH() : size.getW(); 
              if( first ) 
              {
                vmin_text_width = txt_width;
                first=false; 
              }            
              if( txt_width > value_text_max_width) value_text_max_width=txt_width;
              int sh = rotate90 ? size.getW():size.getH();
              if( sh > max_text_h) max_text_h=sh;           
            }
        
        _insets.addB( gc_.getFontMetrics().getHeight() 
                      +  axis_unit_to_dot_spacing
                      +  max_axis_dot_size );
      }
      //axis_dot_size already taking into account.
      else if( _axis.getProperties().get( DAxis.P_STEP_DOT, -1 ) > 0.0 )
      {
        _insets.addB( axis_dot_size );
      }
      //left: portion of units text might be outside rectangle.
      if( step_unit != null )
      {
        _insets.addL( vmin_text_width/2 );   
      }
      //right: portion of units text might be outside rectangle
      if( step_unit != null )
      {
       _insets.addR( value_text_max_width/2 );   
      }
      return _insets;
    }

    /**
     * Compute and return amounts (border) required to render an <b>vertical/horizontal</b> axis on the
     * <b>left</b> or <b>right</b> side of rectangle _R. Axis font must be set in current gc_.
     * A configured pixel scale is requiered to get min/max values and units text locations.
     * Amounts get bounds from untruncated texts.
     */
    public Insets computeYAxisAmounts( boolean vertical, boolean _left, DAxis _axis, IScale _scale, Rect _R, int _title_alignment, Insets _insets )
    {
      if( _insets==null ) _insets = new Insets();
      //prepare (see below)
      Object step_unit  = _axis.getProperties().get( DAxis.P_STEP_UNIT );
      //int vmin_text_width=0;
      //int vmax_text_width=0;
      
      IDAlignment align;
      if (_axis.getProperties().hasKey(DAxis.P_LABEL_ALIGNMENT))
        align = (IDAlignment)_axis.getProperties().get(DAxis.P_LABEL_ALIGNMENT);
      else
        align = new DAlignment(IDAlignment.CENTER);
      boolean rotate90 = align.haveOneOfAlignment(IDAlignment.ROTCW90|IDAlignment.ROTCCW90);
      
      int value_text_max_width=0;
      int max_text_h = 0;
      if( step_unit != null )
      {
          boolean first=true;
          for( Object vs=_scale.stepFirst(step_unit); vs!=null; vs=_scale.stepNext(step_unit,vs) )
          {
            String txt = _scale.valueText( vs );
            if( txt==null ) continue;
            ISize size = gc_.textExtent( txt ); 
            int txt_width;
            if (vertical)
                txt_width = rotate90 ? size.getH() : size.getW();
            else
                txt_width = rotate90 ? size.getW() : size.getH(); 
            if( first ) 
            {
              //vmin_text_width = txt_width;
              first=false; 
            }            
            //vmax_text_width = txt_width; //keep last for vmax
            if( txt_width > value_text_max_width) value_text_max_width=txt_width;
            int sh;
            if (vertical)
                sh = rotate90 ? size.getW() : size.getH(); 
            else
                sh = rotate90 ? size.getH() : size.getW(); 
            if( sh > max_text_h) max_text_h=sh;           
          }
      }

      //right: no amount
      //left: the width of axis title, axis units, axis dots and spacings...
      String title = getAxisTitle( _axis );
      boolean have_title = !DrawUtilIGC.isEmpty(title);
      if( have_title && _axis.getProperties().get( DAxis.P_SHOW_TITLE, have_title ) )
      {
          if (vertical)
              _insets.addL( DrawUtilIGC.textExtent( gc_, title, _title_alignment ).getW()
                      +  axis_title_to_unit_spacing );
          
          else
              _insets.addT( DrawUtilIGC.textExtent( gc_, title, _title_alignment ).getH()
                      +  axis_title_to_unit_spacing );

      }      
      if( step_unit != null  )
      {
          if (vertical)
              _insets.addL( value_text_max_width
                      +  axis_unit_to_dot_spacing
                      +  max_axis_dot_size );
          else
              _insets.addT( value_text_max_width
                      +  axis_unit_to_dot_spacing
                      +  max_axis_dot_size );

      }
      else if( _axis.getProperties().get( DAxis.P_STEP_DOT, -1 ) > 0.0 )
      {
          if (vertical)
              _insets.addL( axis_dot_size );          
          else
              _insets.addT( axis_dot_size );

      }
      //bottom: portion of units text might be outside rectangle (worth case)
      if (vertical)
          _insets.addB( max_text_h/2 );
      else
          _insets.addL( max_text_h/2 );

      //top: portion of units text might be outside rectangle (woth case again)
      if (vertical)
          _insets.addT( max_text_h/2 );
      else          
          _insets.addR( max_text_h/2 );
      
      if( !_left && vertical)
      {
        _insets.setR( _insets.getL() );
        _insets.setL( 0 );
      }

      return _insets;
    }

    
    /**
     * Render an X axis (bottom): DOT/UNIT/LINES and titles.
     * Rectangle, scale and amounts must be computed/updated before calling this method.
     */
    public void renderXAxis( boolean _for_draw, boolean vertical, DAxis _axis, IScale _scale, Rect _R, Insets _insets, IGCDStyle _style, int _title_alignment )
      throws DefaultRenderChartLocation
    {
      //render axis:
      gc_.setFont( _style.getFont() );
      gc_.setBrush( new SolidBrush(_style.getFore()) );
      if( _for_draw )
      {
        if( draw_ )
        {
          onXAxis( vertical, _R, _axis, _scale, DAxis.P_STEP_LINE, _style );
          onXAxis( vertical,_R, _axis, _scale, DAxis.P_STEP_UNIT, _style );
          onXAxis( vertical,_R, _axis, _scale, DAxis.P_STEP_DOT,  _style );
          renderXAxisTitle( vertical, _axis, _R, _insets, _style, _title_alignment );
        }
      } else { //for a locating mode.
        if( locating() )
        {
          //do it in reverse order...
          renderXAxisTitle( vertical,_axis, _R, _insets, _style, _title_alignment );
          onXAxis( vertical,_R, _axis, _scale, DAxis.P_STEP_UNIT, _style );
          onXAxis( vertical,_R, _axis, _scale, DAxis.P_STEP_LINE, _style );
          onXAxis( vertical, _R, _axis, _scale, DAxis.P_STEP_DOT,  _style );
          //locate the (base) line of axis
          if (vertical) {
              int xl = _R.left();
              renderLine( xl, _R.top(), xl, _R.bottom(), DLocated.InAxisXLine, _axis );
          }
          else
             renderLine( _R.left(), _R.bottom(), _R.right(), _R.bottom(), DLocated.InAxisXLine, _axis );
        }
      }
    }

    /**
     * Render an Y axis (left): DOT/UNIT/LINES and titles.
     * Rectangle, scale and amounts must be computed/updated before calling this method.
     */
    public void renderYAxis( boolean vertical, boolean _for_draw, boolean _left, DAxis _axis, IScale _scale, Rect _R, Insets _insets, IGCDStyle _style, int _title_alignment, int _3d_line_x, int _3D_line_y )
      throws DefaultRenderChartLocation
    {
      //render axis:
      gc_.setFont( _style.getFont() );
      gc_.setBrush( new SolidBrush(_style.getFore()) );
      if( _for_draw )
      {
        if( drawing() )
        {
          onYAxis( vertical, _left, _R, _axis, _scale, DAxis.P_STEP_LINE, _style, _3d_line_x, _3D_line_y );
          onYAxis( vertical, _left, _R, _axis, _scale, DAxis.P_STEP_UNIT, _style, _3d_line_x, _3D_line_y );
          onYAxis( vertical, _left, _R, _axis, _scale, DAxis.P_STEP_DOT,  _style, _3d_line_x, _3D_line_y );
          renderYAxisTitle( vertical, _left, _axis, _R, _insets, _style, _title_alignment );
        }
      } 
      else //for a locate
      {
        //do it in reverse order.
        if( locating() )
        {
          renderYAxisTitle( vertical, _left, _axis, _R, _insets, _style, _title_alignment );
          onYAxis( vertical, _left, _R, _axis, _scale, DAxis.P_STEP_DOT,  _style, _3d_line_x, _3D_line_y );
          onYAxis( vertical, _left, _R, _axis, _scale, DAxis.P_STEP_UNIT, _style, _3d_line_x, _3D_line_y );
          onYAxis( vertical, _left, _R, _axis, _scale, DAxis.P_STEP_LINE, _style, _3d_line_x, _3D_line_y );
          //locate for the line of axis:
          if (vertical) {
            int xl = _left ? _R.left() : _R.right();
            renderLine( xl, _R.top(), xl, _R.bottom(), DLocated.InAxisYLine, _axis );
          }
          else {
            renderLine( _R.left(), _R.bottom(), _R.right(), _R.bottom(), DLocated.InAxisYLine, _axis );
            
          }
        }
      }
    }

    /**
     * Render X axis title for bottom axis only.
     */
    protected void renderXAxisTitle( boolean vertical, DAxis _axis, Rect _R, Insets _insets, IGCDStyle _style, int _title_alignment  )
       throws DefaultRenderChartLocation
    {
      if ( !_axis.getProperties().get( DAxis.P_SHOW_TITLE, true ) ) return ;
      
      String title = getAxisTitle( _axis );
      if( DrawUtilIGC.isEmpty(title) ) return ;

      gc_.setFont( _style.getFont() );
      
      if (vertical) {
          ISize size = DrawUtilIGC.textExtent( gc_, title, _title_alignment );

          Rect r_ttl = new Rect();
          int sw = size.getW();
          r_ttl.setRect( _R.x()-_insets.getL(), _R.top(), sw, _R.h() );
          
          String txt = DrawUtilIGC.truncateText( gc_, title, r_ttl.w(), r_ttl.h(), _title_alignment );
          renderText( txt, r_ttl, _title_alignment, _style, DLocated.InAxisXTitle, _axis );  
      }
      else {   
          int th = DrawUtilIGC.textExtent( gc_, title, _title_alignment ).getH();
          Rect r_ttl = new Rect( _R.x(), _R.bottom()+_insets.getB()-th, _R.w(), th );
          String txt = DrawUtilIGC.truncateText( gc_, title, r_ttl.w(), r_ttl.h(), _title_alignment );
          renderText( txt, r_ttl, _title_alignment, _style, DLocated.InAxisXTitle, _axis );
      }
    }
    
    /**
     * Render Y axis title for <b>right</b> axis only.
     */
    protected void renderYAxisTitle( boolean vertical, boolean _left, DAxis _axis, Rect _R, Insets _insets,  IGCDStyle _style, int _title_alignment  )
       throws DefaultRenderChartLocation
    {
      if ( !_axis.getProperties().get( DAxis.P_SHOW_TITLE, true ) ) return ;
      
      String title = getAxisTitle( _axis );
      if( DrawUtilIGC.isEmpty(title) ) return ;
    
      gc_.setFont( _style.getFont() );
      gc_.setBrush( new SolidBrush(_style.getFore()) );
      
      if (vertical) {
          ISize size = DrawUtilIGC.textExtent( gc_, title, _title_alignment );
          
          Rect r_ttl = new Rect();
          int sw = size.getW();
          if( _left )
          {
              r_ttl.setRect( _R.x()-_insets.getL(), _R.top(), sw, _R.h() );
          } else {//right
              r_ttl.setRect( _R.right()+_insets.getR()-sw, _R.top(), sw, _R.h() );
          }
          String txt = DrawUtilIGC.truncateText( gc_, title, r_ttl.w(), r_ttl.h(), _title_alignment );
          renderText( txt, r_ttl, _title_alignment, _style, DLocated.InAxisYTitle, _axis );
      }
      else {
          int th = DrawUtilIGC.textExtent( gc_, title, _title_alignment ).getH();
          Rect r_ttl = new Rect( _R.x(), _R.top()-_insets.getT()-th, _R.w(), th );
          String txt = DrawUtilIGC.truncateText( gc_, title, r_ttl.w(), r_ttl.h(), _title_alignment );
          renderText( txt, r_ttl, _title_alignment, _style, DLocated.InAxisYTitle, _axis );
      }
    }
        
    /**
     * @return a scale for given DAxis, linear by default (if axis's type isn't known).
     * return a LinearDateScale for Date objects.
     */
    public static IScale getScale( DAxis _axis, double scale_min, double scale_max, Object vmin, Object vmax )
    {
      if( _axis==null ) return null;
      
      // search locale
      ULocale locale=null;
      if (_axis.getParent() instanceof DGraphic) {
          DGraphic graph = (DGraphic)_axis.getParent();
          locale = graph.getLocale();
      }    
      
      DPropertyStore prop = _axis.getProperties();
      IScale scale=null;
      Class data_class = (Class)prop.get(DAxis.P_DATA_CLASS);
     
      IScaleProvider sp = (IScaleProvider)prop.get( DAxis.P_DATA_SCALE_PROVIDER );
      if( sp!=null )
      {
        scale = sp.newScaleFor( data_class==null ? Double.class : data_class );
        scale.setScaleRange( scale_min, scale_max );
        scale.setValueRange( vmin, vmax );
      }
      if( scale==null )
      {
        if( data_class == Date.class )
        {
          scale = new LinearDateScale( scale_min, scale_max, vmin, vmax );
        }      
        else
        {
            List categories = _axis.getChildrenOfClass(DCategory.class);
            if (categories.size()>0)
            {
                scale = new CategoryScale(categories);
            }
            else if( _axis.useNumbers() )
            {
                if( _axis.isScaleType( DAxis.S_LOG ) ) {
                    scale = new LogNumberScale( scale_min, scale_max, vmin, vmax );
                } else { // default: LIN           
                    scale = new LinearNumberScale( scale_min, scale_max, vmin, vmax );
                }
            }
        }
      }
      
      
      
      //to avoid null pointer exception, create an invalid axis.
      if( scale==null ) 
      {
        scale = new LinearNumberScale(0,0,0,0);
      }
      
      scale.setLocale(locale);
      
      try{
        UFormat format =  (UFormat)prop.get( DAxis.P_UNIT_FORMAT );
        scale.setTextFormat( format );
      }
      catch( ClassCastException e ) {}
      return  scale;
    }
    
   /**
    * @return an object suitable to get/set stored values in this axis.
    * Simple axis return a VDouble(), if property P_DATA_CLASS is set, try
    * to create an instance of this class. For others case ... return null.
    * Ok, it's not important to understand this, only chart renders have to..
    */
   public static Object getNewInstanceValue( DAxis _axis )   
   {
     if( _axis==null ) return null;
     DPropertyStore prop = _axis.getProperties();
     if( prop==null ) return new VDouble();
     Class c = (Class)prop.get(DAxis.P_DATA_CLASS);
     if( c==null || c==Double.class) return new VDouble();
     try{      
       Object i = c.newInstance();
       return i;
     }
     catch(  Exception e )
     {
e.printStackTrace( System.err);
       return null;
     }
   }


    /** 
     * Render property on X axis, no support for "3D-like".
     * @param _R rectangle to render axis.
     * @param _axis axis to render.
     * @param _property property identifier to render: STEP_LINE,STEP_UNIT,STEP_DOT.
     */
    protected void onXAxis( boolean vertical, Rect _R, DAxis _axis, IScale _scale, String _property, IGCDStyle _style  )
     throws DefaultRenderChartLocation
    {
      if( !_scale.isValid() ) return ;
      if( !_axis.getProperties().hasKey( _property ) ) return ;
      Object step = _axis.getProperties().get( _property );
      if( step== null ) return ;
      
      IDAlignment align;
      if (_axis.getProperties().hasKey(DAxis.P_LABEL_ALIGNMENT))
        align = (IDAlignment)_axis.getProperties().get(DAxis.P_LABEL_ALIGNMENT);
      else
        align = new DAlignment(IDAlignment.CENTER);

      int yd = (vertical) ? _R.left() : _R.bottom();

      IPen save_pen = null;
    try
    {
      //do not compare strings every times:
      int v_prop = 0;
      if( DAxis.P_STEP_UNIT.equals( _property ) )
      {
        v_prop=1;
      }
      else if ( DAxis.P_STEP_DOT.equals( _property ) )
      {
        v_prop =2;
      }
      else if ( DAxis.P_STEP_LINE.equals( _property ) )
      {
        v_prop =3;
        if( draw_ )
        {
          //lines can have color from a property.
          save_pen = gc_.getPen();
          gc_.setPen( new LineStylePen(getLineColor( _axis )) );
        }
      }
      else
      {
        //unknown property...
        return ;
      }
 
      Rect r_txt=new Rect();
      int x_free_for_text =-1; // do not draw text over other text..
      final int h_text_spacing = dpiX(5);
      boolean reversed = _scale.isReversed();
      
      for( Object vs = _scale.stepFirst(step); vs!=null;  vs=_scale.stepNext(step,vs) )
      {
        int xd=(int)_scale.toScale( vs );      
  
        switch( v_prop )
        {
          case 1: { //P_STEP_UNIT           
            String t= _scale.valueText( vs ); 
            if( t==null ) continue;
            
            // alignment
            ISize size = gc_.textExtent( t );
            int size_w, size_h;
            if (align.haveOneOfAlignment(IDAlignment.ROTCW90|IDAlignment.ROTCCW90))
            {
                size_w = size.getH(); size_h = size.getW();  
            }
            else
            {
                size_w = size.getW(); size_h = size.getH(); 
            }
            
            int wt = size_w;
            int x_text = vertical ? xd + size_h/2 : xd-size_w/2;
            
            if( !reversed && (x_text        >= x_free_for_text) 
                        ||  reversed && (x_text+size_w <= x_free_for_text)
                        || x_free_for_text==-1)
            {
              if (vertical)
                  r_txt.setRect( yd-max_axis_dot_size-axis_unit_to_dot_spacing-wt, x_text, size_w, size_h );
              else
                  r_txt.setRect( x_text, yd+max_axis_dot_size+axis_unit_to_dot_spacing, size_w, size_h );
              if(draw_)
              {
                //a "dot" line specialy for unit
                DrawUtilIGC.drawText( gc_, r_txt, align.getAlignment(), t );           
              }
              else //locating
              {
                if ( r_txt.contains(lx_, ly_ ) )
                {
                  throw new DefaultRenderChartLocation( DLocated.InAxisXStepUnitText, _axis, r_txt, t );
                }
              }
              //"dot" corresponding to that unit's text:
              if (vertical)
                  renderLine( yd, xd, yd+axis_unit_dot_size, xd, DLocated.InAxisXStepUnitTextDot, _axis );
              else
                  renderLine( xd, yd, xd, yd+axis_unit_dot_size, DLocated.InAxisXStepUnitTextDot, _axis );
              
              x_free_for_text = x_text + (reversed ? -h_text_spacing : ( size_w + h_text_spacing ) );
            }
            break;
          }
          
          case 2: { //P_STEP_DOT
              if (vertical)
                  renderLine( yd, xd, yd+axis_unit_dot_size, xd, DLocated.InAxisXStepUnitTextDot, _axis );
              else
                  renderLine( xd, yd, xd, yd+axis_dot_size, DLocated.InAxisXStepDot, _axis );
              break;
          }
          
          case 3: { //P_STEP_LINE
              if (vertical)
                  renderLine( (int)yd, _R.top()-xd, _R.right(), _R.top()-xd, DLocated.InAxisXStepLine, _axis );
              else
                  renderLine( (int)xd, _R.top(), (int)xd, _R.bottom(), DLocated.InAxisXStepLine, _axis );
              break;
          }
        }
      }//for
        
    }
    finally
    {
      if( save_pen !=null ) gc_.setPen( save_pen ); 
    }
    }
  
    private void renderLine( int x1, int y1, int x2, int y2, String loc_id, IDItem loc_item)
          throws DefaultRenderChartLocation  
    {
      if( draw_ )
      {
        gc_.drawLine( x1, y1, x2, y2 );
      } else { /*locating */ 
        if( DrawUtilIGC.segmentContains( x1,y1,x2,y2, lx_,ly_) )
        {
          Rect r = new Rect( x1,y1, x2-x1, y2-y1 );
          r.normalize();
          throw new DefaultRenderChartLocation( loc_id, loc_item, r );
        }
      }
    }
    /**
     * Render property on Y axis (left/or right side of graph), support for "3D-like" only for left Y axis.
     * 
     * @param _left true to render property for a left axis
     * @param _get_step_unit_width true to compute step unit width on axis, 
     *                             false to locate/draw.
     * @param _R rectangle to render axis.
     * @param _axis axis to render.
     * @param _property property identifier to render: STEP_LINE,STEP_UNIT,STEP_DOT.
     * @param _3d_line_x simulated 3d on x axis
     * @param _3d_line_y simulated 3d on y axis
     */
    protected void onYAxis( boolean vertical, boolean _left, Rect _R, DAxis _axis, IScale _scale, String _property, IGCDStyle _style,
                         int _3d_line_x, int _3d_line_y )
      throws DefaultRenderChartLocation  
    {
      if( !_scale.isValid() ) return ;
      if( !_axis.getProperties().hasKey( _property ) ) return ;

      Object step = _axis.getProperties().get( _property );
      if( step == null ) return ;
     
      IDAlignment align;
      if (_axis.getProperties().hasKey(DAxis.P_LABEL_ALIGNMENT))
        align = (IDAlignment)_axis.getProperties().get(DAxis.P_LABEL_ALIGNMENT);
      else
        align = new DAlignment(IDAlignment.CENTER);
      
      int xd;
      if (vertical)
         xd = _left ? _R.left() : _R.right();
      else
         xd = _R.top();

      IPen  save_pen = null;
    try
    {
      //do not compare string every times.
      int v_prop=0;
      if( DAxis.P_STEP_UNIT.equals( _property ) )
      {
        v_prop = 1;
      }
      else if ( DAxis.P_STEP_DOT.equals( _property ) )
      {
        v_prop = 2; 
      }
      else if ( DAxis.P_STEP_LINE.equals( _property ) )
      {
        v_prop = 3;
        //lines can have color from property
        if( draw_ )
        {
          save_pen = gc_.getPen();
          gc_.setPen( new LineStylePen(getLineColor( _axis )) );
        }
      }
      else return ; //missing code for new property ?

      int y_free_for_text = -1;//_scale.getPixelMin();
      final int v_text_spacing = dpiY(0);
      Rect r_txt = v_prop==1 ? new Rect() : null;
      
      for( Object vs = _scale.stepFirst(step); vs !=null; vs = _scale.stepNext(step,vs) )
      {        
        int yd=(int)_scale.toScale( vs );      
        
        switch( v_prop )
        {
          case 1: { //P_STEP_UNIT
              String t=_scale.valueText( vs );
              if( t==null ) continue;
              
              // alignment
              ISize size = gc_.textExtent( t );
              int size_w, size_h;
              if (align.haveOneOfAlignment(IDAlignment.ROTCW90|IDAlignment.ROTCCW90))
              {
                  size_w = size.getH(); size_h = size.getW();  
              }
              else
              {
                  size_w = size.getW(); size_h = size.getH(); 
              }
          
              int wt = size_w;
            //  if( wt>max_step_text_width) max_step_text_width=wt;
              int y_text = (vertical) ? yd-size_h/2 : yd - size_w/2;
//TODO: si l'echelle est dessinee haut->bas.
         //     if( y_text >= y_free_for_text )
              boolean text_bounds = (vertical) ? y_text+size_h <= y_free_for_text : y_text+size_w >= y_free_for_text;
              if( text_bounds || y_free_for_text==-1 )
              {
                if (vertical) {
                    if( _left )
                        r_txt.setRect( xd-max_axis_dot_size-axis_unit_to_dot_spacing-wt, y_text, wt, size_h );
                    else //right
                        r_txt.setRect( xd+max_axis_dot_size+axis_unit_to_dot_spacing, y_text, wt, size_h );
                }
                else {
                    r_txt.setRect( y_text, xd-axis_unit_dot_size-max_axis_dot_size-size_h, wt, size_h );
                }
                
                //draw/locate on "dot" near text
                if (vertical) {
                    if( _left )
                        renderLine( xd-axis_unit_dot_size, (int)yd, xd, (int)yd, DLocated.InAxisYStepUnitTextDot, _axis );
                    else  //right
                        renderLine( xd, (int)yd, xd+axis_unit_dot_size, (int)yd, DLocated.InAxisYStepUnitTextDot, _axis );
                }
                else {
                    renderLine( yd, xd-axis_unit_dot_size, yd, xd, DLocated.InAxisYStepUnitTextDot, _axis );
                }
                    
                if( draw_ )
                {
                   DrawUtilIGC.drawText( gc_, r_txt, align.getAlignment(), t );
                }
                else //locating
                {
                  if ( r_txt.contains( lx_, ly_ ) )
                  {                
                    throw new DefaultRenderChartLocation( DLocated.InAxisYStepUnitText, _axis, r_txt, t );
                  }
                }
//TODO: ok si echelle 'reversed'                y_free_for_text = y_text+ext.y+v_text_spacing;
                y_free_for_text = (vertical) ? y_text-v_text_spacing : y_text-v_text_spacing;
              }
              break;
           }
         case 2: {//P_STEP_DOT
             if (vertical) {
                 if( _left )
                     renderLine( xd-axis_dot_size, yd, xd, yd, DLocated.InAxisYStepDot, _axis );
                 else //right
                     renderLine( xd, yd, xd+axis_dot_size, yd, DLocated.InAxisYStepDot, _axis ); 
             }
             else {
                 renderLine( yd, xd-axis_dot_size, yd, xd, DLocated.InAxisYStepDot, _axis );
             }
             break;
         }
         
         case 3: {//P_STEP_LINE
              if (vertical) {
                if( _left && _3d_line_x!=0 && _3d_line_y!=0 )
                {
                  int y3d = yd-_3d_line_y;
                  int x3d = _R.left()+_3d_line_x;
                  renderLine( _R.left(), yd, x3d, y3d, DLocated.InAxisYStepLine, _axis );
                  renderLine( x3d, y3d, _R.right()+_3d_line_x, y3d , DLocated.InAxisYStepLine, _axis);
                } else {
                  renderLine( _R.left(), yd, _R.right(), yd, DLocated.InAxisYStepLine, _axis );
                }
              }
              else {
                  if (_3d_line_x!=0 && _3d_line_y!=0 ) {
                      int y3d = yd+_3d_line_x;
                      int x3d = _R.top()+_3d_line_y;
                      renderLine( yd, _R.top(), y3d, x3d, DLocated.InAxisYStepLine, _axis );
                      renderLine( y3d, x3d, y3d, _R.bottom()+_3d_line_y , DLocated.InAxisYStepLine, _axis);
                  }
                  else {
                      renderLine( yd, _R.top(), yd, _R.bottom(), DLocated.InAxisYStepLine, _axis );
                  }
              }
                  
              break;
           }
        }//switch
      }//for
      
    }
    finally
    {
      if( save_pen !=null ) gc_.setPen( save_pen );
    }
    }

    static boolean horizontal3DContains(  int _lx, int _ly, int _w, int _3d, int _px, int _py )
    {
      // return true if (_px,_py) is in this shape:
      //    _lx+_3d    _lx+_w+_3d
      //       +---------+ _ly-_3d
      //      /         /
      // _ly +---------+
      //    _lx     _lx+_w
      if( _py < (_ly-_3d) ) return false;
      if( _py >  _ly      ) return false;
      if( _py==_ly )
      {
        return (_px>=_lx)&&(_px<=(_lx+_w));
      }
      if( _py==(_ly-_3d))
      {
        return (_px>=(_lx+_3d))&&(_px<=(_lx+_w+_3d));
      }
      
      int dx = _ly-_py; //OK: 45
      return (_px>=(_lx+dx))&&(_px<=(_lx+dx+_w));
    }
    
    static boolean vertical3DContains( int _lx, int _ly, int _h, int _3d, int _px, int _py )
    {
      // return true if (_px,_py) is in this shape:
      //       _lx+_3d
      //          + _ly-_3d
      //         /|
      //    _ly + |
      //        | |
      //        | + _ly+_h-_3d
      //        |/
      // _ly+_h +
      //      _lx
      if( _px <  _lx     ) return false;
      if( _px > (_lx+_3d)) return false;
      if( _px == _lx      ) return (_py>=_ly)&&(_py<=(_ly+_h));
      int dy = _px-_lx; //OK: 45
      int npy = _py+dy;
     return (npy>=_ly)&&(npy<=(_ly+_h));
    }


  /**
   * Render (draw or locate) bar, rectangle with "3d" effect,.
   * @param _r main rectangle of bar.
   * @param _style used to draw bar
   * @param _item thrown for locating mode.
   * @param _3d_z, if not 0 draw 3d (right/top) area.
   */
  public void renderBar( boolean vertical, Rect r, int _back_rgba, IDItem _item, int _3d_z )
  //int _icrv, int _ipt, float _xbar, float _wbar, float _wbar_bs, int _y_bar_zero, boolean _3d, boolean _bs, boolean _im )
    throws DefaultRenderChartLocation
  {
      //main bar
      if( draw_ )
      {
        gc_.setPen( new LineStylePen(RGBA.BLACK) ); //black ://trait exterieur en noir
        gc_.setBrush( new SolidBrush(_back_rgba) );
        gc_.fillRect( r );
        gc_.drawRect( r );
      } 
      else //locating
      {
        if( r.contains( lx_, ly_ ) )
        {
          throw new DefaultRenderChartLocation( DLocated.CurveBar, _item,  r );
        }
      }

      if ( _3d_z!=0 )
      {
        int ar1[] = new int[8],
            ar2[] = new int[8];
              
        if (vertical) {
            ar1[0] = r.left();        ar1[1] = r.top();
            ar1[2] = r.left()+_3d_z;  ar1[3] = r.top()-_3d_z;
            ar1[4] = r.right()+_3d_z; ar1[5] = r.top()-_3d_z;
            ar1[6] = r.right();       ar1[7] = r.top();
            
            ar2[0] = r.right();       ar2[1] = r.top();
            ar2[2] = ar1[4];          ar2[3] = ar1[5]; 
            ar2[4] = r.right()+_3d_z; ar2[5] = r.bottom()-_3d_z;
            ar2[6] = r.right();       ar2[7] = r.bottom();
        }
        else {
            ar1[0] = r.right();        ar1[1] = r.top();
            ar1[2] = r.right()+_3d_z;  ar1[3] = r.top()+_3d_z;
            ar1[4] = r.right()+_3d_z; ar1[5] = r.bottom()+_3d_z;
            ar1[6] = r.right();       ar1[7] = r.bottom();
            
            ar2[0] = r.right();       ar2[1] = r.bottom();
            ar2[2] = ar1[4];          ar2[3] = ar1[5]; 
            ar2[4] =  r.left()+_3d_z; ar2[5] = r.bottom()+_3d_z;
            ar2[6] =r.left();       ar2[7] = r.bottom();
        }

        if( draw_ )
        {
          int dark1;
          int dark2;  
          if (vertical) {
            dark1 = RGBA.Darker( _back_rgba, 0.40f);
            dark2 = RGBA.Darker( _back_rgba, 0.30f);
          }
          else {
            dark1 = RGBA.Lighter( _back_rgba, 0.15f);
            dark2 = RGBA.Darker( _back_rgba, 0.40f);
          }
              
          SolidBrush b = new SolidBrush( dark1 );
          gc_.setBrush( b );
          Polygon poly = new Polygon( ar1 );
          gc_.fillPoly( poly );
          gc_.drawPoly( poly );
          
          b.setRGBA( dark2 );
          gc_.setBrush( b );
          poly.setPoints( ar2 );
          gc_.fillPoly( poly );
          gc_.drawPoly( poly );
        }
        else //locating
        {
          boolean in_ar1; 
          boolean in_ar2;
          if (vertical) {
             in_ar1 = DefaultChartRenderData.horizontal3DContains( r.x(), r.y(),
                                   r.w(), _3d_z, lx_,ly_ );
             in_ar2 = DefaultChartRenderData.vertical3DContains( r.right(), r.y(),
                                         r.h(), _3d_z, lx_,ly_ );
          }
          else {
             in_ar1 = DefaultChartRenderData.horizontal3DContains( r.left(), r.top()+r.h()+_3d_z,
                                       r.w(), _3d_z, lx_,ly_ );
             in_ar2 = DefaultChartRenderData.vertical3DContains( r.right(), r.top()+_3d_z,
                                        r.h(), _3d_z, lx_,ly_ );
          }
          if( in_ar1 )
          {
            Rect br ;
            if (vertical)
                br = new Rect( r.x(), r.y()-_3d_z, r.w()+_3d_z, _3d_z );
            else
                br = new Rect( r.left(), r.top()+r.h(),  r.w(), _3d_z);
            throw new DefaultRenderChartLocation( DLocated.CurveBar, _item, br );
          }
          else if( in_ar2 )
          {
            Rect br;
            if (vertical)
                br = new Rect( r.right(), r.y()-_3d_z, _3d_z, r.h()+_3d_z );
            else
                br = new Rect( r.right(), r.top(), r.h(), _3d_z);
            throw new DefaultRenderChartLocation( DLocated.CurveBar, _item, br );
          }
        }
      }
  }
  
  
  /**
   * Render text for a sector (PieChart and Histor,...).
   * 
   * @param _angle angle where texte is drawn (in radian)
   * @param _ray is distangle where text is drawn from center (must include margin if needed)
   * @param _rc rectangle where the sector are drawn
   * @param _txt text to render
   * @param _point item for location purpose
   * @param _style to render text
   */
  public void renderSectorText( double _angle, double _ray, Rect _rc, String _txt, IDItem _item, IGCDStyle _style, String _located )
    throws DefaultRenderChartLocation
  {
     double arad = Radian.normalize( _angle );

     int tx = (int)(_rc.x()+(_rc.w()>>1)+(_ray)*Math.cos(_angle));
     int ty = (int)(_rc.y()+(_rc.h()>>1)-(_ray)*Math.sin(_angle));

     gc_.setFont( _style.getFont() );
     ISize size = gc_.textExtent( _txt );
     int tw = size.getW();
     int th = size.getH();

     int deg = Radian.iR2D( arad );
     
     //try to place text in a less silly way...
          if ( deg ==  0  ) {             ty -= th/2; }
     else if ( deg <  90  ) {             ty -= th;  }
     else if ( deg == 90  ) { tx -= tw/2; ty -= th; }
     else if ( deg <  180 ) { tx -= tw;   ty -= th; }
     else if ( deg == 180 ) { tx -= tw;   ty-=th/2; }
     else if ( deg <  270 ) { tx -= tw;   }
     else if ( deg == 270 ) { tx -= tw/2; }
     else                   { /*raf*/ };

     Rect rtxt = new Rect( tx, ty, tw, th );

     renderText( _txt, rtxt,  IDAlignment.LEFT|IDAlignment.BOTTOM, _style, _located, _item );
  }
  
  /**
   * renders the marker line for an axis
   */
  public void renderMarkerLine(boolean axis_vertical, boolean for_draw, DAxis axis, Rect area, IScale scale) throws DefaultRenderChartLocation
  {
      List l = axis.getChildrenOfClass(DMarkerLine.class);
      Iterator it = l.iterator();
      while (it.hasNext())
      {
          DMarkerLine ml = (DMarkerLine)it.next();
          
          int y = (int)scale.toScale(ml.getValue());
          
          int x1=(axis_vertical) ? area.left() : y;
          int x2=(axis_vertical) ? area.right() : y;
          int y1=(axis_vertical) ? y : area.top();
          int y2=(axis_vertical) ? y : area.bottom();
          
          if (for_draw) {
             if( drawing() )
             {   
                IDColor c = ml.getColor();
                IPen oldpen;
                LineStylePen newPen;
                if (c!=null)
                    newPen = new LineStylePen(RGBA.Get(c.getRed(), c.getGreen(), c.getBlue()));
                else
                    newPen = new LineStylePen(RGBA.BLACK);
                
                if (ml.getThickness()>0)
                    newPen.setLineWidth((int)ml.getThickness());
                
                oldpen = gc_.setPen( newPen ); 
                gc_.drawLine( x1, y1, x2, y2 );
                gc_.setPen(oldpen);
            }
          }
          else
          {
              if (locating())
              {
                  Rect dr = new Rect(x1,y1-3, x2-x1, 6);
                  if (dr.contains(lx_,ly_))
                     throw new DefaultRenderChartLocation( DLocated.MarkerLine, ml, dr );
              }
          }
      }
  }
}
  

