/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultChartRenderPie3D.java,v 1.3 2008/05/23 14:11:50 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;

import java.util.ArrayList;
import java.util.Arrays;

import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.drawutil.internal.DrawUtilIGC;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.Point3D;
import org.eclipse.tptp.platform.report.drawutil.internal.Vector3D;
import org.eclipse.tptp.platform.report.igc.internal.ISize;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.tools.internal.IDisposable;


/**
 * Default Chart Render for T_PIE3D, "real" 3D for Pie chart,
 * and T_TORUS3D.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DefaultChartRenderPie3D
{
  
  /** variant part of pertistent data for H3D graphic */
  static class H3D implements IDisposable
  {
    DAxis axis;
    Box3D bbox;
    float shadow; //P_3D_SHADOW
    float light; //P_3D_LIGHT
    
    public void dispose()
    {
    }
  }
  
  private static boolean isValidPersistData( H3D p, DefaultChartRenderData d, RenderPersistData rpd )
  {
    if( p==null ) return false;
    return true;
  }

  /** 
   * Render curves
   * Rendering is made AsIs, no Z-Buffer, no hidden face remove algorythm, here is a simple
   * 3D drawing.
   */
  public static void render( DefaultChartRenderData d, RenderPersistData rpd, boolean pie3d )
     throws DefaultRenderChartLocation
  {
    H3D p = null;   
    if( rpd.g instanceof H3D ) p =(H3D)rpd.g;

    //rebuild data ?
    if( !isValidPersistData(p, d, rpd) )
    {
      p = rebuildPersistData( d, rpd );
    }
    
    //missed axis ?
    if ( p.axis==null )
    {
      if( d.drawing())
      {
        String txt = "Error: missing axis";
        d.error( txt, rpd );
      }
      return ;
    }
    
    String axis_unit = p.axis.getUnit();
    if( axis_unit==null ) axis_unit="";
    //just for text formating (..for the moment)
    IScale scale = DefaultChartRenderData.getScale( p.axis, 0,0,null,null);

    boolean show_values = rpd.graphic.getProperties().get( DGraphic.P_SHOW_VALUES, true )
               || rpd.graphic.getProperties().get(DGraphic.P_SHOW_PERCENTAGE, false);    
    p.shadow = rpd.graphic.getProperties().get( DGraphic.P_3D_SHADOW, DGraphic.DEF_3D_SHADOW );
    p.light = rpd.graphic.getProperties().get( DGraphic.P_3D_LIGHT, DGraphic.DEF_3D_LIGHT );

    //get point's values:
    double values[] = new double[20]; //p.npoints doesn't help as it doesn't contains sector points count.
    IDItem items[] = new IDItem[20];
    String labels[] = show_values ? new String[20] : null;
    double shifts[] = new double[20];
    double max_shift = 0.0;
    boolean have_shift = false;
    double sum=0;
    int nvalues=0;
    Object ov=null;
    for( int i=0; i < rpd.curve_styles.length; i++ )
    {
      double curr_shift = rpd.curve_styles[i].curve_.getProperties().get( DCurve.P_SHIFT_SECTORS, 0.0);
      curr_shift = Math.abs( curr_shift );
      have_shift |= (curr_shift > 0.0 );
      max_shift = Math.max( max_shift, curr_shift );
      
      for( IDItem it=rpd.curve_styles[i].curve_.getFirstChild(); it!=null ;it=it.getNext() )
      {
        if(!(it instanceof DPoint)) continue;
        DPoint point = (DPoint)it;
        IDCoord coord = null;
        DText   label = null;
        for( IDItem ic=it.getFirstChild(); ic!=null; ic=ic.getNext() )
        {
          if( ic instanceof IDCoord )
          {
            IDCoord c = (IDCoord)ic;
            if( c.getAxis()==p.axis )
            {
              coord = c;
            }
          }
          else if( ic instanceof DText )
          {
            label = (DText)ic;
          }
        }
        if( coord!=null )
        {
          ov = coord.getValue( ov );
          if( ov instanceof Number )
          {
            double v = Math.abs( ((Number)ov).doubleValue() );//sector require absolute values
            if( v==0.0 ) continue;
            if( nvalues>=values.length)
            {
              int s=nvalues+10;
              double av[] = new double[s];
              System.arraycopy( values, 0, av, 0, nvalues);
              values=av;
              IDItem ap[] = new IDItem[s];
              System.arraycopy( items, 0, ap, 0, nvalues);
              items=ap;
              if( show_values ) {
                String al[] = new String[s];
                System.arraycopy( labels, 0, al, 0, nvalues);
                labels=al;
              }
              av = new double[s];
              System.arraycopy( shifts, 0, av, 0, nvalues);
              shifts = av;
            }
            items [nvalues] = it;
            values[nvalues] = v;  
            shifts[nvalues] = curr_shift;
            sum += v;
            if( show_values ) {
              if( label!=null ) {
                labels[nvalues] = label.getText();
                items [nvalues] = label; //get style of DText and used for location id too.
                if( "".equals( labels[nvalues] )) labels[nvalues]=null;//simplify further works
              } else {
                //label is value plus axis unit.
                labels[nvalues] = scale.valueText( ov );
                if( "".equals( labels[nvalues] ) ) labels[nvalues]=null;//simplify further works
                else labels[nvalues] += axis_unit;
              }
            }
            nvalues++;
          }
        }
      }
    }
    
    if( nvalues == 0 ) return ; //nothing to render.
    
    if( nvalues == 1 ) { have_shift=false; } //no shift if there are one values
    
    {
      //property doesn't apply to T_TORUS3D
      double h = pie3d ? rpd.graphic.getProperties().get( DGraphic.P_3D_HEIGHT, DGraphic.DEF_3D_HEIGHT ) : 0.25;
      double Rx = have_shift ? 2*(max_shift+1)*0.5 : 1.0;
      p.bbox = new Box3D( 0, h, 0, Rx, 0, Rx );
    }
      
    //two angle to drive view
    double phi   = rpd.graphic.getProperties().get( DGraphic.P_XYZ_PHI,   0.3 );
    double theta = rpd.graphic.getProperties().get( DGraphic.P_XYZ_THETA, 0.2 );

    View3D view = new View3D( phi, theta, 0, 1.0f, p.bbox );
    //will be rescaled after insets computing
    view.scaleTo( rpd.ag_rect.x(), rpd.ag_rect.y(), rpd.ag_rect.w(), rpd.ag_rect.h() );
    
    double ntheta = Radian.normalize( view.getTheta() );
    boolean x_incr = (ntheta <= Radian._PI2 || ntheta >= Radian._3PI2 );
    boolean y_incr = (0<= ntheta && ntheta<=Radian._PI);
    double nphi = Radian.normalize( view.getPhi() );
    boolean z_incr = (0<= nphi && nphi <= Radian._PI );
    if( nphi > Radian._PI2 && nphi < Radian._3PI2 )
    {
      x_incr =!x_incr;
      y_incr =!y_incr;
    }

    LRect r_labels []= null;
    Insets insets = new Insets();
    //render labels, here as they don't cover 3D graphic.
    if( show_values )
    {
      r_labels = renderLabels( d,rpd,p, nvalues, values, labels, items, shifts,have_shift, view, sum, x_incr, insets );
    }

    //scale 3D view for rendering if gR rectange
    view.scaleTo( rpd.ag_rect.x()+insets.getL(), rpd.ag_rect.y()+insets.getT(),
                  rpd.ag_rect.w()-insets.getW(), rpd.ag_rect.h()-insets.getH() );

    Vector3D view_vector = view.getViewVector(null);
    view_vector.normalize();

    if( pie3d ) {      
      renderSector( nvalues, sum, values, items, labels, r_labels,shifts,have_shift, d,p, x_incr, view, view_vector );
    } else {
      renderTorus( nvalues, sum, values, items, labels, r_labels,shifts,have_shift, d,p, x_incr, view, view_vector );
    }
  }
  
  private static SolidBrush brush ;
  private static void renderPolygon(IDItem _item, DefaultChartRenderData d, H3D h3d, int rgba, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    if( d.drawing() ) 
    {
      double k = Math.abs(Vector3D.Scalar( v, view_vector ));
      if( brush==null ) brush = new SolidBrush();
      brush.setRGBA( RGBA.Shadow( rgba, h3d.shadow, h3d.light, (float)k ));//, (float)k)); //strength of shadow
      d.gc_.setBrush(brush);
      d.gc_.fillPoly( p );
    } else {
      if( p.contains( d.lx_, d.ly_ ) ) {
        String loc_id  =DLocated.CurveSector;
        if( _item instanceof DText) _item = _item.getParent();
        throw new DefaultRenderChartLocation( loc_id, _item, new Rect(d.lx_,d.ly_,0,0) );
      }   
    }
  }
  
  private static H3D rebuildPersistData( DefaultChartRenderData d, RenderPersistData rpd )
  {
      H3D p = new H3D();
      rpd.g = p;

      //retrieve the 3 required axis ...
      p.axis = null;
      //get the 3 axis ...
      for( IDItem item = rpd.graphic.getFirstChild(); item!=null; item=item.getNext() )
      {
        if( !(item instanceof DAxis) ) continue;
        p.axis = (DAxis)item; 
        break;
      }

      //axis missed, can't go further
      if( p.axis==null )
      {
        return p;
      }
      
      return  p;
  }

  private static class XSort implements Comparable
  {
    public int xpj;
    public int index;
    public XSort( int index, int xpj ) 
    {
      this.xpj=xpj;
      this.index=index;
    }
    public int compareTo(Object o) {
      int oxpj = ((XSort)o).xpj;
      if( oxpj==xpj ) return 0;
      if( xpj > oxpj ) return 1;
      return -1;
    }
  }
  
  /** rearrange labels rectangle on a line trying to center them on projected X points ... */
  private static void sortByXpj( ArrayList list, LRect r_labels[], String labels[], int xl,int xr, int margin )
  {
    if( list.size()>0 )  //1 also as I tried to center label on...
    {
      Object ar[] = list.toArray();
      Arrays.sort( ar );
      //get center of projected points and width of all label on line      
      int xjc=0;
      int labels_width=0;
      for( int j=0; j<ar.length; j++ )
      {
        XSort xs = (XSort)ar[j];
        xjc += xs.xpj;
        labels_width += r_labels[ xs.index ].getW()+margin;
      }
      xjc /= ar.length;
      labels_width -= margin;
      //try to X center labels at projected X center, but take care of xl,xr limits 
      int lx=xjc-labels_width/2;
      if( lx<xl) lx=xl; else if ( lx+labels_width > xr ) lx=xr-labels_width;
      //place labels from lx to ...
      for( int j=0; j<ar.length; j++ )
      {
        int i=((XSort)ar[j]).index;
        r_labels[i].moveTo( lx, r_labels[i].y() );        
        lx += r_labels[i].getW()+margin;
      }
    }
    list.clear();
  }
  
  /** render labels at top/bottom (using several lines if needed), try to sort labels by X ... */   
  private static LRect[] renderLabels( DefaultChartRenderData d, RenderPersistData rpd, H3D p, int nvalues, double []values, String labels[], IDItem items[], double []shifts, boolean have_shifts, View3D view, double sum, boolean x_incr, Insets insets )
     throws DefaultRenderChartLocation
  {
    LRect[]r_labels = new LRect[nvalues];
    insets.reset();
    
    for( int i=0; i<nvalues; i++ )
    {
      if( labels[i]==null ) continue ;
      d.gc_.setFont( IGCDStyle.GetFont( items[i], d.scale_ ));
      ISize size = d.gc_.textExtent( labels[i] );
      r_labels[i] = new LRect( 0,0, size.getW(), size.getH() );
    }
    
    //layout at top or bottom from
    int xl=rpd.ag_rect.x();
    int xr = rpd.ag_rect.right();
    int yt = rpd.ag_rect.top();
    int yb = rpd.ag_rect.bottom();
    int xtop=xl, ytop=yt, top_hmax=0;
    int xbtm=xl, ybtm=yb, btm_hmax=0;
    double a=0.0;
    double xc = x_incr ? p.bbox.getXMax() : p.bbox.getXMin() ;
    double yc = 0.5*p.bbox.getYMax();
    double zc = 0.5*p.bbox.getZMax();
    double R = 0.5;
    Point3D pj = new Point3D();
    view.projection( xc,yc,zc, pj );
    int ycenter = (int)pj.getY();
    ArrayList top_to_sort = new ArrayList();
    ArrayList btm_to_sort = new ArrayList();
    for( int i=0;i<nvalues; i++ )
    {
      double da = Radian._2PI*(values[i]/sum);
      double ma = a+da/2;
      a += da;
      if( r_labels[i]==null ) continue;
      
      double lyc=yc, lzc=zc;
      if( have_shifts && shifts[i] >0.0 )
      {
        double sht = R*shifts[i];
        lyc = yc+sht*Math.cos(ma);
        lzc = zc-sht*Math.sin(ma);
      }
      view.projection( xc,lyc+R*Math.cos(ma), lzc-R*Math.sin(ma), pj );
      boolean on_top = (int)pj.getY() <= ycenter;
      if( on_top )
      {
        if( xtop+r_labels[i].getW() > xr )
        {
          sortByXpj( top_to_sort, r_labels, labels, xl,xr, d.margin_ );
          insets.addT( top_hmax + d.margin_ );
          ytop += top_hmax + d.margin_;
          xtop = xl;
          top_hmax=0;
        }
        r_labels[i].moveTo( xtop, ytop );
        r_labels[i].align=IDAlignment.TOP;
        xtop += r_labels[i].getW()+d.margin_;
        top_hmax = Math.max( top_hmax, r_labels[i].getH() );
        top_to_sort.add( new XSort(i,(int)pj.getX()));
      }
      else //on bottom
      {
        if( xbtm+r_labels[i].getW() > xr )
        {
          sortByXpj( btm_to_sort, r_labels,labels, xl,xr, d.margin_ );
          insets.addB( btm_hmax + d.margin_ );
          ybtm -= btm_hmax + d.margin_;
          xbtm = xl;
          btm_hmax = 0;
        }
        r_labels[i].moveTo( xbtm, ybtm-r_labels[i].getH() );
        r_labels[i].align=IDAlignment.BOTTOM;
        xbtm += r_labels[i].getW()+d.margin_;
        btm_hmax = Math.max( btm_hmax, r_labels[i].getH() );
        btm_to_sort.add( new XSort(i,(int)pj.getX()));
      }
    }//for
    sortByXpj( top_to_sort, r_labels,labels, xl,xr, d.margin_ );
    sortByXpj( btm_to_sort, r_labels,labels, xl,xr, d.margin_ );
    if( btm_hmax > 0 ) insets.addB( btm_hmax+d.margin_ );
    if( top_hmax > 0 ) insets.addT( top_hmax+d.margin_ );
      
    for( int i=0; i<nvalues; i++ )
    {
      if( labels[i]==null) continue;
      
      IGCDStyle style = new IGCDStyle( items[i] );
      String txt = DrawUtilIGC.truncateTextH( d.gc_, labels[i], r_labels[i].getW() );
      String loc_id = items[i] instanceof DText? DLocated.LabelText : DLocated.CurvePointText;
      d.renderText( txt, r_labels[i], IDAlignment.CENTER, style, loc_id, items[i] );
    }
    return r_labels;
  }
  
  private abstract static class AFace implements Comparable
  {
    public double pj_zavg; //must be average of projected  point Z coordinate.
    public int item_index;
    public IDItem item; //item face is created for
    public double medium_angle;
    
    public int compareTo(Object o) 
    {
      AFace f =(AFace)o;
      if( pj_zavg==f.pj_zavg ) return 0;
      if( pj_zavg >f.pj_zavg ) return +1;
      return -1;
    }
  }
  //face using 3 or 4 points
  private static class Face extends AFace
  {
    public int ip0, ip1, ip2, ip3 ;
    public Point3D pj[];// might be null or not
    public Point3D pt[];// might be null or not
    
    public Face() {}
    public Face( int index, IDItem item, int p0, int p1, int p2, int p3, Point3D pt[], Point3D pj[] )
    {
      ip0=p0; ip1=p1; ip2=p2; ip3=p3;
      this.pj=pj;
      this.pt=pt;
      this.item=item;
      item_index=index;
      pj_zavg = (pj[p0].getZ()+pj[p1].getZ()+pj[p2].getZ()+pj[p3].getZ())/4;      
    }
    public Face( int index, IDItem item, int p0, int p1, int p2, Point3D pt[], Point3D pj[] )
    {
      ip0=p0; ip1=p1; ip2=p2; ip3=-1;
      this.pj=pj;
      this.pt=pt;
      this.item=item;
      item_index=index;
      pj_zavg = (pj[p0].getZ()+pj[p1].getZ()+pj[p2].getZ())/3;      
    }
  }
  
  private static void renderTriangle( Face f, DefaultChartRenderData d, H3D h3d, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)f.pj[f.ip0].getX(), (int)f.pj[f.ip0].getY() );
    p.setPoint( 1, (int)f.pj[f.ip1].getX(), (int)f.pj[f.ip1].getY() );
    p.setPoint( 2, (int)f.pj[f.ip2].getX(), (int)f.pj[f.ip2].getY() );   
    v.vectorialProduct( f.pt[f.ip1].getX()-f.pt[f.ip0].getX(),
                        f.pt[f.ip1].getY()-f.pt[f.ip0].getY(),
                        f.pt[f.ip1].getZ()-f.pt[f.ip0].getZ(),
                        f.pt[f.ip2].getX()-f.pt[f.ip1].getX(),
                        f.pt[f.ip2].getY()-f.pt[f.ip1].getY(),
                        f.pt[f.ip2].getZ()-f.pt[f.ip1].getZ() );
    v.normalize();
    renderPolygon( f.item, d,h3d, IGCDStyle.GetBackColor(f.item), p, v, view_vector );
  }
  private static void renderQuad( Face f, DefaultChartRenderData d, H3D h3d, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)f.pj[f.ip0].getX(), (int)f.pj[f.ip0].getY() );
    p.setPoint( 1, (int)f.pj[f.ip1].getX(), (int)f.pj[f.ip1].getY() );
    p.setPoint( 2, (int)f.pj[f.ip2].getX(), (int)f.pj[f.ip2].getY() );   
    p.setPoint( 3, (int)f.pj[f.ip3].getX(), (int)f.pj[f.ip3].getY() );   
    v.vectorialProduct( f.pt[f.ip1].getX()-f.pt[f.ip0].getX(),
                        f.pt[f.ip1].getY()-f.pt[f.ip0].getY(),
                        f.pt[f.ip1].getZ()-f.pt[f.ip0].getZ(),
                        f.pt[f.ip2].getX()-f.pt[f.ip1].getX(),
                        f.pt[f.ip2].getY()-f.pt[f.ip1].getY(),
                        f.pt[f.ip2].getZ()-f.pt[f.ip1].getZ() );
    v.normalize();
    renderPolygon( f.item, d,h3d, IGCDStyle.GetBackColor(f.item), p, v, view_vector );
  }
  
  //face using N points
  private static class FaceN extends AFace
  {
    public int ip[];
    public Point3D pj[]; //link to pj
    public Point3D pt[]; //link to pt
    public FaceN( int index, IDItem item, int []ip, Point3D pt[], Point3D pj[] )
    {
      this.item=item;
      this.ip=ip;
      this.pj=pj;
      this.pt=pt;
      item_index=index;
      //not better: double zn=pj[ip[0]].getZ(), zx=pj[ip[0]].getZ();
      for( int i=0;i<ip.length; i++ )
      {
        double z = pj[ip[i]].getZ();
        pj_zavg += z;
        //if( z<zn ) zn=z; else if( z>zx)zx=z;
      }
      pj_zavg /= ip.length;
      //pj_zavg  = (zx+zn)/2;
    }
    public int getCenterXj()
    {
      double x=0;
      for( int i=0;i<ip.length; i++ )
      {
        x += pj[ip[i]].getX();
      }
      return (int)Math.round( x/ip.length);
    }
    public int getCenterYj()
    {
      double y=0;
      for( int i=0;i<ip.length; i++ )
      {
        y += pj[ip[i]].getY();
      }
      return (int)Math.round( y/ip.length);
    }
  }
  
  private static void renderSector( int nvalues, double sum, double values[], IDItem items[], String labels[], LRect r_labels[], double shifts[], boolean have_shift, DefaultChartRenderData d, H3D p, boolean x_incr, View3D view, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    double xc = 0.5*p.bbox.getXMax(); 
    double yc = 0.5*p.bbox.getYMax();    
    double zc = 0.5*p.bbox.getZMax();
    double x1 = p.bbox.getXMin();
    double x2 = p.bbox.getXMax();
    
    final float R = 0.5f; //radius without shift
    
    if( nvalues==0 ) return ; //nothing to render
    
    final int a_div = 64; //the more you want the slower it is!
    final double angle_incr = Radian._2PI/a_div;
    
    Vector3D v = new Vector3D();
    
    if( nvalues==1 )
    {
      //render just a cylinder, no shift here
      Point3D[] pt = new Point3D[2*a_div];
      Point3D[] pj = new Point3D[2*a_div];
      double a=0.0;
      for( int i=0; i<a_div; i++, a+=angle_incr )
      {
        double c = R*Math.cos(a);
        double s = R*Math.sin(a);
        pt[i      ] = new Point3D( x1, yc+c, zc-s );        
        pt[a_div+i] = new Point3D( x2, yc+c, zc-s );
        view.projection( pt[i], pj[i] =new Point3D() );
        view.projection( pt[i+a_div], pj[i+a_div] =new Point3D() );
      }
      //create faces:
      AFace faces[] = new AFace[a_div+2];
      int last_ip0=a_div-1;
      int last_ip1=2*a_div-1;
      for( int i0=0,i1=a_div; i0<a_div; i0++,i1++ )
      {
        faces[i0] = new Face( 0, items[0], last_ip0, i0, i1, last_ip1, pt, pj );
        last_ip0=i0;
        last_ip1=i1;
      }
      int ip0[] = new int[a_div];
      int ip1[] = new int[a_div];
      for( int i=0; i<a_div; i++ ) { ip0[i]=i; ip1[i]=a_div+i; }
      faces[a_div  ] = new FaceN( 0, items[0], ip0, pt,pj ); //back
      faces[a_div+1] = new FaceN( 0, items[0], ip1, pt,pj ); //front
      
      //sort face by zmin...
      Arrays.sort( faces, 0,faces.length );
      
      Polygon pN = new Polygon( a_div );
      Polygon p4 = new Polygon( 4 );
      for( int i=faces.length-1; i>=0; i-- )
      {
        AFace face = faces[i];
        if( face instanceof FaceN )
        {
          FaceN f = (FaceN)face;
          for( int ip=0; ip<a_div; ++ip ) {
            int ji = f.ip[ip];
            pN.setPoint( ip, (int)pj[ji].getX(), (int)pj[ji].getY() );
          }
          if( f == faces[a_div] )
            v.setVector( -1, 0, 0 );
          else
            v.setVector( 1, 0, 0 );
          renderPolygon( items[0], d,p, IGCDStyle.GetBackColor(items[0]), pN, v, view_vector );         
        } else {
          Face f = (Face)face;
          renderQuad( f, d,p, p4, v,view_vector );
        }
      }
    }
    else //more than one value to render ...
    {
       ArrayList faces = new ArrayList();
       ArrayList front = new ArrayList(); //front and back face list aren't sorted by Zj order
       ArrayList back  = new ArrayList(); //as we know directly which one have to be rendered using x_incr
       
       double last_angle=0.0;
       for( int iv=0; iv<nvalues; iv++ )
       {
         double delta_angle = Radian._2PI*( values[iv]/sum );
         double end_angle   = last_angle + delta_angle;
//System.out.println("value="+values[iv]+" sum="+sum+" delta="+delta_angle+" last_a="+last_angle+" end="+end_angle);         
         double lyc = yc, lzc=zc;
         //sector shifting:
         if( have_shift && shifts[iv] >0.0 )
         {
           double sht = R*shifts[iv];
           double am = (last_angle+end_angle)/2;
           lyc = yc+sht*Math.cos(am);
           lzc = zc-sht*Math.sin(am);
         }

         int npt = (int)Math.round( delta_angle / angle_incr);
         if ( npt<=1 )
         {
           //angle less than ... use a triangle:
           Point3D lpt[] = new Point3D[6];
           Point3D lpj[] = new Point3D[6];
           lpt[0] = new Point3D( x1, lyc, lzc );
           double y1 = lyc+R*Math.cos( last_angle ), z1 = lzc-R*Math.sin(last_angle);
           double y2 = lyc+R*Math.cos(  end_angle ), z2 = lzc-R*Math.sin( end_angle);
           lpt[1] = new Point3D( x1, y1, z1 );
           lpt[2] = new Point3D( x1, y2, z2 );
           lpt[3] = new Point3D( x2, lyc, lzc );
           lpt[4] = new Point3D( x2, y1, z1 );
           lpt[5] = new Point3D( x2, y2, z2 );
           for( int i=0; i<6; i++ ) view.projection( lpt[i], lpj[i]=new Point3D() );

           faces.add( new Face( iv, items[iv], 1,4,5,2, lpt,lpj) );
           if( have_shift ) //sides
           {
             faces.add( new Face( iv, items[iv], 3,4,1,0, lpt,lpj ));
             faces.add( new Face( iv, items[iv], 5,3,0,2, lpt,lpj ));
           }

           Face f = new Face( iv, items[iv], 0,1,2, lpt,lpj);
           f.medium_angle = last_angle+delta_angle/2;
           back .add( f );
           
           f = new Face( iv, items[iv], 3,5,4, lpt,lpj);
           f.medium_angle = last_angle+delta_angle/2;
           front.add( f );
                      
         } 
         else
         {
            double lai = delta_angle/(npt-1);
            double a=last_angle;
            Point3D lpt[] = new Point3D[2*npt+2]; //+2: two centers
            Point3D lpj[] = new Point3D[2*npt+2];
            lpt[0    ] = new Point3D( x1, lyc, lzc );
            lpt[npt+1] = new Point3D( x2, lyc, lzc );
            for(int i=1; i<=npt; i++, a+=lai )
            {
              double y = lyc+R*Math.cos(a), z=lzc-R*Math.sin(a);
              lpt[i] = new Point3D(x1,y,z );
              lpt[i+npt+1] = new Point3D(x2,y,z);
            }
            for( int i=0;i<lpt.length; i++) view.projection( lpt[i], lpj[i]=new Point3D() );
            //create faces
            int fn_ip0[] = new int[npt+1];
            int fn_ip1[] = new int[npt+1];
            fn_ip0[0]=0;
            fn_ip1[0]=npt+1;
            for( int i=1; i<=npt; i++ ) { fn_ip0[i]=i; fn_ip1[i]=i+npt+1; }
            FaceN f = new FaceN( iv, items[iv], fn_ip0, lpt,lpj ); //back
            f.medium_angle = last_angle + delta_angle/2;
            back.add( f );
            f = new FaceN( iv, items[iv], fn_ip1, lpt,lpj ); //front
            f.medium_angle = last_angle + delta_angle/2;
            front.add( f );
            //sides
            int lip0=1, lip1=npt+2;
            for( int ip0=2, ip1=npt+3; ip0<=npt; ip0++, ip1++ )
            {
              Face ff = new Face( iv, items[iv], lip1,ip1,ip0,lip0, lpt,lpj);
              faces.add( ff );
              lip0=ip0;
              lip1=ip1;
            }
            if( have_shift ) //sides
            {
              faces.add( new Face( iv, items[iv], npt+1,npt+2,1,0, lpt,lpj ));
              faces.add( new Face( iv, items[iv], 0,npt,2*npt+1,npt+1, lpt,lpj ));
            }
         }
         last_angle=end_angle;
       }//for
       
       //sort faces:
       Object of[] = faces.toArray();
       Arrays.sort( of, 0, of.length );
       Polygon pN = new Polygon( a_div );
       Polygon p4 = new Polygon( 4 );
       Polygon p3 = new Polygon( 3 );
       
       //only needed if there are transparency...  next day       
       /*if( have_transparency && d.drawing() ) //no location here as it's unvisible faces
       {
         //render back faces...
         ArrayList back_faces = x_incr ? back : front ;       
         boolean  decr= d.drawing();
         int i0 = decr ? back_faces.size()-1 : 0;
         int iX = decr ? 0 : back_faces.size()-1;
         int ii = decr ? -1 : +1;
         for( int i=i0; (ii==1)&&(i<=iX) || (ii==-1)&&(i>=iX); i+=ii )
         {
           renderFace( (AFace)back_faces.get(i), p3,p4,pN, v,view_vector, d);
         }         
       }*/
       
       DefaultRenderChartLocation last_loc=null;
       //draw back to front, located front to back
       boolean decr= d.drawing();
       int i0 = decr ? of.length-1 : 0;
       int iX = decr ? 0 : of.length-1;
       int ii = decr ? -1 : +1;
       DefaultRenderChartLocation loc=null;
       try {
         for( int i=i0; (ii==1)&&(i<=iX) || (ii==-1)&&(i>=iX); i+=ii )
         {
           AFace face = (AFace)of[i];
           renderFace( face, p3,p4,pN, v,view_vector, d,p, false);
         }//for
       } catch ( DefaultRenderChartLocation l ) { loc=l; }
       
       //render front faces...
       {
         ArrayList front_faces = x_incr ? front : back ;       
         decr= d.drawing();
         i0 = decr ? front_faces.size()-1 : 0;
         iX = decr ? 0 : front_faces.size()-1;
         ii = decr ? -1 : +1;
         for( int i=i0; (ii==1)&&(i<=iX) || (ii==-1)&&(i>=iX); i+=ii )
         {
           renderFace( (AFace)front_faces.get(i), p3,p4,pN, v,view_vector, d,p, false);
         }         
         //link to labels ? must be done after face drawing
         if( labels!=null && d.drawing() ) //or show_labels, drawing only as only lines 
         {
           d.gc_.setPen( new LineStylePen( RGBA.BLACK ));
           Point3D pj = new Point3D();
           double xj = x_incr ? x2 : x1 ;
           for( int i=0; i<front_faces.size(); i++ )
           {
              AFace face = (AFace)front_faces.get(i); //FaceN because Face haven't item_index
              int index = face.item_index;
              if( labels[index]==null) continue; //no label for this point
              IDItem item = items[ face.item_index ];
              double ma = face.medium_angle;
              double lyc = yc, lzc=zc;
              if( have_shift && shifts[index] >0.0 )
              {
                double sht = R*shifts[index];
                lyc = yc+sht*Math.cos(ma);
                lzc = zc-sht*Math.sin(ma);
              }
              view.projection( xj, lyc+(0.8*R)*Math.cos(ma), lzc-(0.8*R)*Math.sin(ma), pj );
              int xs = (int)pj.getX();
              int ys = (int)pj.getY();
              int xt, yt ;
              switch( r_labels[index].align )
              {
              default:
              case IDAlignment.TOP: xt = r_labels[index].centerX();
                                    yt = r_labels[index].bottom()+d.margin_/2;
                                    break;
              case IDAlignment.BOTTOM: xt = r_labels[index].centerX();
                                       yt = r_labels[index].top()-d.margin_/2;
                                       break;
              case IDAlignment.LEFT: xt=r_labels[index].right()+d.margin_/2;
                                     yt=r_labels[index].centerY();
                                     break;
              }

              d.gc_.drawLine( xs,ys,xt,yt);
           }
         }
       }
       

       
       if( loc!=null ) throw loc;
    }
  }
  
  
  private static void renderTorus( int nvalues, double sum, double values[], IDItem items[], String labels[], LRect r_labels[], double shifts[], boolean have_shift, DefaultChartRenderData d, H3D p, boolean x_incr, View3D view, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    double xc = 0.5*p.bbox.getXMax(); 
    double yc = 0.5*p.bbox.getYMax();    
    double zc = 0.5*p.bbox.getZMax();
    double x1 = p.bbox.getXMin();
    double x2 = p.bbox.getXMax();
    
    final float R1 = 0.35f; //main radius (without shift)
    final float R2 = 0.15f;
    
    if( nvalues==0 ) return ; //nothing to render
    
    final int a_div1 = 64; //the more you want the slower it is!
    final double angle_incr1 = Radian._2PI/a_div1;

    final int a_div2 = 24; //the more you want the slower it is! (too)
    final double angle_incr2 = Radian._2PI/a_div2;
    
    Vector3D v = new Vector3D();
    
    if( nvalues==1 )
    {
      //render just a tore, no shift here
      Point3D[] pt = new Point3D[a_div1*a_div2];
      Point3D[] pj = new Point3D[a_div1*a_div2];
      double a=0.0;
      int k=0;
      for( int i=0; i<a_div1; i++, a+=angle_incr1 )
      {
        double a2=0.0;
        for( int j=0; j<a_div2; j++, a2+=angle_incr2)
        {
          double x = xc+R2*Math.sin( a2 );
          double R = R1 + R2*Math.cos(a2);
          double y = yc+R*Math.cos(a);
          double z = zc-R*Math.sin(a);
          pt[k] = new Point3D( x, y, z );
          view.projection( pt[k], pj[k] =new Point3D() );
          k++;
        }        
      }
      //create faces:
      Face faces[] = new Face[a_div1*a_div2];
      int last_i=a_div1-1;
      k=0;
      for( int i=0; i<a_div1; i++ )
      {
        int last_j=a_div2-1;
        for( int j=0; j<a_div2; j++ )
        {
          faces[k++] = new Face( 0, items[0], last_i*a_div2+last_j, i*a_div2+last_j, i*a_div2+j, last_i*a_div2+j, pt,pj);
          last_j=j;
        }
        last_i=i;
      }
      
      //sort face by zmin...
      Arrays.sort( faces, 0,faces.length );
      
      Polygon p4 = new Polygon( 4 );
      for( int i=faces.length-1; i>=0; i-- )
      {
        Face f = (Face)faces[i];
        renderQuad( f, d,p, p4, v,view_vector );
      }
    }
    else //more than one value to render ...
    {
       ArrayList faces = new ArrayList();
       
       double last_angle=0.0; //angle of "sector"
       for( int iv=0; iv<nvalues; iv++ )
       {
         double delta_angle = Radian._2PI*( values[iv]/sum );
         double end_angle   = last_angle + delta_angle;         
         double lyc=yc, lzc=zc; //local center of tore, include shift...
         //sector shifting:
         if( have_shift && shifts[iv] >0.0 )
         {
           double sht = R1*shifts[iv];
           double am = (last_angle+end_angle)/2;
           lyc = yc+sht*Math.cos(am);
           lzc = zc-sht*Math.sin(am);
         }

         int npt = (int)Math.round( delta_angle / angle_incr1);
         if ( npt<=1 )
         {
           //two circle only
           Point3D pt[] = new Point3D[2*a_div2];
           Point3D pj[] = new Point3D[2*a_div2];
           
           double a2=0.0;
           int j2=a_div2;
           for( int j=0; j<a_div2; j++, j2++, a2+=angle_incr2)
           {
             double x = xc+R2*Math.sin( a2 );
             double R = R1 + R2*Math.cos(a2);
             double y1 = lyc+R*Math.cos(last_angle);
             double z1 = lzc-R*Math.sin(last_angle);
             double y2 = lyc+R*Math.cos(end_angle);
             double z2 = lzc-R*Math.sin(end_angle);
             pt[j] = new Point3D( x, y1, z1 );
             pt[j2] = new Point3D( x, y2, z2 );
             view.projection( pt[j], pj[j] =new Point3D() );
             view.projection( pt[j2], pj[j2] =new Point3D() );
           }
           //create face
           j2=a_div2;
           int lj=a_div2-1;
           int lj2=2*a_div2-1;
           for( int j=0; j<a_div2; j++, j2++, a2+=angle_incr2)
           {
             faces.add( new Face( iv, items[iv], lj,lj2,j2,j, pt,pj));
             lj=j;
             lj2=j2;
           }
           //sides            
           if( have_shift )
           {
             int ip1[] = new int[a_div2];
             int ip2[] = new int[a_div2];
             for( int i=0; i<a_div2; i++ ) {
               ip1[i] = i;
               ip2[i] = a_div2+i;
             }
             faces.add( new FaceN(iv, items[iv], ip1, pt,pj ) );
             faces.add( new FaceN(iv, items[iv], ip2, pt,pj ) );
           }
         } 
         else
         {
            double lai = delta_angle/(npt-1);
            Point3D pt[] = new Point3D[npt*a_div2];
            Point3D pj[] = new Point3D[npt*a_div2];
            
            double a1=last_angle;
            int k=0;
            for( int i=0; i<npt; i++, a1+=lai )
            {
              if( i==npt-1) a1=end_angle;
              double a2=0.0;
              for( int j=0; j<a_div2; j++, a2+=angle_incr2)
              {
                double x = xc+R2*Math.sin( a2 );
                double R = R1 + R2*Math.cos(a2);
                double y = lyc+R*Math.cos(a1);
                double z = lzc-R*Math.sin(a1);
                pt[k] = new Point3D( x, y, z );
                view.projection( pt[k], pj[k] =new Point3D() );
                k++;
              }        
            }
            //create faces:
            int last_i=0;
            for( int i=1; i<npt; i++ )
            {
              int last_j=a_div2-1;
              for( int j=0; j<a_div2; j++ )
              {
                faces.add( new Face( iv, items[iv], last_i*a_div2+last_j, i*a_div2+last_j, i*a_div2+j, last_i*a_div2+j, pt,pj));
                last_j=j;
              }
              last_i=i;
            }
            //sides
            if( have_shift )
            {
              int ip1[] = new int[a_div2];
              int ip2[] = new int[a_div2];
              for( int i=0; i<a_div2; i++ ) {
                ip1[i] = /* 0*a_div2 +*/i;
                ip2[i] = (npt-1)*a_div2+i;
              }
              faces.add( new FaceN(iv, items[iv], ip1, pt,pj ) );
              faces.add( new FaceN(iv, items[iv], ip2, pt,pj ) );
            }
         }
         last_angle=end_angle;
       }//for
       
       //sort faces:
       Object of[] = faces.toArray();
       Arrays.sort( of, 0, of.length );
       Polygon pN = have_shift ? new Polygon(a_div2) : null;
       Polygon p4 = new Polygon( 4 );
       
       DefaultRenderChartLocation last_loc=null;
       //draw back to front, located front to back
       boolean decr= d.drawing();
       int i0 = decr ? of.length-1 : 0;
       int iX = decr ? 0 : of.length-1;
       int ii = decr ? -1 : +1;
       DefaultRenderChartLocation loc=null;
       try {
         for( int i=i0; (ii==1)&&(i<=iX) || (ii==-1)&&(i>=iX); i+=ii )
         {
           AFace face = (AFace)of[i];
           renderFace( face, null,p4,pN, v,view_vector, d,p, true);
         }//for
       } catch ( DefaultRenderChartLocation l ) { loc=l; }
       
       //link to labels ? must be done after face drawing
       if( labels!=null && d.drawing() ) //or show_labels, drawing only as only lines 
       {
           d.gc_.setPen( new LineStylePen( RGBA.BLACK ));
           Point3D pj = new Point3D();
           double xj = x_incr ? xc+R2 : xc-R2 ;
           last_angle=0.0; //angle of "sector"
           for( int iv=0; iv<nvalues; iv++ )
           {
             double delta_angle = Radian._2PI*( values[iv]/sum );
             double end_angle   = last_angle + delta_angle;         
             double am = (last_angle+end_angle)/2;
             last_angle=end_angle;
             //sector shifting:
             double lyc=yc, lzc=zc; //local center of tore, include shift...
             if( have_shift && shifts[iv] >0.0 )
             {
               double sht = R1*shifts[iv];
               lyc = yc+sht*Math.cos(am);
               lzc = zc-sht*Math.sin(am);
             }
             
             if( labels[iv]==null ) continue; //no label for this sector
                           
             //no R2 here because point it's at _PI2 or _3PI2 =>cos(a)=0
             view.projection( xj, lyc+(R1)*Math.cos(am), lzc-(R1)*Math.sin(am), pj );
             int xs = (int)pj.getX();
             int ys = (int)pj.getY();
             int xt, yt ;
             switch( r_labels[iv].align )
             {
             default:
             case IDAlignment.TOP: xt = r_labels[iv].centerX();
             yt = r_labels[iv].bottom()+d.margin_/2;
             break;
             case IDAlignment.BOTTOM: xt = r_labels[iv].centerX();
             yt = r_labels[iv].top()-d.margin_/2;
             break;
             case IDAlignment.LEFT: xt=r_labels[iv].right()+d.margin_/2;
             yt=r_labels[iv].centerY();
             break;
             }

             d.gc_.drawLine( xs,ys,xt,yt);
           }
       }
       
       if( loc!=null ) throw loc;
    }
  }
  
  private static void renderFace( AFace face, Polygon p3, Polygon p4, Polygon pN, Vector3D v, Vector3D view_vector, DefaultChartRenderData d,H3D h3d, boolean tore )
    throws DefaultRenderChartLocation
  {
    if( face instanceof FaceN )
    {
      FaceN f = (FaceN)face;
      pN.resize( f.ip.length );
      for( int ip=0; ip<f.ip.length; ++ip ) {
        int ji = f.ip[ip];
        pN.setPoint( ip, (int)f.pj[ji].getX(), (int)f.pj[ji].getY() );
      }
      //for tore: must compute:
      if( tore )
      {
        //as it's a circle choose 3 points;
        int i0=0, i1=(int)Math.round(f.ip.length/3.0), i2=(int)Math.round(2*f.ip.length/3.0);
        v.vectorialProduct( f.pt[i1].getX()-f.pt[i0].getX(),
            f.pt[i1].getY()-f.pt[i0].getY(),
            f.pt[i1].getZ()-f.pt[i0].getZ(),
            f.pt[i2].getX()-f.pt[i1].getX(),
            f.pt[i2].getY()-f.pt[i1].getY(),
            f.pt[i2].getZ()-f.pt[i1].getZ() );
        v.normalize();
      } else {
        //for Pie3D:
        //right value for only one ... but as it's the one user see (front one), 
        //keep this value for both avoid computing of vectorial product.
        v.setVector( 1, 0, 0 );
      }
      renderPolygon( f.item, d,h3d, IGCDStyle.GetBackColor(f.item), pN, v, view_vector );         
    } else {
      Face f = (Face)face;      
      if( f.ip3<0 )
        renderTriangle( f, d,h3d, p3, v, view_vector );
      else
        renderQuad( f, d,h3d, p4, v, view_vector );
    }
  }
  
  private static class LRect extends Rect
  {
    public int align; //one of IDAlignment.TOP,LEFT,RIGHT or BOTTOM
    
    public LRect( int x, int y, int w, int h ) {
      super( x, y, w, h );
    }
  }
}
