/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultChartRenderSeries3D.java,v 1.2 2008/05/23 14:11:50 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;


import java.util.Arrays;

import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.Point2D;
import org.eclipse.tptp.platform.report.drawutil.internal.Point3D;
import org.eclipse.tptp.platform.report.drawutil.internal.Vector3D;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Point;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Radian;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.tools.internal.IDisposable;


/**
 * Default Chart Render for T_SERIES3D, real 3D for Histogram including ribbon curves, and others
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DefaultChartRenderSeries3D
{
  
  /** variant part of pertistent data for H3D graphic */
  static class H3D implements IDisposable
  {
    DAxis axis;
    IGCDStyle axis_style;
    IScale scale;
    float shadow;
    float light;
        
    Box3D bbox;
    int npoints;// max points count over all curves
    int ncurves;// number of curves (equal to rpd.curve_styles.length
    boolean thin_bar; //true if bar are disconnected
    float bar_size; //bar size depends on thin_bar ...
    
    public void dispose()
    {
    }
  }
  
  private static boolean isValidPersistData( H3D p, DefaultChartRenderData d, RenderPersistData rpd )
  {
    if( p==null ) return false;
    //as it's change thin_bar, bar_size and box 3d bounds:
    if( isThinBar(rpd) != p.thin_bar ) return false;    
    return true;
  }

  /** 
   * Render XYZ curves (and 'simple' wireframe).
   * Rendering is made AsIs, no Z-Buffer, no hidden face remove algorythm, here is a simple
   * 3D drawing.
   */
  public static void render( DefaultChartRenderData d, RenderPersistData rpd )
     throws DefaultRenderChartLocation
  {
    H3D p = null;   
    if( rpd.g instanceof H3D ) p =(H3D)rpd.g;

    //rebuild data ?
    if( !isValidPersistData(p, d, rpd) )
    {
      p = rebuildPersistData( d, rpd );
    }
    
    //missed axis ?
    if ( p.axis==null )
    {
      if( d.drawing())
      {
        String txt = "Error: missing axis";
        d.error( txt, rpd );
      }
      return ;
    }

    //two angle to drive view
    double phi   = rpd.graphic.getProperties().get( DGraphic.P_XYZ_PHI,   0.3 );
    double theta = rpd.graphic.getProperties().get( DGraphic.P_XYZ_THETA, 0.2 );
    p.shadow = rpd.graphic.getProperties().get( DGraphic.P_3D_SHADOW, DGraphic.DEF_3D_SHADOW );
    p.light  = rpd.graphic.getProperties().get( DGraphic.P_3D_LIGHT, DGraphic.DEF_3D_LIGHT );

    View3D view = new View3D( phi, theta, 0, 1.0f, p.bbox );

    //scale 3D view for rendering if gR rectange
    int rg_w = rpd.ag_rect.w();
    int rg_h = rpd.ag_rect.h();
    view.scaleTo( rpd.ag_rect.x(), rpd.ag_rect.y(), rg_w, rg_h );

    int min_w_h = rg_w ;
    if (min_w_h > rg_h ) min_w_h = rg_h;

    LineStylePen pen = new LineStylePen();
    
    Point2D pt = new Point2D();
    Object  vp = null;
    Point3D p3c = new Point3D();
    Point3D p3l = new Point3D();
    
    double ntheta = Radian.normalize( view.getTheta() );
    boolean x_incr = (ntheta <= Radian._PI2 || ntheta >= Radian._3PI2 );
    boolean y_incr = (0<= ntheta && ntheta<=Radian._PI);
    double nphi = Radian.normalize( view.getPhi() );
    boolean z_incr = (0<= nphi && nphi <= Radian._PI );
    if( nphi > Radian._PI2 && nphi < Radian._3PI2 )
    {
      x_incr =!x_incr;
      y_incr =!y_incr;
    }

    if( d.drawing() )
    {
      //background of bar bounding box
      DefaultChartRender3D.drawBox3D( d, view, p.bbox, x_incr, y_incr, z_incr, false );
    }

    //background part of axis 
    DefaultChartRender3D.renderZAxis3D( d, view, p.bbox, p.axis, p.scale, p.axis_style, x_incr, y_incr, z_incr, false, null );
    
    IDCoord coords[] = new IDCoord[p.npoints];
    DPoint points[] = new DPoint[p.npoints];
    double bar_base = p.scale.getScaleForBar();

    Vector3D view_vector = view.getViewVector(null);
    view_vector.normalize();
    
    // curve iteration order depends on point of view.
    int ic0 = x_incr ? 0 : rpd.curve_styles.length-1;
    int icX = x_incr ? rpd.curve_styles.length-1 : 0;
    int ici = x_incr ? +1 : -1;
    //for( int ic=0; ic<rpd.curve_styles.length; ++ic )
    DefaultRenderChartLocation bar_loc = null;
    for( int ic=ic0; (x_incr&&ic<=icX)||(!x_incr&&ic>=icX); ic+=ici )
    {
      CurveStyle ccurve = rpd.curve_styles[ic];
      //curve is rendered 4 times: X, Y, Z projections and the curve it self.

      if(d.drawing()) 
      { 
        pen.setRGBA( ccurve.style_.getBack() ); d.gc_.setPen(pen); 
      } 
      
      int curve_type;      
           if( ccurve.curve_.getType()==null ) curve_type=0; //T_BAR
      else if( ccurve.curve_.isType( DCurve.T_LINE ) ) curve_type=1;
      else if( ccurve.curve_.isType( DCurve.T_BAR2 ) ) curve_type=2;
      else if( ccurve.curve_.isType( DCurve.T_PYRAMID ) ) curve_type=3; 
      else if( ccurve.curve_.isType( DCurve.T_POINTS ) ) curve_type=4; 
      else if( ccurve.curve_.isType( DCurve.T_CYLINDER ) ) curve_type=5; 
      else if( ccurve.curve_.isType( DCurve.T_AREA_LINE ) ) curve_type=6;
      else curve_type=0; //T_BAR by default.

      //get points and it coordinates
      int npts = 0;
      int ip=0;
      for( IDItem item = ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
      {
        if( !(item instanceof DPoint)) continue;
        for( IDItem icd = item.getFirstChild(); icd!=null; icd=icd.getNext() )
        {
          if(!(icd instanceof IDCoord)) continue;
          if( p.axis  == ((IDCoord)icd).getAxis() )
          {
            npts++;
            coords[ip]=(IDCoord)icd;
            points[ip]=(DPoint)item;
            ip++;
            break;
          }
        }
      }
      
      int xbar = ic;      
      //points iteration order depends on point of view
      int yb0 = y_incr ? 0 : npts-1 ;
      int ybX = y_incr ? npts-1 : 0 ;
      int ybi = y_incr ? +1 : -1;  
      double last_zbar=0.0;
      for( int ybar=yb0; (y_incr&&ybar<=ybX) ||(!y_incr&&ybar>=ybX); ybar+=ybi )
      {          
        double zbar = p.scale.toScale( vp=coords[ybar].getValue(vp) );
        
        try {
          switch( curve_type ) 
          {
          case 0:renderBar( points[ybar], d,p, view, xbar, ybar, bar_base,zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector ); break;
          case 1:
            if( ybar!=yb0 ) { 
              renderLine( points[ybar], d,p, view, xbar, ybar, zbar,last_zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector );
            }
            last_zbar = zbar; 
            break;
          case 2:renderBar2( points[ybar], d,p, view, xbar, ybar, bar_base,zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector ); break; 
          case 3:renderPyramid( points[ybar], d,p, view, xbar, ybar, bar_base,zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector ); break;           
          case 4:renderPoint( points[ybar], d,p, view, xbar, ybar, zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector ); break;           
          case 5:renderOctogon( points[ybar], d,p, view, xbar, ybar, bar_base,zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector ); break;
          case 6:
            if( ybar!=yb0 ) { 
              renderAreaLine( points[ybar], d,p, view, xbar, ybar, bar_base,zbar,last_zbar, x_incr,y_incr,z_incr, p.bar_size, view_vector,
                  ybar==yb0+ybi, ybar==ybX);
            }
            last_zbar = zbar; 
            break;
          }          
        } catch( DefaultRenderChartLocation loc ) {
          bar_loc = loc; //keep last located bar..
        }        
      }//for
    }//for
    
    if( d.drawing() )
    {
      //front of bars bounding box
      DefaultChartRender3D.drawBox3D( d, view, p.bbox, x_incr, y_incr, z_incr, true );
    }
    //foreground part of axis (and location)
    DefaultChartRender3D.renderZAxis3D( d, view, p.bbox, p.axis, p.scale, p.axis_style, x_incr, y_incr, z_incr, true, null );
    
    if( bar_loc != null )
    {
      throw bar_loc;
    }
  }
  
  /** render a part of 3D line (knows as ribbon) */
  private static void renderLine( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double ztop, double last_ztop, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    double y1=y_incr ? (1-w/2) : -(w/2);
    double y2=y_incr ? (w/2) : -(1-w/2);
    if(!y_incr) y++;
    view.projection( x,   y-y1, last_ztop, ptproj[0] );
    view.projection( x+w, y-y1, last_ztop, ptproj[1] );
    view.projection( x,   y+y2, ztop, ptproj[2] );
    view.projection( x+w, y+y2, ztop, ptproj[3] );
    
    int rgba = IGCDStyle.GetBackColor( point );    
    Polygon p = new Polygon( 4 );
    Vector3D v = new Vector3D();
    
    p.setPoint( 0, (int)ptproj[0].getX(), (int)ptproj[0].getY() );
    p.setPoint( 1, (int)ptproj[1].getX(), (int)ptproj[1].getY() );
    p.setPoint( 2, (int)ptproj[3].getX(), (int)ptproj[3].getY() );
    p.setPoint( 3, (int)ptproj[2].getX(), (int)ptproj[2].getY() );
    v.vectorialProduct( w,0,0, 0,y2+y1,ztop-last_ztop );
    v.normalize();    
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  
  /** render a part of 3D Area line  */
  private static void renderAreaLine( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double zbase, double ztop, double last_ztop, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector, boolean is_first, boolean is_last )
    throws DefaultRenderChartLocation
  {
    double y1=y_incr ? (1-w/2) : -(w/2);
    double y2=y_incr ? (w/2) : -(1-w/2);
    if(!y_incr) y++;
    view.projection( x,   y-y1, last_ztop, ptproj[0] );
    view.projection( x+w, y-y1, last_ztop, ptproj[1] );
    view.projection( x,   y+y2, ztop, ptproj[2] );
    view.projection( x+w, y+y2, ztop, ptproj[3] );
        
    int rgba = IGCDStyle.GetBackColor( point );    
    Polygon p = new Polygon( 4 );
    Polygon p3 = null;
    Vector3D v = new Vector3D();
    Point  pt = new Point();
    
    ptvolume[0].setCoord( x,   y-y1, last_ztop );
    ptvolume[1].setCoord( x+w, y-y1, last_ztop );
    ptvolume[2].setCoord( x+w, y+y2, ztop );
    ptvolume[3].setCoord( x,   y+y2, ztop );
    ptvolume[4].setCoord( x,   y-y1, zbase );
    ptvolume[5].setCoord( x+w, y-y1, zbase );
    ptvolume[6].setCoord( x+w, y+y2, zbase );
    ptvolume[7].setCoord( x,   y+y2, zbase );
    for( int i=0; i<8; i++ )
    {
      view.projection( ptvolume[i], ptproj[i] );
    }
    
    int nf=0; //face count
    if( is_first ) faces[nf++].update( 4,5,1,0 ); //left side
    
    //trois cas les 2  Z au dessus, les deux en dessous, un de chaque cote.
    if( ztop >= zbase && last_ztop >= zbase 
     || ztop <  zbase && last_ztop <  zbase )
    {      
      faces[nf++].update( 0,1,2,3); //top
      faces[nf++].update( 7,6,5,4); //bottom
      faces[nf++].update( 1,5,6,2); //front
      faces[nf++].update( 0,3,7,4); //back
    }    
    else //must "cut" part into two convex polygon. 
    {
      //add two point for z=zbase ybase is:
      double ycut = y-y1+(y2+y1)*( zbase-last_ztop)/(ztop-last_ztop);
      ptvolume[8].setCoord( x,   ycut, zbase );
      ptvolume[9].setCoord( x+w, ycut, zbase);
      view.projection( ptvolume[8], ptproj[8] );
      view.projection( ptvolume[9], ptproj[9] );
      
      if( last_ztop >= zbase ) {
        faces[nf++].update( 0,1,9,8); //top (1)
        faces[nf++].update( 8,9,6,7); //top (2)
        faces[nf++].update( 8,9,5,4); //bottom (1)
        faces[nf++].update( 9,8,3,2); //bottom (2)
      } else {
        faces[nf++].update( 4,5,9,8); //top (1)
        faces[nf++].update( 8,9,2,3); //top (2)
        faces[nf++].update( 8,9,1,0); //bottom (1)
        faces[nf++].update( 9,8,7,6); //bottom (2)
      }
      faces[nf++].update( 1,5,9); //front (1)
      faces[nf++].update( 9,2,6); //front (2)
      faces[nf++].update( 0,8,4); //back (1)
      faces[nf++].update( 8,7,3); //back (2)
      
      if(p3==null) p3 = new Polygon(3);
    }
    
    if( is_last ) faces[nf++].update( 7,6,2,3); //right side
    
    Arrays.sort( faces, 0,nf );
    
    for( int i=nf-1; i>=0; i--) 
    {
      if( faces[i].ip3<0 )
        renderTriangle( point, rgba, d,h3d, ptvolume,ptproj, p3, v, view_vector, faces[i].ip0,faces[i].ip1,faces[i].ip2);
      else
        renderQuad( point, rgba, d,h3d, ptvolume,ptproj, p, v, view_vector, faces[i].ip0,faces[i].ip1,faces[i].ip2,faces[i].ip3);
    }
  }
  
  private static Point3D ptproj[] = new Point3D[]{ //projected point of bar, pyramid,...
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
  };
  private static Point3D ptvolume[] = new Point3D[]{ //definition points of volume (bar,...)
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
      new Point3D(), new Point3D(), new Point3D(), new Point3D(),
  };
  
  /** render a 3D bar for histogram curves */
  private static void renderBar( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double zbtm, double ztop, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    boolean z_inverted=false;
    if( ztop < zbtm ) {
      z_inverted=true;
      z_incr = !z_incr; //rever Z to have right order for given z top and bottom
    }
    view.projection( x,   y,   zbtm, ptproj[0] );
    view.projection( x+w, y,   zbtm, ptproj[1] );
    view.projection( x+w, y+w, zbtm, ptproj[2] );
    view.projection( x,   y+w, zbtm, ptproj[3] );
    view.projection( x,   y,   ztop, ptproj[4] );
    view.projection( x+w, y,   ztop, ptproj[5] );
    view.projection( x+w, y+w, ztop, ptproj[6] );
    view.projection( x,   y+w, ztop, ptproj[7] );
    
    int rgba = IGCDStyle.GetBackColor( point );
    
    Polygon p = new Polygon( 4 );
    Vector3D v = new Vector3D();
    Point  pt = new Point();
    if( z_incr ) {
      renderBarBottom( point, rgba, d,h3d, ptproj, p, v, view_vector, z_inverted );        
    } else {
      renderBarTop( point, rgba, d,h3d, ptproj, p, v, view_vector, z_inverted );
    }
    
    if( y_incr ) {
      renderBarLeft( point, rgba, d,h3d, ptproj, p, v, view_vector );
    } else {
      renderBarRight( point, rgba, d,h3d, ptproj, p, v, view_vector );
    }
    
    if( x_incr ) {
      renderBarBack( point, rgba, d,h3d, ptproj, p, v, view_vector );
    } else {
      renderBarFront( point, rgba, d,h3d, ptproj, p, v, view_vector );
    }
    
    if( y_incr ) {
      renderBarRight( point, rgba, d,h3d, ptproj, p, v, view_vector );
    } else {
      renderBarLeft( point, rgba, d,h3d, ptproj, p, v, view_vector );
    }
    
    if( x_incr ) {
      renderBarFront( point, rgba, d,h3d, ptproj, p, v, view_vector );
    } else {
      renderBarBack( point, rgba, d,h3d, ptproj, p ,v, view_vector );
    }
    
    if( z_incr ) {
      renderBarTop( point, rgba, d,h3d, ptproj, p, v, view_vector, z_inverted );
    } else {
      renderBarBottom( point, rgba, d,h3d, ptproj, p, v, view_vector, z_inverted );        
    }
  }
  
  private static SolidBrush brush ;
  private static void renderPolygon(DPoint _point, DefaultChartRenderData d, H3D h3d, int rgba, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    if( d.drawing() ) 
    {
      double k = Math.abs(Vector3D.Scalar( v, view_vector ));
      if( brush==null ) brush = new SolidBrush();
      brush.setRGBA( RGBA.Shadow( rgba, h3d.shadow, h3d.light, (float)k)); //strength of shadow
      d.gc_.setBrush(brush);
      d.gc_.fillPoly( p );
    } else {
      //TODO: take care if label are supported:
      //loc_id = DLocated.LabelText ... cf renderHistogram
      if( p.contains( d.lx_, d.ly_ ) ) {
        String loc_id  =DLocated.CurvePointText;
        throw new DefaultRenderChartLocation( loc_id, _point, new Rect(d.lx_,d.ly_,0,0) );
      }   
    }
  }
  private static void renderBarBottom( DPoint _point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D []ptbar, Polygon p, Vector3D v, Vector3D view_vector, boolean z_inverted )
    throws DefaultRenderChartLocation
  {
    //bottom
    p.setPoint( 0, (int)ptbar[0].getX(), (int)ptbar[0].getY() );
    p.setPoint( 1, (int)ptbar[1].getX(), (int)ptbar[1].getY() );
    p.setPoint( 2, (int)ptbar[2].getX(), (int)ptbar[2].getY() );
    p.setPoint( 3, (int)ptbar[3].getX(), (int)ptbar[3].getY() );
    v.setVector( 0, 0, z_inverted ? +1 : -1 );
    renderPolygon( _point, d,h3d, rgba, p, v, view_vector );
  }  
  private static void renderBarTop( DPoint _point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D []ptbar, Polygon p, Vector3D v, Vector3D view_vector, boolean z_inverted )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)ptbar[4].getX(), (int)ptbar[4].getY() );
    p.setPoint( 1, (int)ptbar[5].getX(), (int)ptbar[5].getY() );
    p.setPoint( 2, (int)ptbar[6].getX(), (int)ptbar[6].getY() );
    p.setPoint( 3, (int)ptbar[7].getX(), (int)ptbar[7].getY() );
    v.setVector( 0, 0, z_inverted ? -1 : +1);
    renderPolygon( _point, d,h3d, rgba, p, v, view_vector );
  }
  private static void renderBarFront( DPoint point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D []ptbar, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)ptbar[1].getX(), (int)ptbar[1].getY() );
    p.setPoint( 1, (int)ptbar[2].getX(), (int)ptbar[2].getY() );
    p.setPoint( 2, (int)ptbar[6].getX(), (int)ptbar[6].getY() );
    p.setPoint( 3, (int)ptbar[5].getX(), (int)ptbar[5].getY() );
    v.setVector( +1, 0, 0 );
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  private static void renderBarBack( DPoint point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D []ptbar, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)ptbar[3].getX(), (int)ptbar[3].getY() );
    p.setPoint( 1, (int)ptbar[0].getX(), (int)ptbar[0].getY() );
    p.setPoint( 2, (int)ptbar[4].getX(), (int)ptbar[4].getY() );
    p.setPoint( 3, (int)ptbar[7].getX(), (int)ptbar[7].getY() );
    v.setVector( -1, 0, 0 );
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  private static void renderBarLeft( DPoint point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D []ptbar, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)ptbar[0].getX(), (int)ptbar[0].getY() );
    p.setPoint( 1, (int)ptbar[1].getX(), (int)ptbar[1].getY() );
    p.setPoint( 2, (int)ptbar[5].getX(), (int)ptbar[5].getY() );
    p.setPoint( 3, (int)ptbar[4].getX(), (int)ptbar[4].getY() );
    v.setVector( 0, -1, 0 );
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  
  private static void renderBarRight( DPoint point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D []ptbar, Polygon p, Vector3D v, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)ptbar[2].getX(), (int)ptbar[2].getY() );
    p.setPoint( 1, (int)ptbar[3].getX(), (int)ptbar[3].getY() );
    p.setPoint( 2, (int)ptbar[7].getX(), (int)ptbar[7].getY() );
    p.setPoint( 3, (int)ptbar[6].getX(), (int)ptbar[6].getY() );
    v.setVector( 0, +1, 0 );
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  
  private static boolean isThinBar( RenderPersistData rpd )
  {
    return rpd.graphic.getProperties().get( DGraphic.P_THIN_BAR, true );
  }
  
  private static H3D rebuildPersistData( DefaultChartRenderData d, RenderPersistData rpd )
  {
      H3D p = new H3D();
      rpd.g = p;

      //retrieve the 3 required axis ...
      p.axis = null;
      //get the 3 axis ...
      for( IDItem item = rpd.graphic.getFirstChild(); item!=null; item=item.getNext() )
      {
        if( !(item instanceof DAxis) ) continue;
        p.axis = (DAxis)item; 
        break;
      }

      //axis missed, can't go further
      if( p.axis==null )
      {
        return p;
      }
      
      //thin bar ?
      p.thin_bar = isThinBar( rpd );
      p.bar_size = p.thin_bar ? 0.7f : 1.0f;    
      
      p.axis_style = new IGCDStyle( p.axis, d.scale_ );

      //keep a little space for text on axis...
      //this is completly choosen "as-is"
      int max_font_h  = 0 ;
      d.gc_.setFont( p.axis_style.getFont() );
      max_font_h = d.gc_.getFontMetrics().getHeight();
      max_font_h = 2*max_font_h +6;
 
      //rectangle of drawing area is...
      rpd.ag_rect.setRect( rpd.ag_rect.left()+max_font_h, rpd.ag_rect.top()+max_font_h,
                           rpd.ag_rect.w()-2*max_font_h,  rpd.ag_rect.h()-max_font_h );

      //bounding box for curve to render
      //curve (axis) can use any Object as value...
      Object min = p.axis.getProperties().get( DAxis.P_MIN );
      Object max = p.axis.getProperties().get( DAxis.P_MAX );
      
      if( min==null || max == null )
      {
        MinMax mm = new MinMax( p.axis );
        
        for( int icrv=0; icrv < rpd.curve_styles.length; ++icrv )
        {
          for( IDItem ip=rpd.curve_styles[icrv].curve_.getFirstChild(); ip!=null; ip=ip.getNext() )
          {
            for( IDItem ic=ip.getFirstChild(); ic!=null; ic=ic.getNext() )
            {
              if( !(ic instanceof IDCoord) ) continue;
              IDCoord coord = (IDCoord)ic;
              DAxis axis = coord.getAxis();
              if( axis==p.axis ) mm.update( coord );
            }
          }
        }
        if( min==null ) min = mm.getVMin();
        if( max==null ) max = mm.getVMax();
      }

      //count (max) number of points
//TODO: subject to invalidate RPD!      
      int npts = 0;
      for( int ic=0; ic<rpd.curve_styles.length; ic++ )
      {
        int np = 0;
        //sector doesn't participate in npoints as it's used to Y axis coordinate range
        if( rpd.curve_styles[ic].curve_.isType( DCurve.T_SECTOR ) ) continue;
        for( IDItem it=rpd.curve_styles[ic].curve_.getFirstChild(); it!=null; it=it.getNext() )
        {
          if( it instanceof DPoint ) np++;
        }
        if( np > npts ) npts=np;
      }
      p.npoints = npts;
//TODO:subject to invalidate RPD!      
      p.ncurves = rpd.curve_styles.length;
      p.scale = DefaultChartRenderData.getScale( p.axis, 0,1, min,max );

      p.bbox = new Box3D( 0, p.ncurves - (p.thin_bar ? (1.0f-p.bar_size) :0f), //curves along X axis 
                          0, p.npoints==0 ? 1 : (p.npoints - (p.thin_bar ? (1.0f-p.bar_size) :0f)), //points along Y axis
                          p.scale.getScaleMin(), p.scale.getScaleMax() );

      return  p;
  }

  private abstract static class AFace implements Comparable
  {
    public double pj_zavg; //must be average of projected  point Z coordinate.
    
    public int compareTo(Object o) 
    {
      AFace f =(AFace)o;
      if( pj_zavg==f.pj_zavg ) return 0;
      if( pj_zavg >f.pj_zavg ) return +1;
      return -1;
    }
  }
  //face using 3 or 4 points
  private static class Face extends AFace
  {
    public int ip0, ip1, ip2, ip3 ;
    public DPoint point; //might be point this face is created for
    public Point3D pj[];// might be null or not
    public Point3D pt[];// might be null or not
    public Face() {}
    public Face( int p0, int p1, int p2, int p3, Point3D pj[] )
    {
      ip0=p0; ip1=p1; ip2=p2; ip3=p3;
      this.pj=pj;
      pj_zavg = (pj[p0].getZ()+pj[p1].getZ()+pj[p2].getZ()+pj[p3].getZ())/4;      
    }
    public Face( int p0, int p1, int p2, Point3D pj[] )
    {
      ip0=p0; ip1=p1; ip2=p2; ip3=-1;
      this.pj=pj;
      pj_zavg = (pj[p0].getZ()+pj[p1].getZ()+pj[p2].getZ())/3;      
    }
    public void update( int p0, int p1, int p2 )
    {
      ip0=p0; ip1=p1; ip2=p2; ip3=-1;
      pj_zavg = (ptproj[p0].getZ()+ptproj[p1].getZ()+ptproj[p2].getZ())/3;
    }
    public void update( int p0, int p1, int p2, int p3 )
    {
      ip0=p0; ip1=p1; ip2=p2; ip3=p3;
      pj_zavg = (ptproj[p0].getZ()+ptproj[p1].getZ()+ptproj[p2].getZ()+ptproj[p3].getZ())/4;
    }
  }
  
  
  private static Face faces[] = new Face[] {
      new Face(), new Face(), new Face(), new Face(),
      new Face(), new Face(), new Face(), new Face(),
      new Face(), new Face(),
  };
  
  /** render a 3D bar for histogram curves */
  private static void renderPyramid( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double zbtm, double ztop, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    boolean z_inverted=false;
    if( ztop < zbtm ) {
      z_inverted=true;
      z_incr = !z_incr; //rever Z to have right order for given z top and bottom
    }
    ptvolume[0].setCoord( x,     y,     zbtm );
    ptvolume[1].setCoord( x+w,   y,     zbtm );
    ptvolume[2].setCoord( x+w,   y+w,   zbtm );
    ptvolume[3].setCoord( x,     y+w,   zbtm );
    ptvolume[4].setCoord( x+w/2, y+w/2, ztop );
    for( int i=0; i<5; i++ ) 
    {
      view.projection( ptvolume[i].getX(), ptvolume[i].getY(), ptvolume[i].getZ(), ptproj[i] );
    }
    int rgba = IGCDStyle.GetBackColor( point );
    
    if( z_inverted )
    {
      faces[0].update( 1,0,4); //left
      faces[1].update( 2,1,4); //front
      faces[2].update( 3,2,4); //right
      faces[3].update( 0,3,4); //back
    } else {
      faces[0].update( 0,1,4); //left
      faces[1].update( 1,2,4); //front
      faces[2].update( 2,3,4); //right
      faces[3].update( 3,0,4); //back      
    }
    
    //sort face by zmin...
    Arrays.sort( faces, 0,4 );
    
    Polygon p4 = new Polygon( 4 );
    Polygon p3 = new Polygon( 3 );
    Vector3D v = new Vector3D();
    Point  pt = new Point();
    if( z_incr ) {
      renderBarBottom( point, rgba, d,h3d, ptproj, p4, v, view_vector, z_inverted );        
    } else {
      //no top:renderBarTop( point, rgba, d, ptbar, strength, p, v, view_vector, z_inverted );
    }
    
    for( int i=3; i>=0; i-- )
    {
      renderTriangle( point, rgba, d,h3d, ptvolume,ptproj, p3, v,view_vector, faces[i].ip0,faces[i].ip1,faces[i].ip2 );
    }
    
    if( z_incr ) { //top
     //no top: renderBarTop( point, rgba, d, ptbar, strength, p, v, view_vector, z_inverted );
    } else { //bottom
      renderBarBottom( point, rgba, d,h3d, ptproj, p4, v, view_vector, z_inverted );        
    }
  }  
  
  private static void renderTriangle( DPoint point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D pt[], Point3D pj[], Polygon p, Vector3D v, Vector3D view_vector, int ip0, int ip1, int ip2 )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)pj[ip0].getX(), (int)pj[ip0].getY() );
    p.setPoint( 1, (int)pj[ip1].getX(), (int)pj[ip1].getY() );
    p.setPoint( 2, (int)pj[ip2].getX(), (int)pj[ip2].getY() );   
    v.vectorialProduct( pt[ip1].getX()-pt[ip0].getX(),
                        pt[ip1].getY()-pt[ip0].getY(),
                        pt[ip1].getZ()-pt[ip0].getZ(),
                        pt[ip2].getX()-pt[ip1].getX(),
                        pt[ip2].getY()-pt[ip1].getY(),
                        pt[ip2].getZ()-pt[ip1].getZ() );
    v.normalize();
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  
  private static void renderBar2( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double zbtm, double ztop, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    boolean z_inverted=false;
    if( ztop < zbtm ) {
      z_inverted=true;
      z_incr = !z_incr; //rever Z to have right order for given z top and bottom
    }
    float w2=w/2;
    ptvolume[0].setCoord( x+w2,  y,     zbtm );
    ptvolume[1].setCoord( x+w,   y+w2,  zbtm );
    ptvolume[2].setCoord( x+w2,  y+w,   zbtm );
    ptvolume[3].setCoord( x,     y+w2,  zbtm );
    ptvolume[4].setCoord( x+w2,  y,     ztop );
    ptvolume[5].setCoord( x+w,   y+w2,  ztop );
    ptvolume[6].setCoord( x+w2,  y+w,   ztop );
    ptvolume[7].setCoord( x,     y+w2,  ztop );
    for( int i=0; i<8; i++ ) 
    {
      view.projection( ptvolume[i].getX(), ptvolume[i].getY(), ptvolume[i].getZ(), ptproj[i] );
    }
    int rgba = IGCDStyle.GetBackColor( point );
    float strength=0.4f;
    
    if( z_inverted )
    {
      faces[0].update( 1,0,4,5 ); //left
      faces[1].update( 2,1,5,6 ); //front
      faces[2].update( 3,2,6,7 ); //right
      faces[3].update( 0,3,7,4 ); //back
    } else {
      faces[0].update( 0,1,5,4 ); //left
      faces[1].update( 1,2,6,5 ); //front
      faces[2].update( 2,3,7,6 ); //right
      faces[3].update( 3,0,4,7 ); //back      
    }
    
    //sort face by zmin...
    Arrays.sort( faces, 0,4 );
    
    Polygon p4 = new Polygon( 4 );
    Vector3D v = new Vector3D();
    Point  pt = new Point();
    if( z_incr ) {
      renderBarBottom( point, rgba, d,h3d, ptproj, p4, v, view_vector, z_inverted );        
    } else {
      renderBarTop( point, rgba, d,h3d,  ptproj, p4, v, view_vector, z_inverted );
    }
    
    for( int i=3; i>=0; i-- )
    {
      renderQuad( point, rgba, d,h3d, ptvolume,ptproj, p4, v,view_vector, faces[i].ip0,faces[i].ip1,faces[i].ip2,faces[i].ip3 );
    }
    
    if( z_incr ) { //top
      renderBarTop( point, rgba, d,h3d, ptproj, p4, v, view_vector, z_inverted );
    } else { //bottom
      renderBarBottom( point, rgba, d,h3d, ptproj, p4, v, view_vector, z_inverted );        
    }
  }  
  private static void renderQuad( DPoint point, int rgba, DefaultChartRenderData d,H3D h3d, Point3D pt[], Point3D pj[], Polygon p, Vector3D v, Vector3D view_vector, int ip0, int ip1, int ip2,int ip3 )
    throws DefaultRenderChartLocation
  {
    p.setPoint( 0, (int)pj[ip0].getX(), (int)pj[ip0].getY() );
    p.setPoint( 1, (int)pj[ip1].getX(), (int)pj[ip1].getY() );
    p.setPoint( 2, (int)pj[ip2].getX(), (int)pj[ip2].getY() );   
    p.setPoint( 3, (int)pj[ip3].getX(), (int)pj[ip3].getY() );
    //normal vector of face get from 3 first points (must be representative of face !)
    v.vectorialProduct( pt[ip1].getX()-pt[ip0].getX(),
                        pt[ip1].getY()-pt[ip0].getY(),
                        pt[ip1].getZ()-pt[ip0].getZ(),
                        pt[ip2].getX()-pt[ip1].getX(),
                        pt[ip2].getY()-pt[ip1].getY(),
                        pt[ip2].getZ()-pt[ip1].getZ() );
    v.normalize();
    renderPolygon( point, d,h3d, rgba, p, v, view_vector );
  }
  
  private static void renderPoint( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double z, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    float w4 = w/4;
    float x1=x+w/2-w4, x2=x+w/2+w4;
    float y1=y+w/2-w4, y2=y+w/2+w4;
    double z1=z-w4, z2=z+w4;
    ptvolume[0].setCoord( x1, y1, z );
    ptvolume[1].setCoord( x2, y1, z );
    ptvolume[2].setCoord( x2, y2, z );
    ptvolume[3].setCoord( x1, y2, z );
    ptvolume[4].setCoord( x+w/2, y+w/2, z2 );
    ptvolume[5].setCoord( x+w/2, y+w/2, z1 );
    for( int i=0; i<6; i++ ) 
    {
      view.projection( ptvolume[i], ptproj[i] );
    }

    int rgba = IGCDStyle.GetBackColor( point );
    
    faces[0].update( 0,1,4); //left top
    faces[1].update( 1,2,4); //front top
    faces[2].update( 2,3,4); //right top
    faces[3].update( 3,0,4); //back top 
    faces[4].update( 1,0,5); //left bottom
    faces[5].update( 2,1,5); //front bottom
    faces[6].update( 3,2,5); //right bottom
    faces[7].update( 0,3,5); //back bottom 
    
    //sort face by zmin...
    Arrays.sort( faces, 0,8 );
    
    Polygon p3 = new Polygon( 3 );
    Vector3D v = new Vector3D();
    Point  pt = new Point();
    
    for( int i=7; i>=0; i-- )
    {
      renderTriangle( point, rgba, d,h3d, ptvolume,ptproj, p3, v,view_vector, faces[i].ip0,faces[i].ip1,faces[i].ip2 );
    }
  } 
  
  private static void renderOctogon( DPoint point, DefaultChartRenderData d,H3D h3d, View3D view, int x, int y, double zbtm, double ztop, boolean x_incr, boolean y_incr, boolean z_incr, float w, Vector3D view_vector )
    throws DefaultRenderChartLocation
  {
    float w2 = w/2;
    float xc=x+w2;
    float yc=y+w2;
    double zn = zbtm, zx=ztop;
    if( ztop < zbtm ) { 
      zn = ztop; zx=zbtm;
    }

    double a=0.0;
    for( int i=0; i<8; i++, a+=Radian._PI4 )
    {
      double c = w2*Math.cos( a ); 
      double s = w2*Math.sin( a );
      ptvolume[i].setCoord( xc+c, yc+s, zn );
      ptvolume[i+8].setCoord( xc+c, yc+s, zx );
    }    
    for( int i=0; i<16; i++ ) 
    {
      view.projection( ptvolume[i], ptproj[i] );
    }

    int rgba = IGCDStyle.GetBackColor( point );
    
    faces[0].update( 0,1,9,8); 
   faces[1].update( 1,2,10,9); 
    faces[2].update( 2,3,11,10);
    faces[3].update( 3,4,12,11); 
    faces[4].update( 4,5,13,12);
    faces[5].update( 5,6,14,13);
    faces[6].update( 6,7,15,14);
    faces[7].update( 7,0,8,15);
    
    //sort face by zmin...
    Arrays.sort( faces, 0,8 );
    
    Polygon p4 = new Polygon( 4 );
    Polygon p8 = new Polygon( 8 );
    Vector3D v = new Vector3D();
    Point  pt = new Point();
    
    if( z_incr ) { //bottom
      for( int i=0; i<8; ++i ) {
        p8.setPoint( i, (int)ptproj[i].getX(), (int)ptproj[i].getY() );
      }
      v.setVector( 0, 0, -1 );
      renderPolygon( point, d,h3d, rgba, p8, v, view_vector );
    } else { //top
      for( int i=0; i<8; ++i ) {
        p8.setPoint( i, (int)ptproj[i+8].getX(), (int)ptproj[i+8].getY() );
      }
      v.setVector( 0, 0, +1 );
      renderPolygon( point, d,h3d, rgba, p8, v, view_vector );
    }

    for( int i=7; i>=0; i-- )
    {
      renderQuad( point, rgba, d,h3d, ptvolume,ptproj, p4, v,view_vector, faces[i].ip0,faces[i].ip1,faces[i].ip2,faces[i].ip3 );
    }
    
    if( z_incr ) {
      for( int i=0; i<8; ++i ) {
        p8.setPoint( i, (int)ptproj[i+8].getX(), (int)ptproj[i+8].getY() );
      }
      v.setVector( 0, 0, +1 );
      renderPolygon( point, d,h3d, rgba, p8, v, view_vector );
    } else { //bottom
      for( int i=0; i<8; ++i ) {
        p8.setPoint( i, (int)ptproj[i].getX(), (int)ptproj[i].getY() );
      }
      v.setVector( 0, 0, -1 );
      renderPolygon( point, d,h3d, rgba, p8, v, view_vector );
    }    
  }
}
