/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DefaultChartRenderXY.java,v 1.8 2008/05/23 14:11:50 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;

import org.eclipse.tptp.platform.report.core.internal.*;
import org.eclipse.tptp.platform.report.drawutil.internal.DrawUtilIGC;
import org.eclipse.tptp.platform.report.drawutil.internal.IGCDStyle;
import org.eclipse.tptp.platform.report.drawutil.internal.ISymbol;
import org.eclipse.tptp.platform.report.igc.internal.IPolygon;
import org.eclipse.tptp.platform.report.igc.internal.IShape;
import org.eclipse.tptp.platform.report.igc.util.internal.LineStylePen;
import org.eclipse.tptp.platform.report.igc.util.internal.Polygon;
import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;
import org.eclipse.tptp.platform.report.igc.util.internal.SolidBrush;
import org.eclipse.tptp.platform.report.tools.internal.IDisposable;
import org.eclipse.tptp.platform.report.tools.internal.VDouble;


/**
 * Default Chart Render for T_XY graphic type and the family of associated curve type...
 * @author ademuyser
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DefaultChartRenderXY
{
  private static class Bubble implements Comparable
  {
    IDCoord cx;
    IDCoord cy;
    IDCoord cr;
    DPoint  point;
    
    public int compareTo(Object o) {
      Bubble b2 = (Bubble)o;
      return ((Comparable)cr.getValue(null)).compareTo( b2.cr.getValue(null) );
    }
    
  }
  private static class BubbleData 
  {
    DAxis  r_axis; //axis used for radius (first seen, first taken)
    Object r_min; //rebuildRPD min radius seen
    Object r_max; //rebuildRPD max radius seen
    IScale r_scale;
    boolean use_y2;
    Bubble bubbles[]; //array of valid bubbles sorted by decreasing radius
  }
  /** variant persist data part for XY graphic */
  private static class XY implements IDisposable
  {
    //note: y2 is optional
    DAxis x_axis, y_axis, y2_axis;
    IGCDStyle x_axis_style, y_axis_style, y2_axis_style;
    IScale x_scale, y_scale, y2_scale;
    Insets xam,yam, y2am;
    
    public void dispose()
    {
/*TODO:suppress IDisposable      
      if( x_axis_style  !=null )  x_axis_style.dispose();
      if( y_axis_style  !=null )  y_axis_style.dispose();
      if( y2_axis_style !=null ) y2_axis_style.dispose();
*/      
    }
    
    //stacked line/area along primary y axis
    StackedArea y1_top_area;
    //stacked line/area along secondy y axis
    StackedArea y2_top_area;
    
    //independant bar width used by curves like VBAR, CANDLESTICK,...
    int ibar_width;
    
    int Rmin, Rmax; //min/max radius (coming from rpd.ag_rect) and common to all bubbles curves
    BubbleData bubbles_data[]; //null if none, [rpd.curve_styles.length] otherwise
  }
  
  private static class SAPoint
  {
    public DPoint point;
    public IDCoord x_coord, y_coord;
    public Object y_value; 
  }
  private static class StackedArea
  {
    /** Stacked area below this one */
    StackedArea below;
    /** index in ccurve variable (ie access to DCurve and style)*/
    int index; 
    /** points of curve are SAPoint*/
    ArrayList points;  
    /** y min/max after cumulation */
    double ymin,ymax;
    /** true if associated curve is T_STACKED_AREA_xxx, false otherwise */
    boolean area;
  }
  
  //axis titles alignment (not configurable)
  private static final int aln_x_title  = IDAlignment.HCENTER; //LEFT/RIGHT
  private static final int aln_y_title  = IDAlignment.ROTCCW90; //VERTICAL a TOP/BOTTOM
  private static final int aln_y2_title = IDAlignment.ROTCW90; //VERTICAL a TOP/BOTTOM
    
  /**
   * Render XY graphic type.
   */
  public static void render( DefaultChartRenderData d, RenderPersistData rpd )
     throws DefaultRenderChartLocation
  {
    XY p = null;
    if( rpd.g instanceof XY )
    {
      p = (XY)rpd.g;
    }
    
    boolean rebuild = (p==null);
    
    //need rebuild?
    if( rebuild )
    {
      p = rebuildRPD( d, rpd );
    } 

    //now can render, remember x,y are mandatory, y2 is optional..
    if ( p.x_axis==null || p.y_axis==null )
    {
      if( d.drawing() )
      {
        String txt;
        if( p.x_axis==null )
        {
          txt = "Error: missing 'x' axis";
        } else {
          txt = "Error: missing 'y' axis";
        }
        d.error( txt, rpd );
      }
      return ;
    }
    
    boolean have_y2 = (p.y2_axis!=null);

    if ( !p.x_scale.isValid() || !p.y_scale.isValid() || (have_y2 && !p.y2_scale.isValid() ) )
    {
      if( d.drawing() ) 
      {
        String n = "";
        if( !p.x_scale.isValid() ) n="X";
        if( !p.y_scale.isValid() ) n=n+((n.length()>0)?",Y":"Y");
        if( have_y2 && !p.y2_scale.isValid() ) n=n+((n.length()>0)?",Y2":"Y2");
        d.error( "Error: can't draw curves, invalid axis:"+n, rpd );
      }
      return;
    }
    
    //basis for histogram bar (xy curve type might be bars)
    int y_bar_zero  = (int)p.y_scale.getScaleForBar();
    int y2_bar_zero = have_y2 ? (int)p.y2_scale.getScaleForBar() : 0 ;

    //clipping (as P_MIN/P_MAX of axis could let curves outside drawing zone
    IShape old_clip=null;
    if(d.drawing()) {
      old_clip = d.gc_.getClipping();
      d.gc_.setClipping( rpd.ag_rect );
    }
    
    //fill surface of area before axis, lines of axis will be draw upon filled area
    boolean stacked_area_done=false;
    if( rpd.graphic.getProperties().get( DGraphic.P_BACK_AREA, false ) )
    {
      if( p.y1_top_area!=null )
      {
        renderStackedAreas( d, rpd, p.y1_top_area, y_bar_zero, p.y_scale, p.x_scale );
      }
      if( have_y2 && p.y2_top_area !=null )
      {
        renderStackedAreas( d, rpd, p.y2_top_area, y2_bar_zero, p.y2_scale, p.x_scale );
      }
      stacked_area_done=true;
      renderAreas( p, d, rpd, y_bar_zero, y2_bar_zero );
    }
    
    //do not clip axis rendering ... they are outside clip rect!
    if(d.drawing()) d.gc_.setClipping( old_clip );    
    d.renderXAxis( false, true, p.x_axis, p.x_scale, rpd.ag_rect, p.xam, p.x_axis_style, aln_x_title );
    d.renderYAxis( true, true, true, p.y_axis, p.y_scale, rpd.ag_rect, p.yam, p.y_axis_style, aln_y_title, 0,0 );
    if( have_y2 )
    {
      d.renderYAxis( true, true, false, p.y2_axis, p.y2_scale, rpd.ag_rect, p.y2am, p.y2_axis_style, aln_y2_title, 0,0 );
    }
    
    if(d.drawing()) 
        d.gc_.setClipping( rpd.ag_rect );
    d.renderMarkerLine(true, false, p.y_axis, rpd.ag_rect, p.y_scale);

    if( d.drawing() )
    {
      d.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
      int btom = rpd.ag_rect.bottom();
      d.gc_.drawLine( rpd.ag_rect.left(), btom, rpd.ag_rect.right(), btom ); // X axis base line
      d.gc_.drawLine( rpd.ag_rect.left(), rpd.ag_rect.top(), rpd.ag_rect.left(),  btom ); // Y axis base line
      if( have_y2 )
      {
        d.gc_.drawLine( rpd.ag_rect.right(), rpd.ag_rect.top(), rpd.ag_rect.right(),  btom ); // Y2 axis base line
      }
    }
    //if(d.drawing()) d.gc_.setClipping( rpd.ag_rect );

    CurveStyle ccurve=null;
    Rect rp=new Rect();
    Rect r_bar=new Rect();
    
    int x_zero = (int)p.x_scale.getScaleForBar();
    
    int symbol_size = (int)(0.025*Math.min(rpd.ag_rect.w(),rpd.ag_rect.h()));

    //limit symbol size from 2 to 5
    symbol_size = Math.max( d.dpiX(2), Math.min( symbol_size, d.dpiX(5) ));
    int symbol_size2 = symbol_size<<1;//*2
    ISymbol symbol=null; //for point curve.

    //try to allocate one object value for each axis, if it's supported only two 'new' for all values.
    Object x_obj = DefaultChartRenderData.getNewInstanceValue( p.x_axis );    
    Object y_obj = DefaultChartRenderData.getNewInstanceValue( p.y_axis );
    Object y2_obj= have_y2 ? DefaultChartRenderData.getNewInstanceValue( p.y2_axis ) :null;
    
    //stacked area curves, as they are filled must be draw first.
    if( p.y1_top_area!=null )
    {
      if( !stacked_area_done )
      {
        renderStackedAreas( d, rpd, p.y1_top_area, y_bar_zero, p.y_scale, p.x_scale );
      }
      //lines and symbol...
      renderStackedLineAndSymbol( d, rpd, p.y1_top_area, p.y_scale, p.x_scale, symbol_size, symbol_size2 );
    }
    if( have_y2 && p.y2_top_area!=null )
    {
      if( !stacked_area_done )
      {
        renderStackedAreas( d, rpd, p.y2_top_area, y2_bar_zero, p.y2_scale, p.x_scale );
      }
      //lines and symbol...
      renderStackedLineAndSymbol( d, rpd, p.y2_top_area, p.y2_scale, p.x_scale, symbol_size, symbol_size2 );
    }
    if( !stacked_area_done )
    {
      renderAreas( p, d, rpd, y_bar_zero, y2_bar_zero );
    }
    
    //for locating mode, curve are checking in reverse order to handle covering.
    boolean reverse = d.locating();
    int ic_inc = reverse ? -1 : +1;
    int ic_size = rpd.curve_styles.length;    
    int rg_x = rpd.ag_rect.left();
    
    for( int ic=reverse ? ic_size-1 : 0;  reverse? (ic>=0) : (ic<ic_size);  ic += ic_inc )
    {
      ccurve = rpd.curve_styles[ic];      
      boolean first=true;
      int last_x = rg_x;
      int last_y = -1;      

      DPoint point=null;
      symbol = ccurve.symbol_;
      int curve_type = 0; //see above
           if( ccurve.curve_.isType( DCurve.T_LINE   ) ) curve_type =0x1;
      else if( ccurve.curve_.isType( DCurve.T_POINTS ) ) curve_type =0x2;
      else if( ccurve.curve_.isType( DCurve.T_LINE_POINTS ) ) curve_type =0x3;
      else if( ccurve.curve_.isType( DCurve.T_BAR    ) ) curve_type =0x4; //exclusive
      else if ( ccurve.curve_.isType( DCurve.T_AREA_LINE ) ) curve_type=0x9;
      else if ( ccurve.curve_.isType( DCurve.T_AREA_LINE_POINTS ) ) curve_type=0xB;
      // stacked area area already done.
      else if ( ccurve.curve_.isType( DCurve.T_STACKED_AREA_LINE ) ) continue;
      else if ( ccurve.curve_.isType( DCurve.T_STACKED_AREA_LINE_POINTS ) ) continue;
      else if ( ccurve.curve_.isType( DCurve.T_STACKED_LINE ) ) continue;
      else if ( ccurve.curve_.isType( DCurve.T_STACKED_LINE_POINTS ) ) continue;
      
      else if ( ccurve.curve_.isType( DCurve.T_CANDLESTICK )) {
        renderCandleStickCurve( ccurve, p, d, rpd, false );
        continue;
      } else if ( ccurve.curve_.isType( DCurve.T_CANDLESTICK_LINE )) {
        renderCandleStickCurve( ccurve, p, d, rpd, true );
        continue;
      }
      else if ( ccurve.curve_.isType( DCurve.T_VBAR )) {
        renderVBar( ccurve, p, d, rpd, y_bar_zero,y2_bar_zero );
        continue;
      }
      else if ( ccurve.curve_.isType( DCurve.T_BUBBLE )) {
        renderBubble( ic, ccurve, p, d, rpd, false );
        continue;
      }
      else if ( ccurve.curve_.isType( DCurve.T_CIRCLE)) {
        renderBubble( ic, ccurve, p, d, rpd, true );
        continue;
      }

      DPoint prev_point = null;
      for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
      {
        if ( !(item instanceof DPoint) ) continue;
        
        point = (DPoint)item;
        
        IDCoord cx = null;
        IDCoord cy = null;
        IDCoord cy2= null;
        for( IDItem itc=point.getFirstChild(); itc!=null; itc=itc.getNext() )
        {
          if(!(itc instanceof IDCoord)) continue;
          IDCoord coord = (IDCoord)itc;
          if( cx==null && coord.getAxis()==p.x_axis) 
          {
            x_obj = coord.getValue( x_obj );
            if( x_obj!=null ) cx = coord;
          }
          else if ( cy ==null && coord.getAxis()==p.y_axis)
          {
            y_obj = coord.getValue( y_obj );
            if( y_obj!=null ) cy = coord;
          }
          else if ( have_y2 && cy2 ==null && coord.getAxis()==p.y2_axis)
          {
            y2_obj = coord.getValue( y2_obj );
            if( y2_obj!=null ) cy2 = coord;
          }
        }
        //cx and one of cy must be there...
        if ( cx==null || ( cy==null && cy2==null ) ) continue;
        
        int px = (int)p.x_scale.toScale( x_obj );
        int py = cy2==null ? (int)p.y_scale.toScale( y_obj ) : (int)p.y2_scale.toScale( y2_obj );
        if ( curve_type == 0x4 ) // DCurve.T_BAR
        {   
          int dx = px-x_zero;
          int ybz = cy2==null ? y_bar_zero : y2_bar_zero;
          r_bar.setRect( last_x, ybz , dx, py-ybz );
          r_bar.normalize();
          if(d.drawing() )
          {
            d.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
            d.gc_.setBrush( new SolidBrush( ccurve.style_.getBack() ) );
            d.gc_.fillRect( r_bar.x(), r_bar.y(), r_bar.w(), r_bar.h() );
            d.gc_.drawRect( r_bar.x(), r_bar.y(), r_bar.w(), r_bar.h() );
          }
          else //locating
          {
            if( r_bar.contains( d.lx_, d.ly_ ) )
            {
              throw new DefaultRenderChartLocation( DLocated.CurveBar, point, r_bar );
            }
          }
          last_x += dx; //last_y not used here.
        }
        else 
        {
          if( (curve_type&0x1)== 0x1 ) // lines ...
          {
            if(first) //line need at least two points
            {
              first=false;
            }
            else
            {
              if( d.drawing() )
              {
                //line-only use back is line color, area use fore ground color
                if( (curve_type&0x8)==0)
                {
                  d.gc_.setPen( new LineStylePen( ccurve.style_.getBack() ) );
                } else {
                  d.gc_.setPen( new LineStylePen( ccurve.style_.getFore() ) );
                }
                d.gc_.drawLine( last_x, last_y, px,py );
              }
              else //locating
              {
                if( DrawUtilIGC.segmentContains( last_x, last_y, px, py, d.lx_, d.ly_ ) )
                {
                  DefaultRenderChartLocation loc= new DefaultRenderChartLocation( DLocated.CurveLine, point, new Rect(last_x,last_y,px-last_x,py-last_y));
                  loc.setPreviousPoint( prev_point );
                  throw loc;
                }
              }
            } 
          }
          if ( (prev_point!=null) && (curve_type&0x2) == 0x2 ) // DCurve.T_POINTS
          {
            //symbol is contained in a rectangle, use last point otherwise next line segment
            //will be drawn over symbol (to bad).
            rp.setRect( last_x-symbol_size, last_y-symbol_size, symbol_size2, symbol_size2 );
            if( d.drawing() )
            {
              d.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
              d.gc_.setBrush( new SolidBrush( ccurve.style_.getBack() ));
              //#bug: 104513
              //IShape clip = d.gc_.setClipping( null );
              IShape clip = d.gc_.getClipping();
              Rect rclip = new Rect(clip.getBounds());
              rclip.setHeight(rclip.getH()+ rp.h()*2);
              rclip.setY(rclip.getY()- rp.h());
              d.gc_.setClipping( rclip );
              symbol.draw( d.gc_, rp );
              d.gc_.setClipping( rpd.ag_rect );
            }
            else //locating
            {
              if( symbol.contains( d.gc_, rp, d.lx_, d.ly_ ) )
              {
                throw new DefaultRenderChartLocation( DLocated.CurvePoint, prev_point, rp );
              }
            }
          }//symbol
          last_x=px;
          last_y=py;
        }//curve type not BAR
        
        prev_point=point;
        
     }//for points
     
     //do not forget last symbol
     if ( (curve_type&0x2) == 0x2 ) // DCurve.T_POINTS
     {
       //symbol is contained in a rectangle
       rp.setRect( last_x-symbol_size, last_y-symbol_size, symbol_size2, symbol_size2 );
       if( d.drawing() )
       {
         d.gc_.setPen( new LineStylePen( RGBA.BLACK ) );
         d.gc_.setBrush( new SolidBrush( ccurve.style_.getBack() ));
         symbol.draw( d.gc_, rp );
       }
       else //locating
       {
         if( symbol.contains( d.gc_, rp, d.lx_, d.ly_ ) )
         {
           throw new DefaultRenderChartLocation( DLocated.CurvePoint, point, rp );
         }
       }
     }
 
    }//for
    if(d.drawing()) d.gc_.setClipping( old_clip );
    
    //time to locate on axis:
    d.renderXAxis( false, false, p.x_axis, p.x_scale, rpd.ag_rect, p.xam, p.x_axis_style, aln_x_title );
    d.renderYAxis( true, false, true, p.y_axis, p.y_scale, rpd.ag_rect, p.yam, p.y_axis_style, aln_y_title, 0,0 );
    if( have_y2 )
    {
      d.renderYAxis( true, false, false, p.y2_axis, p.y2_scale, rpd.ag_rect, p.y2am, p.y2_axis_style, aln_y2_title, 0,0 );
    }
    
    // time to draw the marker line
    d.renderMarkerLine(true, true, p.y_axis, rpd.ag_rect, p.y_scale);
  }
  
  /** used in rebuildRPD data... */
  private static int computeBarWidth( CurveStyle ccurve, XY p, DefaultChartRenderData d, RenderPersistData rpd  )
  {
    //heuristic: to know the width used for candlestick, and in order to candlestick won't
    //overlap each other, we must iterate through the points searching the minimal width
    //between two consecutive candle...
    //Yes, this is a candidate for RPD (in other word computed once)...
    //TBD: need to take care of axis bounds and closer point to compute bar width too ?
    int num_points=0; 
    int last_x=0;
    boolean have_last_x=false;
    int bar_width=rpd.graphic_rect.getW();
    Object x_obj=null;
    for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
    {
      if ( !(item instanceof DPoint) ) continue;
      num_points++;
      DPoint point = (DPoint)item;
        
     // IDCoord cx = null;
      for( IDItem itc=point.getFirstChild(); itc!=null; itc=itc.getNext() )
      {
        if(!(itc instanceof IDCoord)) continue;
        IDCoord coord = (IDCoord)itc;
        if( coord.getAxis()==p.x_axis) 
        {
          x_obj = coord.getValue( x_obj );
          if( x_obj!=null )
          {
            int x = (int)p.x_scale.toScale( x_obj );
            if( have_last_x )
            {
              //means that graphically consecutive value must be consecutive in memory too.
              int w = Math.abs( last_x-x );
              if( w>0 && bar_width > w ) bar_width=w;
            } 
            last_x = x;
            have_last_x=true;
          }
          break;
        }
      }
    }
    if( num_points==0 ) return -1;
    //but the X axis bounds must be checked too..
    bar_width -= d.dpiX( 2 ); //spacing between candles
    //but we need to keep candle visible... in this case two candlestick can overlap.
    bar_width = Math.max( bar_width, d.dpiX(3) );
    
    return bar_width;
  }
  
  /**
   * Render Candle Stick curves.
   */
  private static void renderCandleStickCurve( CurveStyle ccurve, XY p, DefaultChartRenderData d, RenderPersistData rpd, boolean render_lines )
     throws DefaultRenderChartLocation
  {
    int candle_width=p.ibar_width;
    int candle_width_2 = candle_width/2;    
    
    int fore = 0, back = 0;
    int raising=0, falling=0;
    if( d.drawing() )
    {
      fore = ccurve.style_.getFore();
      back = ccurve.style_.getBack();
      if( render_lines )
      {
        //use only line of background color as standard XY curves
        d.gc_.setPen( new LineStylePen(back) );
        raising = falling = back;
      } else {
        d.gc_.setPen( new LineStylePen(fore) );
        d.gc_.setBrush( new SolidBrush(back) );
        raising = back;
        falling = fore;
      }
      //check P_CS_FALL_COLOr property:
      Object prop = ccurve.curve_.getProperties().get( DCurve.P_CS_FALL_COLOR );
      if( prop instanceof IDColor )
      {        
        falling = IGCDStyle.GetRGBA( (IDColor)prop );
      }
    }
    Object y_obj=null;
    Object x_obj=null;
    Rect rect = new Rect();
    int xy[] = new int[8];
    SolidBrush brush = new SolidBrush( RGBA.WHITE );
    LineStylePen lsp = new LineStylePen( RGBA.BLACK );
    Polygon poly = new Polygon((IPolygon)null);
    for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
    {
      if ( !(item instanceof DPoint) ) continue;
        
      DPoint point = (DPoint)item;
        
      IDCoord cx = null;
    //  IDCoord cy_open = null;
    //  IDCoord cy_high = null;
    //  IDCoord cy_low  = null;
    //  IDCoord cy_close= null;
      int x=0, y_open=0, y_high=0, y_low=0, y_close=0;
      int y_curr=0;
      for( IDItem itc=point.getFirstChild(); itc!=null; itc=itc.getNext() )
      {
        if(!(itc instanceof IDCoord)) continue;
        IDCoord coord = (IDCoord)itc;
        if( cx==null && coord.getAxis()==p.x_axis) 
        {
          x_obj = coord.getValue( x_obj );
          if( x_obj!=null )
          {
            cx = coord;
            x = (int)p.x_scale.toScale( x_obj );
          }
        }
        else 
        {
          boolean good_y_axis=false;
          boolean y2_axis=false;
          if( coord.getAxis()==p.y_axis ) 
          {
            good_y_axis=true;
          }
          else if (  coord.getAxis()==p.y2_axis )
          {
            good_y_axis=true;
            y2_axis=true;
          }
          if( good_y_axis )
          {
            y_obj = coord.getValue( y_obj );
            int py = y2_axis ? (int)p.y2_scale.toScale( y_obj ) : (int)p.y_scale.toScale( y_obj );

            switch( y_curr )
            {
              case 0: /*cy_open = coord;*/ y_open = py; y_curr=1; break;
              case 1: /*cy_high = coord;*/ y_high = py; y_curr=2; break;
              case 2: /*cy_low  = coord;*/ y_low  = py; y_curr=3; break;
              case 3: /*cy_close= coord;*/ y_close= py; y_curr=4; break;
            }
          }
        }
      }//for coord
      
      //valid ?
      if( cx==null || y_curr!=4 ) continue;
      
      boolean v_falling = y_close > y_open ; //remember Y are reverse: Val.close < Val.open

      //render type is lines
      if( render_lines )
      {
        if( d.drawing() )
        {
          lsp.setRGBA( v_falling ? falling : raising );
          d.gc_.setPen( lsp );
          d.gc_.drawLine( x, y_high, x, y_low );
          d.gc_.drawLine( x-candle_width_2, y_open, x, y_open );
          d.gc_.drawLine( x, y_close, x+candle_width_2, y_close);
        }
        else //locate
        {
          rect.setRect( x-candle_width_2, y_low, candle_width, y_high-y_low );
          if( rect.contains( d.lx_, d.ly_ ) )
          {
            throw new DefaultRenderChartLocation( DLocated.CurvePoint, point, rect );
          }        
        }
      }
      else //render type is candle stick
      {
        int ymh = 0, yml=0;
        if( v_falling )
        {
          ymh = y_open ; yml = y_close;
        } else {
          ymh = y_close; yml = y_open ;
        }
//System.out.println("CANDLE x="+x+" yo="+y_open+" yh="+y_high+" yl="+y_low+" yc="+y_close+" filled="+filled+" ymh-yml="+(ymh-yml));
        if( d.drawing() )
        {
          xy[0] = xy[2] = x-candle_width_2;
          xy[7] = xy[1] = yml;
          xy[4] = xy[6] = x+candle_width_2;
          xy[3] = xy[5] = ymh;
          if( v_falling )
          {
            brush.setRGBA( falling );            
//TBD: or back ? if back color is light high/low value won't be readable.
            lsp.setRGBA( fore );
          } else {
            brush.setRGBA( raising );
            lsp.setRGBA( fore );
          }
          poly.setPoints( xy );
          d.gc_.setBrush( brush );
          d.gc_.setPen( lsp ); 
          d.gc_.fillPoly( poly );
          d.gc_.drawPoly( poly );
          if( y_high < ymh ) d.gc_.drawLine( x, ymh, x, y_high );
          if( y_low  > yml ) d.gc_.drawLine( x, yml, x, y_low  );
        }
        else //locating
        {
          rect.setRect( x-candle_width_2, y_low, candle_width, y_high-y_low );
          if( rect.contains( d.lx_, d.ly_ ) )
          {
            throw new DefaultRenderChartLocation( DLocated.CurvePoint, point, rect );
          }
        }
      }
    }//for points
    
  }
  
  /**
   * Render VBar curves
   */
  private static void renderVBar( CurveStyle ccurve, XY p, DefaultChartRenderData d, RenderPersistData rpd, int y_bar_zero, int y2_bar_zero)
     throws DefaultRenderChartLocation
  {
    int bar_width=p.ibar_width;
    int bar_width_2 = bar_width/2;    
    
    int /*fore = 0,*/ back = 0;
    SolidBrush brush = new SolidBrush( RGBA.WHITE ); 
    if( d.drawing() )
    {
      //fore = ccurve.style_.getFore();
      back = ccurve.style_.getBack();
      //prefer to use black: d.gc_.setForeground( fore );
      d.gc_.setPen( new LineStylePen(RGBA.BLACK) );
      brush.setRGBA( back );
      d.gc_.setBrush( brush );
    }
    Object y_obj=null;
    Object x_obj=null;
    Rect rect = new Rect();
    for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
    {
      if ( !(item instanceof DPoint) ) continue;
        
      DPoint point = (DPoint)item;
        
      IDCoord cx = null;
      IDCoord cy = null;
      boolean y2_axis=false;
      int x=0, y=0;
      for( IDItem itc=point.getFirstChild(); itc!=null; itc=itc.getNext() )
      {
        if(!(itc instanceof IDCoord)) continue;
        IDCoord coord = (IDCoord)itc;
        if( cx==null && coord.getAxis()==p.x_axis) 
        {
          x_obj = coord.getValue( x_obj );
          if( x_obj!=null )
          {
            cx = coord;
            x = (int)p.x_scale.toScale( x_obj );
          }
        }
        else 
        {
          boolean good_y_axis=false;
          if( coord.getAxis()==p.y_axis ) 
          {
            good_y_axis=true;
          }
          else if (  coord.getAxis()==p.y2_axis )
          {
            good_y_axis=true;
            y2_axis=true;
          }
          if( good_y_axis )
          {
            y_obj = coord.getValue( y_obj );
            int py = y2_axis ? (int)p.y2_scale.toScale( y_obj ) : (int)p.y_scale.toScale( y_obj );

            cy = coord; 
             y = py;
          }
        }
      }//for coord
      
      //valid ?
      if( cx==null || cy==null ) continue;
      
      //render bar 
      int ybz = y2_axis? y2_bar_zero : y_bar_zero;
      rect.setRect( x-bar_width_2, ybz , bar_width, y-ybz );
      rect.normalize();
      if(d.drawing() )
      {
        brush.setRGBA( ccurve.style_.getBack() );
        d.gc_.setBrush( brush );
        d.gc_.fillRect( rect.x(), rect.y(), rect.w(), rect.h() );
        d.gc_.drawRect( rect.x(), rect.y(), rect.w(), rect.h() );
      }
      else //locating
      {
        if( rect.contains( d.lx_, d.ly_ ) )
        {
          throw new DefaultRenderChartLocation( DLocated.CurveBar, point, rect );
        }
      }
    }//for points
  }
  
  /**
   * Render area of T_AREA_LINE and T_AREA_LINE_POINTS curves.
   */
  private static void renderAreas( XY p, DefaultChartRenderData d, RenderPersistData rpd, int y_bar_zero, int y2_bar_zero )
  {
    //as only opaque colors area used, the surface is from curve to y_bar_zero value.
    //this means, curve can hide other curve:
    //dear sir SWT, could you implement transparency please ...

    //no location done for area... one day perhaps..
    if( d.locating() ) return ;
    
    boolean reverse = d.locating();
    int ic_inc = reverse ? -1 : +1;
    int ic_size = rpd.curve_styles.length;    
    //int rg_x = rpd.ag_rect.left();
    
    int pts1[] = new int[100]; //xyxyxy...
    SolidBrush brush = new SolidBrush( RGBA.WHITE );
    Polygon polygon = new Polygon();
    for( int ic=reverse ? ic_size-1 : 0;  reverse? (ic>=0) : (ic<ic_size);  ic += ic_inc )
    {
      CurveStyle cs = rpd.curve_styles[ic]; 
      if( !( cs.curve_.isType( DCurve.T_AREA_LINE )
           ||cs.curve_.isType( DCurve.T_AREA_LINE_POINTS )) ) continue ;

      int pts1_top=0;
      //build a polygon from points.
      boolean known_y_axis=false;
      boolean primary_y_axis=true;
      
      for( IDItem ip= cs.curve_.getFirstChild(); ip!=null; ip=ip.getNext() )
      {        
        if(!(ip instanceof DPoint)) continue;
        //DPoint point=(DPoint)ip;
        IDCoord xc=null, yc=null;
        int px=0, py=0; //, init=0;
        for( IDItem id=ip.getFirstChild(); id!=null; id=id.getNext())
        {
          if(!(id instanceof IDCoord)) continue;
          IDCoord cd = (IDCoord)id;
          DAxis a=cd.getAxis();
          if ( a==p.x_axis )
          {
            xc=cd;
            px = (int)p.x_scale.toScale( cd.getValue(null) );
          }
          else if ( a==p.y_axis && ( !known_y_axis || primary_y_axis )) 
          {
            yc=cd; known_y_axis=true; primary_y_axis=true;
            py = (int)p.y_scale.toScale( cd.getValue(null) );
          }
          else if ( a==p.y2_axis && (!known_y_axis ||!primary_y_axis))
          {
            yc=cd; known_y_axis=true; primary_y_axis=false;
            py = (int)p.y2_scale.toScale( cd.getValue(null) );
          }
        }
        //valid points ?
        if( xc!=null && yc!=null )
        {
          if( pts1_top+2 >= pts1.length )
          {
            int t[] = new int[pts1.length*2];
            System.arraycopy( pts1,0, t,0, pts1_top );
            pts1 = t;
          }
          pts1[pts1_top++] = px;
          pts1[pts1_top++] = py;        
        }
      }
        
      // I need two points to define an area to fill
      if( pts1_top<4 ) continue;
      
      //SWT need exact array length, and we have to add two points
      int poly[] = new int[pts1_top+4];
      poly[0] = pts1[0];
      poly[1] = primary_y_axis ? y_bar_zero : y2_bar_zero;
      System.arraycopy( pts1,0, poly, 2, pts1_top );
      poly[pts1_top+2] = pts1[pts1_top-2];
      poly[pts1_top+3] = primary_y_axis ? y_bar_zero : y2_bar_zero;
      
      if( d.drawing() )
      {
        brush.setRGBA( cs.style_.getBack() );
        d.gc_.setBrush( brush  );
        polygon.setPoints( poly );
        d.gc_.fillPoly( polygon );
      }
//    no locating done for area...
      
    }
  }
  
  /**
   * Render area of stacked area.
   */
  private static void renderStackedAreas( DefaultChartRenderData d, RenderPersistData rpd, StackedArea top_area, int y_bar_zero, IScale y_scale, IScale x_scale )
  {
    //until no location is done for area I can short cut here...
    if( d.locating() ) return ;
    
    int line_above[]=null;
    StackedArea sa_above = null;
    boolean area_above=false;
    
    SolidBrush brush = new SolidBrush( RGBA.WHITE );
    Polygon polygon =new Polygon();
    for( StackedArea sa = top_area; sa!=null; sa=sa.below )
    {
      DCurve curve = rpd.curve_styles[sa.index].curve_;
      
      int size = sa.points.size();
      int curr_line[] = new int[ size*2];
      int ip=0;
      for( Iterator it=sa.points.iterator(); it.hasNext(); )
      {
        SAPoint sp = (SAPoint)it.next();
        double x = x_scale.toScale( sp.x_coord.getValue(null) );
        double y = y_scale.toScale( sp.y_value );
        curr_line[ip++] = (int)x;
        curr_line[ip++] = (int)y;
      }
      
      //need two lines to define an area.
      if( area_above && sa_above != null )
      {        
        //build polygon using previous line
        int poly[] = new int[ curr_line.length + line_above.length ];
        System.arraycopy( line_above,0, poly,0, line_above.length );
        //close polygon using current line.. in reverse points order!
        //(and do not revert xyxyxy coordinates...)
        //do not include points outside
        int j=curr_line.length;
        for( int i=curr_line.length-2; i>=0 && j<poly.length; i-=2 )
        {
          poly[j++] = curr_line[i]; //x
          poly[j++] = curr_line[i+1]; //y
        }        
                
        //true: if( d.drawing() )
        {
          brush.setRGBA( rpd.curve_styles[sa_above.index].style_.getBack() );
          d.gc_.setBrush( brush );
          polygon.setPoints( poly );
          d.gc_.fillPoly( polygon );
        }
        //no locating done for area...
      }
      
      line_above = curr_line;
      sa_above = sa;
      area_above = ( curve.isType( DCurve.T_STACKED_AREA_LINE )
                  || curve.isType( DCurve.T_STACKED_AREA_LINE_POINTS ));
    }//for
    
    //fill bottomest area ?
    if( area_above && line_above !=null )
    {
      //need only two points to close polygon along Y axis.
      int poly[] = new int[ line_above.length + 4 ];
      System.arraycopy( line_above,0, poly,0, line_above.length );
      //only two point to close polygon touching the axis..
      int j=line_above.length;
      poly[j++] = line_above[line_above.length-2];
      poly[j++] = y_bar_zero;
      poly[j++] = line_above[0];
      poly[j++] = y_bar_zero;
      
      //true: if( d.drawing() )
      {
        brush.setRGBA( rpd.curve_styles[sa_above.index].style_.getBack() );
        d.gc_.setBrush( brush );
        polygon.setPoints( poly );
        d.gc_.fillPoly( polygon );
      }
      //no locating done for area...
    }
  }
  
  /**
   * Render line, and symbols or T_STACKED_xxx curve types.
   */
  private static void renderStackedLineAndSymbol( DefaultChartRenderData d, RenderPersistData rpd, StackedArea top_area, IScale y_scale, IScale x_scale, int symbol_size, int symbol_size2 )
    throws DefaultRenderChartLocation
  {
    ISymbol symbol=null;
    Rect rp=new Rect();
      
    SolidBrush brush = new SolidBrush();
    LineStylePen pen = new LineStylePen();
    for( StackedArea sa = top_area; sa!=null; sa=sa.below )
    {
      //int ip=0;
      int lx=0,ly=0; //last points
      DPoint last_point=null;
      
      //curve with symbol (?)
      if( rpd.curve_styles[sa.index].curve_.isType( DCurve.T_STACKED_LINE_POINTS )
       || rpd.curve_styles[sa.index].curve_.isType( DCurve.T_STACKED_AREA_LINE_POINTS ) )
      {
        symbol =  rpd.curve_styles[sa.index].symbol_;
      } else {
        symbol = null;
      }
      
      //as back color is used to fill area, use fore color for the line.
      IGCDStyle style = rpd.curve_styles[sa.index].style_;
      brush.setRGBA( style.getBack() );
      d.gc_.setBrush( brush );
      for( int i=0; i<sa.points.size(); ++i )
      {
        SAPoint sp = (SAPoint)sa.points.get(i);
        int px = (int)x_scale.toScale( sp.x_coord.getValue(null) );
        int py = (int)y_scale.toScale( sp.y_value );
        if( i!=0 )
        {
          if( symbol!=null )
          {
            //symbol at last point, means drawing line do not overlap on symbol...good!
            rp.setRect( lx-symbol_size, ly-symbol_size, symbol_size2, symbol_size2 );
          }
          //two points to draw a line..
          if( d.drawing())
          {
            pen.setRGBA( sa.area ? style.getFore() : style.getBack() );
            d.gc_.setPen( pen );
            d.gc_.drawLine( lx, ly, px, py );
            if( symbol!=null )
            {
              pen.setRGBA( style.getFore() );
              d.gc_.setPen( pen );
//            d.gc_.setForeground( d.getBlack() );
              symbol.draw( d.gc_, rp );
            }
          } else {
            //locate symbol
            if (symbol!=null && symbol.contains( d.gc_, rp, d.lx_, d.ly_ ) )
            {           
              throw new DefaultRenderChartLocation( DLocated.CurvePoint, last_point, rp );
            }
            //locate segment.
            if( DrawUtilIGC.segmentContains(  lx, ly, px, py, d.lx_, d.ly_ ) )
            {
              DefaultRenderChartLocation loc
                =new DefaultRenderChartLocation( DLocated.CurveLine,  sp.point, 
                    new Rect( lx, ly, px-lx, py-ly) );
              loc.setPreviousPoint( ((SAPoint)sa.points.get(i-1)).point );
              throw loc;
            }
          } 
        }
        lx=px;
        ly=py;
        last_point=sp.point;
      }
      
      //do not forgot last symbol 
      if( symbol!=null )
      {
        //symbol at last point, means drawing line do not overlap on symbol...good!
        rp.setRect( lx-symbol_size, ly-symbol_size, symbol_size2, symbol_size2 );
        //two points to draw a line..
        if( d.drawing())
        {
//        d.gc_.setForeground( d.getBlack() );
          symbol.draw( d.gc_, rp );
        } else {
          //locate symbol
          if (symbol.contains( d.gc_, rp, d.lx_, d.ly_ ) )
          {
            throw new DefaultRenderChartLocation( DLocated.CurvePoint, last_point, rp );
          }
        }
      }
    }
  }
  
  /**
   * Rebuild XY chart type private part of persist data.
   */
  private static XY rebuildRPD( DefaultChartRenderData d, RenderPersistData rpd )
  {
     XY p = new XY();
     rpd.g=p;
     
     //search for axis "x" and "y", must exists
     int have_axis=0;
     for( IDItem item=rpd.graphic.getFirstChild(); item!=null; item=item.getNext() )
     {
       if( !(item instanceof DAxis) ) continue;
       DAxis axis = (DAxis)item;
       if( p.x_axis==null && "x".equals( axis.getName() ) ) //$NON-NLS-1$
       {
         p.x_axis = axis;
         have_axis |= 0x1;
       }
       else if ( p.y_axis==null && "y".equals( axis.getName() )) //$NON-NLS-1$
       {
         p.y_axis = axis;
         have_axis |= 0x2;
       }
       else if ( p.y2_axis==null && "y2".equals( axis.getName() )) //$NON-NLS-1$
       {
         p.y2_axis = axis;
         have_axis |= 0x4;
       }
       if( have_axis == 0x7 ) break; //all axis there.
     }
     
     //continue build only if have axis
     if ( p.x_axis!=null && p.y_axis!=null )
     {
       boolean have_y2 = p.y2_axis!=null;
       
       p.x_axis_style = new IGCDStyle( p.x_axis, d.scale_ );
       p.y_axis_style = new IGCDStyle( p.y_axis, d.scale_ );
       if( have_y2 )
       {
         p.y2_axis_style = new IGCDStyle( p.y2_axis, d.scale_ );
       }
       
       //have stacked area ? only if axis use number (x and one of y/y2)
       if( p.x_axis.useNumbers() )
       {  
         for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
         {
           CurveStyle ccurve = rpd.curve_styles[icrv];
           String type = ccurve.curve_.getType();
           
           boolean area=true;           
           if( !( DCurve.T_STACKED_AREA_LINE.equals( type ) 
               || DCurve.T_STACKED_AREA_LINE_POINTS.equals( type ) ) ) 
           {
               if( ( DCurve.T_STACKED_LINE.equals( type ) 
                  || DCurve.T_STACKED_LINE_POINTS.equals( type ) ) )
               {
                 area=false;
               } else {
                 continue;
               }
           }
           
           //few curve belong to y_axis other to y2_axis.
           boolean known_sa_axis = false;
           boolean sa_axis_is_y1 = true;
           
           //check points...
           ArrayList pts = new ArrayList(10);
           boolean all_use_numbers=true;
           for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
           {
               if(!(item instanceof DPoint)) continue;
               DPoint point = (DPoint)item;
               IDCoord xc = null, yc=null;
               //DCoord y1c=null, y2c=null;
               for( IDItem ic=point.getFirstChild(); ic!=null; ic=ic.getNext() )
               {
                 if(!(ic instanceof IDCoord)) continue;
                 IDCoord c = (IDCoord)ic;
                 DAxis  a = c.getAxis();
                 if( a == p.x_axis ) xc=c;
                 else if( a == p.y_axis && (!known_sa_axis || sa_axis_is_y1) ) yc=c;
                 else if( a == p.y2_axis && (!known_sa_axis||!sa_axis_is_y1) ) yc=c;
               }
               //valid point ?
               if( xc!=null && yc!=null )
               {
                 all_use_numbers = yc.getAxis().useNumbers();
                 if( !all_use_numbers ) break;
                 if( !known_sa_axis )
                 {
                   known_sa_axis=true;//now I know which axis is used
                   sa_axis_is_y1 = yc.getAxis()==p.y_axis;
                 }
                 SAPoint pt =new SAPoint();
                 pt.point = point;
                 pt.x_coord = xc;
                 pt.y_coord = yc;
                 pt.y_value = yc.getValue(null);           
                 pts.add(pt);                   
               }
             }
             //area must have at least two points...
             if( all_use_numbers && pts.size() > 2 )
             {             
               StackedArea sa = new StackedArea();
               sa.index=icrv;
               StackedArea top_area = sa_axis_is_y1 ? p.y1_top_area : p.y2_top_area ; 
               sa.below = top_area;
               sa.points = pts;
               sa.area = area;
               //first curve is simple: nothing to do.
               if( top_area != null )
               {
                 //other must cumulate on top, the difficulty is
                 //points aren't x-aligned => must do a linear approximation.
                 StackedArea sab = top_area;
                 //Object vn=null;
                 for( int i=0; i<sa.points.size(); i++ )
                 {
                   SAPoint sp = (SAPoint)sa.points.get(i);
                   //search curve below this point:
                   Object vx = sp.x_coord.getValue(null);
//TBD: linked to number double                 
                   double dx = ((Number)vx).doubleValue();
                   double base_y=0.0;
                   boolean found_candidate=false;
                   for( StackedArea a=sab; (a!=null)&&(!found_candidate); a=a.below )
                   {
                     Object prev_vx=null;
                     SAPoint prev_pt=null;
                     double pdx = 0.0, pdy = 0.0;
                     for( int j=0; (!found_candidate)&&(j<a.points.size()); ++j )
                     {
                       SAPoint pt = (SAPoint)a.points.get(j);
                       Object pvx = pt.x_coord.getValue(null);
                       double cdx = ((Number)    pvx).doubleValue();
                       //double cdy = ((Number)pt.y_coord.getValue(null)).doubleValue();
                       double cdy = ((Number)pt.y_value).doubleValue();
                       if( prev_pt!=null )
                       {
                         if( vx instanceof Comparable )
                         {
                           if( ((Comparable)vx).compareTo( prev_vx ) >=0
                               &&((Comparable)vx).compareTo( pvx ) <= 0 )
                           {
                             //now found y at x=vx of segment [prev_vx..pvx]
                             //y + prev_pt.y+ k*(pt.y-prev_pt.y); k=(vx-prev_pt.x)/(pt.x-prev_pt.x);                           
                             if( cdx==pdx ) //so dx==cdx==pdx
                             {
                               base_y = Math.max( pdy, cdy );                             
                             } else {
                               //that's why I need numbers: compute linear approximation.
                               double k = (dx-pdx)/(cdx-pdx);
                               base_y = pdy + k*(cdy-pdy);
                             }
                             found_candidate=true;
                           }
                         }
                       }
                       prev_pt=pt;
                       prev_vx=pvx;
                       pdx = cdx;
                       pdy = cdy;
                     }
                   }
                   //here is the cumulative
                   double ny = ((Number)sp.y_coord.getValue(null)).doubleValue();                              
                   ny += base_y;
                   sp.y_value = new Double(ny); 
                   if( i==0 )
                   {
                     sa.ymin = sa.ymax = ny;
                   } else {
                     if( sa.ymin > ny ) sa.ymin = ny;
                     else if( sa.ymax < ny ) sa.ymax = ny ;
                   }
                 }
               }
               else
               {
                 //for first curve we have to setup ymin/ymax
                 for( int i=0; i<sa.points.size(); i++ )
                 {
                   SAPoint sp = (SAPoint)sa.points.get(i);
                   double ny=((Number)sp.y_coord.getValue(null)).doubleValue();
                   if(i==0)
                   {
                     sa.ymin=sa.ymax=ny;
                   } else {
                     if(ny<sa.ymin) sa.ymin=ny; else if (ny>sa.ymax) ny=sa.ymax;
                   }
                 }
               }
               if( sa_axis_is_y1 )
               {
                 p.y1_top_area =sa;
               } else {
                 p.y2_top_area =sa; 
               }
             }
           }         
       }//if stacked_area
       
       //have bubble/circle curves ?
       for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
       {
         CurveStyle ccurve = rpd.curve_styles[icrv];
         String type = ccurve.curve_.getType();
         if( DCurve.T_BUBBLE.equals( type ) || DCurve.T_CIRCLE.equals( type ) )
         {
           rebuildRPDForBubble( icrv, ccurve, p, rpd );
         }
       }
       boolean have_bubble = p.bubbles_data!=null; 
       
       //values bounds
       {
         Object x_min = p.x_axis.getProperties().get( DAxis.P_MIN ),
         x_max = p.x_axis.getProperties().get( DAxis.P_MAX ),
         y_min = p.y_axis.getProperties().get( DAxis.P_MIN ),
         y_max = p.y_axis.getProperties().get( DAxis.P_MAX );
         Object y2_min=null, y2_max=null;
         
         if( have_y2 )
         {
           y2_min = p.y2_axis.getProperties().get( DAxis.P_MIN );
           y2_max = p.y2_axis.getProperties().get( DAxis.P_MAX );
         }
         
         if( x_min==null || x_max==null || y_min==null || y_max==null 
             || (have_y2)&& ( y2_min==null || y2_max==null ) )
         {
           
           MinMax x_mm = new MinMax( p.x_axis );
           MinMax y_mm = new MinMax( p.y_axis );
           MinMax y2_mm = have_y2 ? new MinMax( p.y2_axis ) : null;
           
           //update min/max from stacked area...           
           VDouble vd=new VDouble();
           for( StackedArea a=p.y1_top_area; a!=null; a=a.below )
           {
             y_mm.update( vd.setValue( a.ymin ) );
             y_mm.update( vd.setValue( a.ymax ) );
             //now check along X axis.. 
             for( int i=0; i<a.points.size(); ++i)
             {
               SAPoint sp = (SAPoint)a.points.get(i);
               x_mm.update( sp.x_coord );
             }             
           }
           for( StackedArea a=p.y2_top_area; a!=null; a=a.below )
           {
             y2_mm.update( vd.setValue( a.ymin ) );
             y2_mm.update( vd.setValue( a.ymax ) );
             //now check along X axis.. 
             for( int i=0; i<a.points.size(); ++i)
             {
               SAPoint sp = (SAPoint)a.points.get(i);
               x_mm.update( sp.x_coord );
             }             
           }
           
           for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
           {
             CurveStyle ccurve = rpd.curve_styles[icrv];
             //because cumulative are already updated            
             if( ccurve.curve_.isType( DCurve.T_STACKED_LINE )
              || ccurve.curve_.isType( DCurve.T_STACKED_LINE_POINTS )  
              || ccurve.curve_.isType( DCurve.T_STACKED_AREA_LINE )  
              || ccurve.curve_.isType( DCurve.T_STACKED_AREA_LINE_POINTS ) )
             {
               continue;
             }
             for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
             {
               if( !(item instanceof DPoint) ) continue;
               for( IDItem ic=item.getFirstChild(); ic!=null; ic=ic.getNext() )
               {
                 if( !(ic instanceof IDCoord) ) continue;
                 IDCoord coord = (IDCoord)ic;
                 DAxis a = coord.getAxis();
                 if( a == p.x_axis ) { x_mm.update( coord ); }
                 else if( a == p.y_axis ) { y_mm.update( coord ); }
                 else if( have_y2 && a == p.y2_axis ) { y2_mm.update( coord ); }
               }
             }
           }          
           
           if( x_min ==null ) x_min = x_mm.getVMin();
           if( x_max ==null ) x_max = x_mm.getVMax();
           if( y_min ==null ) y_min = y_mm.getVMin();
           if( y_max ==null ) y_max = y_mm.getVMax();      
           if( have_y2 )
           {
             if( y2_min ==null ) y2_min = y2_mm.getVMin();
             if( y2_max ==null ) y2_max = y2_mm.getVMax();      
           }

           if( have_bubble )
           {
             p.x_scale = DefaultChartRenderData.getScale( p.x_axis, rpd.ag_rect.left(), rpd.ag_rect.right(), x_min, x_max );
             p.y_scale = DefaultChartRenderData.getScale( p.y_axis, rpd.ag_rect.bottom(), rpd.ag_rect.top(), y_min, y_max );
             Insets bubble_insets = new Insets();
             Insets bubble_insets2 = new Insets(); //top/bottom for y2 only

             //get insets for each axis, and update value
             bubblesInsets( p, rpd, x_mm,y_mm,y2_mm, bubble_insets,bubble_insets2 );

             //convert back insets to min/max values ...
             //note: use 2* factor as compensation of "estimation" and reducing of ag_rect
             //due to axis insets, ok it's not the right way but if someone have a better idea...
             if( bubble_insets.getW()!=0 ) 
             {
               x_min = p.x_scale.toValue( rpd.ag_rect.left() -2*bubble_insets.getL(), x_min );
               x_max = p.x_scale.toValue( rpd.ag_rect.right()+2*bubble_insets.getR(), x_max );
             }
             if( bubble_insets.getH()!=0 ) 
             {
               y_min = p.y_scale.toValue( rpd.ag_rect.bottom()+2*bubble_insets.getB(), y_min );
               y_max = p.y_scale.toValue( rpd.ag_rect.top()-2*bubble_insets.getT(), y_max );
             }
             if( bubble_insets2.getH()!=0 ) 
             {
               y2_min = p.y2_scale.toValue( rpd.ag_rect.bottom()+2*bubble_insets.getB(), y2_min );
               y2_max = p.y2_scale.toValue( rpd.ag_rect.top()-2*bubble_insets.getT(), y2_max );
             }
           }
         }
         
         if( p.x_scale==null ) {
           p.x_scale = DefaultChartRenderData.getScale( p.x_axis, rpd.ag_rect.left(), rpd.ag_rect.right(), x_min, x_max );
         } else {
           p.x_scale.setValueRange( x_min, x_max );
         }
         if( p.y_scale==null ) {
           p.y_scale = DefaultChartRenderData.getScale( p.y_axis, rpd.ag_rect.bottom(), rpd.ag_rect.top(), y_min, y_max );
         } else {
           p.y_scale.setValueRange( y_min, y_max );
         }
         
         if( !p.x_scale.isValid() ) return p;
         if( !p.y_scale.isValid() ) return p;
         
         //get axis amounts
         d.gc_.setFont( p.x_axis_style.getFont() );
         p.xam = d.computeXAxisAmounts( p.x_axis, p.x_scale, rpd.ag_rect, aln_x_title, null );    
         d.gc_.setFont( p.y_axis_style.getFont() );
         p.yam = d.computeYAxisAmounts(true, true, p.y_axis, p.y_scale, rpd.ag_rect, aln_y_title, null );    
         
         if( have_y2 )
         {
           if( p.y2_scale==null ) {
             p.y2_scale = DefaultChartRenderData.getScale( p.y2_axis, rpd.ag_rect.bottom(), rpd.ag_rect.top(), y2_min, y2_max );
           } else {
             p.y2_scale.setValueRange( y2_min, y2_max );
           }
           if( !p.y2_scale.isValid() ) return p;
           
           d.gc_.setFont( p.y2_axis_style.getFont() );
           //second y axis is on the right
           p.y2am = d.computeYAxisAmounts(true, false, p.y2_axis, p.y2_scale, rpd.ag_rect, aln_y2_title, null );    
         }
         
         //merge two (three) axis, changing ag_rect bounds
         rpd.ag_rect.setLeft  ( rpd.ag_rect.left  ()+Math.max( p.xam.getL(),  p.yam.getL() ));
         rpd.ag_rect.setTop   ( rpd.ag_rect.top   ()+Math.max( p.yam.getT(),   have_y2? p.y2am.getT() : 0) );
         rpd.ag_rect.setRight ( rpd.ag_rect.right ()-Math.max( p.xam.getR(), have_y2 ? p.y2am.getR() : 0 ) );
         rpd.ag_rect.setBottom( rpd.ag_rect.bottom()-Math.max( Math.max( p.xam.getB(), p.yam.getB() ),
             have_y2 ? p.y2am.getB() :0  ) );
         
         //update scales...
         p.x_scale.setScaleRange( rpd.ag_rect.left(), rpd.ag_rect.right() );
         p.y_scale.setScaleRange( rpd.ag_rect.bottom(), rpd.ag_rect.top() );
         if( have_y2 )
         {
           p.y2_scale.setScaleRange( rpd.ag_rect.bottom(), rpd.ag_rect.top() );
         }
         //update bubbles scales 
         if( have_bubble )
         {
           p.Rmax = Math.min( rpd.graphic_rect.getW(), rpd.graphic_rect.getH() )/4;
           p.Rmin = Math.min( p.Rmax/2, 4 ); //4 pixels or less if Rmax is too tiny
           for( int ib=0; ib<p.bubbles_data.length; ib++ )
           {
             BubbleData bdata = p.bubbles_data[ib];
             if( bdata==null ) continue;
             if( bdata.r_scale == null )
             {
               bdata.r_scale = DefaultChartRenderData.getScale( bdata.r_axis, p.Rmin,p.Rmax, bdata.r_min, bdata.r_max );
             } else {
               bdata.r_scale.setScaleRange( p.Rmin, p.Rmax );
             }
           }
         }
         
         //now we can compute ibar_width:
         p.ibar_width = -1;
         for( int icrv=0; icrv<rpd.curve_styles.length; ++icrv )
         {
           CurveStyle ccurve = rpd.curve_styles[icrv];
           //Curves type who will use ibar_width:             
           if(  ccurve.curve_.isType( DCurve.T_CANDLESTICK )
             || ccurve.curve_.isType( DCurve.T_CANDLESTICK_LINE )  
             || ccurve.curve_.isType( DCurve.T_VBAR ) )  
           {
             int lbw = computeBarWidth( ccurve, p, d, rpd );
             if( lbw > 0 ) //do not take if curve have no points / or 0 bar width
             {
               if( p.ibar_width < 0 )
               {
                 p.ibar_width = lbw;
               } else {
                 p.ibar_width = Math.min( p.ibar_width, lbw );
               }
             }
           }
         }
       }
     }
     
     return p;
  }
  
  private static void bubblesInsets( XY p, RenderPersistData rpd, MinMax x_mm, MinMax y_mm, MinMax y2_mm, Insets  bubble_insets, Insets bubble_insets2 )
  {

    //prepare bubbles: use current ag_rect to estimate if bubble bounds are out of
    //ag_rect, if it's the case try to modiy x_min/max y_min/max value to keep bubble in
    //but as ag_rect will be reduced estimation can failed...
    p.Rmax = Math.min( rpd.graphic_rect.getW(), rpd.graphic_rect.getH() )/4;
    p.Rmin = Math.min( p.Rmax/2, 4 ); //4 pixels or less if Rmax is too tiny
    Object r_value=null, x_value=null, y_value=null;
    Rect brect = new Rect();
    for( int ib=0; ib<p.bubbles_data.length; ib++ )
    {
      BubbleData bdata = p.bubbles_data[ib];
      bdata.r_scale = DefaultChartRenderData.getScale( bdata.r_axis, p.Rmin,p.Rmax, bdata.r_min, bdata.r_max );
      for( int jb=0; jb<bdata.bubbles.length; jb++ )
      {
        Bubble bubble = bdata.bubbles[jb];
        //at least update with center (real values)
        x_value = bubble.cx.getValue( x_value );
        y_value = bubble.cy.getValue( y_value );
        x_mm.update( x_value );
        //estimation:
        int cx = (int)p.x_scale.toScale( x_value );
        int cy;
        Insets y_insets;
        if( bdata.use_y2 ) {
          y2_mm.update( y_value ); 
          cy = (int)p.y2_scale.toScale( y_value );
          y_insets = bubble_insets2;
        } else {
          y_mm.update( y_value );
          cy = (int)p.y_scale.toScale( y_value );
          y_insets = bubble_insets;
        }
                 
        int radius = (int)bdata.r_scale.toScale( r_value=bubble.cr.getValue(r_value) );
        brect.setRect( cx-radius, cy-radius, 2*radius, 2*radius );
        //bubble ousite ag_rect ?
        int outside = brect.left() - rpd.ag_rect.left();
        if( outside < 0 ) bubble_insets.maxL( -outside );
        outside = rpd.ag_rect.right() - brect.right();
        if( outside < 0 ) bubble_insets.maxR( -outside );
        outside = brect.top() - rpd.ag_rect.top();
        if( outside < 0 ) y_insets.maxT( -outside );
        outside = rpd.ag_rect.bottom() - brect.bottom();
        if( outside < 0 ) y_insets.maxB( -outside );
      }
    }//for bubble
  }
  
  private static void rebuildRPDForBubble( int icrv, CurveStyle ccurve, XY p, RenderPersistData rpd )
  {
    BubbleData bdata=null;
    MinMax mm = null;
    ArrayList bubbles = new ArrayList();
    int y_axis_is = 0; //0:unknown, 1:y 2:y2
    for( IDItem item=ccurve.curve_.getFirstChild(); item!=null; item=item.getNext() )
    {
      if(!(item instanceof DPoint)) continue;
      IDCoord cx=null;
      IDCoord cy=null, cy2=null;
      IDCoord cr=null;
      for( IDItem ic=item.getFirstChild(); ic!=null; ic=ic.getNext() )
      {
        if(!(ic instanceof IDCoord)) continue;
        IDCoord coord = (IDCoord)ic;
        DAxis  axis  = coord.getAxis();
        if( axis == p.x_axis ) {
          if( cx==null ) cx = coord;
        } else if ( axis ==p.y_axis ) {
          if( cy==null  && y_axis_is!=2 ) cy = coord;
          if( y_axis_is==0 ) y_axis_is=1; //now force all point to use y
        } else if ( axis ==p.y2_axis && y_axis_is!=1 ) {
          if( cy2 == null && y_axis_is!=1 ) cy2 = coord; 
          if( y_axis_is==0 ) y_axis_is=2; //now force all point to use y2
        } else {
          if( bdata==null )  {
            bdata = new BubbleData();
            if( p.bubbles_data==null ) {
              p.bubbles_data = new BubbleData[rpd.curve_styles.length];
            }
            p.bubbles_data[icrv] = bdata;
            
            bdata.r_axis = axis ;
            bdata.r_min = bdata.r_axis.getProperties().get( DAxis.P_MIN );
            bdata.r_max = bdata.r_axis.getProperties().get( DAxis.P_MAX );

            if( bdata.r_min==null || bdata.r_max==null )
            {
              mm = new MinMax( bdata.r_axis );
              mm.update( bdata.r_min );
              mm.update( bdata.r_max );
            } else {
              //do not check for min/max but continue to geather valid points
            }

            cr = coord;
          } else if( axis == bdata.r_axis ) {
            if( cr==null ) cr = coord;
          }
        }
      }//for
      if( cx==null || ( cy==null&&cy2==null) || cr==null )
      {
        continue; //bad point missing coordinate
      }
      if( mm!=null ) mm.update( cr );
      Bubble b = new Bubble();
      b.cx =cx;
      b.cy =cy;
      b.cr =cr;
      b.point = (DPoint)item;
      bubbles.add( b );
    }

    //no valid point found
    if( bdata==null ) return ;
    
    bdata.use_y2 = y_axis_is==2;
    
    if( mm!=null )
    {
      //include min/max hint if user set them.
      mm.update( bdata.r_axis.getProperties().get( DAxis.P_MIN_HINT ) );
      mm.update( bdata.r_axis.getProperties().get( DAxis.P_MAX_HINT ) );
      
      bdata.r_min = mm.getVMin();
      bdata.r_max = mm.getVMax();
    }
    
    //sort bubbles:
    bdata.bubbles = new Bubble[ bubbles.size() ];
    bubbles.toArray( bdata.bubbles );
    
    Arrays.sort( bdata.bubbles );
    //reverse array to have decreasing order:
    int mid = bdata.bubbles.length/2;
    for( int i=0,j=bdata.bubbles.length-1; i<mid; i++,j-- )
    {
      Bubble b = bdata.bubbles[i];
      bdata.bubbles[i] = bdata.bubbles[j];
      bdata.bubbles[j] = b;
    }
  }
  
  /**
   * Render VBar curves
   */
  private static void renderBubble( int icrv, CurveStyle ccurve, XY p, DefaultChartRenderData d, RenderPersistData rpd, boolean circle)
     throws DefaultRenderChartLocation
  {
    if( p.bubbles_data==null ) return ;//no valid bubble/circle curve found.
    
    SolidBrush brush = new SolidBrush( RGBA.WHITE );     
    BubbleData bdata = p.bubbles_data[icrv];
    
    //get radius axis and min max ...  
    if( bdata==null )
    {
      String title = DefaultChartRenderData.getResourceString(ccurve.curve_.getName(), 
                (DI18N)rpd.graphic.getChildOfClass(DI18N.class));
      d.error("No valid points found for "+(circle?"T_CIRCLE":"T_BUBBLE")+" curve (name:"+title+")", rpd );
      return ;
    }
    if( bdata.r_axis == null )
    {
      if( circle )
        d.error( "Missing axis for circle radius", rpd );
      else
        d.error( "Missing axis for bubble radius", rpd );
      return ;
    }
    Object y_obj=null;
    Object x_obj=null;
    Object r_obj=null;
    LineStylePen pen = new LineStylePen(RGBA.BLACK);
    
    DefaultRenderChartLocation loc=null;
    for( int ib=0; ib<bdata.bubbles.length; ib++ )
    {
      Bubble bubble = bdata.bubbles[ib];
      int x = (int)p.x_scale.toScale( x_obj=bubble.cx.getValue(x_obj) );
      int y = bdata.use_y2? (int)p.y2_scale.toScale( y_obj=bubble.cy.getValue(y_obj) )
                          : (int)p. y_scale.toScale( y_obj=bubble.cy.getValue(y_obj) );
      int r = (int)bdata.r_scale.toScale( r_obj=bubble.cr.getValue(r_obj) );

      //render bubble/circle 
      if( circle )
      {
        if(d.drawing() )
        {
          pen.setRGBA( ccurve.style_.getBack() );
          d.gc_.setPen( pen );
          d.gc_.drawCircle( x ,y , r );
        }
        else //locating
        {
          if( DrawUtilIGC.ovalUnder( x-r,y-r,2*r,2*r, d.lx_, d.ly_ ) )
          {
            //as circle might be over another one
            loc = new DefaultRenderChartLocation( DLocated.CurvePoint, bubble.point, new Rect(d.lx_,d.ly_,0,0) );
          }
        }      }
      else //bubble
      {
        if(d.drawing() )
        {
          brush.setRGBA( ccurve.style_.getBack() );
          d.gc_.setBrush( brush );
          d.gc_.fillCircle( x ,y , r );
          d.gc_.drawCircle( x ,y , r );
        }
        else //locating
        {
          if( DrawUtilIGC.ovalContains( x-r,y-r,2*r,2*r, d.lx_, d.ly_ ) )
          {
            //as bubble might be over another one
            loc = new DefaultRenderChartLocation( DLocated.CurvePoint, bubble.point, new Rect(d.lx_,d.ly_,0,0) );
          }
        }
      }
    }//for points
    if(loc!=null) throw loc;
  }
}
