/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LinearDateScale.java,v 1.5 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 14 dc. 2003
 *
 */
package org.eclipse.tptp.platform.report.chart.internal;

import java.util.Date;

import org.eclipse.tptp.platform.report.tools.internal.CalendarField;
import org.eclipse.tptp.platform.report.tools.internal.VLong;

import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.text.UFormat;
import com.ibm.icu.util.Calendar;
import com.ibm.icu.util.ULocale;

/**
 * Implementation of IScale for java.util.Date objects.
 * This scale can used Long object in the meaning of Date.setTime(Long):
 * means milliseconds from the epoch (1 jan 1970).
 * If no date format is set at first invocation of valueText(), 
 * a use SimpleDateFormat is used.
 * 
 * @see java.util.Date, java.util.Calendar, VLong
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class LinearDateScale implements IScale
{
  protected double smin_, smax_;
  protected Date dmin_, dmax_;
  protected long tmin_, tmax_ ;// store Date.getTime();
  protected double k_; //for date this is getTime() value
  protected boolean reverse_;
  protected UFormat format_;
  protected boolean date_;
  protected ULocale locale_;
  
  /** Object must be Date */
  public LinearDateScale( double smin, double smax, Object _vmin, Object _vmax )
  {    
    smin_=smin; smax_=smax;
    reverse_=false;
    date_=false;
    setValueRange( _vmin, _vmax );
  }
  
  /**
   * Change values of pixel scale, Object must be instance of java.util.Date,
   * otherwise scale becomes invalid.
   */
  public void setValueRange( Object v1, Object v2 )
  {
    tmin_=tmax_=0; 
    if( v1 instanceof Date )
    {
      dmin_=(Date)v1; tmin_ = dmin_.getTime();
    } else {
      tmin_ = 0;
    }
    if( v2 instanceof Date )
    {
      dmax_=(Date)v2; tmax_ = dmax_.getTime();
    } else {
      tmax_ = 0;
    }
    if( tmin_ > tmax_ )
    {
      long l = tmin_; tmin_=tmax_; tmax_=l;
      Date d = dmin_; dmin_=dmax_; dmax_=d;
    }
    k_ = tmax_==tmin_ ? Double.NaN : (smax_-smin_)/(double)(tmax_-tmin_);
  }
  
  public boolean isValid() { return tmin_ < tmax_; }
  
  public void setTextFormat( UFormat _format )
  {
    format_=_format;
  }
  
  private boolean output_once=false; //do not bark too much if text formatting failed, once is enough

  public String valueText( Object _value )
  {
    Date date = null;
    if( _value instanceof Date )
    {
      date = (Date)_value;
    }
    else if ( _value instanceof VLong )
    {
      //long means new Date(long), useful for scaleFirst()/scaleNext().
      date = new Date( ((VLong)_value).longValue() );      
    }
    else return null;
    
    if( format_==null )
    {
      //create a default DateFormat...
      if (locale_==null)  
          //format_ = new SimpleDateFormat("yyyy.MM.dd");
          format_ = SimpleDateFormat.getDateInstance(SimpleDateFormat.SHORT);
      else
          format_ = SimpleDateFormat.getDateInstance(SimpleDateFormat.SHORT, locale_);
    }
    
    try {
      String txt = format_.format( date );
      return txt;
    }
    catch( IllegalArgumentException e )
    {
      if( !output_once )
      {
System.err.println("For object's class ="+(date==null?null:date.getClass().getName()));
         e.printStackTrace();
         output_once=true;
      }
    }
    return null;
  }
  
  public void setReversed( boolean b ) { reverse_=b; }
  public boolean isReversed() { return reverse_; }
  

  public Object getValueMax( Object _value )
  {
    if( date_ )
    {
      Date date = _value instanceof Date ? (Date)_value : new Date();
      date.setTime( tmax_ );
      return date;
    }
    return null;
  }
  
  public Object getValueMin( Object _value )
  {
    if( date_ )
    {
      Date date = _value instanceof Date ? (Date)_value : new Date();
      date.setTime( tmin_ );
      return date;
    }
    return null;
  }

  public double getScaleMin() { return smin_; }
  public double getScaleMax() { return smax_; }

  public double toScale( Object _value )
  {
    long value;
    if( _value instanceof Date )
    {
      value = ((Date)_value).getTime();
    }
    else if( _value instanceof VLong )
    {
      value = ((VLong)_value).longValue();
    }
    else return -1;
    double scl;
    if( reverse_ )
    {
//      pix = (int)(pix_max_-(pix_max_-pix_min_)*(value-vmin_)/(vmax_-vmin_));
      scl = smax_-k_*(value-tmin_);
    } else {
      scl = smin_+k_*(value-tmin_);
    }
    return scl;
  }
  
  public Object toValue( double scale, Object _value )
  {
    double val;
    if( reverse_ )
    {
      val = tmin_+(smax_-scale)/k_;
    } else {
      val = tmin_+(scale-smin_)/k_;
    }

    Date date = _value instanceof Date ? (Date)_value : new Date();
    date.setTime( (long)val );
    return date;
  }
  
  public void setScaleRange( double scale_min, double scale_max )
  {
    smin_ = scale_min;
    smax_ = scale_max;
    k_ = tmax_==tmin_ ? Double.NaN : (smax_-smin_)/(double)(tmax_-tmin_);
  }
  
  /**
   * Result is null or a long number, the time in milliseconds.
   * step can be a number : delay in milliseconds
   *  or a calendar : the fields set (YEAR,MONTH,...) is used to compute step.
   */
  public Object stepFirst( Object _step )
  {
    long first = -1;
    if( _step instanceof Number )
    {
      long step = ((Number)_step).longValue(); 
      if(  step <= 0.0f ) return null;
      long vdn = (long)Math.ceil(tmin_/(double)step) ;
      if ( Math.abs(vdn)<1e-6 ) vdn=0; // x*0 = 0
      first = (long)( vdn * step );
    }
    else if ( _step instanceof CalendarField )
    {
      CalendarField cal_step  =(CalendarField)_step;
      Calendar val;
      if (locale_==null)
          val  = Calendar.getInstance();
      else
          val  = Calendar.getInstance(locale_);
      val.setTimeInMillis( tmin_ );
      //round each fields:
      for( int fidx=0; fidx<val.getFieldCount(); fidx++ )
      {
        if( cal_step.isSet( fidx ) )
        {
          int o_val = val.get( fidx ); //Calendar.YEAR);
          int r_val = cal_step.get( fidx );      
          int n_val = r_val > 0 ? r_val*(int)Math.ceil( o_val/(double)r_val) : o_val; //Calendar.YEAR));
          val.set( fidx, n_val );
        }
      }
      first = val.getTimeInMillis();
    }
    else return null; 
    
    return new VLong(first);
  }

  /**
   * @return For given step and value, nxtw 'scaled' value. (>vmax to stop)
   */
  public Object stepNext( Object _step, Object _last )
  {
    if( !(_last instanceof VLong)) return null; //see stepFirst
    VLong v_last = ((VLong)_last);
    long next = v_last.longValue();
    if( _step instanceof Number )
    {
      long step = -1;
      if( _step instanceof Number ) step = ((Number)_step).longValue();
      next +=  step ;
    }
    else if( _step instanceof CalendarField )
    {
      CalendarField cal_step  =(CalendarField)_step;
      Calendar val;
      if (locale_==null)
          val  = Calendar.getInstance();
      else
          val  = Calendar.getInstance(locale_);
      val.setTimeInMillis( v_last.longValue() );
      //round each fields:
      for( int fidx=0; fidx<val.getFieldCount(); fidx++ )
      {
        if( cal_step.isSet( fidx ) )
        {
          int o_val = val.get( fidx ); //Calendar.YEAR);
          int r_val = cal_step.get( fidx );
          int n_val = o_val + r_val; //Calendar.YEAR));
          val.set( fidx, n_val );
        }
      }
      next = val.getTimeInMillis();
    }
    else return null; 
    
    if( next > tmax_ ) return null; //end of iterations
    v_last.setValue( next );
    return v_last;
  }

  /** @return pixel for min value. */
  public double getScaleForBar()
  {

    return smin_;
  }
  
  public void setLocale(ULocale _locale) {
      locale_ = _locale;
  }

}
