/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LinearNumberScale.java,v 1.5 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 14 dc. 2003
 *
 */
package org.eclipse.tptp.platform.report.chart.internal;

import org.eclipse.tptp.platform.report.tools.internal.VDouble;
import org.eclipse.tptp.platform.report.tools.internal.VNumber;

import com.ibm.icu.text.NumberFormat;
import com.ibm.icu.text.UFormat;
import com.ibm.icu.util.ULocale;

/**
 * Implementation of IScale for Object instance of Number.
 * (ie: Integer,Float,Double, ... VFloat,VDouble,... ).
 * This scale can be used also using directly double primitive type.
 *
 * @see IScale.
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class LinearNumberScale implements IScale
{
  protected double dmin_, dmax_;
  protected double vmin_, vmax_, k_;
  protected boolean reverse_;
  protected UFormat format_;
  protected ULocale locale_;
  
  public LinearNumberScale( double dmin, double dmax, double vmin, double vmax )
  {
    dmin_=dmin; 
    dmax_=dmax; 
    reverse_=false;
    setValueRange( vmin, vmax );
  }
  
  public LinearNumberScale( double dmin, double dmax, Object vmin, Object vmax )
  {
    dmin_=dmin; 
    dmax_=dmax; 
    reverse_=false;
    setValueRange( vmin, vmax );
  }
  
  public void setValueRange( Object v1, Object v2 )
  {
    vmin_=-1; vmax_=-2;
    k_ = Double.NaN;
    if( !(v1 instanceof Number) ) return ;
    if( !(v2 instanceof Number) ) return ;
    setValueRange( ((Number)v1).doubleValue(), ((Number)v2).doubleValue() );
  }
  
  public void setValueRange( double d1, double d2 )
  {
    if( d1 < d2 )
    {
      vmin_ = d1; vmax_ = d2 ;
    } else {
      vmin_ = d2; vmax_ = d1 ;
    }
    k_ = vmax_==vmin_ ? Double.NaN : (dmax_-dmin_)/(vmax_-vmin_); 
  }


  public boolean isValid() { return vmin_ < vmax_; }
  
  public void setTextFormat( UFormat _format )
  {
    format_=_format;
  }

  private boolean output_once=false;
  
  public String valueText( Object _value )
  {
    Number number=null;
    if ( _value instanceof Number )
    {
      number = ((Number)_value);
    }
    else 
    {
      return null;
    }
    
    if( format_==null )
    {
      //create default number format
      NumberFormat f;
      if (locale_ == null)
          f = NumberFormat.getInstance();
      else
          f = NumberFormat.getInstance(locale_);
      f.setMaximumFractionDigits(2);
      f.setMinimumFractionDigits(2);
      format_ = f;
    }
    
    try {
      String txt = format_.format( _value );
      return txt;
    }
    catch( IllegalArgumentException e )
    {
      if( !output_once)
      {
System.err.println("For object's class ="+(_value==null?null:number.getClass().getName()));
        e.printStackTrace();
        output_once=true;
      }
    }
    return null;
  }
  
  public void setReversed( boolean b ) { reverse_=b; }
  public boolean isReversed() { return reverse_; }
  

  
  public Object getValueMax( Object _value )
  {
    if( _value instanceof VNumber )
    {
      return ((VNumber)_value).setValue( vmax_ );
    } 
    return new VDouble( vmax_ );
  }
  
  public Object getValueMin( Object _value )
  {
    if( _value instanceof VNumber )
    {
      return ((VNumber)_value).setValue( vmin_ );
    } 
    return new VDouble( vmin_ );
  }

  public double getScaleMin() { return dmin_; }
  public double getScaleMax() { return dmax_; }

  public double toScale( Object _value )
  {
    if( !(_value instanceof Number) ) return -1;
    return toScale( ((Number)_value).doubleValue() );
  }
  
  public double toScale( double _value )
  {
    double dbl;
    if( reverse_ )
    {
//TBD: need more experience on reversed scales ... see toPixel()
//      pix = (pix_max_-k_*(_value-vmin_));
      dbl = dmin_+k_*(_value-vmin_); //k_ must be < 0
    } else {
      dbl = dmin_+k_*(_value-vmin_);
    }
    return dbl;
  }
  
  public Object toValue( double dbl, Object _value )
  {
    double val;
    if( reverse_ )
    {
//TBD: need more experience on reversed scales ... see toPixel()
      val = vmin_+(dmax_-dbl)/k_;
    } else {
      val = vmin_+(dbl-dmin_)/k_;
    }
    if( _value instanceof VDouble )
    {
      return ((VDouble)_value).setValue( val );
    }
    return new VDouble(val);
  }
  
  public void setScaleRange( double dmin, double dmax )
  {
    dmin_ = dmin;
    dmax_ = dmax;
    k_ = vmax_==vmin_ ? Double.NaN : (dmax_-dmin_)/(vmax_-vmin_);
  }
  
  public Object stepFirst( Object _step )
  {
     double step = -1;
     if( _step instanceof Number ) step = ((Number)_step).doubleValue();
     if( step <= 0.0f ) return null;
     double vdn = Math.ceil(vmin_/step) ;
     if ( Math.abs(vdn)<1e-6 ) vdn=0.0f; // x*0 = 0
     vdn = vdn * step ;
     return new VDouble(vdn);
  }
  
  /**
   * @return For given step and value, next 'scaled' value. (>vmax to stop)
   */
  public Object stepNext( Object _step, Object _last )
  {
    double step = -1;
    if( _step instanceof Number ) step = ((Number)_step).doubleValue();
    if( !(_last instanceof VDouble)) return null;
    VDouble v_last = (VDouble)_last;
    double next = v_last.doubleValue() + step ;  
    if( next > vmax_ ) return null;
    v_last.setValue( next );
    return v_last;
  }

  public double getScaleForBar()
  {
    if( vmin_ >= 0.0f ) return dmin_;
    if( vmax_ <= 0.0f ) return dmax_;
    //pixel for 0 value.
    return toScale( 0.0 );
  }

  public String toString()
  {
    return super.toString()+"{pixels="+dmin_+".."+dmax_+", range="+vmin_+".."+vmax_+"}";
  }

  /** A scale_left > scale_right, taking into account of the isReversed */
  public boolean isGT( double scale_left, double scale_right )
  {
    if( reverse_ ) return scale_right > scale_left;
    return scale_left > scale_right ;
  }
  /** A scale_left < scale_right, taking into account of the isReversed */
  public boolean isLT( double scale_left, double scale_right )
  {
    if( reverse_ ) return scale_right < scale_left;
    return scale_left < scale_right ;
  }
  /** A Math.max( scale_left, scale_right ) taking into account of the isReversed */
   public double max( double scale_left, double scale_right )
  {
    if( reverse_ ) return Math.min( scale_left, scale_right );
    return Math.max( scale_left, scale_right );
  }
  /** A min( scale_left, scale_right ) taking into account of the isReversed */
   public double min( double scale_left, double scale_right )
  {
    if( reverse_ ) return Math.max( scale_left, scale_right );
    return Math.min( scale_left, scale_right );
  }
  /** A isGT( scale_left, pixel ) && isLT( pixel, scale_right ) taking into account of the isReversed */
   public boolean isIN( double scale_left, double pixel, double scale_right )
  {
    if( reverse_ ) return (pixel < scale_left ) && ( scale_right < pixel );
    return (scale_left < pixel ) && ( pixel < scale_right );
  }

   public void setLocale(ULocale _locale) {
       locale_ = _locale; 
   }
}
