/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: View3D.java,v 1.2 2008/05/23 14:11:50 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


/*
 * Created on 25 juin 2003
 *
 */
package org.eclipse.tptp.platform.report.chart.internal;

import org.eclipse.tptp.platform.report.drawutil.internal.Point2D;
import org.eclipse.tptp.platform.report.drawutil.internal.Point3D;
import org.eclipse.tptp.platform.report.drawutil.internal.Vector3D;
import org.eclipse.tptp.platform.report.igc.util.internal.Rect;


/**
 * View3D is made to handle projection from 3D space to a 2D screen.
 * It have the ability to transform 3D point before projection using
 * ICoordTransformation.
 * (for example, that can be used to for a 3D drawing on a plan)
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class View3D
{
  /** parameters of the 3D view, angle,angle,distance,distance */
  protected double phi_,theta_,R_,D_;
  /** internal use after update(), coeficient of matrix projection */
  protected double a1,a2,a3,a4,a5,a6,a7,a8;
  /** bounding box used to scale plan */
  protected Box3D bb_; 
  /** internal use, translation for the screen */
  protected double screen_dx_, screen_dy_ ;
  /** internal use, scale for the screen */
  protected double Dx_,Dy_;
  
  /** simplissiest coordinate transformation ... none */
  protected static class NoTransformation implements ICoordTransformation
  {
    public double apply( double _c ) { return _c; }
  }
  protected static ICoordTransformation ct_idem_ = new NoTransformation();

  protected ICoordTransformation tX_, tY_, tZ_;
    
  /** 
   * Create a View 3D.
   * @param _phi angle of the view (radian).
   * @param _theta angle of the view (radian).
   * @param _R distance of the view from (0,0,0) point.
   * @param _D distance of the projection plan from the view.
   * @param _bounding_box 3D bounding box used to scale the view.
   */
  public View3D( double _phi, double _theta, double _R, double _D, Box3D _bounding_box )
  {
    bb_ = _bounding_box;
    phi_=_phi; theta_=_theta; R_=_R; D_=_D;
    tX_ = ct_idem_;
    tY_ = ct_idem_;
    tZ_ = ct_idem_;
    screen_dx_ = screen_dy_ = 0;
    Dx_=Dy_=1.0f;
    update();
  }

  /** update internal coeficient of the view */
  protected void update()
  {
    a1 = Math.sin(theta_);
    a2 = Math.sin(phi_  );
    a3 = Math.cos(theta_);
    a4 = Math.cos(phi_  );
    a5 = a3*a2;
    a6 = a1*a2;
    a7 = a3*a4;
    a8 = a1*a4; 
  }
  
  public double getD() { return D_; }
  public double getR() { return R_; }
  public double getPhi() { return phi_; }
  public double getTheta() { return theta_; }
  
  /** @return view vector (point of view), this vector is not normalized */
  public Vector3D getViewVector( Vector3D v )
  {
    if( v==null ) v=new Vector3D();
    v.setVector( a3*a4, a1*a4, a2 );
    return v;
  }
  
  /** Change D parameter view */
  public void setD( float _d ) { D_=_d; }

  /** Compute project of point (_x,_y,_z), store result in point _p,
   * @return _p (cannot be null).
   */
  public Point2D projection( double _x, double _y, double _z, Point2D _p )
  {
    double nx = tX_.apply(_x), 
           ny = tY_.apply(_y),
           nz = tZ_.apply(_z);
    
    double xo = ( -nx*a1 + ny*a3);
    double yo = ( -nx*a5 - ny*a6 + nz*a4);
//perspective: (non teste, plus lent, et risque de div/0)
    //float zo = -nx*a7 - ny*a8 - nz*a2 + R_;
    //if ( zo == 0.0f ) return false;
    //(*_px) = (int)(_view.left() + _view.width()/2  + D_*xo/zo );
    //(*_py) = (int)(_view.top()  + _view.height()/2 - D_*yo/zo );
//parallele: c'est bien, c'est rapide, c'est adopte quoi!
    _p.setX( screen_dx_ + Dx_*xo );
    _p.setY( screen_dy_ + Dy_*yo ); //les y sont a l'envers.
    return _p;
  }
  
  /** Compute project of point (_x,_y,_z), store result in point _p,
   * @return _p (cannot be null). Z coordinate of projected point contains depth value.
   */
  public void projection( double _x, double _y, double _z, Point3D _p )
  {
    double nx = tX_.apply(_x), 
           ny = tY_.apply(_y),
           nz = tZ_.apply(_z);
    
    double xo = ( -nx*a1 + ny*a3);
    double yo = ( -nx*a5 - ny*a6 + nz*a4);
//perspective: (non teste, plus lent, et risque de div/0)
    double zo = ( -nx*a7 - ny*a8 - nz*a2 + R_ );
    //if ( zo == 0.0f ) return false;
    //(*_px) = (int)(_view.left() + _view.width()/2  + D_*xo/zo );
    //(*_py) = (int)(_view.top()  + _view.height()/2 - D_*yo/zo );
//parallele: c'est bien, c'est rapide, c'est adopte quoi!
    _p.setX( screen_dx_ + Dx_*xo );
    _p.setY( screen_dy_ + Dy_*yo ); //les y sont a l'envers.    
    _p.setZ( zo );
  }
  
  /** Compute projection of point _p and store result in _j, @return _j */
  public final void projection( Point3D _p, Point3D _j )
  {
    projection( _p.getX(), _p.getY(), _p.getZ(), _j );
  }

  /** 
   * @return last parameter given to scaleTo() method, this is screen rectangle
   * where the 3D view is fitted
   */
  public Rect getScaledTo()
  {
    return new Rect( scr_x_, scr_y_, scr_w_, scr_h_ );
  }
  
  private int scr_x_,scr_y_,scr_w_,scr_h_; //result of scaleTo()
  /** Update view to fit given rectangle */
  public void scaleTo( int _scr_x, int _scr_y, int _scr_width, int _scr_height )
  {
    scr_x_ = _scr_x; scr_y_=_scr_y; scr_w_=_scr_width; scr_h_=_scr_height;
    
    //acte#1: compute points of bounding box, with the assumption of scale=1
    //        and no translation on plan.
    screen_dx_ = screen_dy_ = 0;
    D_ = 1.0f;
    Dx_=Dy_=1.0f;

    double scr_x_min= 1e8, scr_y_min= 1e8, scr_x_max=-1e8, scr_y_max=-1e8;
    Point2D p = new Point2D();
    double px,py;

    projection( bb_.getXMin(),bb_.getYMin(),bb_.getZMin(), p);
    px = p.getX(); py=p.getY();
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMin(),bb_.getYMin(),bb_.getZMax(), p);
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMax(),bb_.getYMin(),bb_.getZMax(), p );
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMax(),bb_.getYMin(),bb_.getZMin(), p );
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMin(),bb_.getYMax(),bb_.getZMax(), p );
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMax(),bb_.getYMax(),bb_.getZMax(), p );
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMax(),bb_.getYMax(),bb_.getZMin(), p );
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    projection( bb_.getXMin(),bb_.getYMax(),bb_.getZMin(), p );
    px = p.getX(); py=p.getY();    
    if(px<scr_x_min) scr_x_min=px; if(px>scr_x_max) scr_x_max=px;
    if(py<scr_y_min) scr_y_min=py; if(py>scr_y_max) scr_y_max=py;

    //acte#2: now we can known the dimension of project at scale=1
    //        then a better scale is guess.
    //        after all we can know the plan translation.
    double scr_w = scr_x_max-scr_x_min,
           scr_h = scr_y_max-scr_y_min ;

//    if ( (scr_w==0.0f)||(scr_h==0.0f) ) return false;
    //may throw null pointer exception
    double kx = Math.abs(_scr_width /scr_w); 
    double ky = Math.abs(_scr_height/scr_h); 

    if ( ky < kx ) D_ = ky; else D_=kx;
    Dx_=+D_;
    Dy_=-D_; //because Y are reversed

    int left = _scr_x, right = left + _scr_width,
        top  = _scr_y, bottom= top  + _scr_height;
        
    screen_dx_ = (int)( (left+right - D_*(scr_x_max+scr_x_min))/2 );
    screen_dy_ = (int)( (top+bottom + D_*(scr_y_max+scr_y_min))/2 );
  }

  /** @return X translation coeficient on screen */
  protected double screenDx() { return screen_dx_; }
  /** @return Y translation coeficient on screen */
  protected double screenDy() { return screen_dy_; }

  /** reset any coordinate transformation */
  public void resetCoordTransformation()
  {
    tX_ = tY_ = tZ_ = ct_idem_ ;
  }
  /** set coordinate transformation for X axis */
  public void setXCoordTransformation( ICoordTransformation _ct )
  {
    tX_ = ( _ct==null ) ? ct_idem_ : _ct ;    
  }
  /** set coordinate transformation for X axis */
  public void setYCoordTransformation( ICoordTransformation _ct )
  {
    tY_ = ( _ct==null ) ? ct_idem_ : _ct ;    
  }
  /** set coordinate transformation for X axis */
  public void setZCoordTransformation( ICoordTransformation _ct )
  {
    tZ_ = ( _ct==null ) ? ct_idem_ : _ct ;    
  }
}
