/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IGraphicDocumentGenerator.java,v 1.1 2005/03/30 08:26:49 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IGraphicDocumentGenerator.java,v 1.1 2005/03/30 08:26:49 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

 
import java.io.InputStream;
import java.io.OutputStream;

import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.w3c.dom.Document;


/**
 * <code>IGraphicDocumentGenerator</code> defines the API for an SVG generator.  
 * An SVG generator generates graphic in support of data visualization, such as line charts, 
 * bar charts, pie charts, etc. Input to the SVG generator consists of data and 
 * configuration information, and is formated in XML that conforms to <code>input.xsd</code>.  
 * The output charts are formated in Scalable Vector Graphics (SVG).
 * SVG is a W3C recommendation for describing two-dimensional graphics in XML.  
 */
public interface IGraphicDocumentGenerator {

	/**
	 * Generates a chart in SVG format given the input as an object model. 
	 * 
	 * @param input The document object model (DOM) representation of the input XML.
	 * @return the SVG output
	 * @throws SVGGeneratorException If input is invalid, or generation process failed
	 */
	Document generateGraphicDocument(Chart input) throws SVGGeneratorException;
	
	/**
	 * Generates a chart in SVG format given the input XML as a DOM object. 
	 * The input XML will be validated at runtime if <code>validateInput</code> is 
	 * set to true.
	 * 
	 * @param input The document object model (DOM) representation of the input XML.
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @return the SVG output
	 * @throws SVGGeneratorException If input is invalid, or generation process failed
	 */
	Document generateGraphicDocument(Document input, boolean validateInput) throws SVGGeneratorException;
	
	/**
	 * Generates a chart in SVG format given the input XML as an input stream. 
	 * The input XML will be validated at runtime if <code>validateInput</code> is 
	 * set to true.
	 *  
	 * @param input the input to the SVG generator as an input stream
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @return the SVG output
	 * @throws SVGGeneratorException If input is invalid, or generation process failed
	 */
	Document generateGraphicDocument(InputStream input, boolean validateInput) throws SVGGeneratorException;
	
	/**
	 * Generates a chart in SVG format given the input XML a DOM object, and writes the 
	 * output to the given output stream. The input XML will be validated
	 * at runtime if <code>validateInput</code> is set to true.
	 * 
	 * @param input The document object model (DOM) representation of the input XML.
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @param outputStream output stream
	 * @throws SVGGeneratorException If input is invalid, generation process failed, or failed to write to the given output stream
	 */
	void generateGraphicDocument(Document input, boolean validateInput, OutputStream outputStream) throws SVGGeneratorException;

	/**
	 * Generates a chart in SVG format given the input XML an input stream, and writes the 
	 * output to the given output stream. The input XML will be validated
	 * at runtime if <code>validateInput</code> is set to true.
	 * 
	 * @param input the input to the SVG generator as an input stream
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @param outputStream output stream
	 * @throws SVGGeneratorException If input is invalid, generation process failed, or failed to write to the given output stream
	 */
	void generateGraphicDocument(InputStream input, boolean validateInput, OutputStream outputStream) throws SVGGeneratorException;

	/**
	 * Serializes a <code>Document</code> object to a file in XML format. This is a
	 * convevience method to save the output of the SVG generator to a file.
	 * 
	 * @param generatedDocument the generated graphic
	 * @param filename The location to which the object is serialized.
	 * @return boolean <code>true</code>, if the operation is successful.
	 * @throws SVGGeneratorException If failed to write to the file specified
	 */
	boolean serializeGeneratedDocumentToFile(Document generatedDocument, String uri) throws SVGGeneratorException;
	
	/**
	 * Serializes a <code>Document</code> object to a <code>String</code> in XML format. 
	 * This is a convevience method to save the output of the SVG generator to a string.
	 * 
	 * @param generatedDocument the generated graphic
	 * @return String the XML-serialized form of the <code>Document</code>
	 * @throws SVGGeneratorException If XML serialization failed
	 */	
	String serializeGeneratedDocumentToString(Document generatedDocument) throws SVGGeneratorException;

}