/* ***********************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGGenerator.java,v 1.8 2006/05/08 19:09:26 sleeloy Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal;
/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGGenerator.java,v 1.8 2006/05/08 19:09:26 sleeloy Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.InputDOMDocument;
import org.eclipse.tptp.platform.report.chart.svg.internal.part.ChartArea;
import org.eclipse.tptp.platform.report.chart.svg.internal.part.DefaultResources;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.Utilities;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.XMLLoader;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;



/**
 * <code>SVGGenerator</code> is an implemenation of <code>IGraphicDocumentGenerator</code>.
 * An SVG generator generates graphic in support of data visualization, such as line charts, 
 * bar charts, pie charts, etc. Input to the SVG generator consists of data and 
 * configuration information, and is formated in XML that conforms to <code>input.xsd</code>.  
 * The output charts are formated in Scalable Vector Graphics (SVG).
 * SVG is a W3C recommendation for describing two-dimensional graphics in XML. 
 */
public class SVGGenerator implements IGraphicDocumentGenerator {
	
	static final String JAXP_SCHEMA_SOURCE = "http://java.sun.com/xml/jaxp/properties/schemaSource";
	
	/**
	 * The SVG version is "-//W3C//DTD SVG 1.0//EN".
	 */	
	static public final String SVG_VERSION = "-//W3C//DTD SVG 1.0//EN";
	
	/**
	 * The SVG DTD is "http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd".
	 */	
	static public final String SVG_DTD = "http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd";

	/**
	 * Default org.eclipse.tptp.platform.report.chart.svg.internal.resources
	 */
	private DefaultResources resources;

	/**
	 * Schema location
	 */
	private String schemaLocation;
	public Chart chart;
	/**
	 * @return Returns the chart.
	 */
	public Chart getChart() {
		return chart;
	}
	/**
	 * Instantiates an SVG generator that uses default color palettes and shapes.
	 */
	public SVGGenerator() {
		// create object to hold default org.eclipse.tptp.platform.report.chart.svg.internal.resources for this generator
		resources = new DefaultResources();
	}


	/**
	 * Instantiates an SVG generator that can use user-defined color palettes
	 * and shapes. The parameter to the constructor is an URI to the resource directory 
	 * that contains the specification of color palettes and/or shapes. The color palette
	 * specification is an XML file with filename palettes.xml, and conforms to 
	 * <code>palettes.xsd</code>.  The shape specification is an XML file with filename <code>shapes.xml</code> 
	 * and conforms to <code>shapes.xsd</code>.  
	 *  
	 * @param dir resource directory, in the form of an absolute path or a fully qualified URL
	 */
	public SVGGenerator(String dir) {
		String resourceDir;
		// If dir ends with a file separater, remove the last character.
		if (dir.endsWith("/") || dir.endsWith("\\")) {
			resourceDir = dir.substring(0, dir.length()-1);
		} else {
			resourceDir = dir;
		}

		// Use forward slashes so that paths in ECMA Script will work
		resourceDir = resourceDir.replace('\\', '/');
		
		// create object to hold default org.eclipse.tptp.platform.report.chart.svg.internal.resources for this generator
		resources = new DefaultResources(resourceDir);
	}
	
	/**
	 * Sets the location of the schema used to validate inputs of the chart generator.
	 * The location can be an absolute file path of the schema on the server, or
	 * the location can be specified as an HTTP URL. 
	 * <p>
	 * The schema location set with this API will override the schema location 
	 * specified in the input XML. 
	 * 
	 * @since version 3.0
	 * @param schemaLocation
	 */
	public void setSchemaLocation(String schemaLocation) {
		this.schemaLocation = schemaLocation;
	}


	/**
	 * Generates a chart in SVG format given the input XML as a DOM object. 
	 * The input XML will be validated at runtime if <code>validateInput</code> is 
	 * set to true.
	 * 
	 * @param input The document object model (DOM) representation of the input XML.
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @return the SVG output
	 * @throws SVGGeneratorException If input is invalid, or generation process failed
	 */
public Document generateGraphicDocument(Document input, boolean validateInput) throws SVGGeneratorException {
		if (validateInput) {
			// serialize document to string for validator
			String xml = serializeGeneratedDocumentToString(input);
			StringReader reader = new StringReader(xml);
			InputSource inputSource = new InputSource(reader);
			XMLLoader validator;
			if(schemaLocation != null) {
				validator = new XMLLoader(schemaLocation);
			} else {
				validator = new XMLLoader();
			}
			validator.setValidating(true);
			if (validator.load(inputSource) == false) {
				// xml is invalid
				throw new SVGGeneratorInvalidInputException(validator.getException());
			}
		}
		
		InputDOMDocument inputDoc = new InputDOMDocument(input);
		chart = inputDoc.getChart();		
		return generateGraphicDocument(inputDoc.getChart());
	}


	/**
	 * Generates a chart in SVG format given the input XML as an input stream. 
	 * The input XML will be validated at runtime if <code>validateInput</code> is 
	 * set to true.
	 *  
	 * @param input the input to the SVG generator as an input stream
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @return the SVG output
	 * @throws SVGGeneratorException If input is invalid, or generation process failed
	 */
	public Document generateGraphicDocument(InputStream input,
			boolean validateInput) throws SVGGeneratorException {
		BufferedInputStream in = new BufferedInputStream(input);
		BufferedReader reader = new BufferedReader(new InputStreamReader(in));
		Document inputDom;

		if (validateInput) {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			factory.setAttribute(JAXP_SCHEMA_SOURCE, new File(schemaLocation));
			try {
				InputSource inputSource = new InputSource(reader);
				DocumentBuilder builder = factory.newDocumentBuilder();
				inputDom = builder.parse(inputSource);
			} catch (Exception e) {
				throw new SVGGeneratorInvalidInputException(e);
			}

		} else {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			try {
				InputSource inputSource = new InputSource(reader);
				DocumentBuilder builder = factory.newDocumentBuilder();
				inputDom = builder.parse(inputSource);
			} catch (Exception e) {
				throw new SVGGeneratorInvalidInputException(e);
			}
		}

		InputDOMDocument inputDoc = new InputDOMDocument(inputDom);
		chart = inputDoc.getChart();
		return generateGraphicDocument(inputDoc.getChart());
	}

	/**
	 * Generates a chart in SVG format given the input XML a DOM object, and writes the 
	 * output to the given output stream. The input XML will be validated
	 * at runtime if <code>validateInput</code> is set to true.
	 * 
	 * @param input The document object model (DOM) representation of the input XML.
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @param outputStream output stream
	 * @throws SVGGeneratorException If input is invalid, generation process failed, or failed to write to the given output stream
	 */
	public void generateGraphicDocument(Document input, boolean validateInput, OutputStream outputStream) throws SVGGeneratorException {
		Document svgDoc = generateGraphicDocument(input, validateInput);
		writeDocumentToOutputStream(svgDoc, outputStream);
	}

	/**
	 * Generates a chart in SVG format given the input XML an input stream, and writes the 
	 * output to the given output stream. The input XML will be validated
	 * at runtime if <code>validateInput</code> is set to true.
	 * 
	 * @param input the input to the SVG generator as an input stream
	 * @param validateInput Set to true if the input should be validated, false otherwise.
	 * @param outputStream output stream
	 * @throws SVGGeneratorException If input is invalid, generation process failed, or failed to write to the given output stream
	 */
	public void generateGraphicDocument(InputStream input, boolean validateInput, OutputStream outputStream) throws SVGGeneratorException {
		Document svgDoc = generateGraphicDocument(input, validateInput);
		writeDocumentToOutputStream(svgDoc, outputStream);
	}

	/**
	 * Serializes a <code>Document</code> object to a file in XML format. This is a
	 * convevience method to save the output of the SVG generator to a file.
	 * 
	 * @param generatedDocument the generated graphic
	 * @param filename The location to which the object is serialized.
	 * @return boolean <code>true</code>, if the operation is successful.
	 * @throws SVGGeneratorException If failed to write to the file specified
	 */
	public boolean serializeGeneratedDocumentToFile(Document generatedDocument, String uri) throws SVGGeneratorException {
		if (generatedDocument == null) {
			return false;
		}
		
		OutputStreamWriter fWriter = null;
		FileOutputStream fs = null;
		
		try {
			fs = new FileOutputStream(uri);
		} catch (IOException ioe) {
			throw new SVGGeneratorException(ioe);
		} 
		
		writeDocumentToOutputStream(generatedDocument, fs);
		
		return true;		
	}


	/**
	 * Serializes a <code>Document</code> object to a <code>String</code> in XML format. 
	 * This is a convevience method to save the output of the SVG generator to a string.
	 * 
	 * @param generatedDocument the generated graphic
	 * @return String the XML-serialized form of the <code>Document</code>
	 * @throws SVGGeneratorException If XML serialization failed
	 */	
	public String serializeGeneratedDocumentToString(Document generatedDocument) throws SVGGeneratorException {
		if (generatedDocument == null) {
		return null;
	}
			OutputStreamWriter writer = null;
			ByteArrayOutputStream stream = new ByteArrayOutputStream();
			try {
				writer = new OutputStreamWriter(stream, "UTF-8");
	             DOMSource source = new DOMSource(generatedDocument);
	             StreamResult result = new StreamResult(writer);
				
	             TransformerFactory transFactory = 
		                                  TransformerFactory.newInstance();
	             Transformer transformer = transFactory.newTransformer();

	             transformer.transform(source, result);		
			} catch (Exception e) {
				// should never be here because UTF-8 is supported
				Utilities.assertion(false);
			}
										
		return stream.toString();
		
	}    	

	/**
	 * Generates SVG from object model
	 * 
	 * @since version 3.0
	 * @param input object model
	 * @return the SVG output
	 * @throws SVGGeneratorException
	 */
	public Document generateGraphicDocument(Chart input) throws SVGGeneratorException {
		Document generatedDocument = createSvgDocument();
		try {
			ChartArea svgChart = new ChartArea(input, resources);
	
			Element svgRoot = svgChart.doImplementation(generatedDocument);
			Element docRoot = generatedDocument.getDocumentElement();
			docRoot.appendChild(svgRoot);
			docRoot.setAttribute("onload","SVGGEN_init(evt)");
			docRoot.setAttribute("width", svgChart.getWidth());
			docRoot.setAttribute("height", svgChart.getHeight());
			//For some reason on some machines the xmlns attribute is not written to the file.
			//This is attribute is not necessary.  This attribute is necessary for firefox 1.5
			docRoot.setAttribute("xmlns","http://www.w3.org/2000/svg");
			docRoot.setAttribute("xmlns:xlink","http://www.w3.org/1999/xlink");
		} catch (Exception e) {
			e.printStackTrace();
			// catch any kind of exception during generation
			throw new SVGGeneratorException(e);
		}

		return generatedDocument;
	}

	/**
	 * Creates an SVG document and assigns width and height to the root "svg" element.
	 * 
	 * @param generatedDocument the SVG document
	 * @param data object containing the graphic properties
	 * @return Document the SVG document
	 * @throws SVGGeneratorException
	 */
	private Document createSvgDocument() throws SVGGeneratorException {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
		try {
			builder = factory.newDocumentBuilder();
			DOMImplementation domImpl = builder.getDOMImplementation();
			DocumentType dType = domImpl.createDocumentType("svg", SVG_VERSION, SVG_DTD);
			Document svgDocument = domImpl.createDocument("http://www.w3.org/2000/svg", "svg", dType);
			return svgDocument;
		} catch (ParserConfigurationException e) {
			throw new SVGGeneratorException(e);
		}
	}
	
	/**
	 * Writes the XML document to an output stream
	 * 
	 * @param svgDocument
	 * @param outputStream
	 * @throws SVGGeneratorException
	 */
	public void writeDocumentToOutputStream(Document svgDocument, OutputStream outputStream) throws SVGGeneratorException {
		if (svgDocument != null && outputStream != null) {
				OutputStreamWriter writer = null;

				try {
					writer = new OutputStreamWriter(outputStream, "UTF-8");
				} catch (UnsupportedEncodingException e) {
					// should never be here because UTF-8 is supported
					Utilities.assertion(false);
				}
				try {
										
	             DOMSource source = new DOMSource(svgDocument);
	             StreamResult result = new StreamResult(writer);
				
	             TransformerFactory transFactory = 
		                                  TransformerFactory.newInstance();
	             Transformer transformer = transFactory.newTransformer();

	             transformer.transform(source, result);		
				} catch (Exception ioe) {
					throw new SVGGeneratorException(ioe);
				} 
		}
		
	}    	
//	
//	public void writeDocumentToOutputStream(Document svgDocument, OutputStream outputStream) throws SVGGeneratorException {
//		if (svgDocument != null && outputStream != null) {
//			OutputStreamWriter writer = null;
//			try {
//				try {
//					writer = new OutputStreamWriter(outputStream, "UTF-8");
//				} catch (UnsupportedEncodingException e) {
//					// should never be here because UTF-8 is supported
//					Utilities.assertion(false);
//				}
//				
//				OutputFormat oFormat = new OutputFormat(svgDocument);
//				oFormat.setIndenting(true);
//				oFormat.setEncoding("UTF-8");
//				oFormat.setLineWidth(80);
//				oFormat.setLineSeparator("\n");
//		
//				XMLSerializer s = new XMLSerializer(writer, oFormat);
//				s.serialize(svgDocument);
//				
//				if (writer != null) {
//					writer.close();
//				}
//			} catch (IOException ioe) {
//				throw new SVGGeneratorException(ioe);
//			} 
//		} else {
//			if (outputStream != null) {
//				try {
//					outputStream.close();
//				} catch (IOException ioe) {
//					throw new SVGGeneratorException(ioe);
//				}
//			}
//		}
//	}
}
