/* ***********************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGGeneratorPreferences.java,v 1.2 2006/02/03 19:24:32 sleeloy Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGGeneratorPreferences.java,v 1.2 2006/02/03 19:24:32 sleeloy Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal;


import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.part.IConstants;
import org.eclipse.tptp.platform.report.chart.svg.internal.part.IGraphicTypeConstants;

import com.ibm.icu.util.StringTokenizer;


/**
 * <code>SVGGeneratorPreferences</code> is javabean object that contains getters and setters
 * for the user preferences provided by the SVG Chart Generator.
 * Explioters can use this object to set preferences for the various charts provided by the 
 * SVG Chart Generator.   To apply the preferences to an input model there is a method called
 * applyPreferences that takes in a <code>Chart</code> interface.  Once this method is called 
 * the state of the Preferences are applied to the Chart model.  The <code>SVGGeneratorPreferences</code>
 * provides default preferences.  The default preferences are as follows:
 * <p>
 * - Data value labels are hidden<br>
 * - Data percentage labels are hidden<br>
 * - Font Family is set to 'arial'<br>
 * - Font Weight is set to 'normal'<br>
 * - Font Style is set to 'normal'<br>
 * - Font Size is set to '11'<br>
 * - Font Color is set to '#000000'<br>
 * - Palette Id is set to 'SPRING'<br>
 * </p>
 * <p>
 * 
 * The following is how one would apply preferences.<br>
 * <pre>
 * // Create a line chart object
 * InputFactory factory = InputFactory.INSTANCE;
 * Chart chart = factory.createChart();
 * chart.setType("line");
 * 
 * // Add configuration node
 * Configuration configuration = factory.createConfiguration(chart);
 * chart.setConfiguration(configuration);
 * 
 * //Create a preference object with default settings
 * SVGGeneratorPreferences prefrences = new SVGGeneratorPreferences();
 * 
 * //Set preferences 
 * prefrences.setFontSize(8);
 * prefrences.setShowPercentage(true);
 * prefrences.setShowValues(false);
 * 
 * prefrences.applyPreferences(chart);
 * 
 * //Generate the graph with the preferences applied to the input model
 * Document svgDoc = generator.generateGraphicDocument(chart);
 * </pre>
 * </p>
 */
public class SVGGeneratorPreferences {
	/**
	 * preference diliminator used to create the user preference string blob.
	 */
	public static final String PREF_DELIMINATOR =",";
	/**
	 * preference diliminator used to create the user preference string blob.
	 */
	public static final String FONT_PREF_DELIMINATOR =";";
	/**
	 * preference diliminator used to create the user preference string blob.
	 */
	public static final String FONT_VAL_PREF_DELIMINATOR =":";

   protected boolean showValues = false;
   protected boolean defaultSet = true;
   protected boolean showPercentage = false;
   protected String fontFamily="arial";
   protected String fontWeight="normal";
   protected String fontStyle="normal";
   protected int fontSize=11;
   protected String fontColor="#000000";
   protected String paletteId="SPRING";
   protected String userPreferenceString;
/**
 * Gets the font color.
 * 
 * @return Returns the font color.
 */
public String getFontColor() {
	return fontColor;
}
/**
 * Sets the font color.
 * 
 * @param fontColor The font color to set.
 */
public void setFontColor(String fontColor) {
	this.fontColor = fontColor;
}
/**
 * Gets the font family.
 * 
 * @return Returns the font family.
 */
public String getFontFamily() {
	return fontFamily;
}
/**
 * Sets the font family.
 * 
 * @param fontFamily The font family to set.
 */
public void setFontFamily(String fontFamily) {
	this.fontFamily = fontFamily;
}
/**
 * Gets the font size.
 * 
 * @return Returns the font size.
 */
public int getFontSize() {
	return fontSize;
}
/**
 * Sets the font size.
 * 
 * @param fontSize The font size to set.
 */
public void setFontSize(int fontSize) {
	this.fontSize = fontSize;
}
/**
 * Gets the font style.
 * 
 * @return Returns the font style.
 */
public String getFontStyle() {
	return fontStyle;
}
/**
 * Sets the font style.
 * 
 * @param fontStyle The font style to set.
 */
public void setFontStyle(String fontStyle) {
	this.fontStyle = fontStyle;
}
/**
 * Gets the palette ID
 * @return Returns the palette Id.
 */
public String getPaletteId() {
	return paletteId;
}
/**
 * Sets the palette ID
 * 
 * @param paletteId The palette Id to set.
 */
public void setPaletteId(String paletteId) {
	this.paletteId = paletteId;
}
/**
 * This preference only applies to specific chart types (eg. pie charts).  The percentage labels are shown if this 
 * value is true otherwise the percentage labels are hidden.
 * @return Returns the show percentage.
 */
public boolean getShowPercentage() {
	return showPercentage;
}
/**
 * This preference only applies to specific chart types (eg. pie charts).  The percentage labels are shown if this 
 * value is true otherwise the percentage labels are hidden.
 * @param showPercentage The show percentage to set.
 */
public void setShowPercentage(boolean showPercentage) {
	this.showPercentage = showPercentage;
}
/**
 * Determines whether to show the data value labels, otherwise the data value labels are hidden.
 * @return Returns the show values state.
 */
public boolean getShowValues() {
	return showValues;
}
/**
 * Determines whether to show the data value labels, otherwise the data value labels are hidden.
 * @param showValues The show values to set.
 */
public void setShowValues(boolean showValues) {
	this.showValues = showValues;
	this.defaultSet = false;
}
/**
 * Returns a String blob that represents the state of the user preferences.
 * @return Returns the userPreferenceString.
 */
public String getUserPreferenceString() {
	userPreferenceString = this.paletteId+PREF_DELIMINATOR
							+this.fontSize+PREF_DELIMINATOR
							+"font-weight"+FONT_VAL_PREF_DELIMINATOR+this.fontWeight+FONT_PREF_DELIMINATOR
							+"font-style"+FONT_VAL_PREF_DELIMINATOR+this.fontStyle+FONT_PREF_DELIMINATOR
							+"font-family"+FONT_VAL_PREF_DELIMINATOR+this.fontFamily+FONT_PREF_DELIMINATOR
							+"fill"+FONT_VAL_PREF_DELIMINATOR+this.fontColor+FONT_PREF_DELIMINATOR+PREF_DELIMINATOR
							+(this.showValues?"1":"0")+PREF_DELIMINATOR
							+(this.showValues?"1":"0")+PREF_DELIMINATOR
							+(this.showPercentage?"1":"0");
		
	return userPreferenceString;
}
/**
 * Sets the user preference String blob.  This method is useful when getting the string blob from the client side.
 * @param userPreferenceString The userPreferenceString to set.
 */
public void setUserPreferenceString(String userPreferenceString) {
	this.userPreferenceString = userPreferenceString;
	updatePreferences();
}

protected void updatePreferences(){
	if (userPreferenceString == null) return;
	StringTokenizer st = new StringTokenizer(userPreferenceString, PREF_DELIMINATOR);
	if (st.hasMoreTokens())
		this.paletteId = st.nextToken();
	if (st.hasMoreTokens())
		this.fontSize = Integer.parseInt(st.nextToken());
	if (st.hasMoreTokens()){
		String fontPreferences = st.nextToken();
		StringTokenizer fontSt = new StringTokenizer(fontPreferences, FONT_PREF_DELIMINATOR);
		while (fontSt.hasMoreTokens()){
			setFontPreferences(fontSt.nextToken());
		}
	}
	if (st.hasMoreTokens())
		this.showValues = (Integer.parseInt(st.nextToken()) == 1);
	if (st.hasMoreTokens()){
		boolean showActual = (Integer.parseInt(st.nextToken()) == 1);
		if (showActual) showValues = showActual;
	}
	if (st.hasMoreTokens())
		this.showPercentage = (Integer.parseInt(st.nextToken()) == 1);
}

protected boolean setFontPreferences(String preference){
	StringTokenizer fontPrefSt = new StringTokenizer(preference, FONT_VAL_PREF_DELIMINATOR);
	if (fontPrefSt.hasMoreTokens()){
		String name = fontPrefSt.nextToken();
		if (name.trim().equalsIgnoreCase("font-weight")){
			if (fontPrefSt.hasMoreTokens())
				this.fontWeight = fontPrefSt.nextToken();			
		}
		else if (name.trim().equalsIgnoreCase("font-style")){
			if (fontPrefSt.hasMoreTokens()){
				this.fontStyle = fontPrefSt.nextToken();
			}
		}
		else if (name.trim().equalsIgnoreCase("font-family")){
			if (fontPrefSt.hasMoreTokens())
				this.fontFamily = fontPrefSt.nextToken();			
		}
		else if (name.trim().equalsIgnoreCase("fill")){
			if (fontPrefSt.hasMoreTokens())
				this.fontColor = fontPrefSt.nextToken();			
		}
		else
			return false;
	}
	return true;
}

/**
 * Gets the font weight.
 * 
 * @return Returns the fontWeight.
 */
public String getFontWeight() {
	return fontWeight;
}

/**
 * Sets the font weight.
 * 
 * @param fontWeight The fontWeight to set.
 */
public void setFontWeight(String fontWeight) {
	this.fontWeight = fontWeight;
}

/**
 * This is a helper method that should be used to apply the current state of the user preference object to a chart
 * input model.  Once the input model is created this method should be called to apply the user preference to the 
 * input model then the resulting input model should be fed into the svg chart generator to generate the chart.
 *  
 * @param inputModel the input model that contains the raw data and configuration data.
 * @return an input model that has the preference applied to it.  This is the same copy that is passed into this method.
 */
public Chart applyPreferences(Chart inputModel){
	Configuration configuration = inputModel.getConfiguration();
	configuration.getPalettes().setPaletteId(getPaletteId());
	//for pie charts the values are visible for all other charts the values are hidden.	
	if (defaultSet){
		if (inputModel.getType().equals(IGraphicTypeConstants.PIE_CHART) || inputModel.getType().equals(IGraphicTypeConstants.PIE_CHART3D)){
			this.showValues = true;
			this.showPercentage = true;
		}
	    else{	
			this.showValues = false;
			this.showPercentage = false;
	    }
    }	
	inputModel.getConfiguration().getDataValue().setPercentageVisibility((this.showPercentage)?IConstants.VISIBLE:IConstants.HIDDEN);
	inputModel.getConfiguration().getDataValue().setValueVisibility((this.showPercentage)?IConstants.VISIBLE:IConstants.HIDDEN);
	configuration.getPreferences().setStoredPreferences(getUserPreferenceString());
	return inputModel;
}

//public static final void main(String argv[]){
//	SVGGeneratorPreferences pref = new SVGGeneratorPreferences();
//	pref.setUserPreferenceString("SPRING,11,font-weight:normal; font-style:normal; font-family:arial; fill:#000000;,1,1,0");
//	System.out.println("FONT getFontColor: " + pref.getFontColor());
//	System.out.println("FONT getFontFamily: " + pref.getFontFamily());
//	System.out.println("FONT getFontStyle: " + pref.getFontStyle());
//	System.out.println("FONT getFontSize: " + pref.getFontSize());
//	System.out.println("FONT getPaletteId: " + pref.getPaletteId());
//	System.out.println("FONT getShowPercentage: " + pref.getShowPercentage());
//	System.out.println("FONT getShowValues: " + pref.getShowValues());
//	System.out.println("FONT getUserPreferenceString: " + pref.getUserPreferenceString());
//
//	pref.setUserPreferenceString("SPRING,11,font-weight:normal; font-style:normal; font-family:verdana; fill:#AE0000;,1,1,0");
//	System.out.println("FONT getFontColor: " + pref.getFontColor());
//	System.out.println("FONT getFontFamily: " + pref.getFontFamily());
//	System.out.println("FONT getFontStyle: " + pref.getFontStyle());
//	System.out.println("FONT getFontWeight: " + pref.getFontWeight());
//	pref.setFontColor("#AE9933");
//	pref.setFontSize(23);
//	pref.setShowPercentage(true);
//	pref.setShowValues(false);
//	System.out.println("FONT getFontSize: " + pref.getFontSize());
//	System.out.println("FONT getPaletteId: " + pref.getPaletteId());
//	System.out.println("FONT getShowPercentage: " + pref.getShowPercentage());
//	System.out.println("FONT getShowValues: " + pref.getShowValues());
//	System.out.println("FONT getUserPreferenceString: " + pref.getUserPreferenceString());
//	
//}

}
