/* ***********************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AxisCategory.java,v 1.3 2006/02/03 19:24:30 sleeloy Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AxisCategory.java,v 1.3 2006/02/03 19:24:30 sleeloy Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Axes;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Categories;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Category;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSets;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.FontHelper;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;

import com.ibm.icu.util.StringTokenizer;



/**
 * Implementation of an axis part. 
 */
public abstract class AxisCategory extends Axis {
	
	/**
	 * orientation of label text - horizontal or vertical
	 */
	protected short labelOrientation;
	
	/**
	 * Pointer to the data section of the input
	 */	
	protected Data data;

	/**
	 * Categories in a list.
	 */
	protected List categoryList = null;
	
	/**
	 * Number of categories
	 */
	protected int numOfCategories = 0;
	
	/**
	 * Axis title
	 */
	protected String title = null;

	/**
	 * A category axis has a number equally spaced categories. 
	 * groupBoundaryPosistions is an array of the coordinates that
	 * mark the boundary of the categories.
	 */	
	protected double[] groupBoundaryPositions;
	
	/**
	 * A category axis has a number equally spaced categories.
	 * groupDataPositions is an array of the coordinates that
	 * mark the midpoint of each category.
	 */
	protected double[] groupDataPositions;

	/**
	 * Constructor
	 * @param input
	 * @param axisOrientation
	 */
	public AxisCategory (Chart input, NLString nls) {
		this(input, HORIZONTAL, nls, 0);
	}

	/**
	 * Constructor
	 * @param input
	 * @param axisOrientation
	 * @param labelOrientation
	 */
	public AxisCategory (Chart input, short labelOrientation, NLString nls) {
		this(input, labelOrientation, nls, 0);
	}
	public AxisCategory (Chart input, short labelOrientation, NLString nls, double labelDegrees) {
		super(input, nls, labelDegrees);
		this.labelOrientation = labelOrientation;
		
		showMajorGridLines = false;
		showMinorGridLines = false;

		// get data tag
		data = input.getData();
		
		// get Categories
		DataSets dataSets = null;
		
		if (data != null) {
			dataSets = data.getDataSets();
			Categories categorySet = data.getCategories();
			if (categorySet != null) {
				categoryList = categorySet.getCategory();
				numOfCategories = categoryList.size();
			}
		}
		
		// get axis title
		Configuration config = input.getConfiguration();
		Axes axes = null; 
		if (config != null) {
			axes = config.getAxes();
			if (axes != null) {
				axisDef = axes.getIndependentAxis();
				if (axisDef != null) {
					if (axisDef.isSetLabelRotation())
						setLabelDegrees(axisDef.getLabelRotation());
					title = nls.getString(axisDef.getLabel());
					majorUnitDef = axisDef.getMajorUnit();
					minorUnitDef = axisDef.getMinorUnit();
					if (majorUnitDef != null) {
						if (majorUnitDef.isSetShowTick()) {
							showMajorTicks = majorUnitDef.isShowTick();
						}
						if (majorUnitDef.isSetShowGridLine()) {
							showMajorGridLines = majorUnitDef.isShowGridLine();
						}
					}
					if (minorUnitDef != null) {
						if (minorUnitDef.isSetShowTick()) {
							showMinorTicks = minorUnitDef.isShowTick();
						}
						if (minorUnitDef.isSetShowGridLine()) {
							showMinorGridLines = minorUnitDef.isShowGridLine();
						}
					}
				}
			}
		}
	}

	/**
	 * Constructs horizontal labels 
	 * @return
	 */	
	abstract protected SVGBase drawHorizontalLabels();
	
	/**
	 * Constructs vertical labels 
	 * @return
	 */
	abstract protected SVGBase drawVerticalLabels();
	
	/**
	 * Text wrapping: wrap the x label into rows if the width of the label is too long
	 * @param label x Label
	 */
	protected String[] wrapLabel(String label) {
		ArrayList wrapStrings = new ArrayList();

		StringTokenizer st = new StringTokenizer(label, " ");
		while (st.hasMoreElements()) {
			wrapStrings.add(st.nextElement().toString());
		}
		
		return (String[])wrapStrings.toArray(new String[wrapStrings.size()]);
	}

	/**
	 * Determine the number of rows needed for the axis
	 * @return number of rows
	 */
	protected int getMaxNumOfRowsOfLabels() {
		if (categoryList == null) return 0;
		
		int numOfRows = 0;
		
		for (Iterator i=categoryList.iterator(); i.hasNext();) {
			Category c = (Category) i.next();
			int labelLength = 0;
			if (c.getLabel() != null) {
				StringTokenizer tokenizer = new StringTokenizer(c.getLabel());
				if (tokenizer.countTokens() > numOfRows) {
					numOfRows = tokenizer.countTokens();
				}
			}
		}
		return numOfRows;
	}
	
	/**
	 * Determine the maximum length the labels
	 * @return maximum length of labels in pixels.
	 */
	protected double getMaxLabelLength() {
		if (categoryList == null) return 0;
		
		FontHelper fontHelper = new FontHelper();
		fontHelper.setFontSize(IConstants.AXIS_LABEL_FONT_SIZE);
		double maxLength = 0;
		
		for (Iterator i=categoryList.iterator(); i.hasNext();) {
			Category c = (Category) i.next();
			if (c.getLabel() != null) {
				double len = fontHelper.getStringWidth(nls.getString(c.getLabel()), labelDegrees, true);
				if (len > maxLength) {
					maxLength = len;
				}
			}
		}
		return maxLength;
	}

	/**
	 * Determine the maximum height the labels
	 * @return maximum height of labels in pixels.
	 */
	protected double getMaxLabelHeight() {
		if (categoryList == null) return 0;
		
		FontHelper fontHelper = new FontHelper();
		//need to adjust size
		fontHelper.setFontSize(IConstants.AXIS_LABEL_FONT_SIZE+2);
		double maxLength = 0;
		
		for (Iterator i=categoryList.iterator(); i.hasNext();) {
			Category c = (Category) i.next();
			if (c.getLabel() != null) {
				
				if (categoryList == null) return 0;
				
				int numOfRows = 0;
				
				String maxLabel = "";
				double maxStringHeight = 0;
				double maxStringWidth = 0;
				int maxLine = 1;

				StringTokenizer tokenizer = new StringTokenizer(c.getLabel());
				int lines = tokenizer.countTokens();
				while (tokenizer.hasMoreTokens()){
					String label = tokenizer.nextToken();
					double len = fontHelper.getStringWidth(label, 0);
					if (len > maxStringWidth) {
						maxStringWidth = len;
						maxLabel = label;
					}
				}
				if (lines > maxLine)
					 maxLine = lines;
				maxStringHeight = fontHelper.getHeight(maxLabel, labelDegrees, maxLine);
				if (maxLength < maxStringHeight) maxLength = maxStringHeight;
			}
		}
		return maxLength;
	}

	
	/**
	 * create an array of group boundary positions 
	 *
	 */
	private void setGroupBoundaryPositions() {
		groupBoundaryPositions = new double[numOfCategories + 1];
		if (numOfCategories > 0) {
			for (int i=0; i<groupBoundaryPositions.length; i++) {
				groupBoundaryPositions[i] = getAxisLength() * i / numOfCategories;
			}
		} else {
			groupBoundaryPositions[0] = getAxisLength();
		}
	}
	
	/**
	 * Create an array of data positions .
	 * Data positions are in the middle of each category.
	 *
	 */
	public void setGroupDataPositions() {
		groupDataPositions = new double[numOfCategories];
		double offset = getAxisLength() / (numOfCategories * 2);
		for (int i=0; i<groupDataPositions.length; i++) {
			groupDataPositions[i] = offset + getAxisLength() * i / numOfCategories;
		}
	}
	
	/**
	 * Get the array of group boundary poisitions
	 * @return boundary positions
	 */
	public double[] getGroupBoundaryPositions() {
		if (groupBoundaryPositions == null) {
			setGroupBoundaryPositions();
		}
		return groupBoundaryPositions;
	}
	
	/**
	 * Get the array of data poisitions
	 * @return data positions
	 */
	public double[] getGroupDataPositions() {
		if (groupDataPositions == null) {
			setGroupDataPositions();
		}
		return groupDataPositions;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.Axis#setLabelDegrees(double)
	 */
	public void setLabelDegrees(double labelDegrees) {
		if (this.getAxisOrientation() == VERTICAL) labelDegrees = -(labelDegrees-90); 
		
		super.setLabelDegrees(labelDegrees);
	}

}
