/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AxisNumber.java,v 1.5 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AxisNumber.java,v 1.5 2008/12/12 22:22:09 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.awt.geom.Dimension2D;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.ICustomFormatter;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Axes;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.CategoricalData;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Categories;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Category;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Coordinates;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.CustomFormat;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSet;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DateFormat;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Internationalization;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.NumberFormat;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.FontHelper;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;

import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.util.ULocale;



/**
 * 
 * <code>AxisNumber</code> 
 * $Revision: 1.5 $  
 */
public abstract class AxisNumber extends Axis {

	/**
	 * Minimum value of the number axis
	 */
	protected double min;
	
	/**
	 * Maximum value of the number axis
	 */
	protected double max;
	
	/**
	 * Major unit
	 */
	protected double majorUnit;
	
	/**
	 * Minor unit
	 */
	protected double minorUnit;
	
	/**
	 * Maximum value of the datasets for this axis
	 */ 
	protected double dataMax;

	/**
	 * Minimum value of the datasets for this axis
	 */ 
	protected double dataMin;
		
	/**
	 * Label orientation
	 */	
	protected short labelOrientation = HORIZONTAL;
	
	/**
	 * list of datasets associated with this axis
	 */
	protected List datasets = null;
	
	/**
	 * Values are cumulative (for stack bars)
	 */
	protected boolean cumulative = false;
		
	/**
	 * Margin between the axis and the top edge of the axis part.
	 * (leave room for the label text)
	 */
	protected final static int TOP_MARGIN = 10;

	/**
	 * axis type
	 */
	protected short axisType;
	
	/**
	 * axis title
	 */
	protected String title;

	/**
	 * custom format pattern
	 */
	protected String customPattern;

	/**
	 * internationalization timezone
	 */
	protected String timezone;
	
	/**
	 * Array of labels of this axis (text)
	 * (formatted values of major units)
	 */
	protected String[] labels;
	
	/**
	 * Numerical values of the major units 
	 */
	protected double[] majorUnitValues;
	
	/**
	 * Numerical values of the minor units 
	 */
	protected double[] minorUnitValues;
	
	/**
	 * Maximum number of characters among the labels
	 */
	protected double maxLabelLength = 0;
	protected double maxLabelHeight = 0;
	
	/**
	 * Scale of the axis - true if linear, false if log.
	 */
	protected boolean isLinear = true;
	
	/**
	 * formatter for numbers
	 */
	private com.ibm.icu.text.NumberFormat numberFormatter = null;
	
	/**
	 * custom formatter for numbers
	 */
	private ICustomFormatter customFormatter = null;

	/**
	 * formatter for dates/time
	 */
	private SimpleDateFormat dateFormatter = null;
	
	/**
	 * locale
	 */
	private ULocale locale = null;

	/**
	 * Constructor
	 * @param input
	 */
	public AxisNumber(Chart input, List datasets, short axisType, boolean cumulative, NLString nls) {
		this(input, datasets, axisType, cumulative, HORIZONTAL, nls, 0);
	}

	/**
	 * Constructor
	 * @param input
	 */
	public AxisNumber(Chart input, List datasets, short axisType, boolean cumulative, NLString nls, double labelDegrees) {
		this(input, datasets, axisType, cumulative, HORIZONTAL, nls, labelDegrees);
	}
	/**
	 * Constructor
	 * @param input
	 */
	public AxisNumber(Chart input, List datasets, short axisType, boolean cumulative, short labelOrientation, NLString nls) {
		this(input, datasets, axisType, cumulative, labelOrientation, nls, 0);
	}

	/**
	 * Constructor
	 * @param input
	 */
	public AxisNumber(Chart input, List datasets, short axisType, boolean cumulative, short labelOrientation, NLString nls, double labelDegrees) {
		super(input, nls, labelDegrees);
		this.datasets = datasets;
		this.axisType = axisType;
		this.cumulative = cumulative;
		this.labelOrientation = labelOrientation;
		
		dataMax = Double.MAX_VALUE*-1;
		dataMin = Double.MAX_VALUE;
		
		if (axisType == AXIS_TYPE_PRIMARY_DEPENDENT_AXIS) {
			showMajorGridLines = true;
			showMinorGridLines = true;
		} else if (axisType == AXIS_TYPE_SECONDARY_DEPENDENT_AXIS ||
			axisType == AXIS_TYPE_INDEPENDENT_AXIS) {
			showMajorGridLines = false;
			showMinorGridLines = false;
		}
		
		analyzeDataSets();
		initDataFormatter();
		getTickMarkValuesAndLabels();
	}

	/**
	 * Sets the length of the axis.  The method is used by the parent container.
	 * The information is dependent on other axes.
	 * 
	 * @param length
	 */
	public abstract void setAxisLength(double length);

	/**
	 * Get axis length
	 * @return axis length
	 */
	public abstract double getAxisLength();

	/**
	 * Given the coordinates, relative to the parent container, of the crossing point
	 * of the two axes, translate the axis part to the approprate position.
	 * @param x
	 * @param y
	 */
	abstract public void setCrossPoint(double x, double y);

	/**
	 * Constructs horizontal labels 
	 * @param positions
	 * @return
	 */
	abstract protected SVGBase drawHorizontalLabels(double[] positions);
	
	/**
	 * Constructs vertical labels 
	 * @param positions
	 * @return
	 */
	abstract protected SVGBase drawVerticalLabels(double[] positions);
	
	/**
	 * Get axis orientation
	 * @return axis orientation
	 */
	abstract public short getAxisOrientation();

	/**
	 * Get max,min values for this axis
	 */
	private void analyzeDataSets() {
		// get axis configuration
		Configuration config = input.getConfiguration();
		Axes axes = null;

		// get axis definition		
		if (config != null) {
			axes = config.getAxes();
			dataValueDef = config.getDataValue();
			if (axes != null) {
				if (axisType == AXIS_TYPE_INDEPENDENT_AXIS) {
					axisDef = axes.getIndependentAxis();
				} else if (axisType == AXIS_TYPE_PRIMARY_DEPENDENT_AXIS) {
					axisDef = axes.getPrimaryDependentAxis();
				} else {
					axisDef = axes.getSecondaryDependentAxis();
				}
			}

			// get locale settings
			Internationalization i18n = config.getInternationalization();
			if (i18n != null) {
				String language = i18n.getLanguage() == null ? "" : i18n.getLanguage();
				String country = i18n.getCountry() == null ? "" : i18n.getCountry();
				timezone = i18n.getTimezone() == null ? "" : i18n.getTimezone();
				if ((language != "") && (country != ""))
					locale = new ULocale(language, country);
				else if (language != "")
					locale = new ULocale(language);
			}
		}
			
		if (axisDef != null) {
			// get axis title
			if (axisDef.isSetLabelRotation())
				setLabelDegrees(axisDef.getLabelRotation());
			title = nls.getString(axisDef.getLabel());
			
			if (axisDef.getScale() != null && axisDef.getScale().equals("log")) {
				isLinear = false;
			} else {
				isLinear = true;
			}

			// get or calculate max, min, major unit and minor unit
			boolean findOrderOfMagnitude = false;
			boolean axisMaxKnown = false;
			boolean axisMinKnown = false;
			boolean majorUnitKnown = false;
			double orderOfMagnitude = 1;
			
			// find user defined max, min values
			if (axisDef.isSetMax()) {
				if (isLinear) {
					max = axisDef.getMax();
					axisMaxKnown = true;
				} else if (axisDef.getMax() > 0) {
					max = Math.log(Math.abs(axisDef.getMax()))/Math.log(10);
					axisMaxKnown = true;
				}
			}
			if (axisDef.isSetMin()) {
				if (isLinear) {
					min = axisDef.getMin();
					axisMinKnown = true;
				} else if (axisDef.getMin() > 0) {
					min = Math.log(Math.abs(axisDef.getMin()))/Math.log(10);
					axisMinKnown = true;
				}
			}
			if (axisMaxKnown && axisMinKnown && (max == min)) {
				// max and min cannot be the same value because (1) it doesn't make sense, and 
				// (2) it cause error in coordinate calculations
				// therefore, ignore the max and min settings if they are the same and use the 
				// calculated values
				axisMaxKnown = false;
				axisMinKnown = false;
			}
			
			// find user defined major and minor units
			majorUnitDef = axisDef.getMajorUnit();
			minorUnitDef = axisDef.getMinorUnit();
			if (!isLinear) {
				majorUnit = 1;
				majorUnitKnown = true;
			} else if (majorUnitDef != null && majorUnitDef.isSetValue()) {
				majorUnit = Math.abs(majorUnitDef.getValue());
				if (majorUnit != 0) {
					majorUnitKnown = true;
				}
			} else {
				findOrderOfMagnitude = true;
			}
			
			if (majorUnitDef != null) {
				if (majorUnitDef.isSetShowTick()) {
					showMajorTicks = majorUnitDef.isShowTick();
				}
				if (majorUnitDef.isSetShowGridLine()) {
					showMajorGridLines = majorUnitDef.isShowGridLine();
				}
			}
			if (minorUnitDef != null) {
				if (minorUnitDef.isSetShowTick()) {
					showMinorTicks = minorUnitDef.isShowTick();
				}
				if (minorUnitDef.isSetShowGridLine()) {
					showMinorGridLines = minorUnitDef.isShowGridLine();
				}
			}
			
			if (axisMaxKnown == false || axisMinKnown == false || findOrderOfMagnitude) {
				getDataMaxMin();
				if (findOrderOfMagnitude) {
					double magMin = dataMin;
					double magMax = dataMax;
					if (axisMinKnown) magMin = min;
					if (axisMaxKnown) magMax = max;
					orderOfMagnitude = getOrderOfMagnitude(magMax, magMin);
				}
				if (!isLinear && (dataMax <= 0 || dataMin <= 0)) {
					// If we are using log scale, and if max or min value is zero or negative, 
					// some error must have occurred or the input is invalid.  For example,
					// if all data points in the input are negative number, max will be less than 0.
					// In the case, set max and min to an arbitrary but valid value.
					// Note: the values 1 and 0 below are log values
					max = 1;
					min = 0;
					axisMaxKnown = true;
					axisMinKnown = true;
				}
			}
			
			if (!majorUnitKnown) {
				if ((dataMax - dataMin)/orderOfMagnitude > 5) {
					majorUnit = orderOfMagnitude * 2;
				} else {
					majorUnit = orderOfMagnitude;
				}
			}
			
			if (isLinear) {
				if (!axisMaxKnown) {
					if (dataMax < 0) {
						max = 0;
					} else {
						// find axisMax from dataMax and major unit
						max = Math.floor(dataMax / majorUnit) * majorUnit + majorUnit;
					}
				}
				if (!axisMinKnown) {
					// find axisMin from dataMin and major unit
					if (dataMin >= 0) {
						min = 0;
					} else {
						min = (Math.floor(Math.abs(dataMin) / majorUnit) * majorUnit + majorUnit) * (-1); 
					}
				}
			} else {
				if (!axisMaxKnown) {
					if (dataMax > 0) {
						max = Math.round(Math.log(getOrderOfMagnitude(dataMax) * 10)/Math.log(10));
					} else {
						// This is an error state. We just assign an arbitrary value to max.
						max = 1;
					}
				}
				if (!axisMinKnown) {
					if (dataMin > 0) {
						min = Math.round(Math.log(getOrderOfMagnitude(dataMin))/Math.log(10));
						// make sure min is less than the smallest value.  
						if (Math.round(Math.log(dataMin)/Math.log(10)) == min) {
							min -= 1;
						}
					} else {
						// This is an error state.  We just assign an arbitrary value to min.
						min = 0;
					}
				}
			}
			
			// if max < min, exchange values of max and min.
			if (max < min) {
				double tmp = max;
				max = min;
				min = tmp;
			}
			
			if (axisDef.getMinorUnit() != null && axisDef.getMinorUnit().isSetValue()) {
				minorUnit = Math.abs(minorUnitDef.getValue());
				if (minorUnit == 0) {
					minorUnit = majorUnit / 2; 
				}
			} else {
				// if linear scale, minor unit = major unit / 2
				minorUnit = majorUnit / 2;
			}
 		} else {
 			// no axes preference
 			// find data max and data min
 			getDataMaxMin();
 			// find find magnitude
 			if (isLinear) {
				double orderOfMagnitude = getOrderOfMagnitude(dataMax, dataMin);
				if ((dataMax - dataMin)/orderOfMagnitude > 5) {
					majorUnit = orderOfMagnitude * 2;
				} else {
					majorUnit = orderOfMagnitude;
				}
				
				// minor unit - not used by log scale
				minorUnit = majorUnit / 2;
 			} else {
 				majorUnit = 1;
 			}
 			// find axis max and axis min
 			if (isLinear) {
				if (dataMax < 0) {
					max = 0;
				} else {
					max = Math.floor(dataMax / majorUnit) * majorUnit + majorUnit;
				}
				// find axisMin from dataMin and major unit
				if (dataMin >= 0) {
					min = 0;
				} else {
					min = (Math.floor(Math.abs(dataMin) / majorUnit) * majorUnit + majorUnit) * (-1);
				}
			} else {
				max = Math.round(Math.log(getOrderOfMagnitude(dataMax) * 10)/Math.log(10));
 				min = Math.round(Math.log(getOrderOfMagnitude(dataMin))/Math.log(10));
			}
 		}
	}
	
	/**
	 * Find maximum and minimum values to be plotted.
	 *
	 */
	private void getDataMaxMin() {
		if (datasets != null) {
			if (cumulative == false) {
				for (Iterator i=datasets.iterator(); i.hasNext();) {
					DataSet dataset = (DataSet) i.next();
	
					List dataPoints = dataset.getDataPoint();
					List coordinates = dataset.getCoordinates();
					if (dataPoints != null && dataPoints.isEmpty() == false) {
						// get max, min from datapoint list
						findMaxMinFromDataPoints(dataPoints);
					} else if (coordinates != null && coordinates.isEmpty() == false) {
						boolean useValue1 = false;
						if (axisType == AXIS_TYPE_INDEPENDENT_AXIS) {
							useValue1 = true;
						}
						findMaxMinFromCoordinates(coordinates, useValue1);
					}
				}
				// dataMax < dataMin means that there is no data point.
				if (isLinear && (dataMax < dataMin)) {
					dataMax = 1;
					dataMin = 0;
				}
			} else { 
				// for stack bars, only categorical data allowed
				Data data = input.getData();
				List categoryList = null;
				if (data != null) {
					Categories categories = data.getCategories();
					if (categories != null) {
						categoryList = categories.getCategory();
					}
				}

				if(categoryList != null){
					double[] accumulatedHighValues = new double[categoryList.size()];	
					double[] accumulatedLowValues = new double[categoryList.size()];
		
					// initialize values
					for(int j=0; j < categoryList.size(); j++) {
						accumulatedHighValues[j] = 0;
						accumulatedLowValues[j] = 0;					
					}
		
					for (Iterator i=datasets.iterator(); i.hasNext();) {
						DataSet dataset = (DataSet) i.next();
		
						List dataPoints = dataset.getDataPoint();
						if (dataPoints != null && dataPoints.isEmpty() == false) {
							// get max, min from datapoint list
							findMaxMinFromDataPoints(dataPoints, categoryList, accumulatedHighValues, accumulatedLowValues);
						}
					}
					// dataMax < dataMin means that there is no data point.
					if (isLinear && (dataMax < dataMin)) {
						dataMax = 1;
						dataMin = 0;
					}
				}else {
					// no data in input XML
					if (isLinear) {
						dataMax = 1;
						dataMin = 0;
					} else {
						dataMax = 9;
						dataMin = 1;
					}
				}
			}
		} else {
			// no data in input XML
			if (isLinear) {
				dataMax = 1;
				dataMin = 0;
			} else {
				dataMax = 9;
				dataMin = 1;
			}
		}
	}
	
	private void findMaxMinFromDataPoints(List dataPoints) {
		double currentMax = dataMax;
		double currentMin = dataMin;
		double currentValue;
		
		if (isLinear) {
			double factor = Math.pow(10, findFactor(dataPoints));
			currentMax = currentMax*factor;
			currentMin = currentMin*factor;
			for (Iterator i=dataPoints.iterator(); i.hasNext();) {
				CategoricalData datum = (CategoricalData) i.next();
				currentValue = datum.getValue()*factor;
				if (currentValue < currentMin) {
					currentMin = currentValue;
				}
				
				if (currentValue > currentMax) {
					currentMax = currentValue;
				}
			}
			currentMin = currentMin/factor;
			currentMax = currentMax/factor;
		} else {
			for (Iterator i=dataPoints.iterator(); i.hasNext();) {
				CategoricalData datum = (CategoricalData) i.next();
				currentValue = datum.getValue();
				if (currentValue > 0 && currentValue < currentMin) {
					currentMin = currentValue;
				}
				if (currentValue > 0 && currentValue > currentMax) {
					currentMax = currentValue;
				}
			}
		}
		
		dataMax = currentMax;
		dataMin = currentMin;
	}
	
	private int findMaxFactor(int oldFactor, double value){
		if ((Math.abs(value)*(Math.pow(10, oldFactor)) < 1) &&(Math.abs(value)*(Math.pow(10, oldFactor)) > 0)) {
			return findMaxFactor(oldFactor+1, value);
		}
		else if (Math.abs(value)*(Math.pow(10, oldFactor)) >= 10){
			return findMaxFactor(oldFactor-1, value);
		}
		return oldFactor;
		
	}
//	public static void main(String argv[]){
//		double[] list = new double[3];
//		list[0]= -0.00001;
//		list[1] = -0.001;
//		list[2] = -0.00002;
//		int factor= findFactor(list);
//		System.out.println(Math.pow(10,2) + " " + list[0] + " " + list[1] + " " +list[2]);
////		System.out.println(Math.pow(10,2) + " " + newList[0] + " " + newList[1] + " " +newList[2]);
//	}
	private int findFactor(List values){
	double[] transformedValue = new double[values.size()];
	int factor = 1;
	int maxFactor = Integer.MAX_VALUE;
	boolean assigned = false;
	
	for (Iterator i=values.iterator(); i.hasNext();) {
		CategoricalData datum = (CategoricalData) i.next();
		double currentValue = datum.getValue();
		if (Math.abs(currentValue) < 1){
			factor = findMaxFactor(factor, currentValue);
			if (factor < maxFactor ) maxFactor = factor;
//			if ((newMaxFactor < maxFactor)){
//				maxFactor = newMaxFactor;
//				System.out.println("max " + factor);
//				assigned = true;
//			}
		}
		else
			return 0;
	}
	if (maxFactor == Integer.MAX_VALUE) maxFactor = 0;
//	System.out.println("maxFac " + maxFactor);
//	for (int x = 0; x  < value.length; x++){
//		transformedValue[x] = value[x]*(Math.pow(10,maxFactor));
//	}
	return maxFactor;
	
}
	
	private int findFactor(List values, boolean valueFlag){
		double[] transformedValue = new double[values.size()];
		int factor = 1;
		int maxFactor = Integer.MAX_VALUE;
		boolean assigned = false;
		
		for (Iterator i=values.iterator(); i.hasNext();) {
			Coordinates datum = (Coordinates) i.next();
			double currentValue = (valueFlag ? datum.getValue1() : datum.getValue2());
			if (Math.abs(currentValue) < 1){
				factor = findMaxFactor(factor, currentValue);
				if (factor < maxFactor ) maxFactor = factor;
//				if ((newMaxFactor < maxFactor)){
//					maxFactor = newMaxFactor;
//					System.out.println("max " + factor);
//					assigned = true;
//				}
			}
			else
				return 0;
		}
		if (maxFactor == Integer.MAX_VALUE) maxFactor = 0;
//		System.out.println("maxFac " + maxFactor);
//		for (int x = 0; x  < value.length; x++){
//			transformedValue[x] = value[x]*(Math.pow(10,maxFactor));
//		}
		return maxFactor;
		
	}

	
	private int findFactor(double[] value){
		double[] transformedValue = new double[value.length];
		int factor = 1;
		int maxFactor = Integer.MAX_VALUE;
		boolean assigned = false;
		for (int x = 0; x  < value.length; x++){
			if (Math.abs(value[x]*10) < 1){
				factor = findMaxFactor(factor, value[x]);
				if (factor < maxFactor ) maxFactor = factor;
//				if ((newMaxFactor < maxFactor)){
//					maxFactor = newMaxFactor;
//					assigned = true;
//				}
			}
			else
				return 0;
		}
		if (maxFactor == Integer.MAX_VALUE) maxFactor = 0;
//		System.out.println("maxFac " + maxFactor);
//		for (int x = 0; x  < value.length; x++){
//			transformedValue[x] = value[x]*(Math.pow(10,maxFactor));
//		}
		return maxFactor;
		
	}
	
	private void findMaxMinFromDataPoints(List dataPoints, List categoryList, double[] accumulatedHighValues, double[] accumulatedLowValues) {
		// Add the values of this dataset to the appropriate element in the
		// accumulated values arrays.  Note that the data points may not be 
		// in the order of the categories in the category list.  			
		for (Iterator i=dataPoints.iterator(); i.hasNext();) {
			CategoricalData datum = (CategoricalData) i.next();
			double currentValue = datum.getValue();
			Category category = (Category) datum.getCategoryId();
			int categoryIndex = categoryList.indexOf(category);
			if (currentValue >= 0.0) {
				accumulatedHighValues[categoryIndex] += currentValue;
			} else {
				accumulatedLowValues[categoryIndex] += currentValue;
			}
		}
		int highFactor = findFactor(accumulatedHighValues);
		int lowFactor = findFactor(accumulatedLowValues);
		double factor = Math.pow(10, (highFactor< lowFactor)? highFactor: lowFactor);

		// find dataMax and dataMin
		double currentMax = dataMax;
		double currentMin = dataMin;
		for(int j = 0; j < categoryList.size(); j++) {
			if (isLinear) {
				if (currentMax < accumulatedHighValues[j]*factor) {
					currentMax = accumulatedHighValues[j]*factor;
				}
				if (currentMin > accumulatedLowValues[j]*factor) {
					currentMin = accumulatedLowValues[j]*factor;
				}
			} else {
				// for log scale, we only have positive values
				if (currentMax < accumulatedHighValues[j] && accumulatedHighValues[j] > 0) {
					currentMax = accumulatedHighValues[j];
				}
				if (currentMin > accumulatedHighValues[j] && accumulatedHighValues[j] > 0) {
					currentMin = accumulatedHighValues[j];
				}
			}
		}
		if (isLinear) {
			dataMax = currentMax/factor;
			dataMin = currentMin/factor;
		} 
		else{
			dataMax = currentMax;
			dataMin = currentMin;			
		}
	}

	
	private void findMaxMinFromCoordinates(List coordinates, boolean value1) {
		double currentValue;
		double factor = Math.pow(10, findFactor(coordinates, value1));
		double currentMax = dataMax*factor;
		double currentMin = dataMin*factor;
			
		for (Iterator i=coordinates.iterator(); i.hasNext();) {
			Coordinates datum = (Coordinates) i.next();
			currentValue = (value1 ? datum.getValue1() : datum.getValue2())*factor;
			if (currentValue < currentMin) {
				currentMin = currentValue;
			}
			if (currentValue > currentMax) {
				currentMax = currentValue;
			}
		}
		
		dataMax = currentMax/factor;
		dataMin = currentMin/factor;
	}
	private double getOrderOfMagnitude(double dataMax, double dataMin) {
		if ((Math.abs(dataMax) < 1) && (Math.abs(dataMin) < 1)){
			double minFactor = findMaxFactor(1, dataMin);
			double maxFactor = findMaxFactor(1, dataMax);
			double factor = minFactor;
			if (dataMax == 0)
				factor = Math.pow(10,minFactor);
			else if (dataMin ==0)
				factor = Math.pow(10,maxFactor);
			else
				factor = Math.pow(10, (minFactor < maxFactor)?minFactor:maxFactor);
			return getOrderOfMagnitudeLargeNumber(dataMax*factor, dataMin*factor)/factor;			
		}
		return getOrderOfMagnitudeLargeNumber(dataMax, dataMin);
	}

	private double getOrderOfMagnitudeLargeNumber(double dataMax, double dataMin) {
		double range; 
		double magnitude = 1;
		if (dataMin >= 0) {
			range = dataMax;
		} else if (dataMax <= 0) {
			range = Math.abs(dataMin);
		} else {
			range = dataMax - dataMin;
		}
		while (range > 10) {
			range /= 10;
			magnitude *= 10;
		}
		return magnitude;
	}
	
	private double getOrderOfMagnitude(double num) {
		double magnitude = 1;
		double number = Math.abs(num);
		if (number >= 1) {
			while (number >= 10) {
				number /= 10;
				magnitude *= 10;
			}
		} else if (number != 0) {
			while (number < 1) {
				number *= 10;
				magnitude *= 10;
			}
			magnitude = 1 / magnitude;
		}
		return magnitude;
	}
	
	/**
	 * Find the appoximate width of the labels
	 * @return
	 */
	protected double getLabelWidth() {
		return maxLabelLength+3;
	}
	
	/**
	 * Find the appoximate height of the labels
	 * @return
	 */
	protected double getLabelHeight(){
		return maxLabelHeight;
	}
	
	/**
	 * Determine the labels for a number axis. Labels are shown for major units only.
	 * 
	 */
	protected void getTickMarkValuesAndLabels() {
		FontHelper fontHelper = new FontHelper();
		fontHelper.setFontSize(IConstants.AXIS_LABEL_FONT_SIZE);
		
		int numOfLabels = (int) ((max - min) / majorUnit + 1);
		int numOfMinorDivisions;
		if (isLinear) {
			numOfMinorDivisions = (int) ((max - min) / minorUnit + 1);
		} else {
			numOfMinorDivisions = ((int)(Math.ceil(max) - Math.floor(min))) * 8;
		}
		labels = new String[numOfLabels];
		majorUnitValues = new double[numOfLabels];
		minorUnitValues = new double[numOfMinorDivisions];
		
		if (min <= 0 && max >= 0) {
			// major unit
			int numOfNegLabels = (int) (Math.floor(Math.abs(min)/majorUnit));
			for (int i=0; i<numOfNegLabels; i++) {
				double labelValue = majorUnit * (numOfNegLabels - i) * -1;
				majorUnitValues[i] = labelValue;
				double labelValueForFormat = isLinear ? labelValue : Math.pow(10, labelValue);
				String labelText = formatValue(labelValueForFormat);
				labels[i] = labelText;
				Dimension2D labelLength = fontHelper.getStringBounds(labelText, labelDegrees);
				if (labelLength.getWidth() > maxLabelLength) {
					maxLabelLength = labelLength.getWidth();
				}
				if (labelLength.getHeight() > maxLabelHeight) {
					maxLabelHeight = labelLength.getHeight();
				}
			}
			// zero is in the positive list
			int numOfPositiveLabels = (int) (Math.floor(max/majorUnit) + 1);
			for (int i=0; i<numOfPositiveLabels; i++) {
				double labelValue = i * majorUnit;
				majorUnitValues[numOfNegLabels + i] = labelValue;
				double labelValueForFormat = isLinear ? labelValue : Math.pow(10, labelValue);
				String labelText = formatValue(labelValueForFormat);
				labels[numOfNegLabels + i] = labelText;
				Dimension2D labelLength = fontHelper.getStringBounds(labelText, labelDegrees);
				if (labelLength.getWidth() > maxLabelLength) {
					maxLabelLength = labelLength.getWidth();
				}
				if (labelLength.getHeight() > maxLabelHeight) {
					maxLabelHeight = labelLength.getHeight();
				}
			}
			// minor unit
			if (isLinear) {
				numOfNegLabels = (int) (Math.floor(Math.abs(min)/minorUnit));
				for (int i=0; i<numOfNegLabels; i++) {
					double labelValue = minorUnit * (numOfNegLabels - i) * -1;
					minorUnitValues[i] = labelValue;
				}
				// zero is in the positive list
				numOfPositiveLabels = (int) (Math.floor(max/minorUnit) + 1);
				for (int i=0; i<numOfPositiveLabels; i++) {
					double labelValue = i * minorUnit;
					minorUnitValues[numOfNegLabels + i] = labelValue;
				}
			} else {
				// log scale
				int minorUnitIndex = 0;
				double refValue = Math.floor(min);
				for (int i=0; i<numOfMinorDivisions/8; i++, refValue++) {
					double refValueLinear = Math.pow(10, refValue);
					double order = getOrderOfMagnitude(refValueLinear);
					for (int j=1; j<=8; j++) {
						minorUnitValues[minorUnitIndex++] = Math.log(refValueLinear + order * j)/Math.log(10);
					}
				}
			}
		} else if (min >= 0) {
			// major units
			for (int i=0; i<numOfLabels; i++) {
				double labelValue;
				if (isLinear) {
					labelValue = min + i * majorUnit;
				} else {
					labelValue = Math.ceil(min) + i * majorUnit;
				}
				majorUnitValues[i] = labelValue;
				double labelValueForFormat = isLinear ? labelValue : Math.pow(10, labelValue);
				String labelText = formatValue(labelValueForFormat);
				labels[i] = labelText;
				Dimension2D labelLength = fontHelper.getStringBounds(labelText, labelDegrees);
				if (labelLength.getWidth() > maxLabelLength) {
					maxLabelLength = labelLength.getWidth();
				}
				if (labelLength.getHeight() > maxLabelHeight) {
					maxLabelHeight = labelLength.getHeight();
				}
			}
			// minor units
			if (isLinear) {
				for (int i=0; i<minorUnitValues.length; i++) {
					double labelValue = min + i * minorUnit;
					minorUnitValues[i] = labelValue;
				}
			} else {
				// log scale
				int minorUnitIndex = 0;
				double refValue = Math.floor(min);
				for (int i=0; i<numOfMinorDivisions/8; i++, refValue++) {
					double refValueLinear = Math.pow(10, refValue);
					double order = getOrderOfMagnitude(refValueLinear);
					for (int j=1; j<=8; j++) {
						minorUnitValues[minorUnitIndex++] = Math.log(refValueLinear + order * j)/Math.log(10);
					}
				}
			}
		} else if (max <= 0) {
			// axis with log scale should never reach here
			//major units
			for (int i=1; i <= numOfLabels; i++) {
				double labelValue = max - majorUnit * (numOfLabels - i);
				majorUnitValues[i-1] = labelValue;
				double labelValueForFormat = isLinear ? labelValue : Math.pow(10, labelValue);
				String labelText = formatValue(labelValueForFormat);
				labels[i-1] =  labelText;
				Dimension2D labelLength = fontHelper.getStringBounds(labelText, labelDegrees);
				if (labelLength.getWidth() > maxLabelLength) {
					maxLabelLength = labelLength.getWidth();
				}
				if (labelLength.getHeight() > maxLabelHeight) {
					maxLabelHeight = labelLength.getHeight();
				}
			}
			//minor units
			for (int i=1; i <= minorUnitValues.length; i++) {
				double labelValue = max - minorUnit * (numOfMinorDivisions - i);
				minorUnitValues[i-1] = labelValue;
			}
		}
	}
	
	private void initDataFormatter() {
		if (axisDef != null) {
			CustomFormat customFormat = axisDef.getCustomFormat();
			if (customFormat != null){
					try {
						customPattern = customFormat.getPattern();
						Class customFormatterClass = Class.forName(customFormat.getFormatClass());
						customFormatterClass.getConstructor(new Class[0]);	
						if (customFormatterClass != null)
							customFormatter = (ICustomFormatter)customFormatterClass.newInstance();
					} catch (SecurityException e) {
						customFormatter = null;
					} catch (ClassNotFoundException e) {
						customFormatter = null;
					} catch (NoSuchMethodException e) {
						customFormatter = null;
					} catch (InstantiationException e) {
						customFormatter = null;
					} catch (IllegalAccessException e) {
						customFormatter = null;
					}
							
			}
			else{
				NumberFormat numFormat = axisDef.getNumberFormat();
				if (numFormat == null && dataValueDef != null) numFormat = dataValueDef.getNumberFormat();
				if (numFormat != null) {
					if (locale != null){
						numberFormatter = com.ibm.icu.text.NumberFormat.getInstance(locale);
					} else {
						numberFormatter = com.ibm.icu.text.NumberFormat.getInstance();
					}
					
					if (numberFormatter instanceof DecimalFormat) {
						((DecimalFormat) numberFormatter).applyPattern(numFormat.getPattern());
					}
				} else if (axisDef.getDateFormat() != null) {
					DateFormat dateFormat = axisDef.getDateFormat();
					try {
						if (locale != null) {
							dateFormatter = new SimpleDateFormat(dateFormat.getPattern(), locale);
						} else {
							dateFormatter = new SimpleDateFormat(dateFormat.getPattern());
						}
					} catch (NullPointerException npe) {
						dateFormatter = null;
					} catch (IllegalArgumentException iae) {
						dateFormatter = null;
					}
				}
			}
		}
	}
	
	/**
	 * Formats the value using the number format or datetime format specified in the input. 
	 * This method is used by subclasses and the dataset classes.
	 * @param value
	 * @return
	 */
	public String formatValue(double value) {
		String labelText;
		if (customFormatter != null){
			if (locale == null) locale = ULocale.ENGLISH;			
			labelText = customFormatter.format(new Double(value), locale, timezone, customPattern);
		} else if (numberFormatter != null) {
			labelText = numberFormatter.format(value);
		} else if (dateFormatter != null) {
			labelText = dateFormatter.format(new Date((long)value));
		} else {
			labelText = Double.toString(value);
		}
		return labelText;
	}
	
	/**
	 * Get major unit values
	 * @return major unit values
	 */
	public double[] getMajorUnitValues() {
		return majorUnitValues;
	}

	/**
	 * Get minor unit values
	 * @return minor unit values
	 */
	public double[] getMinorUnitValues() {
		return minorUnitValues;
	}
	
	/**
	 * Get maximum value
	 * @return maximum value
	 */
	public double getMax() {
		return max;
	}

	/**
	 * Get minimum value
	 * @return minimum value
	 */
	public double getMin() {
		return min;
	}
	
	/**
	 * Get scale type of the axis
	 * @return true if scale is linear, false if logarithmic
	 */
	public boolean isLinear() {
		return isLinear;
	}
	
	/**
	 * Since half of the label of the max value goes beyond the right edge of the 
	 * plot area, we need to find the margin required to display the full text
	 * of the max label.
	 *  
	 * @return length of half of the formatted max value
	 */
	public int getMarginForMax() {
		String formattedMax = formatValue(max);
		FontHelper fontHelper = new FontHelper();
		fontHelper.setFontSize(IConstants.AXIS_LABEL_FONT_SIZE);
		marginForMax = (int)((fontHelper.getStringWidth(formattedMax, labelDegrees)/2)- 5);
		return marginForMax;
		//return formattedMax.length()/2 * (AXIS_LABEL_FONT_SIZE);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.Axis#setLabelDegrees(double)
	 */
	public void setLabelDegrees(double labelDegrees) {
		if (this.getAxisOrientation() == VERTICAL) labelDegrees = -(labelDegrees-90); 
		
		super.setLabelDegrees(labelDegrees);
	}
}
