/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AxisNumberHorizontal.java,v 1.2 2005/06/10 17:47:09 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AxisNumberHorizontal.java,v 1.2 2005/06/10 17:47:09 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/



import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGGroup;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGLine;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGPolyline;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGText;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGUse;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.FontHelper;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;


/**
 * 
 */
public class AxisNumberHorizontal extends AxisNumber {
	static final long serialVersionUID = -7315397100775509637L;
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.Axis#getMarginForMax()
	 */
	public int getMarginForMax() {
		int margin =  super.getMarginForMax();
		if (this.labelDegrees > 90) margin = 2*(margin +5) - 5;
		return margin;
	}
	/**
	 * position of the horizontal axis. True if it is at the bottom, false if it is at the top.
	 */
	private boolean bottom = true;
	
	/**
	 * text direction, true if LTR, false if RTL
	 */
	private boolean isLTR = true;
	
	protected String topString = "";
	
	/**
	 * Distance between the left edge of the axis part and the first tick mark on the axis.
	 * 
	 * Half of the label of the first tick mark is on the left of the left of the
	 * tick mark. Since the svg viewbox will clip off that half of the label, 
	 * we need to shift the whole axis to the right. The offset value is quite arbitrary.
	 * It should be longer than the width of the label.
	 */
	private double offset = 100;

	private int AXIS_LABEL_LINE_HEIGHT = AXIS_LABEL_FONT_SIZE + 6;

	/**
	 * @param input
	 * @param datasets
	 * @param axisType
	 * @param cumulative
	 */
	public AxisNumberHorizontal(
		Chart input,
		boolean isLTR,
		List datasets,
		short axisType,
		boolean cumulative,
		NLString nls) {
		super(input, datasets, axisType, cumulative, nls);
	
		if (axisType == AXIS_TYPE_SECONDARY_DEPENDENT_AXIS) {
			bottom = false;
		}
		this.isLTR = isLTR;
	}

	/**
	 * @param input
	 * @param datasets
	 * @param axisType
	 * @param cumulative
	 * @param labelOrientation
	 */
	public AxisNumberHorizontal(
		Chart input,
		boolean isLTR,
		List datasets,
		short axisType,
		boolean cumulative,
		short labelOrientation,
		NLString nls) {
		super(input, datasets, axisType, cumulative, labelOrientation, nls);

		if (axisType == AXIS_TYPE_SECONDARY_DEPENDENT_AXIS) {
			bottom = false;
		}
		this.isLTR = isLTR;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartAxis#setAxisLength(double)
	 */
	public void setAxisLength(double length) {
		axisWidth = length;
		setWidth(Double.toString(length+offset*2));
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartAxis#getAxisLength()
	 */
	public double getAxisLength() {
		return axisWidth;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartAxis#setCrossPoint(double, double)
	 */
	public void setCrossPoint(double x, double y) {
		if (bottom) {
			setXCoordinate(Double.toString(x - offset));
			setYCoordinate(Double.toString(y - TICK_MARK_LENGTH));
		} else {
			setXCoordinate(Double.toString(x - offset));
			setYCoordinate(Double.toString(y - getAxisHeight() + TICK_MARK_LENGTH));
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartNumberAxis#drawHorizontalLabels(double[], boolean)
	 */
	protected SVGBase drawHorizontalLabels(double[] positions) {
		SVGGroup labelGroup = new SVGGroup();
		SVGBase[] svgLabels = new SVGBase[positions.length];
		labelGroup.setChildren(svgLabels);
		topString = "";
		double topY = -Double.MIN_VALUE;
		for (int i=0; i<positions.length; i++) {
			int currentY;
			if (bottom) {
				currentY = BASE_FONT_SIZE/2 + TICK_MARK_LENGTH + AXIS_SPACE_BTW_AXIS_AND_LABEL;
			} else {
				currentY = getTitleHeight() + AXIS_LABEL_LINE_HEIGHT;
			}
			if (labels[i] != null) {
				SVGGroup label = new SVGGroup();
				SVGBase[] rowText = new SVGBase[1];
				label.setChildren(rowText);
				AxisLabel labelLine = new AxisLabel(AxisLabel.EAST, AxisLabel.BELOW);
				labelLine.setDegrees(labelDegrees);
				labelLine.setXCoordinate(Double.toString(positions[i]));
				labelLine.setYCoordinate(Integer.toString(currentY));
				labelLine.setIdentifier("SVGGEN_axisLabel" + EventTools.getUniqueID());
				labelLine.setText(labels[i]);
				rowText[0] = labelLine;
				if (currentY > topY){
					topY = currentY;
					topString = labels[i];
				}

				svgLabels[i] = label;
			}
		}		
		return labelGroup;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartNumberAxis#drawVerticalLabels(double[], boolean)
	 */
	protected SVGBase drawVerticalLabels(double[] positions) {
		// TODO support vertical labels
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartNumberAxis#getAxisOrientation()
	 */
	public short getAxisOrientation() {
		return HORIZONTAL;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartAxis#getAxisWidth()
	 */
	public double getAxisWidth() {
		// note: this value of the length of the axis line. 
		// It does not include the offset 
		return axisWidth;
	}

	/**
	 * Get height of axis title section
	 * @return height of axis title
	 */
	private int getTitleHeight() {
		if (title == null) {
			return 0;
		}
		return AXIS_TITLE_FONT_SIZE + AXIS_SPACE_BTW_TITLE_AND_LABEL;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPartAxis#getAxisHeight()
	 */
	public double getAxisHeight() {
		// (LABEL_FONT_SIZE + 6) is the the line height
		return TICK_MARK_LENGTH + AXIS_SPACE_BTW_AXIS_AND_LABEL + getLabelHeight() + getTitleHeight();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
	 */
	protected void constructPart() {
		double[] tick_x_coordinates = new double[majorUnitValues.length];
		
		// create axis group
		SVGGroup axisGroup = new SVGGroup();
		SVGBase[] axisParts = new SVGBase[5]; //axis line, major tick marks, minor tick marks and labels, title
		axisGroup.setChildren(axisParts);
//		if (axisDef != null) {
//			axisGroup.setIdentifier(axisDef.getId());
//		}
		
		// add axis group to axis svg
		SVGBase[] child = new SVGBase[1];
		child[0] = axisGroup;
		setChildren(child);

		// draw major axis line
		SVGLine axis = new SVGLine();
		String axis_y;
		if (bottom) {
			axis_y = Integer.toString(TICK_MARK_LENGTH);
		} else {
			axis_y = Double.toString(getAxisHeight() - TICK_MARK_LENGTH);
		}
		axis.setX1(Double.toString(offset));
		axis.setY1(axis_y);
		axis.setX2(Double.toString(offset + axisWidth));
		axis.setY2(axis_y);
		axis.setStyleClass("axisLines");
		
		SVGGroup axisGroup2 = new SVGGroup();
		SVGBase[] axisGroup2Children = new SVGBase[2];
		axisGroup2Children[0] = axis;
		axisGroup2.setChildren(axisGroup2Children);
		
		if (axisDef != null) {
			// Handle user tooltips
			axisGroup2Children[1] = EventTools.generateUserTooltip(axisGroup2, axisDef.getTooltip(), nls);

			// Handle user events
			EventTools.generateUserEvents(axisGroup2, axisDef.getEventHandler());

			// Handle accessibility
			EventTools.generateAccessibility(axisGroup2, axisDef.getAccessibility(), nls);
		}
		
		axisParts[0] = axisGroup2;
		
		// draw ticks (in the middle of each group)
		// 1. define tick mark
		//		<g id="xtickline">
		//			<polyline class="axisTicks" points="0 0 0 5"/>
		//		</g>

		SVGGroup tickDefGroup = new SVGGroup();
		tickDefGroup.setIdentifier("tickline");
		
		SVGBase tickDef = drawTickMarks();
		tickDef.setStyleClass("axisTicks");
		tickDef.setIdentifier("verticaltickline");
		
		SVGBase[] tick = new SVGBase[1];
		tick[0] = tickDef; 
		tickDefGroup.setChildren(tick);
		
		addDefinition(tickDef);

		// 2. create group for ticks
		// tick_y is the y coordinate of the tick marks (same of all tick marks)
		String tick_y;
		if (bottom) {
			tick_y = "0";
		} else {
			tick_y = Double.toString(getAxisHeight() - TICK_MARK_LENGTH);
		}
		// major ticks
		if (showMajorTicks) {
			SVGGroup tickGroup = new SVGGroup();
			SVGBase[] ticks = new SVGBase[majorUnitValues.length];
			axisParts[1] = tickGroup;
			tickGroup.setChildren(ticks);

			for (int i=0; i < majorUnitValues.length; i++) {
				double tick_x;
				if (isLTR) {
					tick_x = getXCoordinate(majorUnitValues[i], offset + axisWidth, offset, max, min);
				} else {
					tick_x = getXCoordinate(majorUnitValues[i], offset, offset + axisWidth, max, min);
				}
				tick_x_coordinates[i] = tick_x;
				SVGUse useTick = new SVGUse();
				useTick.setHref("#verticaltickline");
				useTick.setXCoordinate(Double.toString(tick_x));
				useTick.setYCoordinate(tick_y);
				ticks[i] = useTick;
			}
		} else {
			// calculate positions for labels only
			for (int i=0; i < majorUnitValues.length; i++) {
				double tick_x;
				if (isLTR) {
					tick_x = getXCoordinate(majorUnitValues[i], offset + axisWidth, offset, max, min);
				} else {
					tick_x = getXCoordinate(majorUnitValues[i], offset, offset + axisWidth, max, min);
				}
				tick_x_coordinates[i] = tick_x;
			}
		}
		
		// minor ticks
		if (showMinorTicks) {
			SVGGroup tickGroup = new SVGGroup();
			SVGBase[] ticks = new SVGBase[minorUnitValues.length];
			axisParts[2] = tickGroup;
			tickGroup.setChildren(ticks);
			
			for (int i=0; i < minorUnitValues.length; i++) {
				double tick_x;
				if (isLTR) {
					tick_x = getXCoordinate(minorUnitValues[i], offset + axisWidth, offset, max, min);
				} else {
					tick_x = getXCoordinate(minorUnitValues[i], offset, offset + axisWidth, max, min);
				}
				SVGUse useTick = new SVGUse();
				useTick.setHref("#verticaltickline");
				useTick.setXCoordinate(Double.toString(tick_x));
				useTick.setYCoordinate(tick_y);
				ticks[i] = useTick;
			}
		}
		
		// 3. draw labels
		if (labelOrientation == HORIZONTAL) {
			axisParts[3] = drawHorizontalLabels(tick_x_coordinates);
		} else {
			axisParts[3] = drawVerticalLabels(tick_x_coordinates);
		}
		
		// 4. draw axis title
		if (title != null) {
			String title_y;
			if (bottom) {
				title_y = Double.toString(getAxisHeight());
			} else {
				title_y = Double.toString(getTitleHeight());
			}
			SVGText svgTitle = new SVGText();
			svgTitle.setText(title);
			svgTitle.setXCoordinate(Double.toString(offset + getAxisLength()/2));
			svgTitle.setYCoordinate(title_y);
			svgTitle.setStyleClass("axisTitles anchorAtMiddle");
			if (axisType == AXIS_TYPE_PRIMARY_DEPENDENT_AXIS) {
				svgTitle.setIdentifier("SVGGEN_labelsValues1");
			} else {
				svgTitle.setIdentifier("SVGGEN_labelsValues2");
			}
			axisParts[4] = svgTitle;
		}

	}

	/**
	 * Calculate coordinate in linear scale
	 * @param value value to be plotted
	 * @param max_coor coordinate of the maximum value of this axis
	 * @param min_coor coordinate of the minimum value of this axis
	 * @param max maximum value of this axis
	 * @param min minimum value of this axis
	 * @return
	 */
	protected double getXCoordinate(double value, double max_coor, double min_coor, double max, double min) {
		return (value - max) / (min - max) * (min_coor - max_coor) + max_coor;
	}

	protected SVGBase drawTickMarks(){
		SVGPolyline tickDef = new SVGPolyline();
		tickDef.setPoints("0 0 0 " + TICK_MARK_LENGTH); // "0 0 0 5"
		return tickDef;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.Axis#axisExtendedLength()
	 */
	public double axisExtendedLength() {
		FontHelper fontHelper = new FontHelper();
		fontHelper.setFontSize(IConstants.AXIS_LABEL_FONT_SIZE);
		return fontHelper.getHeight(topString, labelDegrees);
	}
}
