/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ChartArea.java,v 1.7 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ChartArea.java,v 1.7 2008/12/12 22:22:09 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGGroup;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGScript;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGStyle;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSets;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Ecmascript;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Internationalization;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Palettes;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Preferences;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Scripts;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Shapes;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Size;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.TitleBar;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;

import com.ibm.icu.util.ULocale;


/**
 * Chart Area
 */
public class ChartArea extends Part implements IGraphicTypeConstants, IGraphicDocumentStyle {
	static final long serialVersionUID = 2696039115291774673L;
	/**
	 * color palettes
	 */
	private SVGColorPalettes palettes;
	
	/**
	 * shapes
	 */
	private SVGShapes shapes;
	
	/**
	 * use shapes in the chart - true/false
	 */
	private boolean useShapes;
	
	/**
	 * background color
	 */
	private String bkgroundColor;
	
	/**
	 * A flag to indicate whether the border of chart area should be drawn
	 */
	private boolean drawBorder = true;
	
	/**
	 * Location of the legend
	 */
	private String legendLocation = POSITION_TRAILING;
	
	/**
	 * text direction - true if LTR, false if RTL
	 */
	private boolean isLTR = true;
	
	/**
	 * chartArea element from input
	 */
	private org.eclipse.tptp.platform.report.chart.svg.internal.input.ChartArea chartarea = null;
	
	/**
	 * Default org.eclipse.tptp.platform.report.chart.svg.internal.resources - default shapes and palettes
	 */
	private DefaultResources resources;
	
	/**
	 * A simple structure for holding information about each part:
	 * 1. a flag to indicate whether it should displayed
	 * 2. the x coordinate of the part relative to its container, 
	 * 3. the y coordinate of the part relative to its container, 
	 * 4. the width of the part 
	 * 5. the height of the part 
	 */
	private class SVGPartInfo {
		public boolean show = false;
		public double x = 0;
		public double y = 0;
		public double width = 0;
		public double height = 0;
		public String toString() {
			return "x: " + x + " y: " + y + " width: " + width + " height: " + height;
		}
	}

	/**
	 * Part info for title
	 */
	private SVGPartInfo titleInfo;
	
	/**
	 * Part info for legend
	 */
	private SVGPartInfo legendInfo;
	
	/**
	 * Part info for graph area
	 */
	private SVGPartInfo graphAreaInfo;
	
	/**
	 * Part info for timestamp area
	 */
	private SVGPartInfo timestampInfo;

	/**
	 * Number of data sets
	 */
	private int numOfDataSets = 0;
	
	/**
	 * An object for retriving translated text
	 */
	private NLString nls;
	
	/**
	 * Locale of this chart.
	 */
	private ULocale locale;

	// calculate part dimensions
	private double east = width;
	private double west = 0;
	private double north = 0;
	private double south = height;
		
	/**
	 * Constructor
	 * @param input
	 */
	public ChartArea(Chart input, DefaultResources resources) {
		super(input);
		
		this.resources = resources;
		
		Configuration config = input.getConfiguration();

		// find out text direction (LTR/RTL)
		if (config != null) {
			Internationalization i18n = config.getInternationalization();
			String textDirection = null;
			if (i18n != null) {
				textDirection = i18n.getTextDirection();
				if (textDirection != null) {
					if (textDirection.equals("LTR")) {
						isLTR = true;
					} else if (textDirection.equals("RTL")){
						isLTR = false;
					} else {
						isLTR = true;
					}
				}
			}
		}

		// Determine the positions and dimensions of the four parts in chart area
		titleInfo = new SVGPartInfo();
		legendInfo = new SVGPartInfo();
		graphAreaInfo = new SVGPartInfo();
		timestampInfo = new SVGPartInfo();
		determinePartDimensions(titleInfo, legendInfo, graphAreaInfo, timestampInfo);
		
		// find out number of datasets
		Data data = input.getData();
		if (data != null) {
			DataSets datasets = data.getDataSets();
			if (datasets != null) {
				numOfDataSets = datasets.getDataSet().size(); 
			}
		}
		
		// instantiate shapes
		initShapes(input.getType(), config);
		
		// get background color and chart border settings
		if (config != null) {
			chartarea = config.getChartArea();
			if (chartarea != null) {
				bkgroundColor = chartarea.getBackgroundColor();
				if (chartarea.isSetBorder()) {
					drawBorder = chartarea.isBorder();
				}
			}
		}
		
		if (config != null) {
			Internationalization i18n = config.getInternationalization();
			if (i18n != null) {
				String resourceBundle = i18n.getResourceBundle();
				String language = i18n.getLanguage() == null ? "" : i18n.getLanguage();
				String country = i18n.getCountry() == null ? "" : i18n.getCountry();
				if ((language != "") && (country != ""))
					locale = new ULocale(language, country);				
				else if (language.equals("") == false) {
					locale = new ULocale(language);
				} else {
					locale = ULocale.getDefault();
				}
				
				if (resourceBundle != null) {
					nls = new NLString(resourceBundle, locale);
				} else {
					nls = new NLString();
				}
			} else {
				locale = ULocale.getDefault();
				nls = new NLString();
			}
		} else {
			locale = ULocale.getDefault();
			nls = new NLString();
		}
		
		// instantiate color palettes
		initColorPalettes(config, locale);

	}
	
	/**
	 * Initialize color palettes
	 * @param config
	 */
	private void initColorPalettes(Configuration config, ULocale locale) {
		palettes = new SVGColorPalettes(resources, locale);
		if (config != null) {
			Palettes palettesConfig = config.getPalettes();
			if (palettesConfig != null) {
				String palettesLocation = palettesConfig.getLocation();
				String paletteId = palettesConfig.getPaletteId();
				String paletteSetId = palettesConfig.getPaletteSetId();
				if (palettesLocation != null || paletteId != null || paletteSetId != null) {
					palettes.setCustomPalettes(palettesLocation, paletteId, paletteSetId);
				}
				
				// Users can assign colors to data sets to override the colors of the palette. 
				List colorList = palettesConfig.getDataSetColor();
				palettes.setDataSetColors(colorList);
			}
		}
	}
	
	/**
	 * Initialize shapes
	 * @param chartType
	 * @param config
	 */
	private void initShapes(String chartType, Configuration config) {
		
		if (chartType.equals(IGraphicTypeConstants.LINE_CHART) ||
			chartType.equals(IGraphicTypeConstants.AREA_CHART) || 
			chartType.equals(IGraphicTypeConstants.STACK_AREA_CHART) ||
			chartType.equals(IGraphicTypeConstants.SCATTER_CHART) ){
				
			useShapes = true;
			shapes = new SVGShapes(resources);

			// use user-defined shapes if location of shape definition file is specified 
			if (config != null) {
				Shapes shapesConfig = config.getShapes();
				if (shapesConfig != null) {
					String shapesLocation = shapesConfig.getLocation();
					if (shapesLocation != null) {
						shapes.setCustomShapes(shapesLocation);
					}

					// Users can assign shapes to data sets explicitly
					List shapeList = shapesConfig.getShape();
					shapes.setDataSetShapes(shapeList);
				}
			}
			
		}
		
	}

	/**
	 * 	The four top level parts are title, graph area, legend and timestamp.
	 * 	Graph area is always created.  The other three can be suppressed.
	 * 	Depending on whether each of these parts need to be shown, and 
	 * 	their preferred positions in the chart, they will have different
	 * 	dimensions. This method determines the dimensions of each part.
	 *
	 * @param titleInfo
	 * @param legendInfo
	 * @param graphAreaInfo
	 * @param timestampInfo
	 */
	protected void determinePartDimensions(SVGPartInfo titleInfo, SVGPartInfo legendInfo, SVGPartInfo graphAreaInfo, SVGPartInfo timestampInfo) {
		// user-defined width and height values for legend
		double userDefinedLegendWidth = 0;
		double userDefinedLegendHeight = 0;

		// set default values
		legendInfo.show = true;
		titleInfo.show = true;
		timestampInfo.show = true;
		graphAreaInfo.show = true;
		
		// set default chart width and height
		if (input.getType().equals(METER)) {
			width = CHART_AREA_METER_DEFAULT_WIDTH;
		} else {
			width = CHART_AREA_DEFAULT_WIDTH;
		}
		height = CHART_AREA_DEFAULT_HEIGHT;

		Configuration config = input.getConfiguration(); 
		if (config != null) {
			// If chart size is specified in the input, override the default value(s)
			Size size = config.getSize();
			if (size != null) {
				if (size.isSetWidth() && size.getWidth() > 0) {
					width = size.getWidth();
				}
				if (size.isSetHeight() && size.getHeight() > 0) {
					height = size.getHeight();
				}					
			}

			// Determine if any of the parts is not required
			// Legend: find out if it should be generated and its location
			org.eclipse.tptp.platform.report.chart.svg.internal.input.Legend legend = config.getLegend();
			if (legend != null) {
				if (legend.isSetShow() && legend.isShow() == false) {
					legendInfo.show = false;
				} else if (legend.getLocation() != null){
					legendLocation = legend.getLocation();
				}
				if (legend.isSetWidth() && legend.getWidth() > 0 && legend.getWidth() < width) {
					userDefinedLegendWidth = legend.getWidth();
				}
				if (legend.isSetHeight() && legend.getHeight() > 0 && legend.getHeight() < height) {
					userDefinedLegendHeight = legend.getHeight();
				}
			}

			// Title bar: find out if it should be generated
			TitleBar titleBar = config.getTitleBar();
			Preferences pref = config.getPreferences();
			boolean showTitle = true;
			boolean showPref = false;
			if (titleBar != null) {
				// show title bar by default
				if (titleBar.isSetShow() && titleBar.isShow() == false) {
					showTitle = false;
				}
			}
			if (pref != null) {
				// preferences is not shown by default  
				if (pref.isSetShow() && pref.isShow()) {
					showPref = true;
				}
			}
			if (showTitle || showPref) {
				titleInfo.show = true;
			}else{
				titleInfo.show = false;
			}

			// show timestamp if it is provided and not explicitly supprssed
			if (input.getData() != null && input.getData().getTimestamp() != null) {
				if (config.getTimestamp() != null) {
					if (config.getTimestamp().isSetShow() && !config.getTimestamp().isShow()) {
						timestampInfo.show = false;
					}
				} 
			}else{
				timestampInfo.show = false;
			}
		}

		// meter charts do not need legend
		if (input.getType().equals(METER)) {
			legendInfo.show = false;
		}
		
		// init component bounds
		north = 0;
		south = height;
		east = width;
		west = 0;
		
		if (titleInfo.show) {
			// set title dimensions
			titleInfo.width = width;
			titleInfo.height = TITLE_HEIGHT;  // constant
			titleInfo.x = 0;
			titleInfo.y = 0;

			north = titleInfo.height;
		}
		
		if (timestampInfo.show) {
			// set dimenstions of the timestamp region
			timestampInfo.width = width;
			timestampInfo.height = TIMESTAMP_HEIGHT; // constant
			timestampInfo.x = 0;
			timestampInfo.y = height - timestampInfo.height;
			
			south = timestampInfo.y;
		}
		
		// The dimensions of the legend and the graph area are calculated in constructPart method, after the initializing the legend.
		// The width of the legend will be dependent on how much content is in the legend.  
		
	}


	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
	 */
	protected void constructPart() {
		// set svg width and height
		setWidth(Double.toString(width));
		setHeight(Double.toString(height));
		
		if (input.getId() != null) {
			setIdentifier(input.getId());
		}

		int numUserScripts = 0;
		
		{
			// Determine the number of user scripts
			if (input.getConfiguration() != null) {
				Scripts scripts = input.getConfiguration().getScripts();
				if (scripts != null) {
					numUserScripts = scripts.getEcmascript().size();
				}
			}
		}
		
		// parts: scripts, background, border, graph area, pref, title, legend, timestamp, tooltip		
		SVGBase[] parts = new SVGBase[8+numUserScripts];

		SVGGroup group = new SVGGroup();
		group.setChildren(parts);
		SVGBase[] groupChild = new SVGBase[1];
		groupChild[0] = group;

		// Add default keypress handler for Accessibility
		//group.addEvent("onkeypress", "SVGGEN_handle_keypress(evt)");
		
		// group.addEvent("onload", "SVGGEN_init(evt)");
		
		if (chartarea != null) {
			// Handle user events
			EventTools.generateUserEvents(group, chartarea.getEventHandler());
		
			// Handle accessibility
			EventTools.generateAccessibility(group, chartarea.getAccessibility(), nls);
		}
		
		setChildren(groupChild);
		
		{
			{
				// Add Default ECMAScripts
				SVGScript scriptEmitter = new SVGScript();
				EventTools.generateDefaultScripts(input, scriptEmitter, palettes, legendInfo.show, locale);
				parts[0] = scriptEmitter;
			}
			
			if (numUserScripts > 0) {
				// Add user ECMAScripts
				List ecmaScriptList = input.getConfiguration().getScripts().getEcmascript();
				for (int i = 0; i < numUserScripts; ++i) {
					String scriptName = ((Ecmascript) ecmaScriptList.get(i)).getHref();
					if (scriptName != null){
						SVGScript scriptEmitter = new SVGScript();
						scriptEmitter.setUserScript(scriptName);
						parts[i+1] = scriptEmitter;
					}
				}
			}
		}
		
		// add style definitions
		SVGStyle style = new SVGStyle();
		style.setId(DEFAULT_STYLE_ID);
		StringBuffer styleRules = new StringBuffer(COMMON_STYLE);
		if (input.getType().equals(IGraphicTypeConstants.METER)){
			styleRules.append(METER_STYLE);
		}
		else if (input.getType().equals(IGraphicTypeConstants.PIE_CHART) || input.getType().equals(IGraphicTypeConstants.PIE_CHART3D)){ 
			styleRules.append(PIE_STYLE);
		}
		else{
			styleRules.append(XYCHART_STYLE);
		}
		
		// add color palettes
		String[] palette = palettes.getPalette();
		for (int i = 0; i < numOfDataSets; i++) {
			int paletteIndex = i % palette.length;
			
			if (input.getType().equals(IGraphicTypeConstants.PIE_CHART) || input.getType().equals(IGraphicTypeConstants.PIE_CHART3D)) {
				styleRules.append(" .dataset" + i + "{stroke:black; fill:" + palette[paletteIndex] + ";}");
				styleRules.append(" .shape" + i + "{stroke:none; fill:" + palette[paletteIndex] + ";}");
			}
			if (input.getType().equals(IGraphicTypeConstants.VBAR_CHART) 
			|| input.getType().equals(IGraphicTypeConstants.VSTACKBAR_CHART) 
			|| input.getType().equals(IGraphicTypeConstants.HBAR_CHART) 
			|| input.getType().equals(IGraphicTypeConstants.VSTACKBAR_CHART3D) 
			|| input.getType().equals(IGraphicTypeConstants.HBAR_CHART3D) 
			|| input.getType().equals(IGraphicTypeConstants.HSTACKBAR_CHART3D) 
			|| input.getType().equals(IGraphicTypeConstants.VBAR_CHART3D) 
			|| input.getType().equals(IGraphicTypeConstants.HSTACKBAR_CHART)){

				styleRules.append(" .dataset" + i + "{stroke-width:0.75pt; stroke:" + palette[paletteIndex] + ";}");
//					if (hasLegend) {
					styleRules.append(" .shape" + i + "{stroke:none; fill:" + palette[paletteIndex] + ";}");
					styleRules.append(" .bars" + i + "{stroke:" + palette[paletteIndex] + "; fill:" + palette[paletteIndex] + ";}");		  	
//					  }
			}
			styleRules.append(" .strokecolor" + i + "{stroke:"+ palette[paletteIndex] +";}");
			styleRules.append(" .fillcolor" + i + "{fill:"+ palette[paletteIndex] +";}");
		}

		style.setType(STYLE_TYPE);
		style.setCdata(styleRules.toString());
		addDefinition(style);
		
		// add shape definitions
		if (useShapes) {
			addDefinition(shapes.getShapeDefinition(numOfDataSets));
		}
		
		// add title bar 
		if (titleInfo.show) {
			Title titlePart = new Title(input, isLTR, titleInfo.x, titleInfo.y, titleInfo.width, titleInfo.height, nls);
			parts[4+numUserScripts] = titlePart;
		}
		
		// add legend
		if (legendInfo.show) {
			int margin = (int)LEGEND_WIDE_SIDE_MARGIN;
			if ((input.getConfiguration() != null) && (input.getConfiguration().getLegend() != null))
				margin = input.getConfiguration().getLegend().getMargin();
			Legend legendPart = Legend.getInstance(legendLocation, isLTR, input, legendInfo.x, legendInfo.y, legendInfo.width, legendInfo.height, east-west, south-north, margin, palettes, shapes, nls);
			parts[5+numUserScripts] = legendPart;
			
			double legendHeight = legendPart.getLegendHeight();

			// adjust legend position for leading/trailing based on LTR/RTL orientation
			if (legendLocation.equals(POSITION_LEADING)) {
				legendLocation = (isLTR) ? POSITION_WEST : POSITION_EAST;
			} else if (legendLocation.equals(POSITION_TRAILING)) {
				legendLocation = (isLTR) ? POSITION_EAST : POSITION_WEST;
			}

			if (legendLocation.equals(POSITION_EAST)) {
//				legendInfo.width = legendPart.getLegendWidth();
				legendInfo.height = south - north;
				legendInfo.x = width - legendPart.getLegendWidth();
				legendInfo.y = north;
				
				east = legendInfo.x;
			} else if (legendLocation.equals(POSITION_WEST)) {
//				legendInfo.width = legendPart.getLegendWidth();
				legendInfo.height = south - north;
				legendInfo.x = 0;
				legendInfo.y = north;
				
				west = legendPart.getLegendWidth();
			} else 
			if (legendLocation.equals(POSITION_NORTH)) {
				legendInfo.width = width;
//				legendInfo.height = userDefinedLegendHeight > 0 ? userDefinedLegendHeight : LEGEND_HORIZONTAL_HEIGHT;  
				legendInfo.x = 0;
				legendInfo.y = north;
				
				north = north + legendHeight; 
			} else { // "S"
				legendInfo.width = width;
//				legendInfo.height = userDefinedLegendHeight > 0 ? userDefinedLegendHeight : LEGEND_HORIZONTAL_HEIGHT;  
				legendInfo.x = 0;
				legendInfo.y = south - legendHeight;
				
				south = legendInfo.y; 
			}
			legendPart.setXCoordinate(Double.toString(legendInfo.x));
			legendPart.setYCoordinate(Double.toString(legendInfo.y));
			if (legendLocation.equals(POSITION_NORTH) || legendLocation.equals(POSITION_SOUTH)) {
				legendPart.setWidth(Double.toString(legendInfo.width));
			} else {
				legendPart.setHeight(Double.toString(legendInfo.height));
			}
		}
		
		graphAreaInfo.width = east - west;
		graphAreaInfo.height = south - north;  
		graphAreaInfo.x = west;
		graphAreaInfo.y = north;
		
//		System.out.println("titleInfo:");
//		System.out.println(titleInfo);
//		System.out.println("timestampInfo:");
//		System.out.println(timestampInfo);
//		System.out.println("legendInfo: ");
//		System.out.println(legendInfo);
//		System.out.println("graphAreaInfo:");
//		System.out.println(graphAreaInfo);

		// add graph area
		GraphArea graphArea = GraphArea.getInstance(input, isLTR, graphAreaInfo.x, graphAreaInfo.y, graphAreaInfo.width, graphAreaInfo.height, palettes, shapes);
		graphArea.setNLString(nls);

		
		parts[3+numUserScripts] = graphArea; 

		// add timestamp section
		if (timestampInfo.show) {
			Timestamp timestampPart = new Timestamp(input, isLTR, timestampInfo.x, timestampInfo.y, timestampInfo.width, timestampInfo.height, nls);
			parts[6+numUserScripts] = timestampPart;
		}
		
		// draw background of the chart area
		if (bkgroundColor != null && bkgroundColor.equals("") == false) {
			SVGRectangle bkground = new SVGRectangle();
			bkground.setXCoordinate("0");
			bkground.setYCoordinate("0");
			bkground.setWidth(getWidth());
			bkground.setHeight(getHeight());
			bkground.setFill(bkgroundColor);
			parts[1+numUserScripts] = bkground;
		}
		
		// draw border for the chart area
		if (drawBorder) {
			SVGRectangle border = new SVGRectangle();
			border.setXCoordinate("0");
			border.setYCoordinate("0");
			border.setWidth(Double.toString(width-1));
			border.setHeight(Double.toString(height-1));
			border.setFill("none");
			border.setStroke("#666666");
			parts[2+numUserScripts] = border;
		}
		
		{
			// Generate Tooltips placeholder SVG
			TooltipPlaceholder tooltipSvg = new TooltipPlaceholder(input, 0, 0, width, height);
			parts[7+numUserScripts] = tooltipSvg;
		}
		
	}
	
}
