/* ***********************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventTools.java,v 1.3 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.chart.svg.internal.part;

import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGGroup;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGScript;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGText;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Accessibility;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSets;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataValue;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Ecmascript;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.EventHandler;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Preferences;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Scripts;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Tooltip;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.IIDGenerator;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.Utilities;

import com.ibm.icu.util.ULocale;


/**
 * Methods which deal with events, event handlers, tooltips and java scripts
 *
 */
public class EventTools implements IGraphicTypeConstants {
	
	/**
	 * Holds the unique ID number
	 */
	public static int idNumber = 0;
	
	protected static IIDGenerator IDGenerator;
	public static void registerIDGenerator(IIDGenerator IDGenerator){
		EventTools.IDGenerator = IDGenerator;
	}
	
	/**
	 * Generates default ECMAScripts
	 * 
	 * @param input the chart
	 * @param scriptEmitter an SVGScript object for generating the <script> element
	 */
	static public void generateDefaultScripts(Chart input, SVGScript scriptEmitter, 
		SVGColorPalettes palettes, boolean hasLegend, ULocale locale) {
		
		ECMAScriptGenerator jscript = new ECMAScriptGenerator();
		StringBuffer functions = new StringBuffer();
		
		String graphicType = input.getType();
		
		Data data = input.getData();
		List dataSetList = null;
		if (data != null) {
			DataSets datasets = data.getDataSets();
			if (datasets != null) {
				dataSetList = datasets.getDataSet(); 
			}
		}
		int numberOfDataSets = dataSetList == null ? 0 : dataSetList.size();
		
		Preferences preferencesInput = null;
		if (input.getConfiguration() != null) {
			preferencesInput = input.getConfiguration().getPreferences();
		}
		
		String cookieName = null;
		if (preferencesInput != null && preferencesInput.getBrowserCookieName() != null) {
			cookieName = preferencesInput.getBrowserCookieName();
			if (cookieName.length() == 0) cookieName = null;
		}
		
		String preferencesVariable = null;
		if (preferencesInput != null && preferencesInput.getUpdatedPreferencesVariable() != null) {
			preferencesVariable = preferencesInput.getUpdatedPreferencesVariable();
			if (preferencesVariable.length() == 0) preferencesVariable = null;
		}
		
		String storedUserPreferences = null;
		if (preferencesInput != null && preferencesInput.getStoredPreferences() != null) {
			storedUserPreferences = preferencesInput.getStoredPreferences();
			if (storedUserPreferences.length() == 0) storedUserPreferences = null;
		}
		else 
			storedUserPreferences = getDefaultPreferences(input, palettes, graphicType, IGraphicDocumentStyle.BASIC_FONT);
		
		String preferencePage = IUserPreferencesConstants.LINE_PREFS;
		String preferencePageHeight = IUserPreferencesConstants.LINE_PREFS_HEIGHT;
		String preferencePageHeightNS = IUserPreferencesConstants.NS_LINE_PREFS_HEIGHT;
		if (graphicType.equals(AREA_CHART)) {
			preferencePage = IUserPreferencesConstants.AREA_PREFS;
			preferencePageHeight = IUserPreferencesConstants.AREA_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_AREA_PREFS_HEIGHT;
		} else if (graphicType.equals(STACK_AREA_CHART)) {
			preferencePage = IUserPreferencesConstants.AREA_PREFS;
			preferencePageHeight = IUserPreferencesConstants.AREA_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_AREA_PREFS_HEIGHT;
		} else if (graphicType.equals(SCATTER_CHART)) {
			preferencePage = IUserPreferencesConstants.SCATTER_PREFS;
			preferencePageHeight = IUserPreferencesConstants.SCATTER_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_SCATTER_PREFS_HEIGHT;
		} else if (graphicType.equals(VBAR_CHART) || graphicType.equals(HBAR_CHART)) {
			preferencePage = IUserPreferencesConstants.BAR_PREFS;
			preferencePageHeight = IUserPreferencesConstants.BAR_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_BAR_PREFS_HEIGHT;
		} else if (graphicType.equals(VSTACKBAR_CHART) || graphicType.equals(HSTACKBAR_CHART)) {
			preferencePage = IUserPreferencesConstants.STACKBAR_PREFS;
			preferencePageHeight = IUserPreferencesConstants.STACKBAR_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_STACKBAR_PREFS_HEIGHT;
		} else if (graphicType.equals(VBAR_CHART3D) || graphicType.equals(HBAR_CHART3D)) {
			preferencePage = IUserPreferencesConstants.BAR_PREFS3D;
			preferencePageHeight = IUserPreferencesConstants.BAR_PREFS_HEIGHT3D;
			preferencePageHeightNS = IUserPreferencesConstants.NS_BAR_PREFS_HEIGHT3D;
		} else if (graphicType.equals(VSTACKBAR_CHART3D) || graphicType.equals(HSTACKBAR_CHART3D)) {
			preferencePage = IUserPreferencesConstants.STACKBAR_PREFS3D;
			preferencePageHeight = IUserPreferencesConstants.STACKBAR_PREFS_HEIGHT3D;
			preferencePageHeightNS = IUserPreferencesConstants.NS_STACKBAR_PREFS_HEIGHT3D;
		} else if (graphicType.equals(PIE_CHART)) {
			preferencePage = IUserPreferencesConstants.PIE_PREFS;
			preferencePageHeight = IUserPreferencesConstants.PIE_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_PIE_PREFS_HEIGHT;
		} else if (graphicType.equals(PIE_CHART3D)) {
			preferencePage = IUserPreferencesConstants.PIE_PREFS3D;
			preferencePageHeight = IUserPreferencesConstants.PIE_PREFS_HEIGHT3D;
			preferencePageHeightNS = IUserPreferencesConstants.NS_PIE_PREFS_HEIGHT3D;
		} else if (graphicType.equals(METER)) {
			preferencePage = IUserPreferencesConstants.METER_PREFS;
			preferencePageHeight = IUserPreferencesConstants.METER_PREFS_HEIGHT;
			preferencePageHeightNS = IUserPreferencesConstants.NS_METER_PREFS_HEIGHT;
		}

		String url = null;
		if (preferencesInput != null) {
			Utilities.assertion(preferencesInput.getUrl() != null);
			url = preferencesInput.getUrl();

			// If url ends with a file separater, remove the last character.
			if (url.endsWith("/") || url.endsWith("\\")) {
				url = url.substring(0, url.length()-1);
			}	
			
			// Use forward slashes so that paths in ECMA Script will work
			url = url.replace('\\', '/');

			url = url + "/";

			if (locale.getLanguage().equals(new ULocale("de", "", "").getLanguage())
				|| locale.getLanguage().equals(new ULocale("es", "", "").getLanguage())
				|| locale.getLanguage().equals(new ULocale("fr", "", "").getLanguage())
				|| locale.getLanguage().equals(new ULocale("it", "", "").getLanguage())
				|| locale.getLanguage().equals(new ULocale("ja", "", "").getLanguage())
				|| locale.getLanguage().equals(new ULocale("ko", "", "").getLanguage())){
					url = url + locale.getLanguage();
					url = url + "/";
				}
			else if (locale.getLanguage().equals(new ULocale("pt", "", "").getLanguage())){
				url = url + "pt/BR/";
			}
			else if (locale.getLanguage().equals(new ULocale("zh", "", "").getLanguage())){
				if (locale.getCountry().equals(new ULocale("zh","CN", "").getCountry())){
					url = url + "zh/CN/";
				}
				else if(locale.getCountry().equals(new ULocale("zh", "TW", "").getCountry())){
					url = url + "zh/TW/";
				}
				else {
					url = url + "zh/CN/";
				}
			}
			else {
				url = url + "en/";
			}

		}
		
		Scripts scriptsInput = null;
		if (input.getConfiguration() != null) {
			scriptsInput = input.getConfiguration().getScripts();
		}

		String svgDocumentVariable = null;
		String svgWindowVariable = null;
		if (scriptsInput != null) {
			svgDocumentVariable = scriptsInput.getSvgDocumentVariable();
			if (svgDocumentVariable != null && svgDocumentVariable.length() == 0) svgDocumentVariable = null;

			svgWindowVariable = scriptsInput.getSvgWindowVariable();
			if (svgWindowVariable != null && svgWindowVariable.length() == 0) svgWindowVariable = null;
		}

		int numOfCategories = 0; 
		
		if ((data != null) && data.getCategories() != null && data.getCategories().getCategory() != null)
			numOfCategories = data.getCategories().getCategory().size();		
		// User preference scripts
		if (preferencesInput != null){
			functions.append(jscript.generateClientSnifferVariables());		
			functions.append(jscript.generateGetPreferencesFunction(cookieName));
			functions.append(jscript.generateStorePreferencesFunction(cookieName));
			functions.append(jscript.generateGetUserPreferencesFunction());
			functions.append(jscript.generateOpenPreferencePageFunction(url, preferencePage, graphicType, IUserPreferencesConstants.PREFS_WIDTH, preferencePageHeight));		
			functions.append(jscript.generateUseNetscapePreferencePage(url, IUserPreferencesConstants.NS_PREFS, graphicType, IUserPreferencesConstants.NS_PREFS_WIDTH, preferencePageHeightNS));
		}
		//determine if we have to show shapes
		boolean show_Shapes = showShapes(input);
		boolean showUserPref = showUserPref(preferencesInput);
		functions.append(jscript.generateGlobalVariables());
		functions.append(jscript.generateInitFunction(storedUserPreferences, palettes, IGraphicDocumentStyle.BASIC_FONT, cookieName, graphicType, dataSetList, numOfCategories, numberOfDataSets, preferencesVariable, svgDocumentVariable, svgWindowVariable, show_Shapes));		
		functions.append(jscript.generateGetCurrentValuesFunction());
		functions.append(jscript.generateGetPalettesFunction());
		functions.append(jscript.generateCookieFunctions());
		functions.append(jscript.generateResetStylesFunction());
		functions.append(jscript.generateUpdateLegendStylesFunction(numberOfDataSets, graphicType));
		functions.append(jscript.generateUpdateStylesFunction(numberOfDataSets, cookieName, graphicType, hasLegend, preferencesVariable, numOfCategories, showUserPref));	
		functions.append(jscript.generateToggleVisibilityFunction(numberOfDataSets, graphicType, show_Shapes));
		functions.append(jscript.generateBuildHelper());

		// Event scripts
		functions.append(jscript.generateTooltipScripts());	
		functions.append(jscript.generateToggleVisibilityDatavalue());	
		//functions.append(jscript.generateKeyPressHandler());	

		if (input.getType().equals(AREA_CHART)  
			||input.getType().equals(STACK_AREA_CHART))
		{
			functions.append(jscript.generateMoveNode());	
			functions.append(jscript.generateChangeOpacity());	
		} 
		
		if (input.getType().equals(STACK_AREA_CHART)) {
			functions.append(jscript.generateTakeOut());	
			functions.append(jscript.generatePutIn());
			functions.append(jscript.generateToggle());
			functions.append(jscript.generateRegen());	
			functions.append(jscript.generateGetCoordinateFunction());	
		}
				
		if (input.getType().equals(PIE_CHART) || input.getType().equals(PIE_CHART3D)) {

			functions.append(jscript.generatePieScripts());	
		}
		 
		if (input.getType().equals(SCATTER_CHART)) {
			functions.append(jscript.generateToggleVisibilityShadow(show_Shapes));	
		}

		if (input.getType().equals(VBAR_CHART)
			||input.getType().equals(HBAR_CHART)
			||input.getType().equals(HSTACKBAR_CHART)
			||input.getType().equals(HBAR_CHART3D)
			||input.getType().equals(HSTACKBAR_CHART3D)
			||input.getType().equals(VBAR_CHART3D)
			||input.getType().equals(HSTACKBAR_CHART3D)
			||input.getType().equals(VSTACKBAR_CHART)
			||input.getType().equals(VSTACKBAR_CHART3D))
		{
			functions.append(jscript.generateMoveNode());	
		} 

		if (input.getType().equals(VBAR_CHART) || input.getType().equals(HBAR_CHART) || input.getType().equals(VBAR_CHART3D) || input.getType().equals(HBAR_CHART3D)) {
			functions.append(jscript.generateToggleVisibilityDatasetBar());	
		}
		else if (input.getType().equals(SCATTER_CHART)) {
			functions.append(jscript.generateToggleVisibilityDatasetScatter());	
		}
		else {
			functions.append(jscript.generateToggleVisibilityDataset());	
		}

		inlineScripts(input, functions);
		scriptEmitter.setDefaultScripts(functions.toString());
		
		
	}

	static protected void inlineScripts(Chart input, StringBuffer functions){
		if (input.getConfiguration() != null) {
			Scripts scripts = input.getConfiguration().getScripts();
			if (scripts != null) {
				for (int x = 0; x < scripts.getEcmascript().size(); x++){
					String scriptContent = ((Ecmascript) scripts.getEcmascript().get(x)).getValue();
					if (scriptContent != null)
						functions.append(scriptContent);					
				}
			}
		}		
	}
	
	static protected boolean showShapes(Chart input){
		boolean showShapes = true;
		if (input.getConfiguration() != null && input.getConfiguration().getShapes() != null){
			if (input.getConfiguration().getShapes().getVisibility() == null)
				input.getConfiguration().getShapes().setVisibility(IConstants.VISIBLE);
			showShapes = (input.getConfiguration().getShapes().getVisibility().trim().equals(IConstants.VISIBLE));
		}			
		return showShapes;
	}
	
	static protected boolean showUserPref(Preferences pref) {
		if (pref != null) {
			if (pref.isSetShow() && pref.isShow()) {
			 	return true;
			}
		}
		return false;
	}
	
	static protected String getDefaultPreferences(Chart input, SVGColorPalettes palettes, String type, String currentFont){
		StringBuffer preference = new StringBuffer();
		String defaultFont = IGraphicDocumentStyle.BASIC_FONTSTYLE;
		String newDefaultFont = defaultFont.substring(0, defaultFont.indexOf("font-family") + "font-family".length() + 1)
								+ currentFont 
								+ defaultFont.substring(defaultFont.indexOf("; fill"), defaultFont.length());
				
		preference.append(palettes.getPaletteDisplayName());
		preference.append(",");
		preference.append(IGraphicDocumentStyle.BASIC_FONTSIZE);
		preference.append(",");
		preference.append(newDefaultFont);
		preference.append(",");

		String showShapes = "1";
		if (input.getConfiguration() != null && input.getConfiguration().getShapes() != null){
			if (input.getConfiguration().getShapes().getVisibility() == null)
				input.getConfiguration().getShapes().setVisibility(IConstants.VISIBLE);
			showShapes = (input.getConfiguration().getShapes().getVisibility().trim().equals(IConstants.VISIBLE))?"1":"0";
		}			
		preference.append(showShapes);
		preference.append(",");
		String showValues = "0";
		String showPercentage = "0";
		if (type.equals(IGraphicTypeConstants.PIE_CHART) || type.equals(IGraphicTypeConstants.PIE_CHART3D)){
			showValues = "1";
			showPercentage = "1";
		}
		if (input.getConfiguration() != null && input.getConfiguration().getDataValue() != null){
			if (input.getConfiguration().getDataValue().getValueVisibility() == null){
				// default for show values in pie chart is ON, all others OFF
				if (type.equals(IGraphicTypeConstants.PIE_CHART) || type.equals(IGraphicTypeConstants.PIE_CHART3D))
					input.getConfiguration().getDataValue().setPercentageVisibility(IConstants.VISIBLE);
				else 	
					input.getConfiguration().getDataValue().setPercentageVisibility(IConstants.HIDDEN);
			}
			if (input.getConfiguration().getDataValue().getPercentageVisibility() == null){
				// default for show values in pie chart is ON, all others OFF
				if (type.equals(IGraphicTypeConstants.PIE_CHART) || type.equals(IGraphicTypeConstants.PIE_CHART3D))
					input.getConfiguration().getDataValue().setPercentageVisibility(IConstants.VISIBLE);
				else 	
					input.getConfiguration().getDataValue().setPercentageVisibility(IConstants.HIDDEN);
			}
			Configuration config = input.getConfiguration();
			if (config != null) {
				DataValue dataValue = config.getDataValue();
				if (dataValue != null) {
					String valueVisibility = dataValue.getValueVisibility();
					if (valueVisibility != null) {
						showValues = valueVisibility.trim().equals(IConstants.VISIBLE) ? "1" : "0";
					}
					String percentageVisibility = dataValue.getPercentageVisibility();
					if (percentageVisibility != null) {
						showPercentage = percentageVisibility.trim().equals(IConstants.VISIBLE) ? "1" : "0";
					}
				}
			}
		}
		preference.append(showValues);
		preference.append(",");
		preference.append(showPercentage);
		return preference.toString();
	}

	/**
	 * Generates user specified event handlers into the specified <g> group element
	 * 
	 * @param group the <g> element
	 * @param eventHandlerList the list of events to generate
	 */
	static public void generateUserEvents(SVGGroup group, List eventHandlerList) {
		for (int i = 0; i < eventHandlerList.size(); ++i) {
			EventHandler eventHandler = (EventHandler) eventHandlerList.get(i);
			group.addEvent(eventHandler.getType(), eventHandler.getValue());
		}
	}
	
	/**
	 * Generates user specified event handlers into the specified <g> group element
	 * 
	 * @param group the <g> element
	 * @param eventHandlerList the list of events to generate
	 */
	static public void generateStackAreaProcessedData(SVGScript scriptEmitter, String data) {
		ECMAScriptGenerator jscript = new ECMAScriptGenerator();
		StringBuffer functions = new StringBuffer();
		functions.append(data);	
		scriptEmitter.setDefaultScripts(functions.toString());
	}

	/**
	 * Generates user specified event handlers into the specified <g> group element
	 * 
	 * @param group the <g> element
	 * @param eventHandlerList the list of events to generate
	 */
	static public void generateStackAreaJSFunction(SVGScript scriptEmitter, String data) {
		ECMAScriptGenerator jscript = new ECMAScriptGenerator();
		StringBuffer functions = new StringBuffer();
		functions.append(data);	
		scriptEmitter.setDefaultScripts(functions.toString());
	}
		
	/** Generates user specified tooltips
	 * 
	 * @param group the <g> element which holds the mouse events
	 * @param tooltip the tooltip object
	 * @return an SVGText object holding the tooltip text
	 */
	static public SVGBase generateUserTooltip(SVGGroup group, Tooltip tooltip, NLString nls) {
		SVGBase result = null;

		if (tooltip != null) {
			String tooltipID;
			
			if (tooltip.getId() != null) {
				tooltipID = tooltip.getId();
			}
			else {
				tooltipID = "SVGGEN_tooltip" + getUniqueID();
			}
								
			SVGGroup toolTipGroup = new SVGGroup();
			SVGBase[] toolTips = new SVGBase[2];
			toolTipGroup.setChildren(toolTips);
			toolTipGroup.setVisibility("hidden");
						
			SVGText tooltipText = new SVGText();
			tooltipText.setText(nls.getString(tooltip.getLabel()));
			tooltipText.setIdentifier(tooltipID + "text");
			toolTips[0] = tooltipText;
			
			SVGRectangle tooltipRect = new SVGRectangle();
			tooltipRect.setIdentifier(tooltipID + "rect");
			tooltipRect.setXCoordinate("0");
			tooltipRect.setYCoordinate("0");
			tooltipRect.setWidth("0");
			tooltipRect.setHeight("0");
			toolTips[1] = tooltipRect;
			
			result = toolTipGroup;


			group.addEvent("onmouseover", "SVGGEN_showTooltip(evt,'"+tooltipID+"')");
			group.addEvent("onmouseout", "SVGGEN_hideTooltip(evt,'"+tooltipID+"')");
		}
		
		return result;
	}

	/**
	 * Generates value tooltips
	 * 
	 * @param group the <g> element which holds the mouse events
	 * @param value the value to display in the tooltip
	 * @return an SVGText object holding the tooltip text
	 */
	static public SVGBase generateValueTooltip(SVGGroup group, String value) {
		SVGBase result = null;

		String tooltipID = "SVGGEN_tooltip" + getUniqueID();
				
		SVGGroup toolTipGroup = new SVGGroup();
		SVGBase[] toolTips = new SVGBase[2];
		toolTipGroup.setChildren(toolTips);
		toolTipGroup.setVisibility("hidden");
						
		SVGText tooltipText = new SVGText();
		tooltipText.setText(value);
		tooltipText.setIdentifier(tooltipID + "text");
		toolTips[0] = tooltipText;
			
		SVGRectangle tooltipRect = new SVGRectangle();
		tooltipRect.setIdentifier(tooltipID + "rect");
		tooltipRect.setXCoordinate("0");
		tooltipRect.setYCoordinate("0");
		tooltipRect.setWidth("0");
		tooltipRect.setHeight("0");		
		toolTips[1] = tooltipRect;
			
		result = toolTipGroup;
				
		group.addEvent("onmouseover", "SVGGEN_showTooltip(evt,'"+tooltipID+"')");
		group.addEvent("onmouseout", "SVGGEN_hideTooltip(evt,'"+tooltipID+"')");
		
		return result;
	}
	
	/**
	 * Adds user specified accessibility information to the group
	 * 
	 * @param group the SVGGroup to which to add the information
	 * @param accessibility the Accessibility object containing the title and desc
	 */
	static public void generateAccessibility(SVGGroup group, Accessibility accessibility, NLString nls) {
		if (accessibility != null) {			
			String title = nls.getString(accessibility.getTitle());
			if (title != null && title.length() != 0) {
				group.setTitle(title);
			}
			
			String desc = nls.getString(accessibility.getDescription());
			if (desc != null && desc.length() != 0) {
				group.setDesc(desc);
			}
		}
	}

	/**
	 * Generates unique IDs between "000000000" and "999999999"
	 * 
	 * @return a zero padded string representing the unique ID
	 */
	static synchronized public String getUniqueID() {
		if (EventTools.IDGenerator != null)
			return EventTools.IDGenerator.getUniqueID();
		String result = Integer.toString(idNumber);
		
		int zeros = 9 - result.length();
		for (int i = 0; i < zeros; ++i) {
			result = '0' + result;
		}
		
		++idNumber;
		if (idNumber == 999999999) {
			idNumber = 0;
		}
		
		return result;
	}
    
    /**
     * Generates unique IDs between "000000000" and "999999999"
     * 
     * @return a zero padded string representing the unique ID
     */
    static synchronized public int getHashCode(Object obj) {
        if (EventTools.IDGenerator != null)
            return EventTools.IDGenerator.getUniqueHashcode(obj);
        return obj.hashCode();
    }    

}
