/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GraphAreaMeter.java,v 1.4 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;
/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GraphAreaMeter.java,v 1.4 2008/12/12 22:22:09 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGCircle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGGroup;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGLine;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGPath;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGPolyline;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGText;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Axes;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.AxisDefinition;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.CategoricalData;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Coordinates;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSet;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSets;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataValue;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Internationalization;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.MarkerLine;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.NumberFormat;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.UnitDefinition;

import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.util.ULocale;


/**
 * 
 */
public class GraphAreaMeter extends GraphArea {
	static final long serialVersionUID = 2017681070254741260L;
	/**
	 * locale
	 */
	private ULocale locale = null;
	private double min;
	private double max;
	private double pointerValue = 0.0;
	private double majorUnit;
	private double minorUnit;
	private String label;
	private boolean pointerValueKnown = false;
	private com.ibm.icu.text.NumberFormat numberFormatter = null;
	
	private DataSet dataset = null;
	private CategoricalData datumCat = null;
	private Coordinates datumCoor = null;
	private boolean useDatumCat = false;
	private String pointerId = null;
		
	private TreeSet sortedMarkers; 
	/**
	 * @param input
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 */
	public GraphAreaMeter(
		Chart input,
		double x,
		double y,
		double width,
		double height) {
		super(input, x, y, width, height);
		
		analyzeData();
	}
	
	/**
	 * Get data from input
	 *
	 */
	private void analyzeData() {
		// get data value for pointer
		getPointerValue();
		
		// determine dial max, min values
		double orderOfMag;
		boolean maxKnown = false;
		boolean minKnown = false;
		boolean majorUnitKnown = false;
		boolean minorUnitKnown = false;
		
		Configuration config = input.getConfiguration();
		if (config != null) {
			// get locale settings
			Internationalization i18n = config.getInternationalization();
			if (i18n != null) {
				String language = i18n.getLanguage() == null ? "" : i18n.getLanguage();
				String country = i18n.getCountry() == null ? "" : i18n.getCountry();
				if ((language != "") && (country != ""))				
					locale = new ULocale(language, country);
				else if (language != "")
					locale = new ULocale(language);
			}
			
			Axes axes = config.getAxes();
			DataValue dataValueDef = config.getDataValue();
			if (axes != null) {
				AxisDefinition axisDef = axes.getIndependentAxis();
				
				// init data formatter
				initDataFormatter(axisDef,dataValueDef);
				
				// get values related to the axis (dial scale)
				if (axisDef != null) {
					// get max value
					if (axisDef.isSetMax()) {
						max = axisDef.getMax();
						maxKnown = true;
					}
					//get min value
					if (axisDef.isSetMin()) {
						min = axisDef.getMin();
						minKnown = true;
					}
					// get major unit
					UnitDefinition majorUnitDef = axisDef.getMajorUnit();
					if (majorUnitDef != null) {
						if (majorUnitDef.isSetValue()) {
							majorUnit = Math.abs(majorUnitDef.getValue());
							if (majorUnit != 0) {
								majorUnitKnown = true;
							}
						}
					}
					//get minor unit
					UnitDefinition minorUnitDef = axisDef.getMinorUnit();
					if (minorUnitDef != null) {
						if (minorUnitDef.isSetValue()) {
							minorUnit = Math.abs(minorUnitDef.getValue());
							if (minorUnit != 0) {
								minorUnitKnown = true;
							}
						}
					}
					// get marker values 
					//(for marking regions such as "safe", "warning", "danger", ...
					List markerList = axisDef.getMarkerLine();
					if (markerList != null && markerList.size() > 0) {
						sortedMarkers = new TreeSet(
							new Comparator() {
								public int compare(Object o1, Object o2) {
								   MarkerLine m1 = (MarkerLine)o1;
								   MarkerLine m2 = (MarkerLine)o2;
								   double d1 = m1.getValue();
								   double d2 = m2.getValue();
								   return (d1 == d2) ? 0 : (d1 < d2) ? -1 : 1;
								}
							}
						);
						sortedMarkers.addAll(markerList);
					}
				}
			}
			if (!maxKnown || !minKnown || !majorUnitKnown) {
				if (maxKnown)
					orderOfMag = getOrderOfMagnitude(pointerValue, max);
//					orderOfMag = getOrderOfMagnitude(pointerValue);
				else
					orderOfMag = getOrderOfMagnitude(pointerValue);
				if (!majorUnitKnown) {
					majorUnit = orderOfMag;
				}
				if (!minorUnitKnown) {
					minorUnit  = majorUnit / 2;
				}
//				if (sortedMarkers != null && sortedMarkers.size() > 0) {
//					max = 0;
//					for (Iterator i=sortedMarkers.iterator(); i.hasNext();) {
//						MarkerLine marker = (MarkerLine) i.next();
//						double value = marker.getValue();
//						if (!maxKnown && value > max) {
//							max = value;
//						}
//						if (!minKnown && value < min) {
//							min = value;
//						} 
//					}
//					maxKnown = true;
//					minKnown = true;
//				}

				// find the largest marker line value
				double maxMarkerValue = 0;
				boolean maxMarkerValueSet = false;
				if (sortedMarkers != null && sortedMarkers.size() > 0) {
					for (Iterator i=sortedMarkers.iterator(); i.hasNext();) {
						MarkerLine marker = (MarkerLine) i.next();
						if (maxMarkerValueSet == false) {
							maxMarkerValue = marker.getValue();
							maxMarkerValueSet = true;
						} else {
							if (marker.getValue() > maxMarkerValue) {
								maxMarkerValue = marker.getValue();
							}
						}
					}
				}

				if (!maxKnown) {
					if (pointerValue < 0) {
						max = 0;
					} else {
						// find axisMax from dataMax and major unit
						max = Math.floor(pointerValue / majorUnit) * majorUnit + majorUnit;
						// Use the largest marker value as the max if available
						if (maxMarkerValueSet && maxMarkerValue > max) {
							max = maxMarkerValue;
						}
					}
				}
				if (!minKnown) {
					// find axisMin from dataMin and major unit
					if (min >= 0) {
						min = 0;
					} else {
						min = (Math.floor(Math.abs(pointerValue) / majorUnit) * majorUnit + majorUnit) * (-1); 
					}
				}
			}
		}
	}
	
	private void initDataFormatter(AxisDefinition axisDef, DataValue dataValueDef) {
		if (axisDef != null) {
			NumberFormat numFormat = axisDef.getNumberFormat();
			if (numFormat == null && dataValueDef != null) numFormat = dataValueDef.getNumberFormat();
			if (numFormat != null) {
				if (locale != null) {
					numberFormatter = com.ibm.icu.text.NumberFormat.getInstance(locale);
				} else {
					numberFormatter = com.ibm.icu.text.NumberFormat.getInstance();
				}
				if (numberFormatter instanceof DecimalFormat) {
					((DecimalFormat) numberFormatter).applyPattern(numFormat.getPattern());
				}
			}
		}
	}
	
	private String formatValue(double value) {
		String labelText;
		if (numberFormatter != null) {
			labelText = numberFormatter.format(value);
		} else {
			labelText = Double.toString(value);
		}
		return labelText;
	}
	
	private void getPointerValue() {
		Data data = input.getData();
		if (data != null) {
			DataSets datasets = data.getDataSets();
			if (datasets != null) {
				List datasetList = datasets.getDataSet();
				if (datasetList.size() != 0) {
					dataset = (DataSet) datasetList.get(0);
					label = dataset.getLabel();
					
					List dataList = dataset.getDataPoint();
					if (dataList.size() != 0) {
						datumCat = (CategoricalData) dataList.get(0);
						if (datumCat.isSetValue()) {
							pointerValue = datumCat.getValue();
							pointerId = datumCat.getId();
							pointerValueKnown = true;
							useDatumCat = true;
							return;
						}
					}
					dataList = dataset.getCoordinates();
					if (dataList.size() != 0) {
						datumCoor = (Coordinates) dataList.get(0);
						if (datumCoor.isSetValue1()) {
							pointerValue = datumCoor.getValue1();
							pointerId = datumCoor.getId();
							pointerValueKnown = true;
						}
					}
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
	 */
	protected void constructPart() {
		double x, y;
	
		// centre points for everything
		double gWidth = width;
		double gHeight = height;
		double cx = gWidth / 2;
		double cy = gHeight / 2 + 5;		

		// get the smaller value of graphicHeight/graphicWidth
		double baseLength = gHeight >= gWidth ? gWidth : gHeight; 
		
		// create meter element parts
		// parts: outer rim, inner rim, bkground,
		SVGBase[] meterParts = new SVGBase[11];
		setChildren(meterParts);
		
		// outer rim
		SVGCircle outerRim = new SVGCircle();
		outerRim.setStyleClass("outerRim");
		outerRim.setCx(Double.toString(cx));
		outerRim.setCy(Double.toString(cy));
		outerRim.setRadius(Double.toString(baseLength * 0.360));
		meterParts[0] = outerRim;
		
		// inner rim
		SVGCircle innerRim = new SVGCircle();
		innerRim.setStyleClass("innerRim");
		innerRim.setCx(Double.toString(cx));
		innerRim.setCy(Double.toString(cy));
		innerRim.setRadius(Double.toString(baseLength * 0.352));
		meterParts[1] = innerRim;
		
		// meter background
		SVGCircle bkground = new SVGCircle();
		bkground.setStyleClass("speedometerBg");
		bkground.setCx(Double.toString(cx));
		bkground.setCy(Double.toString(cy));
		bkground.setRadius(Double.toString(baseLength * 0.344));
		meterParts[2] = bkground;

		double boxWidth = baseLength * 0.262;
		double boxYOffset = baseLength * 0.1;
		short boxHeight = 14;

		// the line of text that goes below the dial
		// group for font changes		
		if (label != null) {
			SVGGroup labelTextGroup = new SVGGroup();
			labelTextGroup.setIdentifier("rateDefinition");
			labelTextGroup.setStyleClass("rateDef anchorAtMiddle");
			meterParts[5] = labelTextGroup;
			SVGBase[] groupChildren = new SVGBase[1];
			labelTextGroup.setChildren(groupChildren);
			
			SVGText labelText = new SVGText();
			labelText.setXCoordinate(Double.toString(cx));
			labelText.setYCoordinate(Double.toString(cy + boxYOffset/2));
			labelText.setText(nls.getString(label));
			groupChildren[0] = labelText;
		}

		// draw dial range (a grey semicircle)
		double radius = baseLength * 0.25;	
		String path;
		path =
			"M "
			+ (cx - radius)
			+ " "
			+ cy
			+ " A "
			+ radius
			+ " "
			+ radius
			+ " 0 1 1 "
			+ (cx + radius)
			+ " "
			+ cy
			+ " Z";

		SVGPath rangeBackground = new SVGPath();
		rangeBackground.setStyleClass("safeRange");
		rangeBackground.setPathData(path);
		meterParts[6] = rangeBackground;

		// draw color ranges 
		if (sortedMarkers != null && sortedMarkers.size() > 0) {
			SVGGroup sectorGroup = new SVGGroup();
			SVGBase[] sectors = new SVGBase[sortedMarkers.size()];
			meterParts[7] = sectorGroup;
			sectorGroup.setChildren(sectors);
			int sectorIndex = 0;
			double angleSum = 0;
			
			for (Iterator i=sortedMarkers.iterator(); i.hasNext();) {
				MarkerLine marker = (MarkerLine) i.next();
				double value = marker.getValue();
				if (value > max) {
					value = max;
				} else if (value <= min) {
//					value = min;
					continue;
				}
				String color = marker.getColor();
				String markerlabel = nls.getString(marker.getLabel());
				double angle = 180 / (max - min) * (value - min) - angleSum;
				double arcx = cx + (radius) * Math.cos(Math.toRadians(180 - angle));
				double arcy = cy - (radius) * Math.sin(Math.toRadians(180 - angle));
				
				String sectorPath =
					"M "
					+ cx
					+ " "
					+ cy
					+ " L "
					+ (cx - radius)
					+ " "
					+ cy
					+ " A "
					+ radius
					+ " "
					+ radius
					+ " 0 0 1 "
					+ arcx
					+ " "
					+ arcy
					+ " L "
					+ cx
					+ " "
					+ cy
					+ " Z";

				SVGGroup pathGroup = new SVGGroup();
				SVGBase[] thePath = new SVGBase[2];
				pathGroup.setChildren(thePath);
				sectors[sectorIndex++] = pathGroup;
				thePath[1] = EventTools.generateValueTooltip(pathGroup, markerlabel);
				
				SVGPath sector = new SVGPath();
				sector.setStyle("fill:" + color);
				sector.setPathData(sectorPath);
				sector.setTransformation("rotate(" + angleSum +"," + cx + ","+ cy + ")");
				thePath[0] = sector;
				angleSum += angle;
			}
		}
		
		// draw tick marks
		// tick mark definition
		SVGPolyline tickDef = new SVGPolyline();
		tickDef.setPoints("0 0 " + TICK_MARK_LENGTH + " 0");
		tickDef.setStyleClass("ticks");
		tickDef.setIdentifier("tick");
		addDefinition(tickDef);
		
		// major tick marks and labels
		int numOfTicks = (int) ((max-min) / majorUnit);
		if ((max-min) % majorUnit == 0) {
			numOfTicks++;
		}

		SVGGroup tickGroup = new SVGGroup();
		SVGBase[] ticks = new SVGBase[numOfTicks*2];
		meterParts[8] = tickGroup;
		tickGroup.setChildren(ticks);
		
		// the coordinates of the unrotated tick mark
		String tickPoints = new String((cx-radius) + " " + cy + " " + (cx-radius+5) + " " + cy);
		
		for (int i=0; i<numOfTicks; i++) {
			// draw tick
			double angle = 180 / (max - min) * majorUnit * i;
			SVGPolyline newTick = new SVGPolyline();
			newTick.setPoints(tickPoints);
			newTick.setStyleClass("ticks");
			newTick.setTransformation("rotate(" + angle +"," + cx + ","+ cy + ")");
			ticks[i*2] = newTick;
			
			// draw label
			SVGText label = new SVGText();
			label.setText(formatValue(min + majorUnit * i));
			double labelx = cx + (radius + 5) * Math.cos(Math.toRadians(180 - angle));
			double labely = cy - (radius + 5) * Math.sin(Math.toRadians(180 - angle));
			if (labelx < cx) {
				label.setStyleClass("anchorAtEnd");
			} else if (labelx > cx) {
				label.setStyleClass("anchorAtStart");
			} else {
				label.setStyleClass("anchorAtMiddle");
			}
			label.setXCoordinate(Double.toString(labelx));
			label.setYCoordinate(Double.toString(labely));
			ticks[i*2+1] = label;
		}
		
		// minor tick marks
		numOfTicks = (int) ((max-min) / minorUnit);
		if ((max-min) % minorUnit == 0) {
			numOfTicks++;
		}

		tickGroup = new SVGGroup();
		ticks = new SVGBase[numOfTicks];
		meterParts[9] = tickGroup;
		tickGroup.setChildren(ticks);
		
		for (int i=0; i<numOfTicks; i++) {
			double angle = 180 / (max - min) * minorUnit * i;
			SVGPolyline newTick = new SVGPolyline();
			newTick.setPoints(tickPoints);
			newTick.setStyleClass("ticks");
			newTick.setTransformation("rotate(" + angle +"," + cx + ","+ cy + ")");
			ticks[i] = newTick;
		}
		
		// pointer
		if (pointerValueKnown && pointerValue >= min && pointerValue <= max) {
			SVGGroup pointerGroup = new SVGGroup();
			meterParts[10] = pointerGroup;

			SVGBase[] pointerParts = new SVGBase[8];
			pointerGroup.setChildren(pointerParts);
			
			// Handle user tooltips (dataset)
			pointerParts[4] = EventTools.generateUserTooltip(pointerGroup, dataset.getTooltip(), nls);

			// Handle user events (dataset)
			EventTools.generateUserEvents(pointerGroup, dataset.getEventHandler());
		
			// Handle accessibility (dataset)
			EventTools.generateAccessibility(pointerGroup, dataset.getAccessibility(), nls);

			// Datapoint events
			if (useDatumCat) {
				// Handle user tooltips
				pointerParts[5] = EventTools.generateUserTooltip(pointerGroup, datumCat.getTooltip(), nls);

				// Handle user events
				EventTools.generateUserEvents(pointerGroup, datumCat.getEventHandler());
		
				// Handle accessibility
				EventTools.generateAccessibility(pointerGroup, datumCat.getAccessibility(), nls);
			}
			else {
				// Handle user tooltips
				pointerParts[5] = EventTools.generateUserTooltip(pointerGroup, datumCoor.getTooltip(), nls);

				// Handle user events
				EventTools.generateUserEvents(pointerGroup, datumCoor.getEventHandler());
		
				// Handle accessibility
				EventTools.generateAccessibility(pointerGroup, datumCoor.getAccessibility(), nls);
			}

			// the box that shows the value of the pointer
			SVGRectangle rateBox = new SVGRectangle();
			rateBox.setStyleClass("actualRate");
			rateBox.setWidth(Double.toString(boxWidth));
			rateBox.setHeight(Short.toString(boxHeight));
			rateBox.setXCoordinate(Double.toString(cx - (boxWidth/2)));
			rateBox.setYCoordinate(Double.toString(cy + boxYOffset));
			pointerParts[6] = rateBox;

			// text of the value of the pointer
			if (pointerValueKnown) {
				String actualRate = formatValue(pointerValue);	
				if (actualRate != null) {
					// group for font changes
					SVGGroup rateTextGroup = new SVGGroup();
					rateTextGroup.setIdentifier("minmaxRate");
					rateTextGroup.setStyleClass("minmaxRate anchorAtMiddle");
					pointerParts[7] = rateTextGroup;
					SVGBase[] groupChildren = new SVGBase[1];
					rateTextGroup.setChildren(groupChildren);
			
					SVGText rateText = new SVGText();
					rateText.setXCoordinate(Double.toString(cx));
					rateText.setYCoordinate(Double.toString(cy + boxYOffset + boxHeight - 3));
					rateText.setText(actualRate);
					groupChildren[0] = rateText;
				}
			}
		
			double pointerLength = radius * 0.95;
			double angle = (pointerValue - min) / (max - min) * 180;
			SVGLine pointer = new SVGLine();
			pointer.setX1(Double.toString(cx - pointerLength));
			pointer.setY1(Double.toString(cy));
			pointer.setX2(Double.toString(cx));
			pointer.setY2(Double.toString(cy));
			pointer.setIdentifier(pointerId);
			pointer.setStyleClass("needle");
			pointer.setTransformation("rotate(" + angle +"," + cx + ","+ cy + ")");
			pointerParts[0] = pointer;
		
			SVGCircle outerCircle = new SVGCircle();
			outerCircle.setCx(Double.toString(cx));
			outerCircle.setCy(Double.toString(cy));
			outerCircle.setRadius(Double.toString(3));
			outerCircle.setStyleClass("outerCircle");
			pointerParts[1] = outerCircle;
		
			SVGCircle innerCircle = new SVGCircle();
			innerCircle.setCx(Double.toString(cx));
			innerCircle.setCy(Double.toString(cy));
			innerCircle.setRadius(Double.toString(2));
			innerCircle.setStyleClass("innerCircle");
			pointerParts[2] = innerCircle;

			SVGCircle center = new SVGCircle();
			center.setCx(Double.toString(cx));
			center.setCy(Double.toString(cy));
			center.setRadius(Double.toString(0.5));
			center.setStyleClass("center");
			pointerParts[3] = center;
		}
	}
	/**
	 * Find the order of magnitude of a number given the maximum value.
	 * @param num
	 * @return order of magnitude
	 */
	private double getOrderOfMagnitude(double num, double max) {
		double numMagnitude = getOrderOfMagnitude(num);
		double magnitude = 1;
		double number = Math.abs(max)/10;
		if (number >= 1) {
			while (number >= 10) {
				number /= 10;
				magnitude *= 10;
			}
		} else if (number != 0){
			while (number < 1) {
				number *= 10;
				magnitude *= 10;
			}
			magnitude = 1 / magnitude;
		}
		if (numMagnitude > magnitude) magnitude = numMagnitude;
		return magnitude;
	}

	/**
	 * Find the order of magnitude of a number.
	 * @param num
	 * @return order of magnitude
	 */
	private double getOrderOfMagnitude(double num) {
		double magnitude = 1;
		double number = Math.abs(num);
		if (number >= 1) {
			while (number >= 10) {
				number /= 10;
				magnitude *= 10;
			}
		} else if (number != 0){
			while (number < 1) {
				number *= 10;
				magnitude *= 10;
			}
			magnitude = 1 / magnitude;
		}
		return magnitude;
	}
	
}
