/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Grid.java,v 1.3 2005/06/10 17:47:09 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Grid.java,v 1.3 2005/06/10 17:47:09 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


package org.eclipse.tptp.platform.report.chart.svg.internal.part;

import java.util.Vector;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGPolyline;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGUse;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.PlotArea;


/**
 * 
 */
public class Grid extends Part {
	static final long serialVersionUID = 1674478110608378637L;
	protected Axis indepAxis = null;
	protected AxisNumber primaryDepAxis = null;
	protected AxisNumber secondaryDepAxis = null;
	
	protected boolean drawBorder = true;
	protected String bkgroundColor;
	
	protected boolean isLTR = true;

	/**
	 * @param input
	 */
	public Grid(Chart input, boolean isLTR) {
		super(input);
		this.isLTR = isLTR;
	}

	/**
	 * @param input
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 */
	public Grid(Chart input, boolean isLTR, double x, double y, double width, double height) {
		super(input, x, y, width, height);
		this.isLTR = isLTR;
		
		Configuration config = input.getConfiguration();
		if (config != null) {
			PlotArea plotarea = config.getPlotArea();
			if (plotarea != null) {
				bkgroundColor = plotarea.getBackgroundColor();
				if (plotarea.isSetBorder()) {
					drawBorder = plotarea.isBorder();
				}
			}
		}
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
	 */
	protected void constructPart() {
		// The pixels on the right and bottom of an SVG cannot be shown after SVG transcoding.
		// Hence, don't use the rightmost pixels to show the right edge of the grid border.
		// The height of the grid does not need adjustment because the bottom of the grid is 
		// always covered by an axis.
		double gridWidth = width - 1;
		
		// grid line defintions
		SVGPolyline horizontalSolidLineDef = new SVGPolyline();
		horizontalSolidLineDef.setPoints("0 0 " + gridWidth + " 0");
		horizontalSolidLineDef.setStyleClass("gridline");
		horizontalSolidLineDef.setIdentifier("horizontalSolidLine");
		
		SVGPolyline verticalSolidLineDef = new SVGPolyline();
		verticalSolidLineDef.setPoints("0 0 0 " + getHeight());
		verticalSolidLineDef.setStyleClass("gridline");
		verticalSolidLineDef.setIdentifier("verticalSolidLine");
		
		SVGPolyline horizontalDashLineDef = new SVGPolyline();
		horizontalDashLineDef.setPoints("0 0 " + gridWidth + " 0");
		horizontalDashLineDef.setStyleClass("griddashline");
		horizontalDashLineDef.setIdentifier("horizontalDashLine");
		
		SVGPolyline verticalDashLineDef = new SVGPolyline();
		verticalDashLineDef.setPoints("0 0 0 " + getHeight());
		verticalDashLineDef.setStyleClass("griddashline");
		verticalDashLineDef.setIdentifier("verticalDashLine");
		
		SVGPolyline horizontalZeroLineDef = new SVGPolyline();
		horizontalZeroLineDef.setPoints("0 0 " + gridWidth + " 0");
		horizontalZeroLineDef.setStyleClass("axisLines");
		horizontalZeroLineDef.setIdentifier("horizontalZeroLine");
		
		SVGPolyline verticalZeroLineDef = new SVGPolyline();
		verticalZeroLineDef.setPoints("0 0 0 " + getHeight());
		verticalZeroLineDef.setStyleClass("axisLines");
		verticalZeroLineDef.setIdentifier("verticalZeroLine");
		
		addDefinition(horizontalSolidLineDef);
		addDefinition(verticalSolidLineDef);
		addDefinition(horizontalDashLineDef);
		addDefinition(verticalDashLineDef);
		addDefinition(horizontalZeroLineDef);
		addDefinition(verticalZeroLineDef);
		
		// draw grid lines
		Vector gridLines = new Vector();
		if (primaryDepAxis != null) {
			// draw major grid lines for primary axis
			if (primaryDepAxis.isShowMajorGridLines()) {
				if (primaryDepAxis.getAxisOrientation() == VERTICAL) {
					gridLines.addAll(drawHorizontalGridLines(primaryDepAxis, true));
				} else {
					gridLines.addAll(drawVerticalGridLines(primaryDepAxis, true));
				}
			}
			// draw minor grid lines for primary axis
			if (primaryDepAxis.isShowMinorGridLines()) {
				if (primaryDepAxis.getAxisOrientation() == VERTICAL) {
					gridLines.addAll(drawHorizontalGridLines(primaryDepAxis, false));
				} else {
					gridLines.addAll(drawVerticalGridLines(primaryDepAxis, false));
				}
			}
			// draw zero line for primary axis
			if (primaryDepAxis.isLinear && primaryDepAxis.getMin() <= 0 && primaryDepAxis.getMax() >= 0) {
				if (primaryDepAxis.getAxisOrientation() == VERTICAL) {
					gridLines.add(drawHorizontalZeroLine(primaryDepAxis));
				} else {
					gridLines.add(drawVerticalZeroLine(primaryDepAxis));
				}
			}
		}
		if (secondaryDepAxis != null) {
			// draw major grid lines for secondary axis
			if (secondaryDepAxis.isShowMajorGridLines()) {
				if (secondaryDepAxis.getAxisOrientation() == VERTICAL) {
					gridLines.addAll(drawHorizontalGridLines(secondaryDepAxis, true));
				} else {
					gridLines.addAll(drawVerticalGridLines(secondaryDepAxis, true));
				}
			}
			// draw minor grid lines for secondary axis
			if (secondaryDepAxis.isShowMinorGridLines()) {
				if (secondaryDepAxis.getAxisOrientation() == VERTICAL) {
					gridLines.addAll(drawHorizontalGridLines(secondaryDepAxis, false));
				} else {
					gridLines.addAll(drawVerticalGridLines(secondaryDepAxis, false));
				}
			}
			// draw zero lines for secondary axis
			if (secondaryDepAxis.isLinear && secondaryDepAxis.getMin() <= 0 && secondaryDepAxis.getMax() >= 0) {
				if (secondaryDepAxis.getAxisOrientation() == VERTICAL) {
					gridLines.add(drawHorizontalZeroLine(secondaryDepAxis));
				} else {
					gridLines.add(drawVerticalZeroLine(secondaryDepAxis));
				}
			}
		}
		
		if (indepAxis != null) {
			// draw major grid lines for independent axis
			if (indepAxis.isShowMajorGridLines()) {
				if (indepAxis.getAxisOrientation() == VERTICAL) {
					if (indepAxis instanceof AxisNumber) {
						gridLines.addAll(drawHorizontalGridLines((AxisNumber)indepAxis, true));
					} else {
						gridLines.addAll(drawHorizontalGridLines((AxisCategory)indepAxis, true));
					}
				} else {
					if (indepAxis instanceof AxisNumber) {
						gridLines.addAll(drawVerticalGridLines((AxisNumber)indepAxis, true));
					} else {
						gridLines.addAll(drawVerticalGridLines((AxisCategory)indepAxis, true));
					}
				}
			}
			
			// draw minor grid lines for independent axis
			if (indepAxis.isShowMinorGridLines()) {
				if (indepAxis.getAxisOrientation() == VERTICAL) {
					if (indepAxis instanceof AxisNumber) {
						gridLines.addAll(drawHorizontalGridLines((AxisNumber)indepAxis, false));
					} else {
						gridLines.addAll(drawHorizontalGridLines((AxisCategory)indepAxis, false));
					}
				} else {
					if (indepAxis instanceof AxisNumber) {
						gridLines.addAll(drawVerticalGridLines((AxisNumber)indepAxis, false));
					} else {
						gridLines.addAll(drawVerticalGridLines((AxisCategory)indepAxis, false));
					}
				}
			}
			
			// draw zero line for independent axis
			if (indepAxis instanceof AxisNumber) {
				AxisNumber numAxis = (AxisNumber) indepAxis;
				if (numAxis.isLinear && numAxis.getMin() <= 0 && numAxis.getMax() >= 0) {
					if (indepAxis.getAxisOrientation() == VERTICAL) {
						gridLines.add(drawHorizontalZeroLine(numAxis));
					} else {
						gridLines.add(drawVerticalZeroLine(numAxis));
					}
				}
			}
		}
		
		// number of children = Number of grid lines + border + background
		SVGBase[] children = new SVGBase[gridLines.size() + 2];
		
		if (bkgroundColor != null && bkgroundColor.equals("") == false) {
			SVGRectangle bkground = new SVGRectangle();
			bkground.setXCoordinate("0");
			bkground.setYCoordinate("0");
			bkground.setWidth(Double.toString(gridWidth));
			bkground.setHeight(getHeight());
			bkground.setFill(bkgroundColor);
			children[0] = bkground;
		}
		
		if (drawBorder) {
			SVGRectangle border = new SVGRectangle();
			border.setXCoordinate("0");
			border.setYCoordinate("0");
			border.setWidth(Double.toString(gridWidth));
			border.setHeight(getHeight());
			border.setFill("none");
			border.setStroke("#cccccc");
			children[1] = border;
		}
		
		for (int i=0; i<gridLines.size(); i++) {
			children[i+2] = (SVGBase) gridLines.elementAt(i);
		}
		setChildren(children);
	}

	/**
	 * Set independent axis
	 * @param axis
	 */
	public void setIndepAxis(Axis axis) {
		indepAxis = axis;
	}

	/**
	 * Set primary dependent axis
	 * @param axis
	 */
	public void setPrimaryDepAxis(AxisNumber axis) {
		primaryDepAxis = axis;
	}

	/**
	 * Set secondary dependent axis
	 * @param axis
	 */
	public void setSecondaryDepAxis(AxisNumber axis) {
		secondaryDepAxis = axis;
	}
	
	/**
	 * Draw horizonatal grid lines
	 * @param axis
	 * @param major true if drawing major grid lines, false if minor grid lines
	 * @return a vector of grid lines
	 */
	protected Vector drawHorizontalGridLines(AxisNumber axis, boolean major) {
		double[] values;
		if (major) {
			values = axis.getMajorUnitValues();
		} else {
			values = axis.getMinorUnitValues();
		}
		double max_y = 0;
		double min_y = Double.parseDouble(getHeight());
		double max = axis.getMax();
		double min = axis.getMin();
		
		Vector lines = new Vector(values.length);

		for (int i=0; i < values.length; i++) {
			double line_y = getCoordinate(values[i], max_y, min_y, max, min);
			SVGUse useLine = new SVGUse();
			if (major) {
				useLine.setHref("#horizontalSolidLine");
			} else {
				useLine.setHref("#horizontalDashLine");
			}
			useLine.setXCoordinate("0");
			useLine.setYCoordinate(Double.toString(line_y));
			lines.add(useLine);
		}
		
		return lines;
	}

	/**
	 * Draw horizontal "zero line".  Draws a solid line perpendicular to the
	 * axis at the value 0.
	 * 
	 * @param axis
	 * @return
	 */	
	protected SVGBase drawHorizontalZeroLine(AxisNumber axis) {
		double max_y = 0;
		double min_y = Double.parseDouble(getHeight());
		double max = axis.getMax();
		double min = axis.getMin();
		
		double line_y = getCoordinate(0.0, max_y, min_y, max, min);
		SVGUse useLine = new SVGUse();
		useLine.setHref("#horizontalZeroLine");
		useLine.setXCoordinate("0");
		useLine.setYCoordinate(Double.toString(line_y));

		return useLine;
	}

	/**
	 * Draw horizonatal grid lines
	 * @param axis
	 * @param major true if drawing major grid lines, false if minor grid lines 
	 * @return a vector of grid lines
	 */
	protected Vector drawVerticalGridLines(AxisNumber axis, boolean major) {
		double[] values;
		if (major) {
			values = axis.getMajorUnitValues();
		} else {
			values = axis.getMinorUnitValues();
		}
		double max_x = Double.parseDouble(getWidth());
		double min_x = 0;
		double max = axis.getMax();
		double min = axis.getMin();
		
		Vector lines = new Vector(values.length);

		for (int i=0; i < values.length; i++) {
			double line_x;
			if (isLTR) {
				line_x = getCoordinate(values[i], max_x, min_x, max, min);
			} else {
				line_x = getCoordinate(values[i], min_x, max_x, max, min);
			}
			SVGUse useLine = new SVGUse();
			if (major) {
				useLine.setHref("#verticalSolidLine");
			} else {
				useLine.setHref("#verticalDashLine");
			}
			useLine.setXCoordinate(Double.toString(line_x));
			useLine.setYCoordinate("0");
			lines.add(useLine);
		}
		
		return lines;
	}
	
	protected Vector drawVerticalGridLines(AxisCategory axis, boolean major) {
		double[] values;
		if (major) {
			values = axis.getGroupBoundaryPositions();
		} else {
			values = axis.getGroupDataPositions();
		}

		Vector lines = new Vector(values.length);

		for (int i=0; i < values.length; i++) {
			SVGUse useLine = new SVGUse();
			if (major) {
				useLine.setHref("#verticalSolidLine");
			} else {
				useLine.setHref("#verticalDashLine");
			}
			useLine.setXCoordinate(Double.toString(values[i]));
			useLine.setYCoordinate("0");
			lines.add(useLine);
		}
		
		return lines;
	}
	
	protected Vector drawHorizontalGridLines(AxisCategory axis, boolean major) {
		double[] values;
		if (major) {
			values = axis.getGroupBoundaryPositions();
		} else {
			values = axis.getGroupDataPositions();
		}

		Vector lines = new Vector(values.length);

		for (int i=0; i < values.length; i++) {
			SVGUse useLine = new SVGUse();
			if (major) {
				useLine.setHref("#horizontalSolidLine");
			} else {
				useLine.setHref("#horizontalDashLine");
			}
			useLine.setXCoordinate("0");
			useLine.setYCoordinate(Double.toString(values[i]));
			lines.add(useLine);
		}
		
		return lines;
	}
	
	/**
	 * Draw vertical "zero line".  Draws a solid line perpendicular to the
	 * axis at the value 0.
	 * 
	 * @param axis
	 * @return
	 */	
	protected SVGBase drawVerticalZeroLine(AxisNumber axis) {
		double max_x = Double.parseDouble(getWidth());
		double min_x = 0;
		double max = axis.getMax();
		double min = axis.getMin();
		
		double line_x;
		if (isLTR) {
			line_x = getCoordinate(0.0, max_x, min_x, max, min);
		} else {
			line_x = getCoordinate(0.0, min_x, max_x, max, min);
		}
		SVGUse useLine = new SVGUse();
		useLine.setHref("#verticalZeroLine");
		useLine.setXCoordinate(Double.toString(line_x));
		useLine.setYCoordinate("0");

		return useLine;
	}

	/**
	 * Calculate coordinate in linear scale
	 * @param value value to be plotted
	 * @param max_coor maximum coordinate of this axis
	 * @param min_coor minimum coordinate of this axis
	 * @param max maximum value of this axis
	 * @param min minimum value of this axis
	 * @return
	 */
	protected double getCoordinate(double value, double max_coor, double min_coor, double max, double min) {
		return (value - max) / (min - max) * (min_coor - max_coor) + max_coor;
	}

}
