/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Grid3D.java,v 1.2 2005/06/10 17:47:09 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Grid3D.java,v 1.2 2005/06/10 17:47:09 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.util.Vector;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGFeComponentTransfer;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGFeFunc;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGFilter;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGPolygon;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGPolyline;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGUse;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;



/**
 * <code>Grid3D</code> class draws the grid lines in a 3-D space.
 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.Grid
 */
public class Grid3D extends Grid {
    protected double x3Doffset;
    protected double y3Doffset;
    static final long serialVersionUID = 8771697403582307286L;
    /**
     * Constructor
     * 
     * @param input chart input data
     * @param isLTR a boolean value that determines if the layout should be displayed left-to-right
     * 
     */
    public Grid3D(Chart input, boolean isLTR) {
        super(input, isLTR);
    }

    /**
     * Constructor
     * 
     * @param input chart input data
     * @param x the x position of the grid
     * @param y the y position of the grid
     * @param width the width of the grid
     * @param height the height of the grid
     * @param x3Doffset the x 3-D offset that determines the depth of the 3-D grid
     * @param y3Doffset the y 3-D offset that determines the depth of the 3-D grid
     */
    public Grid3D(Chart input, boolean isLTR, double x, double y, double width,
        double height, double x3Doffset, double y3Doffset) {
        super(input, isLTR, x, y, width, height);
        this.x3Doffset = x3Doffset;
        this.y3Doffset = -y3Doffset;
    }

    /* (non-Javadoc)
     * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
     */
    protected void constructPart() {
        // The pixels on the right and bottom of an SVG cannot be shown after Batik transcoding.
        // Hence, don't use the rightmost pixels to show the right edge of the grid border.
        // The height of the grid does not need adjustment because the bottom of the grid is 
        // always covered by an axis.
        double gridWidth = width - 1;

        // grid line defintions
        SVGPolyline horizontalSolidLineDef = new SVGPolyline();
        horizontalSolidLineDef.setFill("none");
        if (isLTR)
	        horizontalSolidLineDef.setPoints("0 0 " + x3Doffset + " " + y3Doffset +
    	        " " + (gridWidth) + " " + y3Doffset);
    	else
	        horizontalSolidLineDef.setPoints(x3Doffset + " " + y3Doffset +
    	        " " + (gridWidth) + " " + y3Doffset);
        horizontalSolidLineDef.setStyleClass("gridline");
        horizontalSolidLineDef.setIdentifier("horizontalSolidLine");

        SVGPolyline verticalSolidLineDef = new SVGPolyline();
        verticalSolidLineDef.setFill("none");
        verticalSolidLineDef.setPoints(x3Doffset + " 0 " + x3Doffset + " " +
            (height + y3Doffset) + " 0 " + height);
        verticalSolidLineDef.setStyleClass("gridline");
        verticalSolidLineDef.setIdentifier("verticalSolidLine");

        SVGPolyline horizontalDashLineDef = new SVGPolyline();
        horizontalDashLineDef.setFill("none");
        if (isLTR)
	        horizontalDashLineDef.setPoints("0 0 " + x3Doffset + " " + y3Doffset +
    	        " " + (gridWidth) + " " + y3Doffset);
    	else
	        horizontalDashLineDef.setPoints(x3Doffset + " " + y3Doffset +
    	        " " + (gridWidth) + " " + y3Doffset);
        horizontalDashLineDef.setStyleClass("griddashline");
        horizontalDashLineDef.setIdentifier("horizontalDashLine");

        SVGPolyline verticalDashLineDef = new SVGPolyline();
        verticalDashLineDef.setFill("none");
        verticalDashLineDef.setPoints(x3Doffset + " 0 " + x3Doffset + " " +
            (height + y3Doffset) + " 0 " + height);
        verticalDashLineDef.setStyleClass("griddashline");
        verticalDashLineDef.setIdentifier("verticalDashLine");

        SVGPolygon horizontalZeroLineDef = new SVGPolygon();
        horizontalZeroLineDef.setPoints("0,0 " + (gridWidth - x3Doffset) +
            ",0 " + gridWidth + ", " + y3Doffset + " " + x3Doffset + "," +
            y3Doffset);
        horizontalZeroLineDef.setStyleClass("zero3DAxisLines");
        horizontalZeroLineDef.setIdentifier("horizontalZeroLine");

        SVGPolygon verticalZeroLineDef = new SVGPolygon();
        verticalZeroLineDef.setPoints("0," + (-y3Doffset) + " " + x3Doffset +
            ",0 " + x3Doffset + "," + (height + y3Doffset) + " 0," + (height));
        verticalZeroLineDef.setStyleClass("zero3DAxisLines");
        verticalZeroLineDef.setIdentifier("verticalZeroLine");

        SVGPolygon horizontalLineDef = new SVGPolygon();
        horizontalLineDef.setPoints("0,0 " + (gridWidth - x3Doffset) +
            ",0 " + gridWidth + ", " + y3Doffset + " " + x3Doffset + "," +
            y3Doffset);
        horizontalLineDef.setStyleClass("axis3DLimitLines");
        horizontalLineDef.setIdentifier("horizontalLimitLine");
        horizontalLineDef.setFilter("url(#top-dimension)");
        

        SVGPolygon verticalLineDef = new SVGPolygon();
        verticalLineDef.setPoints("0," + (-y3Doffset) + " " + x3Doffset +
            ",0 " + x3Doffset + "," + (height + y3Doffset) + " 0," + (height));
        verticalLineDef.setStyleClass("axis3DLimitLines");
        verticalLineDef.setIdentifier("verticalLimitLine");
        verticalLineDef.setFilter("url(#right-dimension)");
        

        //add 3d filters
        SVGFilter filter = new SVGFilter();
        filter.setIdentifier("right-dimension");

        SVGFeComponentTransfer[] feComponentTransferPart = new SVGFeComponentTransfer[1];
        SVGFeComponentTransfer feComponentTransfer = new SVGFeComponentTransfer();
        feComponentTransferPart[0] = feComponentTransfer;
        filter.setChildren(feComponentTransferPart);

        SVGFeFunc[] feFuncPart = new SVGFeFunc[3];
        SVGFeFunc feFunc = new SVGFeFunc("R");
        feFunc.setSlope("1");
        feFunc.setIntercept("-0.1");
        feFunc.setType("linear");
        feFuncPart[0] = feFunc;

        feFunc = new SVGFeFunc("G");
        feFunc.setSlope("1");
        feFunc.setIntercept("-0.1");
        feFunc.setType("linear");
        feFuncPart[1] = feFunc;

        feFunc = new SVGFeFunc("B");
        feFunc.setSlope("1");
        feFunc.setIntercept("-0.1");
        feFunc.setType("linear");
        feFuncPart[2] = feFunc;

        feComponentTransfer.setChildren(feFuncPart);

        addDefinition(filter);

        filter = new SVGFilter();
        filter.setIdentifier("top-dimension");

        feComponentTransferPart = new SVGFeComponentTransfer[1];
        feComponentTransfer = new SVGFeComponentTransfer();
        feComponentTransferPart[0] = feComponentTransfer;
        filter.setChildren(feComponentTransferPart);

        feFuncPart = new SVGFeFunc[3];
        feFunc = new SVGFeFunc("R");
        feFunc.setSlope("1");
        feFunc.setIntercept("0.1");
        feFunc.setType("linear");
        feFuncPart[0] = feFunc;

        feFunc = new SVGFeFunc("G");
        feFunc.setSlope("1");
        feFunc.setIntercept("0.1");
        feFunc.setType("linear");
        feFuncPart[1] = feFunc;

        feFunc = new SVGFeFunc("B");
        feFunc.setSlope("1");
        feFunc.setIntercept("0.1");
        feFunc.setType("linear");
        feFuncPart[2] = feFunc;

        feComponentTransfer.setChildren(feFuncPart);

        addDefinition(filter);

        //end of add 3d filters        

        addDefinition(horizontalSolidLineDef);
        addDefinition(verticalSolidLineDef);
        addDefinition(horizontalDashLineDef);
        addDefinition(verticalDashLineDef);
        addDefinition(horizontalZeroLineDef);
        addDefinition(verticalZeroLineDef);
        addDefinition(horizontalLineDef);
        addDefinition(verticalLineDef);

        // draw grid lines
        Vector gridLines = new Vector();

        if (primaryDepAxis != null) {
            // draw major grid lines for primary axis
            if (primaryDepAxis.isShowMajorGridLines()) {
                if (primaryDepAxis.getAxisOrientation() == VERTICAL) {
                    gridLines.addAll(drawHorizontalGridLines(primaryDepAxis,
                            true));
                } else {
                    gridLines.addAll(drawVerticalGridLines(primaryDepAxis, true));
                }
            }

            // draw minor grid lines for primary axis
            if (primaryDepAxis.isShowMinorGridLines()) {
                if (primaryDepAxis.getAxisOrientation() == VERTICAL) {
                    gridLines.addAll(drawHorizontalGridLines(primaryDepAxis,
                            false));
                } else {
                    gridLines.addAll(drawVerticalGridLines(primaryDepAxis, false));
                }
            }

        }

        if (secondaryDepAxis != null) {
            // draw major grid lines for secondary axis
            if (secondaryDepAxis.isShowMajorGridLines()) {
                if (secondaryDepAxis.getAxisOrientation() == VERTICAL) {
                    gridLines.addAll(drawHorizontalGridLines(secondaryDepAxis,
                            true));
                } else {
                    gridLines.addAll(drawVerticalGridLines(secondaryDepAxis,
                            true));
                }
            }

            // draw minor grid lines for secondary axis
            if (secondaryDepAxis.isShowMinorGridLines()) {
                if (secondaryDepAxis.getAxisOrientation() == VERTICAL) {
                    gridLines.addAll(drawHorizontalGridLines(secondaryDepAxis,
                            false));
                } else {
                    gridLines.addAll(drawVerticalGridLines(secondaryDepAxis,
                            false));
                }
            }
        }


        if (indepAxis != null) {
            // draw major grid lines for independent axis
            if (indepAxis.isShowMajorGridLines()) {
                if (indepAxis.getAxisOrientation() == VERTICAL) {
                    if (indepAxis instanceof AxisNumber) {
                        gridLines.addAll(drawHorizontalGridLines(
                                (AxisNumber) indepAxis, true));
                    } else {
                        gridLines.addAll(drawHorizontalGridLines(
                                (AxisCategory) indepAxis, true));
                    }
                } else {
                    if (indepAxis instanceof AxisNumber) {
                        gridLines.addAll(drawVerticalGridLines(
                                (AxisNumber) indepAxis, true));
                    } else {
                        gridLines.addAll(drawVerticalGridLines(
                                (AxisCategory) indepAxis, true));
                    }
                }
            }

            // draw minor grid lines for independent axis
            if (indepAxis.isShowMinorGridLines()) {
                if (indepAxis.getAxisOrientation() == VERTICAL) {
                    if (indepAxis instanceof AxisNumber) {
                        gridLines.addAll(drawHorizontalGridLines(
                                (AxisNumber) indepAxis, false));
                    } else {
                        gridLines.addAll(drawHorizontalGridLines(
                                (AxisCategory) indepAxis, false));
                    }
                } else {
                    if (indepAxis instanceof AxisNumber) {
                        gridLines.addAll(drawVerticalGridLines(
                                (AxisNumber) indepAxis, false));
                    } else {
                        gridLines.addAll(drawVerticalGridLines(
                                (AxisCategory) indepAxis, false));
                    }
                }
            }
        }

        if (primaryDepAxis != null) {
            // draw zero line for primary axis
            if (primaryDepAxis.isLinear && (primaryDepAxis.getMin() <= 0) &&
                    (primaryDepAxis.getMax() >= 0)) {
                if (primaryDepAxis.getAxisOrientation() == VERTICAL) {
                    gridLines.add(drawHorizontalZeroLine(primaryDepAxis));
                } else {
                	//for horizontal bidi graphs we need to draw the zero line differently due to drawing order
                	//drawing 3-D graphs are very much dependent on the drawing order of svg elements.
                	if (!((!isLTR) && (input.getType().equals(IGraphicTypeConstants.HSTACKBAR_CHART3D))))
	                    gridLines.add(drawVerticalZeroLine(primaryDepAxis));
                }
            }
        }
        if (secondaryDepAxis != null) {
            // draw zero lines for secondary axis
            if (secondaryDepAxis.isLinear && (secondaryDepAxis.getMin() <= 0) &&
                    (secondaryDepAxis.getMax() >= 0)) {
                if (secondaryDepAxis.getAxisOrientation() == VERTICAL) {
                    gridLines.add(drawHorizontalZeroLine(secondaryDepAxis));
                } else {
                    gridLines.add(drawVerticalZeroLine(secondaryDepAxis));
                }
            }
        }
        if (indepAxis != null) {
            // draw zero line for independent axis
            if (indepAxis instanceof AxisNumber) {
                AxisNumber numAxis = (AxisNumber) indepAxis;

                if (numAxis.isLinear && (numAxis.getMin() <= 0) &&
                        (numAxis.getMax() >= 0)) {
                    if (indepAxis.getAxisOrientation() == VERTICAL) {
                        gridLines.add(drawHorizontalZeroLine(numAxis));
                    } else {
                        gridLines.add(drawVerticalZeroLine(numAxis));
                    }
                }
            }
        }

        // number of children = Number of grid lines + border + background
        SVGBase[] children = new SVGBase[gridLines.size() + 4];
        if (isLTR){
	        children[0] = drawXAxisLimitLines();
    	    children[1] = drawYAxisLimitLines();        
        }
        else{
	        children[1] = drawXAxisLimitLines();
    	    children[0] = drawYAxisLimitLines();        
        }

        if ((bkgroundColor != null) && (bkgroundColor.equals("") == false)) {
            SVGRectangle bkground = new SVGRectangle();
            bkground.setXCoordinate("0");
            bkground.setYCoordinate("0");
            bkground.setWidth(Double.toString(gridWidth));
            bkground.setHeight(getHeight());
            bkground.setFill(bkgroundColor);
            children[2] = bkground;
        }

        if (drawBorder) {
            SVGPolygon border = new SVGPolygon();
            border.setPoints(x3Doffset +
                ",0 " + gridWidth + ",0 " + gridWidth + "," +
                (height - Math.abs(y3Doffset)) + " " + (x3Doffset) +
                "," +  (height - Math.abs(y3Doffset)));
            border.setFill("none");
            border.setStroke("#cccccc");
            children[3] = border;
        }

        for (int i = 0; i < gridLines.size(); i++) {
            children[i + 4] = (SVGBase) gridLines.elementAt(i);
        }
        

        setChildren(children);
    }

    /**
     * Draw horizonatal grid lines
     * @param axis
     * @param major true if drawing major grid lines, false if minor grid lines
     * @return a vector of grid lines
     */
    protected Vector drawHorizontalGridLines(AxisNumber axis, boolean major) {
        double[] values;

        if (major) {
            values = axis.getMajorUnitValues();
        } else {
            values = axis.getMinorUnitValues();
        }

        double max_y = Math.abs(y3Doffset);
        double min_y = Double.parseDouble(getHeight());
        double max = axis.getMax();
        double min = axis.getMin();

        Vector lines = new Vector(values.length);

        for (int i = 0; i < values.length; i++) {
            double line_y = getCoordinate(values[i], max_y, min_y, max, min);
            SVGUse useLine = new SVGUse();

            if (major) {
                useLine.setHref("#horizontalSolidLine");
            } else {
                useLine.setHref("#horizontalDashLine");
            }

            useLine.setXCoordinate("0");
            useLine.setYCoordinate(Double.toString(line_y));
            lines.add(useLine);
        }

        //draw 3d line
        SVGUse verticalLine = new SVGUse();
        verticalLine.setHref("#verticalSolidLine");
        verticalLine.setXCoordinate("0");
        verticalLine.setYCoordinate("0");
        lines.add(verticalLine);

        return lines;
    }

    /**
     * Draw horizontal "zero line".  Draws a solid line perpendicular to the
     * axis at the value 0.
     *
     * @param axis
     * @return
     */
    protected SVGBase drawHorizontalZeroLine(AxisNumber axis) {
        double max_y = Math.abs(y3Doffset);
        double min_y = Double.parseDouble(getHeight());
        double max = axis.getMax();
        double min = axis.getMin();

        double line_y = getCoordinate(0.0, max_y, min_y, max, min);
        SVGUse useLine = new SVGUse();
        useLine.setHref("#horizontalZeroLine");
        useLine.setXCoordinate("0");
        useLine.setYCoordinate(Double.toString(line_y));

        return useLine;
    }

    /**
     * Draw vertical grid lines
     * @param axis
     * @param major true if drawing major grid lines, false if minor grid lines
     * @return a vector of grid lines
     */
    protected Vector drawVerticalGridLines(AxisNumber axis, boolean major) {
        double[] values;

        if (major) {
            values = axis.getMajorUnitValues();
        } else {
            values = axis.getMinorUnitValues();
        }

        double max_x = Double.parseDouble(getWidth()) - Math.abs(x3Doffset);
        double min_x = 0;
        double max = axis.getMax();
        double min = axis.getMin();

        Vector lines = new Vector(values.length);

        for (int i = 0; i < values.length; i++) {
            double line_x;

            if (isLTR) {
                line_x = getCoordinate(values[i], max_x, min_x, max, min);
            } else {
                line_x = getCoordinate(values[i], min_x, max_x, max, min);
            }

            SVGUse useLine = new SVGUse();

            if (major) {
                useLine.setHref("#verticalSolidLine");
            } else {
                useLine.setHref("#verticalDashLine");
            }

            useLine.setXCoordinate(Double.toString(line_x));
            useLine.setYCoordinate("0");
            lines.add(useLine);
        }

        //draw 3d line
        SVGUse horizontalLine = new SVGUse();
        horizontalLine.setHref("#horizontalSolidLine");
        horizontalLine.setXCoordinate("0");
        horizontalLine.setYCoordinate("" + getHeight());
        lines.add(horizontalLine);

        return lines;
    }


    /**
     * Draw vertical "grid line".  Draws a grid lines perpendicular to the
     * axis.  Depending whether the grid lines are major or minor the lines 
     * are drawn with a different style.
     *
     * @param axis
     * @return
     */
    protected Vector drawVerticalGridLines(AxisCategory axis, boolean major) {
        double[] values;

        if (major) {
            values = axis.getGroupBoundaryPositions();
        } else {
            values = axis.getGroupDataPositions();
        }

        Vector lines = new Vector(values.length);

        for (int i = 0; i < values.length; i++) {
            SVGUse useLine = new SVGUse();

            if (major) {
                useLine.setHref("#verticalSolidLine");
            } else {
                useLine.setHref("#verticalDashLine");
            }

            useLine.setXCoordinate(Double.toString(values[i]));
            useLine.setYCoordinate("0");
            lines.add(useLine);
        }

        return lines;
    }

    /**
     * Draw horizontal "grid line".  Draws a grid lines perpendicular to the
     * axis.  Depending whether the grid lines are major or minor the lines 
     * are drawn with a different style.
     *
     * @param axis
     * @return
     */
    protected Vector drawHorizontalGridLines(AxisCategory axis, boolean major) {
        double[] values;

        if (major) {
            values = axis.getGroupBoundaryPositions();
        } else {
            values = axis.getGroupDataPositions();
        }

        Vector lines = new Vector(values.length);

        for (int i = 0; i < values.length; i++) {
            SVGUse useLine = new SVGUse();

            if (major) {
                useLine.setHref("#horizontalSolidLine");
            } else {
                useLine.setHref("#horizontalDashLine");
            }

            useLine.setXCoordinate("0");
            useLine.setYCoordinate(Double.toString(values[i] - y3Doffset));
            lines.add(useLine);
        }

        return lines;
    }

    /**
     * Draw vertical "zero line".  Draws a solid line perpendicular to the
     * axis at the value 0.
     *
     * @param axis
     * @return
     */
    protected SVGBase drawVerticalZeroLine(AxisNumber axis) {
        double max_x = Double.parseDouble(getWidth()) - Math.abs(x3Doffset);
        double min_x = 0;
        double max = axis.getMax();
        double min = axis.getMin();

        double line_x;

        if (isLTR) {
            line_x = getCoordinate(0.0, max_x, min_x, max, min);
        } else {
            line_x = getCoordinate(0.0, min_x, max_x, max, min);
        }

        SVGUse useLine = new SVGUse();
        useLine.setHref("#verticalZeroLine");
        useLine.setXCoordinate(Double.toString(line_x));
        useLine.setYCoordinate("0");

        return useLine;
    }

    /**
     * Draw axis lines.  Draws two solid lines that are parallel to the x and y axis
     *
     * @param axis
     * @return
     */
    protected SVGBase drawXAxisLimitLines() {
        double yAxisX = 0;
        if (!isLTR) yAxisX = Double.parseDouble(getWidth())-x3Doffset;       

        SVGUse useLine = new SVGUse();
        useLine.setHref("#verticalLimitLine");
        useLine.setXCoordinate(Double.toString(yAxisX));
        useLine.setYCoordinate("0");

        return useLine;
    }
    
    /**
     * Draw axis lines.  Draws two solid lines that are parallel to the x and y axis
     *
     * @param axis
     * @return
     */
    protected SVGBase drawYAxisLimitLines() {
        double xAxisY = Double.parseDouble(getHeight());
        

        SVGUse useLine = new SVGUse();
        useLine.setHref("#horizontalLimitLine");
        useLine.setXCoordinate("0");
        useLine.setYCoordinate(Double.toString(xAxisY));

        return useLine;
    }
}
