/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Legend.java,v 1.2 2005/05/30 15:53:43 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Legend.java,v 1.2 2005/05/30 15:53:43 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

 
import java.util.Iterator;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGGroup;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGLine;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGText;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGUse;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSet;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSets;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.FontHelper;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;



/**
 * <code>Legend</code> 
 * $Revision: 1.2 $  
 */
public abstract class Legend extends Part {

	/**
	 * legend identifier
	 */
	protected String legendId = null;
	
	/**
	 * legend title
	 */
	protected String legendTitle = null;
	
	/**
	 * an array of legend labels
	 */
	protected String[] legendLabels = null;
	
	/**
	 * an array of Datasets
	 */
	protected DataSet[] datasetArray = null;
	
	/**
	 * position of the legend
	 */
	protected String location = POSITION_EAST;  // default location
	
	/**
	 * A flag to show if shapes should be used
	 */ 
	protected boolean useShapes = true;

	/**
	 * Shapes to be used in the legend
	 */
	protected SVGShapes shapes;
	
	/**
	 * Palettes for legend shape color
	 */
	protected SVGColorPalettes palettes;
	
	/**
	 * Number of data sets
	 */
	protected int numOfDataSets = 0;
	
	/**
	 * background color of legend
	 */
	protected String backgroundColor;

	/**
	 * text direction
	 */	
	protected boolean isLTR = true;
	
	/**
	 * An object for retriving translated text
	 */
	protected NLString nls;
	
	/**
	 * For calculating string lengths
	 */
	protected FontHelper fontHelper;
	
	/**
	 * Width of the chart area
	 */
	protected double chartWidth;
	
	/**
	 * Margin between the plotarea and legend
	 */
	protected double margin;
	
	/**
	 * Height of the chart area
	 */
	protected double chartHeight;
	
	protected double customHeight;
	protected boolean isSetCustomHeight;
	protected double customWidth;
	protected boolean isSetCustomWidth;
	
	/**
	 * @param input
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 */
	public Legend(Chart input, double x, double y, double width, double height, double chartWidth, double chartHeight, int margin, SVGColorPalettes palettes, SVGShapes shapes, NLString nls) {
		super(input, x, y, width, height);

		this.shapes = shapes;
		this.palettes = palettes;
		this.chartWidth = chartWidth;
		this.chartHeight = chartHeight;
		this.margin = margin;
		this.nls = nls;

		// bar charts and pie charts don't use shapes
		if (input.getType().endsWith("Bar") || input.getType().endsWith("Bar3D")|| input.getType().equals("pie")|| input.getType().equals("pie3D")) {
			useShapes = false;
		}
		
		// get data from input		
		getDataFromInput();
		
		// init fontHelper
		fontHelper = new FontHelper();
	}
	
	/**
	 * get tile and data set labels from input 
	 */
	private void getDataFromInput() {
		// get data from input
		Configuration config = input.getConfiguration();
		
		// get legend title, location and background color
		if (config != null) {
			org.eclipse.tptp.platform.report.chart.svg.internal.input.Legend legendInput = config.getLegend();
			if (legendInput != null) {
				legendTitle = legendInput.getTitle();
				if (legendInput.isSetLocation()) {
					location = legendInput.getLocation();
				}
				backgroundColor = legendInput.getBackgroundColor();
				legendId = legendInput.getId();
				customHeight = legendInput.getHeight();
				isSetCustomHeight = legendInput.isSetHeight();
				customWidth = legendInput.getWidth();
				isSetCustomWidth = legendInput.isSetWidth();
			}
		}
		
		// get data set labels
		Data data = input.getData();
		if (data != null) {
			DataSets dataSets = data.getDataSets();
			if (dataSets != null) {
				List dataSetList= dataSets.getDataSet();
				numOfDataSets = dataSetList.size();
				legendLabels = new String[numOfDataSets];
				datasetArray = new DataSet[numOfDataSets];
				int count = 0;
				for (Iterator i=dataSetList.iterator(); i.hasNext(); count++) {
					datasetArray[count] = (DataSet) i.next();
					legendLabels[count] = datasetArray[count].getLabel(); 
				}
			}
		}
	}

	/**
	 * draws the legend title
	 * @param x the x coordinate of the title
	 * @param y the y coordinate of the title
	 * @return title in SVGText
	 */
	protected SVGText drawTitle(double x, double y) {
		SVGText title = new SVGText();
		
		title.setText(nls.getString(legendTitle));

		if (isLTR) {
			title.setXCoordinate(Double.toString(x));
			title.setYCoordinate(Double.toString(y));
			title.setStyleClass("legendTitle");
			title.setIdentifier("SVGGEN_legendTitle");
		} else {
			// for RTL, title is anchored on the right
			// set x, y coordinates
			title.setXCoordinate(Double.toString(x));
			title.setYCoordinate(Double.toString(y));
			title.setStyleClass("legendTitle anchorAtEnd");
			title.setIdentifier("SVGGEN_legendTitle");
		}
		
		return title;
	}
	
	/**
	 * Draws a legend item in the legend
	 * @param x the x coordinate of the item
	 * @param y the y coordinate of the item
	 * @param itemNum the item number, use for determining the color or shape
	 * @return a legend item
	 */
	protected SVGBase drawLegendItem(double x, double y, int itemNum) {

		String dataSetColor = palettes.getDataSetColor(datasetArray[itemNum].getId());

		SVGGroup group = new SVGGroup();
		SVGBase[] children = new SVGBase[2];
		group.setChildren(children);

		SVGGroup shapeGroup = new SVGGroup();
		SVGBase[] shapeGroupChildren = new SVGBase[1];
		shapeGroup.setChildren(shapeGroupChildren);
		children[0] = shapeGroup;
		shapeGroup.setIdentifier("SVGGEN_legendShape" + itemNum);
		shapeGroup.setStyleClass("shape" + itemNum);
		int index = itemNum +1;
		if (input.getType().equals(AREA_CHART) || input.getType().equals(STACK_AREA_CHART)) {
			shapeGroup.addEvent("onmouseover", "SVGGEN_ChangeOpacity('SVGGEN_poly" + itemNum + "','0.6')");
			shapeGroup.addEvent("onmouseout", "SVGGEN_ChangeOpacity('SVGGEN_poly" + itemNum + "','0.1')");
		}
		else if (input.getType().equals(SCATTER_CHART)) {
			shapeGroup.addEvent("onmouseover", "SVGGEN_toggleVisibilityShadow(evt,'SVGGEN_shadowGroup" + datasetArray[itemNum].getId() + "','visible')");
			shapeGroup.addEvent("onmouseout", "SVGGEN_toggleVisibilityShadow(evt,'SVGGEN_shadowGroup" + datasetArray[itemNum].getId() + "','hidden')");
		}
		
		if (!input.getType().equals(HSTACKBAR_CHART3D) && !input.getType().equals(VSTACKBAR_CHART3D) && !input.getType().equals(VSTACKBAR_CHART) && !input.getType().equals(HSTACKBAR_CHART) && !input.getType().equals(PIE_CHART) && !input.getType().equals(PIE_CHART3D)) {
			// Make line disappear when user clicks on the legend shape
			if (input.getType().equals(AREA_CHART)) {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "');" +
					"SVGGEN_toggleVisibilityDataset(evt,'SVGGEN_area" + itemNum + "');");
			}else if (input.getType().equals(SCATTER_CHART)) {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "','SVGGEN_shadowGroup" + datasetArray[itemNum].getId() + "')");
			}else if (input.getType().equals(STACK_AREA_CHART)) {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "');" +
					"SVGGEN_toggleVisibilityDataset(evt,'SVGGEN_area" + itemNum + "');SVGGEN_toggle("+index +");");
			}else {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "')");
			}
		}
			
		// Handle user events
		EventTools.generateUserEvents(shapeGroup, datasetArray[itemNum].getEventHandler());
			
		SVGGroup legendLabelGroup = new SVGGroup();
		SVGBase[] legendLabelGroupChildren = new SVGBase[2];
		legendLabelGroup.setChildren(legendLabelGroupChildren);
			
		// Handle user tooltips
		legendLabelGroupChildren[1] = EventTools.generateUserTooltip(legendLabelGroup, datasetArray[itemNum].getTooltip(), nls);

		// Handle accessibility
		EventTools.generateAccessibility(legendLabelGroup, datasetArray[itemNum].getAccessibility(), nls);
			
		children[1] = legendLabelGroup;

		if (isLTR) {
			// draw square
			SVGRectangle square = new SVGRectangle();
			square.setXCoordinate(Double.toString(x));
			square.setYCoordinate(Double.toString(y));
			square.setHeight(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			square.setWidth(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			if (dataSetColor != null) {
				square.setFill(dataSetColor);
			}

			shapeGroupChildren[0] = square;
			
			// draw text
			SVGText legendLabel = new SVGText();
			legendLabel.setText(nls.getString(legendLabels[itemNum]));
			legendLabel.setXCoordinate(Double.toString(x + LEGEND_ITEM_FONT_SIZE + LEGEND_SPACE_BETWEEN_SYMBOL_AND_TEXT));
			legendLabel.setYCoordinate(Double.toString(y + LEGEND_ITEM_FONT_SIZE));
			legendLabel.setStyleClass("legendData");
			legendLabel.setIdentifier("SVGGEN_legendData" + itemNum);
			
			legendLabelGroupChildren[0] = legendLabel;
		} else {
			// draw a legend item that has the square on the right of the text
			// text should be right aligned
			SVGRectangle square = new SVGRectangle();
			square.setXCoordinate(Double.toString(x - LEGEND_ITEM_FONT_SIZE));
			square.setYCoordinate(Double.toString(y));
			square.setHeight(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			square.setWidth(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			if (dataSetColor != null) {
				square.setFill(dataSetColor);
			}

			shapeGroupChildren[0] = square;
			
			// draw text
			SVGText legendLabel = new SVGText();
			legendLabel.setText(nls.getString(legendLabels[itemNum]));
			legendLabel.setXCoordinate(Double.toString(x - LEGEND_ITEM_FONT_SIZE - LEGEND_SPACE_BETWEEN_SYMBOL_AND_TEXT));
			legendLabel.setYCoordinate(Double.toString(y + LEGEND_ITEM_FONT_SIZE));
			legendLabel.setStyleClass("legendData anchorAtEnd");
			legendLabel.setIdentifier("SVGGEN_legendData" + itemNum);
			
			legendLabelGroupChildren[0] = legendLabel;
		}
		
		return group;
	}

	protected SVGBase drawLegendItem(String shapeId, double shapeWidth, double shapeHeight, double x, double y, int itemNum) {
		
		String dataSetColor = palettes.getDataSetColor(datasetArray[itemNum].getId());

		SVGGroup group = new SVGGroup();
		SVGBase[] children = new SVGBase[2];
		group.setChildren(children);
		
		SVGGroup shapeGroup = new SVGGroup();
		children[0] = shapeGroup;
		SVGBase[] shapeGroupChildren = new SVGBase[2];
		shapeGroup.setChildren(shapeGroupChildren);
		shapeGroup.setStyleClass("fillcolor" + itemNum + " strokecolor" + itemNum);
		
		int index = itemNum +1;
		if (input.getType().equals(AREA_CHART) || input.getType().equals(STACK_AREA_CHART)) {
			// Change opacity when mouse moves over the legend shape
			shapeGroup.addEvent("onmouseover", "SVGGEN_ChangeOpacity('SVGGEN_poly" + itemNum + "','0.6')");
			shapeGroup.addEvent("onmouseout", "SVGGEN_ChangeOpacity('SVGGEN_poly" + itemNum + "','0.1')");
		} 
		else if (input.getType().equals(SCATTER_CHART)) {
			// Show and hide shadows when mouse moves over the legend shape
			shapeGroup.addEvent("onmouseover", "SVGGEN_toggleVisibilityShadow(evt,'SVGGEN_shadowGroup" + datasetArray[itemNum].getId() + "','visible')");
			shapeGroup.addEvent("onmouseout", "SVGGEN_toggleVisibilityShadow(evt,'SVGGEN_shadowGroup" + datasetArray[itemNum].getId() + "','hidden')");
		}
			
		if (!input.getType().equals(VSTACKBAR_CHART3D) && !input.getType().equals(HSTACKBAR_CHART3D) && !input.getType().equals(VSTACKBAR_CHART) && !input.getType().equals(HSTACKBAR_CHART) && !input.getType().equals(PIE_CHART) && !input.getType().equals(PIE_CHART3D)) {
			// Make line disappear when user clicks on the legend shape
			if (input.getType().equals(AREA_CHART)) {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "');" +
					"SVGGEN_toggleVisibilityDataset(evt,'SVGGEN_area" + itemNum + "');");
			}else if (input.getType().equals(SCATTER_CHART)) {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "','SVGGEN_shadowGroup" + datasetArray[itemNum].getId() + "')");
			}else if (input.getType().equals(STACK_AREA_CHART)) {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "');" +
					"SVGGEN_toggleVisibilityDataset(evt,'SVGGEN_area" + itemNum + "');SVGGEN_toggle("+index+");");				
			}else {
				shapeGroup.addEvent("onclick", "SVGGEN_toggleVisibilityDataset(evt,'" + datasetArray[itemNum].getId() + "')");
			}
		}

		// Handle user events
		EventTools.generateUserEvents(shapeGroup, datasetArray[itemNum].getEventHandler());

		// legend label group			
		SVGGroup legendLabelGroup = new SVGGroup();
		children[1] = legendLabelGroup;

		SVGBase[] legendLabelGroupChildren = new SVGBase[2];
		legendLabelGroup.setChildren(legendLabelGroupChildren);

		// Handle user tooltips
		legendLabelGroupChildren[1] = EventTools.generateUserTooltip(legendLabelGroup, datasetArray[itemNum].getTooltip(), nls);

		// Handle accessibility
		EventTools.generateAccessibility(legendLabelGroup, datasetArray[itemNum].getAccessibility(), nls);

		if (isLTR) {
			SVGGroup shapeWithLineGroup = new SVGGroup();
			SVGBase[] shapeAndLine = new SVGBase[2];
			shapeWithLineGroup.setChildren(shapeAndLine);
			shapeWithLineGroup.setIdentifier("SVGGEN_legendShape" + itemNum);
			shapeGroupChildren[0] = shapeWithLineGroup;
			if (dataSetColor != null) {
				shapeWithLineGroup.setFill(dataSetColor);
				shapeWithLineGroup.setStroke(dataSetColor);
			}

			// draw line through the shape
			if (input.getType().equals(SCATTER_CHART) == false) {
				SVGLine line = new SVGLine();
				line.setX1(Double.toString(x));
				line.setY1(Double.toString(y+shapeHeight/2));
				line.setX2(Double.toString(x + LEGEND_SYMBOL_LINE_LENGTH));
				line.setY2(Double.toString(y+shapeHeight/2));
				//line.setStyleClass("linestyle strokecolor" + itemNum);
				shapeAndLine[0] = line;
			}
			
			// draw shape 
			if ((shapeId != null) && (!shapeId.trim().equals(""))){
				SVGUse useShape = new SVGUse();
				//useShape.setIdentifier(dp.getId());
				useShape.setHref("#" + shapeId);
				double shapeX = x + (LEGEND_SYMBOL_LINE_LENGTH - shapeWidth) / 2;
				useShape.setTransformation("translate(" + (shapeX) + "," + y + ")");
				shapeAndLine[1] = useShape;
			}
			
			// draw square
			// The visibility of the square will be set to hidden when the chart is loaded.
			SVGGroup squareGroup = new SVGGroup();
			SVGBase[] squareMember = new SVGBase[1];
			squareGroup.setChildren(squareMember);
			shapeGroupChildren[1] = squareGroup;
			if (dataSetColor != null) {
				squareGroup.setFill(dataSetColor);
				squareGroup.setStroke(dataSetColor);
			}

			SVGRectangle square = new SVGRectangle();
			square.setXCoordinate(Double.toString(x+5));
			square.setYCoordinate(Double.toString(y));
			square.setHeight(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			square.setWidth(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			square.setIdentifier("SVGGEN_legendsquareshapeWithoutLine" + itemNum);
			square.setVisibility("hidden");

			squareMember[0] = square;
			
			// draw text
			SVGText legendLabel = new SVGText();
			legendLabel.setText(nls.getString(legendLabels[itemNum]));
			legendLabel.setXCoordinate(Double.toString(x + LEGEND_SYMBOL_LINE_LENGTH + LEGEND_SPACE_BETWEEN_SYMBOL_AND_TEXT));
			legendLabel.setYCoordinate(Double.toString(y + LEGEND_ITEM_FONT_SIZE));
			legendLabel.setStyleClass("legendData");
			legendLabel.setIdentifier("SVGGEN_legendData" + itemNum);

			legendLabelGroupChildren[0] = legendLabel;
			
		} else {
			// RTL setting
			// draw a legend item that has the shape on the right of the text
			// text should be right aligned

			SVGGroup shapeWithLineGroup = new SVGGroup();
			SVGBase[] shapeAndLine = new SVGBase[2];
			shapeWithLineGroup.setChildren(shapeAndLine);
			shapeWithLineGroup.setIdentifier("SVGGEN_legendShape" + itemNum);
			//shapeWithLineGroup.setStyleClass("fillcolor" + itemNum);
			shapeGroupChildren[0] = shapeWithLineGroup;

			// draw line through the shape
			if (input.getType().equals(SCATTER_CHART) == false) {
				SVGLine line = new SVGLine();
				line.setX1(Double.toString(x - LEGEND_SYMBOL_LINE_LENGTH));
				line.setY1(Double.toString(y+shapeHeight/2));
				line.setX2(Double.toString(x));
				line.setY2(Double.toString(y+shapeHeight/2));
				//line.setStyleClass("linestyle strokecolor" + itemNum);
				shapeAndLine[0] = line;
			}
			
			// draw shape 
			if ((shapeId != null) && (!shapeId.trim().equals(""))){
				SVGUse useShape = new SVGUse();
				//useShape.setIdentifier(dp.getId());
				useShape.setHref("#" + shapeId);
				double shapeX = x - LEGEND_SYMBOL_LINE_LENGTH + (LEGEND_SYMBOL_LINE_LENGTH - shapeWidth) / 2;
				useShape.setTransformation("translate(" + shapeX + "," + y + ")");
				shapeAndLine[1] = useShape;
			}

			// draw square
			// The visibility of the square will be set to hidden when the chart is loaded.
			SVGGroup squareGroup = new SVGGroup();
			SVGBase[] squareMember = new SVGBase[1];
			squareGroup.setChildren(squareMember);
			shapeGroupChildren[1] = squareGroup;

			SVGRectangle square = new SVGRectangle();
			square.setXCoordinate(Double.toString(x - LEGEND_ITEM_FONT_SIZE - 5));
			square.setYCoordinate(Double.toString(y));
			square.setHeight(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			square.setWidth(Integer.toString(LEGEND_ITEM_FONT_SIZE));
			//square.setStyleClass("fillcolor" + itemNum);
			square.setIdentifier("SVGGEN_legendsquareshapeWithoutLine" + itemNum);
			square.setVisibility("hidden");

			squareMember[0] = square;
			
			// draw text
			SVGText legendLabel = new SVGText();
			legendLabel.setText(nls.getString(legendLabels[itemNum]));
			legendLabel.setXCoordinate(Double.toString(x - LEGEND_SYMBOL_LINE_LENGTH - LEGEND_SPACE_BETWEEN_SYMBOL_AND_TEXT));
			legendLabel.setYCoordinate(Double.toString(y + LEGEND_ITEM_FONT_SIZE));
			legendLabel.setStyleClass("legendData anchorAtEnd");
			legendLabel.setIdentifier("SVGGEN_legendData" + itemNum);

			legendLabelGroupChildren[0] = legendLabel;
		}
		
		return group;
	}

	/**
	 * Return a subclass given the legend position and text direction.
	 * @param legendPosition
	 * @param isLTR
	 * @param input
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param shapes
	 * @return
	 */
	public static Legend getInstance(
		String legendPosition, 
		boolean isLTR, 
		Chart input, 
		double x, 
		double y, 
		double width, 
		double height, 
		double chartWidth,
		double chartHeight,
		int margin,
		SVGColorPalettes palettes, 
		SVGShapes shapes, 
		NLString nls) 
	{
		if (isLTR) {
			if (legendPosition.equals(POSITION_EAST) || legendPosition.equals(POSITION_WEST) ||
				legendPosition.equals(POSITION_LEADING) || legendPosition.equals(POSITION_TRAILING)) {
				return new LegendVerticalLTR(input, x, y, width, height, chartWidth, chartHeight, margin, palettes, shapes, legendPosition, nls);
			} else {
				return new LegendHorizontalLTR(input, x, y, width, height, chartWidth, chartHeight, margin, palettes, shapes, nls);
			}
		} else {
			if (legendPosition.equals(POSITION_EAST) || legendPosition.equals(POSITION_WEST) ||
				legendPosition.equals(POSITION_LEADING) || legendPosition.equals(POSITION_TRAILING)) {
				return new LegendVerticalRTL(input, x, y, width, height, chartWidth, chartHeight, margin, palettes, shapes, legendPosition, nls);
			} else {
				return new LegendHorizontalRTL(input, x, y, width, height, chartWidth, chartHeight, margin, palettes, shapes, nls);
			}
		}
	}
	
	/**
	 * Sets the NLString object to retrive translated text
	 * @param nls
	 */
	public void setNLString(NLString nls) {
		this.nls = nls;
	}
	
	public double getLegendHeight() {
		return height;
	}
	
	public double getLegendWidth() {
		return width;
	}
	
}
