/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LegendVerticalLTR.java,v 1.3 2005/06/10 17:47:09 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGGroup;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGRectangle;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.SVGShapeIterator;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.Utilities;


/**
 * Vertical legend
 */
public class LegendVerticalLTR extends Legend {
	static final long serialVersionUID = -7153936797793941014L;
	/**
	 * legend position
	 */
	private String legendPosition;
	
	protected double scalingFactor = 1;
	
	class Coordinates {
		public double x;
		public double y;
		public Coordinates(double x, double y) {
			this.x = x;
			this.y = y;
		}
	}
	
	protected Coordinates[] legendItemCoordinates;
	
	/**
	 * @param input
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param shapes
	 */
	public LegendVerticalLTR(
		Chart input,
		double x,
		double y,
		double width,
		double height,
		double chartWidth,
		double chartHeight,
		int margin,
		SVGColorPalettes palettes,
		SVGShapes shapes,
		String legendPosition, 
		NLString nls) {
		super(input, x, y, width, height, chartWidth, chartHeight, margin, palettes, shapes, nls);
		this.legendPosition = legendPosition; 
		if (legendLabels != null) {
			legendItemCoordinates = new Coordinates[legendLabels.length];
		} else {
			legendItemCoordinates = new Coordinates[0];
		}
		analyzeData();
	}
	
	protected void analyzeData() {
		double maxX = 0;
		double itemLineHeight;
		double leftMargin = getLeftMargin();
		double currentY = LEGEND_VERTICAL_TOP_MARGIN;
		double verticalScalingFactor = 1;
		double horizontalScalingFactor = 1;

		
		if (legendTitle != null) {
			fontHelper.setFontSize(LEGEND_TITLE_FONT_SIZE);
			double titleWidth = fontHelper.getStringWidth(nls.getString(legendTitle));
			if (titleWidth > maxX) {
				maxX = titleWidth;
			}
			currentY += fontHelper.getHeight() + LEGEND_SPACE_AFTER_TITLE;
		}
		
		fontHelper.setFontSize(LEGEND_ITEM_FONT_SIZE);
		itemLineHeight = LEGEND_ITEM_FONT_SIZE + LEGEND_SPACE_BETWEEN_ITEMS;
		
		if (useShapes && legendLabels != null) {
			Utilities.assertion(shapes != null);

			for (int i=0; i<legendLabels.length; i++) {
				double itemWidth = LEGEND_SYMBOL_LINE_LENGTH + 
								   LEGEND_SPACE_BETWEEN_SYMBOL_AND_TEXT + 
								   fontHelper.getStringWidth(nls.getString(legendLabels[i])) + 
								   LEGEND_SPACE_BETWEEN_ITEMS;

				if (itemWidth > maxX) {
					maxX = itemWidth;
				}
				
				legendItemCoordinates[i] = new Coordinates(leftMargin, currentY);
				
				currentY += itemLineHeight;
			}
		} else if (!useShapes && legendLabels != null) {
			for (int i=0; i<legendLabels.length; i++) {
				double itemWidth = LEGEND_ITEM_FONT_SIZE +    // width of the square
								   LEGEND_SPACE_BETWEEN_SYMBOL_AND_TEXT + 
								   fontHelper.getStringWidth(nls.getString(legendLabels[i])) + 
								   LEGEND_SPACE_BETWEEN_ITEMS;

				if (itemWidth > maxX) {
					maxX = itemWidth;
				}

				legendItemCoordinates[i] = new Coordinates(leftMargin, currentY);
				
				currentY += itemLineHeight;
			}
		}
		
		maxX += getLeftMargin() + getRightMargin();
		
		double naturalLegendHeight = currentY + LEGEND_WIDE_SIDE_MARGIN;
		double naturalLegendWidth = maxX;
		if (isSetCustomWidth == false) {
			if (maxX < chartWidth * LEGEND_MAX_WIDTH_FRACTION) {
				width = maxX;
			} else {
				width = chartWidth * LEGEND_MAX_WIDTH_FRACTION;
				// shrink legend to fit
				horizontalScalingFactor = width / naturalLegendWidth;
			}
		} else {
			width = customWidth;
			horizontalScalingFactor = width / naturalLegendWidth;
		}
		if (naturalLegendHeight > chartHeight) {
			verticalScalingFactor = chartHeight / naturalLegendHeight;
		}
		scalingFactor = horizontalScalingFactor < verticalScalingFactor ? horizontalScalingFactor : verticalScalingFactor;
		if (scalingFactor < 1) {
			width = width * scalingFactor;
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
	 */
	protected void constructPart() {
		// parts: background + legend group
		SVGBase[] subElements = new SVGBase[2];
		setChildren(subElements);
		setIdentifier(legendId);
		
		// parts: n data sets, 1 title, 
		SVGGroup legendGroup = new SVGGroup();
		SVGBase[] legendMembers = new SVGBase[numOfDataSets + 1];
		legendGroup.setChildren(legendMembers);
		subElements[1] = legendGroup;

		// draw title
		if (legendTitle != null) {
			legendMembers[0] = drawTitle(getLeftMargin(), LEGEND_VERTICAL_TOP_MARGIN + LEGEND_TITLE_FONT_SIZE);
		}

		if (useShapes && legendLabels != null) {
			Utilities.assertion(shapes != null);
			SVGShapeIterator shapeIterator = shapes.getShapeIterator();
			for (int i=0; i<legendLabels.length; i++) {
				legendMembers[i+1] = drawLegendItem(shapeIterator.getNextShapeId(datasetArray[i].getId()), shapeIterator.getShapeWidth(), shapeIterator.getShapeHeight(), legendItemCoordinates[i].x, legendItemCoordinates[i].y, i);
			}
		} else if (!useShapes && legendLabels != null) {
			for (int i=0; i<legendLabels.length; i++) {
				legendMembers[i+1] = drawLegendItem(legendItemCoordinates[i].x, legendItemCoordinates[i].y, i);
			}
		}
		
		if (scalingFactor < 1) {
			legendGroup.setTransformation("scale(" + scalingFactor + ")");
		}
		
		setWidth(Double.toString(width));
		
		if (backgroundColor != null && backgroundColor.equals("") == false) {
			SVGRectangle bkground = new SVGRectangle();
			bkground.setXCoordinate("0");
			bkground.setYCoordinate("0");
			bkground.setWidth(getWidth());
			bkground.setHeight(getHeight());
			bkground.setFill(backgroundColor);
			subElements[0] = bkground;
		}
	}
	
	private double getLeftMargin() {
		double sideMargin;
		if (legendPosition.equals(POSITION_WEST) || legendPosition.equals(POSITION_LEADING)) {
			sideMargin = margin;
		} else {
			sideMargin = LEGEND_NARROW_SIDE_MARGIN;
		}
		return sideMargin;
	}

	private double getRightMargin() {
		double rightMargin;
		if (legendPosition.equals(POSITION_WEST) || legendPosition.equals(POSITION_LEADING)) {
			rightMargin = LEGEND_NARROW_SIDE_MARGIN;
		} else {
			rightMargin = margin;
		}
		return rightMargin;
	}
}
