/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGColorPalettes.java,v 1.3 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;


import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSet;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSetColor;
import org.eclipse.tptp.platform.report.chart.svg.internal.palettes.ColorPaletteSets;
import org.eclipse.tptp.platform.report.chart.svg.internal.palettes.PaletteSetType;
import org.eclipse.tptp.platform.report.chart.svg.internal.palettes.PaletteType;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.PalettesDOMDocument;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.Utilities;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.XMLLoader;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;

import com.ibm.icu.util.StringTokenizer;
import com.ibm.icu.util.ULocale;


/**
 * <code>SVGColorPalettes</code> is a container that contains a number of Color Palettes. 
 * 
 * $Revision: 1.3 $
 */
public class SVGColorPalettes {
	
	/**
	 * Flag to determine default or custom palettes to use.
	 */	
	private boolean useDefault = true;

	/**
	 * Custom palettesId.  
	 */		
	private String customPalettesId;

	/**
	 * Set of color palettes to add to the user preferences
	 */	
	private HashMap customGraphicColorPalettes = null;
	
	/**
	 * A HashMap of display names of the palette under use.  Key is palette id.
	 */
	private HashMap customDisplayNameMap = null;
	
	/**
	 * Custom color palettes
	 */	
	private String [] customColorPalettes = null;

	/**
	 * Object ref to custom palette-set id
	 */	
	private  Object  customPaletteSetId = null;

	/**
	 * List of all the custom color palette sets
	 */	
	private  List customColorPaletteSet = null;	
	
	/**
	 * default org.eclipse.tptp.platform.report.chart.svg.internal.resources
	 */
	private DefaultResources defaultResources = null;
	
	/**
	 * palette display name
	 */
	private String customPaletteDisplayName = null;
	
	/**
	 * object for retriving translated text from resource bundles
	 */
	private NLString nls = null;
	
	/**
	 * locale for the chart
	 */
	private ULocale locale = null;
	
	/**
	 * A hash map for association data set ID with a color. 
	 * This color will override the palette color.
	 */
	private HashMap dataSetColors = null;
	
	/**
	 * Constructor
	 *
	 */
	public SVGColorPalettes(DefaultResources defaultResources, ULocale locale) {
		this.defaultResources = defaultResources;
		this.locale = locale;
		this.dataSetColors = new HashMap();

		if (defaultResources.isDefaultPalettesInitialized() == false) {
			initColorPalettes();
		}
		
		initNLString();
	}


	/**
	 * Returns the set of color palettes.
	 * 
	 * @return HashMap
	 */
	public HashMap getGraphicColorPalettes() {
		if (useDefault) {
			return defaultResources.getDefaultGraphicColorPalettes();
		}else{
			return customGraphicColorPalettes;
		}
	}
		
	/**
	 * Gets the palette. Use this palette for during generation.
	 * @return 
	 */
	public String[] getPalette() {
		if (useDefault) {
			return defaultResources.getDefaultColorPalettes();
		}else{
			return customColorPalettes;
		}
	}
	
	/**
	 * Gets a color from the palette given an index
	 * @param colorIndex the index of the color to return
	 * @return the selected color
	 */
	public String getPaletteColor(int colorIndex) {
		if (useDefault) {
			String[] colors = defaultResources.getDefaultColorPalettes();
			int numColors = colors.length; 
			return colors[colorIndex % numColors];
		}else{
			int numColors = customColorPalettes.length;
			return customColorPalettes[colorIndex % numColors];
		}
	}
	
	/**
	 * Gets the palette. Use this palette for during generation.
	 * @return 
	 */
	public String getPaletteId() {
		if (useDefault) { 
			return defaultResources.getDefaultPalettesId();
		}else{
			return customPalettesId;
		}
	}
	
	public String getPaletteDisplayName() {
		if (useDefault) {
			String displayNameKey = defaultResources.getDefaultColorPaletteDisplayName();
			if (displayNameKey != null && nls.getString(displayNameKey) != null) {
				return nls.getString(displayNameKey);
			}
			return getPaletteId();
		} else {
			if (customPaletteDisplayName != null && nls.getString(customPaletteDisplayName) != null) {
				return nls.getString(customPaletteDisplayName);
			}
			return getPaletteId();
		}
	}
	
	public String getPaletteDisplayName(String paletteId) {
		HashMap displayNameMap = getDisplayNameMap();
		if (useDefault) {
			String displayNameKey = (String) displayNameMap.get(paletteId);
			if (displayNameKey != null && nls.getString(displayNameKey) != null) {
				return nls.getString(displayNameKey);
			}
			return getPaletteId();
		} else {
			if (customPaletteDisplayName != null && nls.getString(customPaletteDisplayName) != null) {
				return nls.getString(customPaletteDisplayName);
			}
			return getPaletteId();
		}
	}
	
	/**
	 * Assign colors to datasets to override the palettes.
	 * 
	 * @param colorList a list of color for datasets. type of object is DataSetColor
	 */
	public void setDataSetColors(List colorList) {
		for (Iterator iter=colorList.iterator(); iter.hasNext(); ) {
			DataSetColor dataSetColor = (DataSetColor) iter.next();
			try {
				DataSet dataSet = (DataSet) dataSetColor.getDataSetId();
				String dataSetId = dataSet.getId();
				String color = dataSetColor.getColor();
				dataSetColors.put(dataSetId, color);
			} catch (ClassCastException e) {
				// do nothing
			}
		}
	}
	
	/**
	 * Gets the color given a data set ID.
	 * @param dataSetId
	 * @return
	 */
	public String getDataSetColor(String dataSetId) {
		String color = (String) dataSetColors.get(dataSetId);
		return color;
	}
	
	private HashMap getDisplayNameMap() {
		if (useDefault) {
			return defaultResources.getDefaultPaletteDisplayNames();
		} else {
			return customDisplayNameMap;
		}
	}
	
	/**
	 * Initialize color palettes
	 */
	private void initColorPalettes() {

		synchronized (defaultResources) {
		
			if (defaultResources.isDefaultPalettesInitialized()) {
				return;
			}
			
			PalettesDOMDocument palettesDOM = null;
			boolean initialized = false;
			if (defaultResources.getResourceDir() != null) {
				String paletteXMLFile = defaultResources.getResourceDir() + "/palettes.xml";
				palettesDOM = getPalettesFromInputFile(paletteXMLFile);
				if (palettesDOM != null) {
					initialized = parsePaletteDOM(palettesDOM);
				}
			}
			if (initialized == false) {
				// no resource directory is specified or 
				// problem occurred during parsing palette definition file
				palettesDOM = getPalettesFromDefaultLocation();
				if (palettesDOM != null) {
					initialized = parsePaletteDOM(palettesDOM);
				}
			}
			if (initialized) {
				defaultResources.setDefaultPalettesDOM(palettesDOM);
				defaultResources.setDefaultPalettesInitialized(true);
			} else {
				// The default location should always work.
				Utilities.assertion(defaultResources.isDefaultPalettesInitialized());
			}
		}
	}
	
	private void initNLString() {
		String resourceBundle = defaultResources.getPaletteResourceBundle();
		if (resourceBundle != null && locale != null) {
			nls = new NLString(resourceBundle, locale);
		} else {
			nls = new NLString();
		}
	}
	
	/**
	 * Load Default palettes from a file. These palettes are used during init.	 
	 * @param paletteXMLFile
	 * @return
	 */
	private PalettesDOMDocument getPalettesFromInputFile(String paletteXMLFile) {
		Document doc = null;

		InputSource inputSource = new InputSource(paletteXMLFile);
		XMLLoader xmlLoader = new XMLLoader();
		
		if (xmlLoader.load(inputSource) == false) {
			return null;
		}

		doc = xmlLoader.getDOM();
		PalettesDOMDocument palettesDom = new PalettesDOMDocument(doc);
		return palettesDom;
	}
	
	private PalettesDOMDocument getPalettesFromDefaultLocation() {
		Document doc = null;

		XMLLoader xmlLoader = new XMLLoader();

		InputStream defaultShapes = Utilities.getResourceAsStream("org/eclipse/tptp/platform/report/chart/svg/internal/resources/palettes.xml");
		InputSource inputSource = new InputSource(defaultShapes);
		
		if (xmlLoader.load(inputSource) == false) {
			return null;
		}

		doc = xmlLoader.getDOM();
		PalettesDOMDocument palettesDom = new PalettesDOMDocument(doc);
		return palettesDom;
	}
	
	private boolean parsePaletteDOM(PalettesDOMDocument palettesDom) {
		Object defaultPaletteSetId;
		List defaultColorPaletteSet;
		String[] defaultColorPalettes;
		String defaultPalettesId;
		String defaultDisplayName;

		//Update defaultColorPalettes with the default paletteset and default palette specified in the input. 
		if(palettesDom != null){
			getDefaultHashColorPaletteset(palettesDom);
			ColorPaletteSets defaultColorPaletteSets = palettesDom.getPalettes();
			if (defaultColorPaletteSets != null){
				// default color palette set
				defaultPaletteSetId = defaultColorPaletteSets.getDefaultSetId();
				defaultResources.setDefaultPaletteSetId(defaultPaletteSetId);
				
				// list of color palettes of the default palette set
				defaultColorPaletteSet = defaultColorPaletteSets.getColorPaletteSet();
				defaultResources.setDefaultColorPaletteSet(defaultColorPaletteSet);

				defaultColorPalettes = getDefaultPalette();
				defaultResources.setDefaultColorPalettes(defaultColorPalettes);

				// ID of the default palette
				PaletteSetType defaultPaletteSet = (PaletteSetType) defaultColorPaletteSets.getDefaultSetId();
				PaletteType dp = (PaletteType) defaultPaletteSet.getDefaultPaletteId();
				defaultPalettesId = dp.getId();
				defaultResources.setDefaultPalettesId(defaultPalettesId);
				
				// set default display name
				// if display name does not exist, use ID as display name
				defaultDisplayName = dp.getDisplayName();
				if (defaultDisplayName == null || defaultDisplayName.equals("")) {
					defaultDisplayName = defaultPalettesId;
				}
				defaultResources.setDefaultColorPaletteDisplayName(defaultDisplayName);
				defaultResources.setDefaultPaletteDisplayNames(getDisplayNameMap(defaultPaletteSet));
				
				String resourceBundle = defaultColorPaletteSets.getResourceBundle();
				defaultResources.setPaletteResourceBundle(resourceBundle);

				return true;
			}
		}
		return false;
	}

	/**
	 * Get the defaultColorPalettes string which is define in default palettes.xml
	 * @return String [] defaultColorPalettes
	 */
	private String[] getDefaultPalette() {
		PaletteSetType dps = (PaletteSetType) defaultResources.getDefaultPaletteSetId();
		PaletteType dp = (PaletteType) dps.getDefaultPaletteId();
		String paletteId = dp.getId();
		String colorList = dp.getColorList();
		return getColors(colorList);
	}

	/**
	 * Update the hash table for the default color paletteset
	 * @param palettesDom
	 */
	private void getDefaultHashColorPaletteset(PalettesDOMDocument palettesDom) {
		ColorPaletteSets colorPaletteSets = palettesDom.getPalettes();
		List paletteSetList = colorPaletteSets.getColorPaletteSet();
		if(paletteSetList != null){
			PaletteSetType paletteSet = (PaletteSetType) colorPaletteSets.getDefaultSetId();
			List paletteList = paletteSet.getPalette();
			HashMap defaultGraphicColorPalettes = new HashMap();
			for (int j = 0; j < paletteList.size(); ++j) {
				PaletteType dp = (PaletteType)paletteList.get(j);
				String colorList = dp.getColorList();
				String id  = dp.getId();
				defaultGraphicColorPalettes.put(id, getColors(colorList));
			}
			defaultResources.setDefaultGraphicColorPalettes(defaultGraphicColorPalettes);
		}
	}
	
	/**
	 * Update the hash table for the custom color paletteset
	 * @param palettesDom
	 * @param paletteSetId
	 */
	private PaletteSetType getCustomHashColorPaletteset(PalettesDOMDocument palettesDom, String paletteSetId) {
		ColorPaletteSets colorPaletteSets = palettesDom.getPalettes();
		List paletteSetList = colorPaletteSets.getColorPaletteSet();
		PaletteSetType paletteSet = null;
		if(paletteSetList != null){
			if(paletteSetId != null && paletteSetId != ""){
				for (int i = 0; i < paletteSetList.size(); ++i) {
						paletteSet = (PaletteSetType) paletteSetList.get(i);
						if(paletteSetId.equals(paletteSet.getId())){
							break;
						}else{
							paletteSet = (PaletteSetType) colorPaletteSets.getDefaultSetId();
						}
				}
			}else{
				paletteSet = (PaletteSetType) colorPaletteSets.getDefaultSetId();
			}
			customGraphicColorPalettes = new HashMap();
			if(paletteSet != null){
				List paletteList = paletteSet.getPalette();
				for (int j = 0; j < paletteList.size(); ++j) {
					PaletteType dp = (PaletteType)paletteList.get(j);
					String colorList = dp.getColorList();
					String id  = dp.getId();
					customGraphicColorPalettes.put(id, getColors(colorList));
				}
			}
		}
		return paletteSet;
	}

	/**
	 * Load customized palettes from a file. These palettes are only used for this chart.
	 * 
	 * @param palettesLocation
	 */
	public void setCustomPalettes(String palettesLocation, String paletteId, String paletteSetId) {
		Document doc = null;
		PalettesDOMDocument palettesDom;
		if(palettesLocation == null){
			if (defaultResources.isDefaultPalettesInitialized() == false) {
				initColorPalettes();
			}
			palettesDom = defaultResources.getDefaultPalettesDOM();
		} else {
			palettesDom = getPalettesFromInputFile(palettesLocation);
			if (palettesDom == null) {
				if (defaultResources.isDefaultPalettesInitialized() == false) {
					initColorPalettes();
				}
				palettesDom = defaultResources.getDefaultPalettesDOM();
			}
		}

		Utilities.assertion(palettesDom != null);

		PaletteSetType paletteSet = null;
		paletteSet = getCustomHashColorPaletteset(palettesDom, paletteSetId);
		customDisplayNameMap = getDisplayNameMap(paletteSet);
		Object[] keys = customGraphicColorPalettes.keySet().toArray();
		//Update customColorPalettes with paletteId  specified in the input else update with default set
		if(paletteSetId != null && paletteSetId != ""){
			if (keys != null && keys.length != 0) {
				// Check if this paletteId is in the paletteset. If not get the default Palette Id
				paletteId = getCorrectPaletteId(paletteSet, paletteId);
				customColorPalettes = (String[]) customGraphicColorPalettes.get(paletteId);
				if(customColorPalettes != null){
					customPalettesId = paletteId;
					customPaletteDisplayName = getPaletteDisplayName(paletteSet, paletteId);
					useDefault = false;
				}
			}
		}else{
			ColorPaletteSets customColorPaletteSets = palettesDom.getPalettes();
			if(customColorPaletteSets != null){
				paletteSet = (PaletteSetType) customColorPaletteSets.getDefaultSetId();
				if(paletteSet != null){
					paletteId = getCorrectPaletteId(paletteSet, paletteId);
					customColorPalettes = (String[]) customGraphicColorPalettes.get(paletteId);
					if(customColorPalettes != null){
						customPalettesId = paletteId;
						customPaletteDisplayName = getPaletteDisplayName(paletteSet, paletteId);
						useDefault = false;
					}
				}
			}
		}
	}
		
	/**
	 * This method return the valid Palette Id
	 * @param paletteSet Valid palette Set
	 * @param paletteId User set Palette Id
	 * @return String Valid palette Id
	 */
	private String getCorrectPaletteId(PaletteSetType paletteSet, String paletteId) {
		PaletteType ddp = (PaletteType)paletteSet.getDefaultPaletteId();
		String DefaultPaletteId = ddp.getId();
		if(paletteId != null){
			List plist = paletteSet.getPalette();
			for (int j = 0; j < plist.size(); ++j) {
				PaletteType dp = (PaletteType)plist.get(j);
				if(paletteId.equals(dp.getId())){
					return paletteId;
				}
			}
		}
		return DefaultPaletteId;
	}
	
	private String getPaletteDisplayName(PaletteSetType paletteSet, String paletteId) {
		if (paletteId != null) {
			List paletteList = paletteSet.getPalette();
			for (Iterator iter = paletteList.iterator(); iter.hasNext(); ) {
				PaletteType palette = (PaletteType) iter.next();
				if (palette.getId().equals(paletteId)) {
					return palette.getDisplayName();
				}
			}
		}
		return paletteId;
	}
	
	private HashMap getDisplayNameMap(PaletteSetType paletteSet) {
		HashMap displayNameMap = new HashMap();
		List paletteList = paletteSet.getPalette();
		for (Iterator iter = paletteList.iterator(); iter.hasNext(); ) {
			PaletteType palette = (PaletteType) iter.next();
			if (palette.getDisplayName() != null) {
				displayNameMap.put(palette.getId(), palette.getDisplayName());
			} else {
				displayNameMap.put(palette.getId(), palette.getId());
			}
		}
		return displayNameMap;
	}

	/**
	 * Text Tokenizer: seperate the colors into string array. 
	 * @param label x Label
	 */
	private String[] getColors(String label) {
		ArrayList wrapStrings = new ArrayList();

		StringTokenizer st = new StringTokenizer(label);
		while (st.hasMoreElements()) {
			wrapStrings.add(st.nextElement().toString());
		}
		return (String[])wrapStrings.toArray(new String[wrapStrings.size()]);
	}
}
