/* ***********************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGShapes.java,v 1.4 2006/05/08 19:09:28 sleeloy Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;
/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGShapes.java,v 1.4 2006/05/08 19:09:28 sleeloy Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.io.InputStream;
import java.io.Serializable;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.tptp.platform.report.chart.svg.internal.SVGGenerator;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DataSet;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Shape;
import org.eclipse.tptp.platform.report.chart.svg.internal.shapes.ShapeType;
import org.eclipse.tptp.platform.report.chart.svg.internal.shapes.Shapes;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.SVGShapeIterator;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.ShapesDOMDocument;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.Utilities;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.XMLLoader;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;


/**
 * <code>SVGShapes</code> is a container that contains a number of Shapes. 
 * $Revision: 1.4 $
 */

public class SVGShapes implements Serializable {
	static final long serialVersionUID = 7766275120609290096L;	
	private DefaultResources defaultResources;
	
	private String shapeIds[];
	private double[] shapeWidths;	
	private double[] shapeHeights;
	private Element[] shapeDefinitions;

	private boolean useDefault = true;
	
	private HashMap dataSetShapes;
	
	/**
	 * Constructor
	 *
	 */
	public SVGShapes(DefaultResources resources) {
		this.defaultResources = resources;
		this.dataSetShapes = new HashMap();
	}

	
	/**
	 * Initialize shapes
	 * @param shapeXMLFile
	 * @throws SVGGeneratorException
	 */
	private void initShapes() {
		
		synchronized (defaultResources) {

			if (defaultResources.isDefaultShapesInitialized()) {
				return;
			}
		
			Shapes shapes = null;
			boolean initialized = false;
			if (defaultResources.getResourceDir() != null) {
				String shapeXMLFile = defaultResources.getResourceDir() + "/shapes.xml";
				shapes = getShapesFromShapeXMLFile(shapeXMLFile);
				if (shapes != null) {
					initialized = parseShapes(shapes);
				}
			}
		
			if (initialized == false) {
				// no resource directory is specified or 
				// problem occurred during parsing shape definition file
				shapes = getShapesFromDefaultLocation();
				if (shapes != null) {
					initialized = parseShapes(shapes);
				}
			}
			if (initialized) {
				defaultResources.setDefaultShapesInitialized(true);
			} else {
				// The default location should always work
				Utilities.assertion(false);
			}

		}
	}

	/**
	 * get shape definitions from shapes.xml
	 * 
	 * @param shapes
	 * @return true if no error occurred during parsing shapes.xml, false otherwise.
	 */
	private boolean parseShapes(Shapes shapes) {
		Utilities.assertion(shapes != null);

		boolean parseFailed = false;
		List shapeList = shapes.getShape();

		String[] defaultShapeIds = new String[shapeList.size()];
		double[] defaultShapeWidths = new double[shapeList.size()];
		double[] defaultShapeHeights = new double[shapeList.size()];
		Element[] defaultShapeDefinitions = new Element[shapeList.size()];

		int shapeIndex = 0;
		for (Iterator i = shapeList.iterator(); i.hasNext(); shapeIndex++) {
			ShapeType shape = (ShapeType) i.next();
			String shapeDef = shape.getValue();
			Element shapeElem;
			shapeElem = getSymbolElementFromShapeDef(shapeDef);
			if (shapeElem == null) {
				parseFailed = true;
				defaultShapeIds = null;
				defaultShapeWidths = null;
				defaultShapeHeights = null;
				defaultShapeDefinitions = null;
				break;
			}
			// TODO prepend SVGGEN_ ???
			shapeElem.setAttribute("id", shape.getId());
			defaultShapeWidths[shapeIndex] = shape.getWidth();
			defaultShapeHeights[shapeIndex] = shape.getHeight();
			defaultShapeIds[shapeIndex] = shape.getId();
			defaultShapeDefinitions[shapeIndex] = shapeElem;
		}
		if (parseFailed == false) {
			defaultResources.setDefaultShapeIds(defaultShapeIds);
			defaultResources.setDefaultShapeWidths(defaultShapeWidths);
			defaultResources.setDefaultShapeHeights(defaultShapeHeights);
			defaultResources.setDefaultShapeDefinitions(defaultShapeDefinitions);
			return true;
		}
		return false;
	}

	/**
	 * Create shape document
	 * @return
	 */
	private Document createShapeDefDoc() {
		// create DOM for shape definitions		
		Document shapeDefDoc = null;
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
		try {
			builder = factory.newDocumentBuilder();
			DOMImplementation domImpl = builder.getDOMImplementation();
			DocumentType dType = domImpl.createDocumentType("svg", SVGGenerator.SVG_VERSION, SVGGenerator.SVG_DTD);			
			
			shapeDefDoc = domImpl.createDocument("http://www.w3.org/2000/svg", "g", dType);
		} catch (ParserConfigurationException e) {
			Utilities.assertion(false);
		}
		return shapeDefDoc;
	}
	
	/**
	 * Get shapes from XML file
	 * @param shapeXMLFile
	 * @return
	 */
	private Shapes getShapesFromShapeXMLFile(String shapeXMLFile) {
		XMLLoader xmlLoader = new XMLLoader();
		InputSource inputSource = new InputSource(shapeXMLFile);
	
		if (xmlLoader.load(inputSource) == false) {
			return null;
		}

		ShapesDOMDocument shapesInput = new ShapesDOMDocument(xmlLoader.getDOM());
		return shapesInput.getShapes();
	}
	
	/**
	 * get shapes from defualt location
	 * @return
	 */
	private Shapes getShapesFromDefaultLocation() {
		XMLLoader xmlLoader = new XMLLoader();

		InputStream defaultShapes = Utilities.getResourceAsStream("org/eclipse/tptp/platform/report/chart/svg/internal/resources/shapes.xml");
		InputSource inputSource = new InputSource(defaultShapes);

		if (xmlLoader.load(inputSource) == false) {
			return null;
		}

		ShapesDOMDocument shapesInput = new ShapesDOMDocument(xmlLoader.getDOM());
		return shapesInput.getShapes();
	}

	/**
	 * Gets the shape element.
	 * @param xmlFragments XML definitions of a shape
	 * @return
	 * @throws SVGGeneratorException
	 */
	private Element getSymbolElementFromShapeDef(String xmlFragments) {
		StringBuffer xml = new StringBuffer("<symbol>" + xmlFragments + "</symbol>");
		
		XMLLoader xmlLoader = new XMLLoader();
		StringReader reader = new StringReader(xml.toString());
		InputSource inputSource = new InputSource(reader);
		if (xmlLoader.load(inputSource) == false) {
			return null;
		}
		return xmlLoader.getDOM().getDocumentElement();
	}

	/**
	 * Load customized shapes from a file. These shapes are only used for this chart.
	 * 
	 * @param shapesLocation
	 */
	public void setCustomShapes(String shapesLocation) {
		boolean configInvalid = false;

		Document shapeDefDoc = createShapeDefDoc();
		Utilities.assertion(shapeDefDoc != null);
	
		Shapes shapes = getShapesFromShapeXMLFile(shapesLocation);
		if (shapes == null) {
			return;
		}
	
		List shapeList = shapes.getShape();

		shapeIds = new String[shapeList.size()];
		shapeWidths = new double[shapeList.size()];
		shapeHeights = new double[shapeList.size()];
		shapeDefinitions = new Element[shapeList.size()];

		int shapeIndex = 0;
		for (Iterator i = shapeList.iterator(); i.hasNext(); shapeIndex++) {
			ShapeType shape = (ShapeType) i.next();
			String shapeDef = shape.getValue();
			Element shapeElem = getSymbolElementFromShapeDef(shapeDef);
			if (shapeElem == null) {
				configInvalid = true;
				break;
			}
			shapeElem.setAttribute("id", shape.getId());
			shapeWidths[shapeIndex] = shape.getWidth();
			shapeHeights[shapeIndex] = shape.getHeight();
			shapeIds[shapeIndex] = shape.getId();
			shapeDefinitions[shapeIndex] = shapeElem;
		}

		if (configInvalid == false) {
			useDefault = false;
		}
	}
	
	/**
	 * Gets the DOM representation of the shapes.
	 * @param numOfShapes number of shapes
	 * @return a group of shapes
	 */
	public Element getShapeDefinition(int numOfShapes) {
		Document shapeDefDoc = createShapeDefDoc();
		Utilities.assertion(shapeDefDoc != null);
		Element shapeGroup = shapeDefDoc.getDocumentElement();
		shapeGroup.setAttribute("id", "symbols");
		
		Element[] shapeElementArray;
		String[] shapeIdArray;
		if (useDefault) {
			if (defaultResources.isDefaultShapesInitialized() == false) {
				initShapes();
			}
			shapeElementArray = defaultResources.getDefaultShapeDefinitions();
			shapeIdArray = defaultResources.getDefaultShapeIds();
		} else {
			shapeElementArray = shapeDefinitions;
			shapeIdArray = shapeIds;
		}
		
		// first add the shapes explicitly assigned to data sets
		int numberOfShapeElementsAdded = 0;
		for (int i=0; i<shapeElementArray.length; i++) {
			if (dataSetShapes.containsValue(shapeIdArray[i])) {
				synchronized (defaultResources) {
					shapeGroup.appendChild(shapeDefDoc.importNode(shapeElementArray[i], true));
				}
				numberOfShapeElementsAdded++;
			}
		}
		
		int numOfShapesLeft = shapeElementArray.length - numberOfShapeElementsAdded;
		int numOfShapesNeeded = numOfShapes - numberOfShapeElementsAdded;
		
		int numOfDefs = numOfShapesNeeded < numOfShapesLeft ? numOfShapesNeeded : numOfShapesLeft;
		int shapeIndex = 0;
		numberOfShapeElementsAdded = 0;
		while (numberOfShapeElementsAdded < numOfDefs) {
			if (dataSetShapes.containsValue(shapeIdArray[shapeIndex]) == false) {
				if (useDefault) {
					// since xerces library is not thread safe, we need to make sure
					// the shared variable is used by one thread at a time.
					synchronized (defaultResources) {
						shapeGroup.appendChild(shapeDefDoc.importNode(shapeElementArray[shapeIndex], true));
					}
				} else {
					shapeGroup.appendChild(shapeDefDoc.importNode(shapeElementArray[shapeIndex], true));
				}
				numberOfShapeElementsAdded++;
			}
			shapeIndex++;
		}

		return shapeGroup;
	}

	public void setDataSetShapes(List shapeList) {
		for (Iterator iter=shapeList.iterator(); iter.hasNext(); ) {
			Shape dataSetShape = (Shape) iter.next();
			try {
				DataSet dataSet = (DataSet) dataSetShape.getDataSetId();
				String dataSetId = dataSet.getId();
				String shapeId = dataSetShape.getShapeId();
				dataSetShapes.put(dataSetId, shapeId);
			} catch (ClassCastException e) {
				// do nothing
			}
		}
	}
	
	public class ShapeIterator implements SVGShapeIterator {

		private int nextShapeIndex = 0;
		private double currentShapeWidth = 0;
		private double currentShapeHeight = 0;
		private String currentShapeId;
		
		/* (non-Javadoc)
		 * @see org.eclipse.tptp.platform.report.chart.svg.internal.util.SVGShapeIterator#getNextShapeId(java.lang.String)
		 */
		public String getNextShapeId(String dataSetId) {
			String[] shapeIdArray;
			double[] widths;
			double[] heights;
			if (useDefault) {
				if (defaultResources.isDefaultShapesInitialized() == false) {
					initShapes();
				}
				shapeIdArray = defaultResources.getDefaultShapeIds();
				widths = defaultResources.getDefaultShapeWidths();
				heights = defaultResources.getDefaultShapeHeights();
			} else {
				shapeIdArray = shapeIds;
				widths = shapeWidths;
				heights = shapeHeights;
			}

			String shapeId = (String)dataSetShapes.get(dataSetId);
			if (shapeId == null) {
				int startIndex = nextShapeIndex;
			
				// we have to assume at least 1 shape exists
				Utilities.assertion(shapeIdArray.length > 0);
			
				do {
					nextShapeIndex = nextShapeIndex % shapeIdArray.length;
					shapeId = shapeIdArray[nextShapeIndex];
					if (dataSetShapes.containsValue(shapeId) == false) {
						currentShapeId = shapeId;
						currentShapeWidth = widths[nextShapeIndex % widths.length];
						currentShapeHeight = heights[nextShapeIndex % heights.length];
						nextShapeIndex++;
						break;
					}
					nextShapeIndex++;
				} while (nextShapeIndex != startIndex);
			} else {
				currentShapeId = shapeId;
				// find shape index
				int shapeIndex = 0;
				for (int i=0; i<shapeIdArray.length; i++) {
					if (shapeIdArray[i].equals(shapeId)) {
						shapeIndex = i;
						break;
					}
				}
				currentShapeWidth = widths[shapeIndex];
				currentShapeHeight = heights[shapeIndex];
			}
			return shapeId;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.tptp.platform.report.chart.svg.internal.util.SVGShapeIterator#getShapeWidth()
		 */
		public double getShapeWidth() {
			return currentShapeWidth;
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.tptp.platform.report.chart.svg.internal.util.SVGShapeIterator#getShapeHeight()
		 */
		public double getShapeHeight() {
			return currentShapeHeight;
		}
		
	}
	
	public SVGShapeIterator getShapeIterator() {
		return new ShapeIterator();
	}
}
