/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Timestamp.java,v 1.4 2008/12/12 22:22:09 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.part;


import java.util.Date;

import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGBase;
import org.eclipse.tptp.platform.report.chart.svg.internal.generator.SVGText;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Chart;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Configuration;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Data;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.DateFormat;
import org.eclipse.tptp.platform.report.chart.svg.internal.input.Internationalization;
import org.eclipse.tptp.platform.report.chart.svg.internal.util.NLString;

import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.util.GregorianCalendar;
import com.ibm.icu.util.ULocale;


/**
 *
 * <code>Timestamp</code> is class to add the timestamp for each svg graph 
 * $Revision: 1.4 $  
 */

public class Timestamp extends Part {
	static final long serialVersionUID = 6366274088283355495L;
	/**
	 * timestamp text
	 */
	private String timestamp;
	
	/**
	 * text direction setting, true if LTR, false if RTL
	 */
	private boolean isLTR;
	
	/**
	 * An object for retriving translated text
	 */
	private NLString nls;
	
	/**
	 * @param input Chart
	 * @param x position for timestamp
	 * @param y position for timestamp
	 * @param width position for timestamp
	 * @param height position for timestamp
	 */
	public Timestamp(Chart input, boolean isLTR, double x, double y, double width, double height, NLString nls) {
		super(input, x, y, width, height);

		this.isLTR = isLTR;
		this.nls = nls;
		String prefix = null;
		String pattern = null;
		ULocale locale = null;

		// get the input info from config, how time stamp is tobe shown (prefix, pattern and locale)		
		SimpleDateFormat dateFormatter = null;
		Configuration config = input.getConfiguration();
		if (config != null) {
			if(config.getTimestamp() != null ){
				if(config.getTimestamp().getPrefix() != null) {
					prefix = nls.getString(config.getTimestamp().getPrefix());
				}

				if(config.getTimestamp().getDateFormat() != null) {
					DateFormat dateFormat = config.getTimestamp().getDateFormat();
					if(dateFormat != null){
						if(dateFormat.getPattern() != null){
							pattern = dateFormat.getPattern();
						}
					}
				}
			}
			Internationalization i18n = config.getInternationalization();
			if (i18n != null) {
				String language = i18n.getLanguage() == null ? EMPTY_STRING : i18n.getLanguage();
				String country = i18n.getCountry() == null ? EMPTY_STRING : i18n.getCountry();
				if ((language != EMPTY_STRING) || (country != EMPTY_STRING))
					locale = new ULocale(language, country);
				else if (language != EMPTY_STRING)
					locale = new ULocale(language);
			}
		}

		// format timestamp based on local, pattern and prefix.
		Data data = input.getData();
		if (data != null) {
			if(pattern != null && pattern != EMPTY_STRING){
				try {
					if (locale != null) {
						dateFormatter = new SimpleDateFormat(pattern, locale);
					} else {
						dateFormatter = new SimpleDateFormat(pattern);
					}
				} catch (NullPointerException npe) {
					dateFormatter = null;
				} catch (IllegalArgumentException iae) {
					dateFormatter = null;
				}

			}else{
				try {
					dateFormatter = new SimpleDateFormat();
				} catch (NullPointerException npe) {
					dateFormatter = null;
				} catch (IllegalArgumentException iae) {
					dateFormatter = null;
				}
			}
			
			String timeStampText;
			if (dateFormatter != null) {
				timeStampText = dateFormatter.format(parseISO8601Date(data.getTimestamp()));
			}else{
				timeStampText = data.getTimestamp();
			}
			if(prefix != null){
				if(!prefix.equals(EMPTY_STRING)) {
					if (isLTR) {
						timestamp = prefix + "  " + timeStampText;
					} else {
						timestamp = timeStampText + "  " + prefix;
					}
				} else {
					timestamp = timeStampText;
				}
			}else{
				timestamp = timeStampText;
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.report.chart.svg.internal.part.SVGPart#constructPart()
	 */
	protected void constructPart() {
		String title, stylesheetClass = "graphTimeStamp anchorAtEnd";
		if (timestamp != null) {
			SVGText graphTimeStamp = new SVGText(); 
			graphTimeStamp.setText(timestamp);
			if (isLTR) {
				graphTimeStamp.setXCoordinate(Double.toString(width - TIMESTAMP_SIDE_MARGIN));
				graphTimeStamp.setYCoordinate(Double.toString(height - TIMESTAMP_BOTTOM_MARGIN));
				graphTimeStamp.setStyleClass("graphTimeStamp anchorAtEnd");
				graphTimeStamp.setIdentifier("SVGGEN_timeStamp");
			} else {
				// TODO test RTL setting
				graphTimeStamp.setXCoordinate(Double.toString(TIMESTAMP_SIDE_MARGIN));
				graphTimeStamp.setYCoordinate(Double.toString(height - TIMESTAMP_BOTTOM_MARGIN));
				graphTimeStamp.setStyleClass("graphTimeStamp anchorAtStart");
				graphTimeStamp.setIdentifier("SVGGEN_timeStamp");
			}
			
			SVGBase[] children = new SVGBase[2];
			children[1] = graphTimeStamp;
			setChildren(children);
		}
	}
	
	/**
	 * Parse a date in ISO8601 format CCYY-MM-DDThh:mm:ss
	 * @param s date in ISO8601 format
	 * @return null if the format is invalid
	 */
	static public Date parseISO8601Date(String s) {
		if (s == null) {
			return null;
		}
		
		int year = 0, month = 0, day = 0, hrs = 0, min = 0, sec = 0;
		try {
			int len = s.length();
			if (len >= 4) {
				year = Integer.parseInt(s.substring(0,4));
			}
			if (len >= 7) {
				month = Integer.parseInt(s.substring(5,7))-1;
			}
			if (len >= 10) {
				day = Integer.parseInt(s.substring(8,10));
			}
			if (len >= 13) {
				hrs = Integer.parseInt(s.substring(11,13));
			}
			if (len >= 16) {
				min = Integer.parseInt(s.substring(14,16));
			}
			if (len >= 19) {
				sec = Integer.parseInt(s.substring(17,19));
			}
			return new GregorianCalendar(year, month, day, hrs, min, sec).getTime();
		}
		catch(Exception e) {
			return null;
		}
	}


}
