package org.eclipse.tptp.platform.report.chart.svg.internal.util;

/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SVGTranscode.java,v 1.6 2005/06/17 15:17:28 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

 
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;

import org.apache.batik.dom.svg.SAXSVGDocumentFactory;
import org.apache.batik.transcoder.DefaultErrorHandler;
import org.apache.batik.transcoder.TranscoderException;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.TranscodingHints;
import org.apache.batik.transcoder.image.JPEGTranscoder;
import org.apache.batik.transcoder.image.PNGTranscoder;
import org.apache.batik.transcoder.svg2svg.SVGTranscoder;
import org.apache.batik.util.XMLResourceDescriptor;
import org.apache.fop.svg.PDFTranscoder;
import org.w3c.dom.Document;

/*******************************************************************************
* This class to used to convert a SVG image data to a static image format
* for display when SVG is not supported.
*
* NOTE: This class provides a set of static methods that wrapper portions of 
* the open source Apache Batik project binaries.  Use of the Batik binaries in
* a product requires specific approval from the IBM Open Source Steering Committee.
* Each product must obtain individual approval for this usage.  
* 
* This class is a wrapper for the Apache Batik software which performs the
* actual transcoding of the SVG format data into the selected static image
* format data. 
*
* Several static methods are provided to allow a variety of input and output
* styles.
* 
* Batik does not provide sufficient information to correct errors during the
* processing of the SVG document, so applications must choose to either ignore
* any exceptions or to take the appropriate action to handle the exception.
* 
* You can read more on batik at http://xml.pache.org/batik/.  It should be noted that
* the SVG Chart Generator 1.1 has been tested with Batik v1.5.  The Batik
* v1.5 distribution can be downloaded from the download page on
* http://xml.apache.org/batik site.
* 
* @author Warren Williams
*******************************************************************************/
public class SVGTranscode
{
    /**
     * Transcode a SVG data file and write it to an output stream as a JPEG image.
     *
     * @param svgFile  the SVG data file to transcode
     * @param ostream  the output stream
     *
     * @throws IOException
     * @throws TranscoderException
     */
    public static void transcode2JPEG(File svgFile, OutputStream ostream)
        throws IOException, TranscoderException
    {
        Reader r = new FileReader(svgFile);
        transcode2JPEG(r, ostream);
    }

    /**
     * Transcode a string of SVG XML data and write it to an output stream as a JPEG image.
     *
     * @param svg  the string containing the SVG XML data to transcode
     * @param ostream  the output stream
     *
     * @throws IOException
     * @throws TranscoderException
     */
    public static void transcode2JPEG(String svg, OutputStream ostream)
        throws IOException, TranscoderException
    {
        Reader r = new StringReader(svg);
        transcode2JPEG(r, ostream);
    }

    /**
     * Transcode a SVG document and write it to an output stream as a JPEG image.
     *
     * @param document the SVG document to transcode
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2JPEG(Document document, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(document);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2JPEG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
     * Transcode a SVG format input stream and write it to an output stream as a JPEG image.
     *
     * @param istream  the input stream
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2JPEG(InputStream istream, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(istream);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2JPEG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
     * Transcode a SVG format Reader and write it to an output stream as a JPEG image.
     *
     * @param reader  the input reader
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2JPEG(Reader reader, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(reader);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2JPEG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
    
    /**
     * Transcode SVG format data from a transcoder input into JPEG format data and
     * write it to a transcoder output.
     *
     * @param tin  the TranscoderInput
     * @param tout  the TranscoderOutput
     * 
     * @throws TranscoderException
     */
    public static void transcode2JPEG(TranscoderInput tcin, TranscoderOutput tcout)
        throws TranscoderException
    {
        // create the JPEG transcoder
        JPEGTranscoder t = new JPEGTranscoder();
        t.addTranscodingHint(JPEGTranscoder.KEY_QUALITY, new Float(.8));

        // set the custom error handler
        t.setErrorHandler(new DefaultErrorHandler()
        {
            public void error(TranscoderException te)
            {
            }
            public void warning(TranscoderException te)
            {
            }
        });

        // transcode the data
        t.transcode(tcin, tcout);
    }

    /**
     * Transcode a SVG data file and write it to an output stream as a PNG image.
     *
     * @param svgFile  the SVG data file to transcode
     * @param ostream  the output stream
     *
     * @throws IOException
     * @throws TranscoderException
     */
    public static void transcode2PNG(File svgFile, OutputStream ostream)
        throws IOException, TranscoderException
    {
        Reader r = new FileReader(svgFile);
        transcode2PNG(r, ostream);
    }

    /**
     * Transcode a string of SVG XML data and write it to an output stream as a PNG image.
     *
     * @param svg  the string containing the SVG XML data to transcode
     * @param ostream  the output stream
     *
     * @throws IOException
     * @throws TranscoderException
     */
    public static void transcode2PNG(String svg, OutputStream ostream)
        throws IOException, TranscoderException
    {
        Reader r = new StringReader(svg);
        transcode2PNG(r, ostream);
    }

    /**
     * Transcode a SVG document and write it to an output stream as a PNG image.
     *
     * @param document the SVG document to transcode
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2PNG(Document document, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(document);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2PNG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore flush errors
        }
    }

    /**
     * Transcode a SVG format input stream and write it to an output stream as a PNG image.
     *
     * @param istream  the input stream
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2PNG(InputStream istream, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(istream);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2PNG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
     * Transcode a SVG format Reader and write it to an output stream as a PNG image.
     *
     * @param reader  the input reader
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2PNG(Reader reader, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(reader);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2PNG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
     * Transcode SVG format data from a transcoder input into PNG format data and
     * write it to a transcoder output.
     *
     * @param tin  the TranscoderInput
     * @param tout  the TranscoderOutput
     *
     * @throws TranscoderException
     */
    public static void transcode2PNG(TranscoderInput tcin, TranscoderOutput tcout)
        throws TranscoderException
    {   	
        // create the PNG transcoder
        PNGTranscoder t = new PNGTranscoder();

        // set the custom error handler
        t.setErrorHandler(new DefaultErrorHandler()
        {
            public void error(TranscoderException te)
            {
            }
            public void warning(TranscoderException te)
            {
            }
        });

        // transcode the data
        t.transcode(tcin, tcout);
    }

    /**
     * Transcode a string of SVG XML data and write it to an output stream as a PDF image.
     *
     * @param svg  the string containing the SVG XML data to transcode
     * @param ostream  the output stream
     *
     * @throws IOException
     * @throws TranscoderException
     */
    public static void transcode2PDF(String svg, OutputStream ostream)
        throws IOException, TranscoderException
    {
        Reader r = new StringReader(svg);
        transcode2PDF(r, ostream);
    }

    /**
     * Transcode a SVG document and write it to an output stream as a PDF image.
     *
     * @param document the SVG document to transcode
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2PDF(Document document, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(document);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2PDF(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore flush errors
        }
    }

    /**
     * Transcode a SVG format input stream and write it to an output stream as a PDF image.
     *
     * @param istream  the input stream
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2PDF(InputStream istream, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(istream);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2PDF(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
     * Transcode a SVG format Reader and write it to an output stream as a PDF image.
     *
     * @param reader  the input reader
     * @param ostream  the output stream
     *
     * @throws TranscoderException
     */
    public static void transcode2PDF(Reader reader, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(reader);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        transcode2PDF(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }
    /**
     * Transcode SVG format data from a transcoder input into PNG format data and
     * write it to a transcoder output.
     *
     * @param tin  the TranscoderInput
     * @param tout  the TranscoderOutput
     *
     * @throws TranscoderException
     */
    public static void transcode2PDF(TranscoderInput tcin, TranscoderOutput tcout)
        throws TranscoderException
    {
        // create the PNG transcoder
        PDFTranscoder t = new PDFTranscoder();

        // set the custom error handler
        t.setErrorHandler(new DefaultErrorHandler()
        {
            public void error(TranscoderException te)
            {
            	te.printStackTrace();
            }
            public void warning(TranscoderException te)
            {
            	te.printStackTrace();
            }
        });

        // transcode the data
        t.transcode(tcin, tcout);
    }
    /**
     * 'Pretty print' an SVG input stream to an SVG output stream
     *
     * This method will 'pretty print' an SVG input stream to an SVG output stream.
     *
     *
     * @param istream the SVG input stream
     * @param ostream the SVG output stream
     *
     * @return the created SVG document
     *
     * @throws TranscoderException
     */
    public static void prettyPrintSVG(InputStream istream, OutputStream ostream)
        throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(istream);
        TranscoderOutput tcout = new TranscoderOutput(ostream);
        prettyPrintSVG(tcin, tcout);

        // flush the output stream
        try
        {
            ostream.flush();
        } catch (IOException ioe)
        {
            // ignore output stream flush error
        }
    }

    /**
     * 'Pretty print' an SVG Reader to an SVG Writer
     *
     * This method will 'pretty print' an SVG Reader to an SVG Writer.
     *
     *
     * @param rdr the SVG Reader
     * @param writer the SVG Writer
     *
     * @throws TranscoderException
     */
    public static void prettyPrintSVG(Reader rdr, Writer writer) throws TranscoderException
    {
        // transcode the data
        TranscoderInput tcin = new TranscoderInput(rdr);
        TranscoderOutput tcout = new TranscoderOutput(writer);
        prettyPrintSVG(tcin, tcout);

        // flush the output stream
        try
        {
            writer.flush();
        } catch (IOException ioe)
        {
            // ignore writer flush error
        }
    }

    /**
    * Transcode SVG format data from a transcoder input into a 'pretty printed' SVG format data and
    * write it to a transcoder output.
    *
    * @param tin  the TranscoderInput
    * @param tout  the TranscoderOutput
    *
    * @throws TranscoderException
    */
    public static void prettyPrintSVG(TranscoderInput tcin, TranscoderOutput tcout)
        throws TranscoderException
    {
        // create the SVG transcoder
        SVGTranscoder t = new SVGTranscoder();

        // set the transcoding hints to 'pretty print' the output
        TranscodingHints hints = new TranscodingHints();
        hints.put(SVGTranscoder.KEY_FORMAT, SVGTranscoder.VALUE_FORMAT_ON);
        //		hints.put(SVGTranscoder.KEY_DOCUMENT_WIDTH, "150");
        //		hints.put(SVGTranscoder.KEY_TABULATION_WIDTH, "3");

        t.setTranscodingHints(hints);

        // set the custom error handler
        t.setErrorHandler(new DefaultErrorHandler()
        {
            public void error(TranscoderException te)
            {
            }
            public void warning(TranscoderException te)
            {
            }
        });

        // transcode the data
        t.transcode(tcin, tcout);
    }

    /**
    * Transcode SVG format data from an input String into a 'pretty printed' SVG format output String
    *
    * @param svgInput the SVG input string
    * @return the 'pretty printed' output string
    *
    * @throws TranscoderException
    */
    public static String prettyPrintSVG(String svgInput) throws TranscoderException
    {
        // create the SVG StringReader and StringWriter
        StringReader rdr = new StringReader(svgInput);
        StringWriter writer = new StringWriter();

        // pretty print the SVG data
        prettyPrintSVG(rdr, writer);

        // flush the Writer
        writer.flush();

        // return the formatted string
        return writer.toString();
    }

    /**
    * Create a SVG document from a URI String
    *
    * This method is provided as a possibly more convenient way to
    * create a SVGDocument for use with the transcode methods in this class.
    * Since the underlying DocumentFactory requires the uri of the document,
    * the uri is the sole parameter for this method.
    *
    * @param uri the SVG uri string
    * @return the created SVG document
    *
    * @throws IOException
    */
    public static Document createSVGDocument(String uri) throws IOException
    {
        String parser = XMLResourceDescriptor.getXMLParserClassName();
        SAXSVGDocumentFactory df = new SAXSVGDocumentFactory(parser);
        Document doc = df.createDocument(uri);
        return doc;
    }

} // end of SVGTranscode
