/* ***********************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Utilities.java,v 1.2 2005/06/07 18:30:37 dguilbaud Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/

package org.eclipse.tptp.platform.report.chart.svg.internal.util;
/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Utilities.java,v 1.2 2005/06/07 18:30:37 dguilbaud Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/


import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.tptp.platform.report.chart.svg.internal.input.impl.InputBase;
import org.w3c.dom.CDATASection;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class holds general Utility methods.
 *
 */
public class Utilities {
	
	/**
	 * Outputs a message indicating the that the specified function is 
	 * not implemented.
	 * 
	 * @param name the name of the unimplemented function
	 */
	static public void unimplementedFunction() {
		System.err.println("Unimplemented Function.");
		Exception e = new Exception();
		e.printStackTrace();
	}
	
	/**
	 * Given an Element, this function locates the immediate child which is an
	 * Element and whose name matches the name given.  It then returns the Element
	 * found.
	 * 
	 * @param element the Element to search
	 * @param name the name to match to the node name
	 * @return the Element found
	 */
	static public Element getElement(Element element, String name) {
		NodeList children = element.getChildNodes();
		for (int i = 0; i < children.getLength(); ++i) {
			Node child = children.item(i);
			if (child.getNodeType() == Node.ELEMENT_NODE && child.getNodeName() == name) {
				return (Element) child;
			}
		}
		
		return null;
	}

	/**
	 * Given an Element, this function locates all the immediate children which
	 * are Elements and whose name matches the name given.  It then returns the list
	 * of Elements found.
	 * 
	 * @param element the Element to search
	 * @param name the name to match to the node name
	 * @return a List of Element nodes found
	 */
	static public List getElementList(Element element, String name) {
		NodeList children = element.getChildNodes();
		ArrayList resultList = new ArrayList();
		for (int i = 0; i < children.getLength(); ++i) {
			Node child = children.item(i);
			if (child.getNodeType() == Node.ELEMENT_NODE && child.getNodeName() == name) {
				resultList.add(child);
			}
		}
		return resultList;
	}
	
	/**
	 * Given an Element, this function assumes the first child of that Element
	 * is a Text node and extracts and return the text.
	 * 
	 * @param element the input Element
	 * @return the value of the text node child
	 */
	static public String getText(Element element) {
		Node child = element.getFirstChild();
		String result = null;
		if ((child != null) && (child.getNodeType() == Node.TEXT_NODE)) {
			result = child.getNodeValue();
		}
		return result;
	}
	
	public static void copyAttributes(Element from, Element to) {
		NamedNodeMap attrMap = from.getAttributes();
		for (int i=0; i<attrMap.getLength(); i++) {
			Node attr = attrMap.item(i);
			String name = attr.getNodeName();
			String value = attr.getNodeValue();
			to.setAttribute(name, value);
		}
	}
	
	public static void copyChildren(Element from, Element to) {
		while (from.getFirstChild() != null) {
			Node childToMove = from.removeChild(from.getFirstChild());
			to.appendChild(childToMove);
		}
	}
	
//	public static Element appendChild(Element parent, Element child, String tagName) {
//		String childTag = child.getTagName();
//		if (childTag.equals(tagName) == false) {
//			Element newElement = parent.getOwnerDocument().createElement(tagName);
//			// copy attributes
//			Utilities.copyAttributes(child, newElement);
//			// copy children elements
//			Utilities.copyChildren(child, newElement);
//			// append newElement to parent
//			parent.appendChild(newElement);
//			// delete the "old" element
//			parent.getOwnerDocument().removeChild(child);
//			return newElement;
//		} else {
//			parent.appendChild(child);
//			return child;
//		}
//	}
	
	public static Element appendChild(InputBase parentObj, Object childObj, String tagName) {
		Element parent;
		Element child;
		Utilities.assertion(childObj instanceof InputBase);
		try {
			parent = parentObj.getElement();
			child = ((InputBase) childObj).getElement();
		
			String childTag = child.getTagName();
			if (childTag.equals(tagName) == false) {
				Element newElement = parent.getOwnerDocument().createElement(tagName);
				// copy attributes
				Utilities.copyAttributes(child, newElement);
				// copy children elements
				Utilities.copyChildren(child, newElement);
				// append newElement to parent
				parent.appendChild(newElement);
				((InputBase)childObj).setElement(newElement);
				return newElement;
			} else {
				parent.appendChild(child);
				return child;
			}
		} catch (ClassCastException e) {
			return null;
		}
	}
	
	public static void setInputObjectElement(Object obj, Element element) {
		try {
			InputBase inputObj = (InputBase) obj;
			inputObj.setElement(element);
		} catch (ClassCastException e) {
			// ignore
		}
	}

	
	/**
	 * Given an Element, returns the content of the first CDATA node of this element.
	 * If this element has no CDATA node, null will be returned. 
	 *
	 * @param element the input Element
	 * @return the text in the CDATA node
	 */
	static public String getCDATA(Element element) {
		String result = null;
		NodeList nodeList = element.getChildNodes();
		for (int i=0; i<nodeList.getLength(); i++) {
			Node child = nodeList.item(i);
			if (child.getNodeType() == Node.CDATA_SECTION_NODE) {
				result = ((CDATASection) child).getData();
				break;
			}
		}
		return result;
	}
	
	/**
	 * Displays a traceback if the assertion expression evaluates to false.
	 * 
	 * @param assertion the expression to evaluate
	 */
	static public void assertion(boolean assertion) {
		if (assertion == false) {
			System.err.println("Assertion Failed.");
			Exception e = new Exception();
			e.printStackTrace();
		}
	}

	/**
	 * Returns an input stream for reading the specified resource.
	 *
	 * The search order is described in the documentation for {@link
	 * #getResource(String)}.<p>
	 *
	 * @param  name the resource name
	 * @return an input stream for reading the resource, or <code>null</code>
	 *         if the resource could not be found
	 */
	public static InputStream getResourceAsStream( String name )
	{
		ClassLoader clsLoader = null;
		InputStream is = null;
		for ( Iterator iter=getClassLoaders(); iter.hasNext(); )
		{
			clsLoader = (ClassLoader) iter.next();
			is = clsLoader.getResourceAsStream( name );
			if ( is != null )
			{
				return is;
			}
		}
		return null;
	}

	/**
	 * Returns a URL for reading the specified resource.
	 *
	 * The search order is described in the documentation for {@link
	 * #getResource(String)}.<p>
	 *
	 * @param  name the resource name
	 * @return a URL for reading the resource, or <code>null</code>
	 *         if the resource could not be found
	 */
	public static URL getResourceAsUrl( String name )
	{
		ClassLoader clsLoader = null;
		URL url = null;
		for ( Iterator iter=getClassLoaders(); iter.hasNext(); )
		{
			clsLoader = (ClassLoader) iter.next();
			url = clsLoader.getResource( name );
			if ( url != null )
			{
				return url;
			}
		}
		return null;
	}
	
	/**
	 * Returns an iterator of class loaders. The class loaders are available
	 * using this priority:
	 * <ol>
	 *   <li>the class loader of this class
	 *   <li>the context class loader of the current thread
	 *   <li>the system class loader
	 * </ol>
	 * Subclasses may wish to override this method to change the class loaders
	 * or their priorities.
	 */
	protected static Iterator getClassLoaders()
	{
		ArrayList clsLoaders = new ArrayList();

		ClassLoader clsLoader = null;

		//@03A
		// class loader of this class. if the wcl.jar is packaged with the
		// application war, then this loader should be the only they need.
		// if the wcl.jar is installed in the was_home/lib/app directory,
		// then this loader provides access to PS classes only.
		try
		{
			clsLoader = Utilities.class.getClassLoader();
			if ( clsLoader != null )
			{
				clsLoaders.add( clsLoader );
			}
		}
		catch ( SecurityException se )
		{
		}

		// context class loader of the current thread. if the wcl.jar is
		// installed in the was_home/lib/app directory, then this loader
		// provides access to the application jars and classes.
		Thread t = Thread.currentThread();
		if ( t != null )
		{
			try //@02A
			{
				clsLoader = t.getContextClassLoader();
				if ( clsLoader != null )
				{
					clsLoaders.add( clsLoader );
				}
			}
			catch ( SecurityException se ) //@02A
			{
			}
		}

		// system class loader. this is just another class loader that is
		// well known and available, but in general, I wouldn't expect to
		// find anything out of this one.
		try //@02A
		{
			clsLoader = ClassLoader.getSystemClassLoader();
			if ( clsLoader != null )
			{
				clsLoaders.add( clsLoader );
			}
		}
		catch ( SecurityException se ) //@02A
		{
		}

		return clsLoaders.iterator();
	}
}
