/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DCell.java,v 1.3 2008/05/23 14:11:48 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.core.internal;


/**
 * This is a cell of table. <br>
 * The cell is the smallest part of a table: table contains rows and each row 
 * contains cells, that's why you need a line number and a column number to define 
 * a cell.<br>
 * A cell can contain all paragraph items
 * and support alignment,all the items added to the cell will be displayed 
 * by the driver in the specified alignment.<br> 
 * You can create a cell occupating two or more columns by using the setColSpan 
 * method.
 * <br><br>
 * <pre>
 * Exemple:<br>
 * DTable tab = new DTable(1,true);<br>
 * DRow r[2];<br>
 * r[0]=new DRow();<br>
 * r[1]=new DRow();<br>
 * DCell c1 = new DCell(), c2= new DCell();<br>
 * r[0].addChild(c1);<br>
 * r[0].insertChild(c2, c1);<br>
 * c1.addChild(new DText("cell 0,0"));<br>
 * c2.addChild(new DText("cell 0,1"));<br>
 * c1 = new DCell();<br>
 * c1.setColSpan(1);  // <- this cell takes 2 columns<br>
 * r[1].addChild(c1);<br>
 * c1.addChild(new DText("cell 1,0 on two columns"));<br>
 * </pre>
 * <br>
 * Result:<br>
 * The cell c1 of the second line has one "colspan"<br>
 * ______________________ <br>
 * | cell 0,0  .......| cell 0,1.......|<br>
 * |_____________________|<br>
 * | cell 1,0 on two columns ..|<br>
 * |_____________________|<br>
 * 
 * @see DTable
 * @see DRow
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class DCell extends DAbstractParagraph
{
   public static final int DEFAULT_COLSPAN = 0;
   public static final int DEFAULT_ROWSPAN = 0;

   private int colSpan = DEFAULT_COLSPAN;
   private int rowSpan = DEFAULT_ROWSPAN;
   
   /**
    * Creates a cell.
    * 
    * @param cspan is the number of column that the cell can occupate to the right.
    * @param rspan is the number of line that the cell can occupate to the bottom.
    * @param align is the global alignment for the cell.
    * 
    * @see IDAlignment
    */
   public DCell(int cspan, int rspan, int align) 
   {
   	super(align);
   	colSpan = cspan;
   	rowSpan = rspan;    
   }
   
   /**
    * Creates a cell.
    * 
    * @param align is the global alignment for the cell.
    * 
    * @see IDAlignment
    */
   public DCell(int align) 
   {
   	super(align);
   }
   
   
   /**
    * Creates a cell.
    * 
    * @param cspan is the number of column that the cell can occupate to the right.
    * @param rspan is the number of line that the cell can occupate to the bottom.
   */
   public DCell(int cspan, int rspan) 
   {
   	  super();
      colSpan = cspan;
      rowSpan = rspan;    
   }
   
   /**
    * Creates a cell with no column span and row span.
    */
   public DCell() 
   {
      super();
   }
   
   /**
    * Access method for the colSpan property.
    * 
    * @return   the current value of the colSpan property
    */
   public int getColSpan() 
   {
      return colSpan;    
   }
   
   /**
    * Sets the value of the colSpan property.
    * 
    * @param aColSpan the new value of the colSpan property
    */
   public void setColSpan(int aColSpan) 
   {
      colSpan = aColSpan;    
   }
   
   /**
    * Access method for the rowSpan property.
    * 
    * @return   the current value of the rowSpan property
    */
   public int getRowSpan() 
   {
      return rowSpan;    
   }
   
   /**
    * Sets the value of the rowSpan property.
    * 
    * @param aRowSpan the new value of the rowSpan property
    */
   public void setRowSpan(int aRowSpan) 
   {
      rowSpan = aRowSpan;    
   }
   
   /**
    * Returns the line number (with the row span) that the cell is taking.
    */
   public int getLineNumber() 
   {
	  DRow r = (DRow)getParent();
      if (r == null) return -1;
      
      DTable t = (DTable)r.getParent();
	  if (t == null) return -1;
      
      IDItem i;
      int nbrow = 0;
   	  for (i = t.getFirstChild(); i != r; i = i.getNext())
	  {
		 if (i instanceof DRow)
		 {
		 	nbrow = nbrow + 1;
		 } 
	  }
   	
      return nbrow;    
   }
   
   /**
    * Returns the column number of the cell (with the col span value) that the cell is taking.
    */
   public int getColumnNumber() 
   {
   	  IDItem p = getParent();
   	  if (p == null) return -1;
      if (p.getFirstChild()==null) return -1;
   	  
   	  IDItem i;
	  int nbcell = 0;
	  int nbcolspan = 0;
   	  for (i = p.getFirstChild(); i != this; i = i.getNext())
   	  {
   	  	if (i instanceof DCell)
   	  	{
   	  		DCell c = (DCell)i;
   	  		nbcell = nbcell + 1;
   	  		nbcolspan = nbcolspan + c.getColSpan();
   	  	}
   	  }
      return nbcell + nbcolspan;    
   }
}
