/* ***********************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DColor.java,v 1.3 2008/05/23 14:11:47 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/


package org.eclipse.tptp.platform.report.core.internal;

import org.eclipse.tptp.platform.report.igc.util.internal.RGBA;


/**
 * DColor describes a color with RGB values.
 * 
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 * 
 */
public class DColor implements IDColor
{
   private int red;
   private int blue;
   private int green;
   private int alpha = 255;
   
   /**
    * Creates a DColor with 0,0,0 component. (black color)
    */
   public DColor() 
   {    
   }
   
   /**
    * Creates a DColor. 
    * The three values passed in argument represent the RGB values  (Red Green Blue) 
    * and set the color.  
    * 
    * @param r red value between 0..255
    * @param g green value between 0..255
    * @param b blue value between 0..255
    */
   public DColor( int r, int g, int b) 
   {
      setRGB(r,g,b);    
   }
   
   /**
    * Creates a DColor copying R,G,B field from given parameter.
    * 
    * @param clr color to copy, or 0,0,0 if null
    */
   public DColor( IDColor clr ) 
   {
     if( clr==null )
     {
       setRGB( 0,0,0 );
     } else {
       setRGB(clr.getRed(),clr.getGreen(),clr.getBlue());
     }
   }
   
   /**
    * Returns the Red value of DColor.
    */
   public int getRed() 
   {
    return red;    
   }
   
   /**
    * Returns the Green value of DColor.
    */
   public int getGreen() 
   {
    return green;    
   }
   
   /**
    * Returns the blue value of DColor.
    */
   public int getBlue() 
   {
    return blue;    
   }
   
   /**
    * Sets the Red (0.255) value of DColor.
    */
   public void setRed(int r) 
   {
       if (r>255) r=255;
       red = r;    
   }
   
   /**
    * Sets the Green (0..255) value of DColor.
    */
   public void setGreen(int g) 
   {
       if (g>255) g=255;
       green = g;    
   }
   
   /**
    * Sets the Blue (0..255) value of DColor.
    */
   public void setBlue(int b) 
   {
       if (b>255) b=255;
       blue = b;    
   }
   
   /**
    * Returns true if the values R(red), G(green ans B(Blue) are the same.
    */
   public boolean equals(Object c) 
   {
     if( c==this) return true;
     if( !(c instanceof IDColor) ) return false;

     IDColor cl = (IDColor)c;
     return ((cl.getRed()== getRed()) && (cl.getGreen() == getGreen()) && (cl.getBlue() == getBlue()));	
   }
   
   /**
    * Sets the color with R(red), G(green) and B(blue) values.
    * 
    * @param r red value
    * @param g green value
    * @param b blue value
    */
   public void setRGB(int r, int g, int b) 
   {
      setRed(r);
      setGreen(g);
      setBlue(b);    
   }
   
   /**
    * Sets the color with R(red), G(green), B(blue) and A(alpha) values.
    * 
    * @param r red value 0..255
    * @param g green value 0.255
    * @param b blue value 0..255
    * @param a alpha value 0..255
    */
   public void setRGBA(int r, int g, int b, int a) 
   {
      setRed(r);
      setGreen(g);
      setBlue(b);    
      setAlpha(a);
   }
   
   /* * @return string representation of object, adding the three color field at
    * the end of string returned by parent's class */
   public String toString()
   {
     String s = super.toString();
     s += "{"+red+","+green+","+blue+"}"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
     return s;
   }
   
   /** @return RGBA color corresponding to current color */
   public int getRGBA()
   {
     return RGBA.Get( red, green, blue );
   }
   
   /** @return RGBA color from given IDColor, must be be null. */
   public static  int RGBA( IDColor c ) 
   {
     return RGBA.Get( c.getRed(), c.getGreen(), c.getBlue() );
   }
   
   /** @return RGBA color from given IDColor, if IDColor is null, default value is returned */
   public static  int RGBA( IDColor c, int rgb_default ) 
   {
     if( c==null ) return rgb_default;
     return RGBA.Get( c.getRed(), c.getGreen(), c.getBlue() );
   }
   
   /** @return the transparency value */
   public int getAlpha() {
      return alpha;
   }
   
   /** sets the transparency value (0..255) */
   public void setAlpha(int alpha) {
       if (alpha>255) alpha = 255;
       this.alpha = alpha;
   }


   /**
    * @returns the color definition in a string with the format:
    * #RRGGBBAA: RRGGBB are the color components in hexa in the range 00..FF
    * AA is the tranparency value in hexa in the range 00..FF
    * If the color is a solid color (alpha = FF) the AA is not generated
    * ex:
    *    Solid light gray :  #777777
    *    50% Transp light gray : #777777F0
    */
   public String serializeToString() {
   
      String r = Integer.toHexString(getRed());
      if (getRed()<=0xF) r = "0" + r;
      String g = Integer.toHexString(getGreen());
      if (getGreen()<=0xF) g = "0" + g;
      String b = Integer.toHexString(getBlue());
      if (getBlue()<=0xF) b = "0" + b;
      
      String ret = "#" + r + g + b;
      
      if (getAlpha() < 0xFF)
      {
          String a = Integer.toHexString(getAlpha());
          if (getAlpha()<=0xF) a = "0" + a;
          ret = ret + a;
      }
      return ret;
   }

   /**
    * Initializes a string with the format #RRGGBBAA  
    */
   public void serializeFromString(String s) {
      if ((s.length() < 7) || (s.charAt(0)!='#')) return;
      
      setRed(Integer.decode("0x" + s.substring(1,3)).intValue());
      setGreen(Integer.decode("0x" + s.substring(3,5)).intValue());
      setBlue(Integer.decode("0x" + s.substring(5,7)).intValue());
      
      if (s.length()==9)
          setAlpha(Integer.decode("0x" + s.substring(7,9)).intValue());
   }

  /* (non-Javadoc)
   * @see org.eclipse.tptp.platform.report.core.internal.IDColor#getID()
   */
  public String getID() 
  {
    return Integer.toString( this.hashCode(), 16 );
  }
}
